/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <glib/gstdio.h>
#include <glib/gi18n.h>
#include "grdcfile.h"
#include "grdcfilemanager.h"
#include "grdcfileeditor.h"
#include "grdcconnectionwindow.h"
#include "grdcabout.h"
#include "grdcpixmaps.h"
#include "grdcpref.h"
#include "grdcprefdialog.h"
#include "grdcmain.h"

typedef struct _GrdcMain
{
    GtkWidget *main_window;
    GtkWidget *file_list;
    GtkTreeModel *file_model;
    GtkUIManager *uimanager;
    GtkWidget *toolbar;
    GtkWidget *statusbar;

    GtkTreeViewColumn *group_column;

    GtkActionGroup *main_group;
    GtkActionGroup *file_sensitive_group;

    gboolean initialized;

    gchar *selected_filename;
    gchar *selected_name;
    GtkTreeIter *selected_iter;
} GrdcMain;

enum
{
    PROTOCOL_COLUMN,
    NAME_COLUMN,
    GROUP_COLUMN,
    SERVER_COLUMN,
    FILENAME_COLUMN,
    N_COLUMNS
};

typedef enum
{
    GRDC_LAUNCH_MAIN,
    GRDC_LAUNCH_QUICK,
    GRDC_LAUNCH_FILE
} GrdcLaunchType;

static void
grdc_main_launcher (GrdcLaunchType launch_type, const gchar *param1)
{
    gint argc;
    gchar *argv[50];
    gint i;
    GError *error = NULL;
    gboolean ret;
    GtkWidget *dialog;

    argc = 0;
    argv[argc++] = g_strdup ("grdc");

    switch (launch_type)
    {
    case GRDC_LAUNCH_MAIN:
        break;

    case GRDC_LAUNCH_QUICK:
        argv[argc++] = g_strdup ("-q");
        if (param1)
        {
            argv[argc++] = g_strdup ("-t");
            argv[argc++] = g_strdup (param1);
        }
        break;

    case GRDC_LAUNCH_FILE:
        argv[argc++] = g_strdup ("-c");
        argv[argc++] = g_strdup (param1);
        break;
    }    

    argv[argc++] = NULL;

    ret = g_spawn_async (
        NULL, /* working_directory: take current */
        argv, /* argv[0] is the executable, parameters start from argv[1], end with NULL */
        NULL, /* envp: take the same as parent */
        G_SPAWN_SEARCH_PATH, /* flags */
        NULL, /* child_setup: function to be called before exec() */
        NULL, /* user_data: parameter for child_setup function */
        NULL, /* child_pid */
        &error);

    for (i = 0; i < argc; i++) g_free (argv[i]);

    if (!ret)
    {
        dialog = gtk_message_dialog_new (NULL,
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
            error->message, NULL);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
    }
}

static void
grdc_main_save_size (GrdcMain *grdcmain)
{
    if ((gdk_window_get_state (grdcmain->main_window->window) & GDK_WINDOW_STATE_MAXIMIZED) == 0)
    {
        gtk_window_get_size (GTK_WINDOW (grdcmain->main_window), &grdc_pref.main_width, &grdc_pref.main_height);
        grdc_pref.main_maximize = FALSE;
    }
    else
    {
        grdc_pref.main_maximize = TRUE;
    }
    grdc_pref_save ();
}

static void
grdc_main_quit (GrdcMain *grdcmain)
{
    GtkWidget *dialog;
    gint cnt;

    cnt = grdc_connection_window_count ();
    if (cnt > 0)
    {
        dialog = gtk_message_dialog_new (GTK_WINDOW (grdcmain->main_window),
            GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
            ngettext(
                "Found %i active connection.\nAre you sure to quit?",
                "Found %i active connections.\nAre you sure to quit?",
                cnt), cnt);
        if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES)
        {
            grdc_connection_window_quit_all ();
            grdc_main_save_size (grdcmain);
            gtk_widget_destroy (grdcmain->main_window);
        }
        gtk_widget_destroy (dialog);
    }
    else
    {
        grdc_main_save_size (grdcmain);
        gtk_widget_destroy (grdcmain->main_window);
    }
}

static gboolean
grdc_main_delete_event (GtkWidget *widget, GdkEvent *event, gpointer data)
{
    grdc_main_quit ((GrdcMain*) data);
    return TRUE;
}

static void
grdc_main_destroy (GtkWidget *widget, gpointer data)
{
    gtk_main_quit ();
}

static void
grdc_main_clear_selection_data (GrdcMain *grdcmain)
{
    g_free (grdcmain->selected_filename);
    g_free (grdcmain->selected_name);
    grdcmain->selected_filename = NULL;
    grdcmain->selected_name = NULL;

    gtk_action_group_set_sensitive (grdcmain->file_sensitive_group, FALSE);
}

static gboolean
grdc_main_selection_func (
    GtkTreeSelection *selection,
    GtkTreeModel     *model,
    GtkTreePath      *path,
    gboolean          path_currently_selected,
    gpointer          user_data)
{
    GrdcMain *grdcmain;
    guint context_id;
    GtkTreeIter iter;
    gchar buf[1000];

    grdcmain = (GrdcMain*) user_data;
    if (path_currently_selected) return TRUE;

    if (!gtk_tree_model_get_iter (model, &iter, path)) return TRUE;

    grdc_main_clear_selection_data (grdcmain);

    gtk_tree_model_get (model, &iter,
        NAME_COLUMN, &grdcmain->selected_name,
        FILENAME_COLUMN, &grdcmain->selected_filename,
        -1);

    context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (grdcmain->statusbar), "status");
    gtk_statusbar_pop (GTK_STATUSBAR (grdcmain->statusbar), context_id);
    if (grdcmain->selected_filename)
    {
        gtk_action_group_set_sensitive (grdcmain->file_sensitive_group, TRUE);
        g_snprintf (buf, sizeof (buf), "%s (%s)", grdcmain->selected_name, grdcmain->selected_filename);
        gtk_statusbar_push (GTK_STATUSBAR (grdcmain->statusbar), context_id, buf);
    }
    else
    {
        gtk_statusbar_push (GTK_STATUSBAR (grdcmain->statusbar), context_id, grdcmain->selected_name);
    }
    return TRUE;
}

static void
grdc_main_load_file_list_callback (gpointer data, gpointer user_data)
{
    GrdcMain *grdcmain;
    GtkTreeIter iter;
    GtkListStore *store;
    GrdcFile *grdcfile;
    GdkPixbuf *icon;

    grdcfile = (GrdcFile*) data;
    grdcmain = (GrdcMain*) user_data;
    store = GTK_LIST_STORE (grdcmain->file_model);

    icon = grdc_file_get_icon (grdcfile, 24);

    gtk_list_store_append (store, &iter);
    gtk_list_store_set (store, &iter,
        PROTOCOL_COLUMN, icon,
        NAME_COLUMN, grdcfile->name,
        GROUP_COLUMN, grdcfile->group,
        SERVER_COLUMN, grdcfile->server,
        FILENAME_COLUMN, grdcfile->filename,
        -1);

    g_object_unref (icon);

    if (grdcmain->selected_filename && g_strcmp0 (grdcfile->filename, grdcmain->selected_filename) == 0)
    {
        grdcmain->selected_iter = gtk_tree_iter_copy (&iter);
    }
}

static void
grdc_main_load_file_tree_group (GtkTreeStore *store)
{
    GtkTreeIter iter;
    GtkWidget *image;
    GdkPixbuf *icon;
    gchar *groups, *ptr1, *ptr2;

    groups = g_strdup (grdc_pref.groups);
    if (groups == NULL || groups[0] == '\0') return;

    image = gtk_image_new ();
    icon = gtk_widget_render_icon (image, GTK_STOCK_DIRECTORY, GTK_ICON_SIZE_SMALL_TOOLBAR, NULL);

    ptr1 = groups;
    while (ptr1)
    {
        ptr2 = strchr (ptr1, ',');
        if (ptr2) *ptr2++ = '\0';

        gtk_tree_store_append (store, &iter, NULL);
        gtk_tree_store_set (store, &iter,
            PROTOCOL_COLUMN, icon,
            NAME_COLUMN, ptr1,
            FILENAME_COLUMN, NULL,
            -1);

        ptr1 = ptr2;
    }

    g_object_unref (icon);
    gtk_widget_destroy (image);
    g_free (groups);
}

static void
grdc_main_load_file_tree_callback (gpointer data, gpointer user_data)
{
    GrdcMain *grdcmain;
    GtkTreeIter iter, child;
    GtkTreeStore *store;
    GrdcFile *grdcfile;
    GdkPixbuf *icon;
    gboolean ret, match;
    gchar *name, *filename;

    grdcfile = (GrdcFile*) data;
    grdcmain = (GrdcMain*) user_data;
    store = GTK_TREE_STORE (grdcmain->file_model);

    icon = grdc_file_get_icon (grdcfile, 24);

    ret = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (store), &iter);
    match = FALSE;
    while (ret)
    {
        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, NAME_COLUMN, &name, FILENAME_COLUMN, &filename, -1);
        match = (filename == NULL && g_strcmp0 (name, grdcfile->group) == 0);
        g_free (name);
        g_free (filename);
        if (match) break;
        ret = gtk_tree_model_iter_next (GTK_TREE_MODEL (store), &iter);
    }

    gtk_tree_store_append (store, &child, (match ? &iter : NULL));
    gtk_tree_store_set (store, &child,
        PROTOCOL_COLUMN, icon,
        NAME_COLUMN, grdcfile->name,
        GROUP_COLUMN, grdcfile->group,
        SERVER_COLUMN, grdcfile->server,
        FILENAME_COLUMN, grdcfile->filename,
        -1);

    g_object_unref (icon);

    if (grdcmain->selected_filename && g_strcmp0 (grdcfile->filename, grdcmain->selected_filename) == 0)
    {
        grdcmain->selected_iter = gtk_tree_iter_copy (&child);
    }
}

static void
grdc_main_load_files (GrdcMain *grdcmain)
{
    gint n;
    gchar buf[200];
    guint context_id;
    GtkTreePath *path;

    switch (grdc_pref.view_file_mode)
    {

    case GRDC_VIEW_FILE_TREE:
        gtk_tree_view_column_set_visible (grdcmain->group_column, FALSE);
        grdcmain->file_model = GTK_TREE_MODEL (gtk_tree_store_new (N_COLUMNS,
            GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING));
        grdc_main_load_file_tree_group (GTK_TREE_STORE (grdcmain->file_model));
        n = grdc_file_manager_iterate (grdc_main_load_file_tree_callback, grdcmain);
        break;

    case GRDC_VIEW_FILE_LIST:
    default:
        gtk_tree_view_column_set_visible (grdcmain->group_column, TRUE);
        grdcmain->file_model = GTK_TREE_MODEL (gtk_list_store_new (N_COLUMNS,
            GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING));
        n = grdc_file_manager_iterate (grdc_main_load_file_list_callback, grdcmain);
        break;
    }

    gtk_tree_view_set_model (GTK_TREE_VIEW (grdcmain->file_list), grdcmain->file_model);
    gtk_tree_view_expand_all (GTK_TREE_VIEW (grdcmain->file_list));

    if (grdcmain->selected_iter)
    {
        gtk_tree_selection_select_iter (
            gtk_tree_view_get_selection (GTK_TREE_VIEW (grdcmain->file_list)),
            grdcmain->selected_iter);
        path = gtk_tree_model_get_path (grdcmain->file_model, grdcmain->selected_iter);
        gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (grdcmain->file_list),
            path, NULL, TRUE, 0.5, 0.0);
        gtk_tree_path_free (path);
        gtk_tree_iter_free (grdcmain->selected_iter);
        grdcmain->selected_iter = NULL;
    }

    g_snprintf (buf, 200, ngettext("Total %i item.", "Total %i items.", n), n);
    context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (grdcmain->statusbar), "status");
    gtk_statusbar_pop (GTK_STATUSBAR (grdcmain->statusbar), context_id);
    gtk_statusbar_push (GTK_STATUSBAR (grdcmain->statusbar), context_id, buf);
}

static void
grdc_main_action_file_quick_connect_full (GrdcMain *grdcmain, const gchar *protocol)
{
    if (grdc_pref.separated_process)
    {
        grdc_main_launcher (GRDC_LAUNCH_QUICK, protocol);
    }
    else
    {
        grdc_file_editor_open_temp_full (GTK_WINDOW (grdcmain->main_window), FALSE, NULL, protocol);
    }
}

static void
grdc_main_action_file_quick_connect (GObject *object, GrdcMain *grdcmain)
{
    grdc_main_action_file_quick_connect_full (grdcmain, NULL);
}

static void
grdc_main_action_file_quick_connect_rdp (GObject *object, GrdcMain *grdcmain)
{
    grdc_main_action_file_quick_connect_full (grdcmain, "RDP");
}

static void
grdc_main_action_file_quick_connect_vnc (GObject *object, GrdcMain *grdcmain)
{
    grdc_main_action_file_quick_connect_full (grdcmain, "VNC");
}

static void
grdc_main_action_file_quick_connect_vnci (GObject *object, GrdcMain *grdcmain)
{
    grdc_main_action_file_quick_connect_full (grdcmain, "VNCI");
}

static void
grdc_main_action_file_quick_connect_sftp (GObject *object, GrdcMain *grdcmain)
{
    grdc_main_action_file_quick_connect_full (grdcmain, "SFTP");
}

static void
grdc_main_action_file_connect (GtkAction *action, GrdcMain *grdcmain)
{
    if (grdc_pref.separated_process)
    {
        grdc_main_launcher (GRDC_LAUNCH_FILE, grdcmain->selected_filename);
    }
    else
    {
        grdc_connection_window_open (GTK_WINDOW (grdcmain->main_window), grdcmain->selected_filename, FALSE);
    }
}

static void
grdc_main_action_file_new (GtkAction *action, GrdcMain *grdcmain)
{
    gint ret;

    ret = grdc_file_editor_create (GTK_WINDOW (grdcmain->main_window), FALSE);
    if (ret == GTK_RESPONSE_OK || ret == GTK_RESPONSE_APPLY)
    {
        grdc_main_load_files (grdcmain);
    }
}

static void
grdc_main_action_file_edit (GtkAction *action, GrdcMain *grdcmain)
{
    gint ret;

    if (grdcmain->selected_filename == NULL) return;
    ret = grdc_file_editor_open (GTK_WINDOW (grdcmain->main_window), grdcmain->selected_filename, FALSE);
    if (ret == GTK_RESPONSE_OK || ret == GTK_RESPONSE_APPLY)
    {
        grdc_main_load_files (grdcmain);
    }
}

static void
grdc_main_action_file_delete (GtkAction *action, GrdcMain *grdcmain)
{
    GtkWidget *dialog;

    if (grdcmain->selected_filename == NULL) return;

    dialog = gtk_message_dialog_new (GTK_WINDOW (grdcmain->main_window),
        GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
        _("Are you sure to delete '%s'"), grdcmain->selected_name);
    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES)
    {
        g_unlink (grdcmain->selected_filename);
        grdc_main_load_files (grdcmain);
    }
    gtk_widget_destroy (dialog);
}

static void
grdc_main_action_file_preferences (GtkAction *action, GrdcMain *grdcmain)
{
    grdc_pref_dialog_run (GTK_WINDOW (grdcmain->main_window), 0);
}

static void
grdc_main_action_file_quit (GtkAction *action, GrdcMain *grdcmain)
{
    grdc_main_quit (grdcmain);
}

static void
grdc_main_action_view_toolbar (GtkToggleAction *action, GrdcMain *grdcmain)
{
	gboolean toggled;

	toggled = gtk_toggle_action_get_active (action);
    if (toggled)
    {
        gtk_widget_show (grdcmain->toolbar);
    }
    else
    {
        gtk_widget_hide (grdcmain->toolbar);
    }
    if (grdcmain->initialized)
    {
        grdc_pref.hide_toolbar = !toggled;
        grdc_pref_save ();
    }
}

static void
grdc_main_action_view_statusbar (GtkToggleAction *action, GrdcMain *grdcmain)
{
	gboolean toggled;

	toggled = gtk_toggle_action_get_active (action);
    if (toggled)
    {
        gtk_widget_show (grdcmain->statusbar);
    }
    else
    {
        gtk_widget_hide (grdcmain->statusbar);
    }
    if (grdcmain->initialized)
    {
        grdc_pref.hide_statusbar = !toggled;
        grdc_pref_save ();
    }
}

static void
grdc_main_action_view_small_toolbutton (GtkToggleAction *action, GrdcMain *grdcmain)
{
	gboolean toggled;

	toggled = gtk_toggle_action_get_active (action);
    if (toggled)
    {
        gtk_toolbar_set_icon_size (GTK_TOOLBAR (grdcmain->toolbar), GTK_ICON_SIZE_MENU);
    }
    else
    {
        gtk_toolbar_unset_icon_size (GTK_TOOLBAR (grdcmain->toolbar));
    }
    if (grdcmain->initialized)
    {
        grdc_pref.small_toolbutton = toggled;
        grdc_pref_save ();
    }
}

static void
grdc_main_action_view_file_mode (GtkRadioAction *action, GtkRadioAction *current, GrdcMain *grdcmain)
{
    grdc_pref.view_file_mode = gtk_radio_action_get_current_value (action);
    grdc_pref_save ();
    grdc_main_load_files (grdcmain);
}

static void
grdc_main_action_help_about (GtkAction *action, GrdcMain *grdcmain)
{
    grdc_about_open (GTK_WINDOW (grdcmain->main_window));
}

static const gchar *grdc_main_ui_xml = 
"<ui>"
"  <menubar name='MenuBar'>"
"    <menu name='FileMenu' action='File'>"
"      <menuitem name='FileQuickConnectMenu' action='FileQuickConnect'/>"
"      <menu name='FileQuickConnectProtoMenu' action='FileQuickConnectProto'>"
"        <menuitem name='FileQuickConnectProtoRDPMenu' action='FileQuickConnectProtoRDP'/>"
#ifdef HAVE_LIBVNCCLIENT
"        <menuitem name='FileQuickConnectProtoVNCMenu' action='FileQuickConnectProtoVNC'/>"
"        <menuitem name='FileQuickConnectProtoVNCIMenu' action='FileQuickConnectProtoVNCI'/>"
#endif
#ifdef HAVE_LIBSSH
"        <menuitem name='FileQuickConnectProtoSFTPMenu' action='FileQuickConnectProtoSFTP'/>"
#endif
"      </menu>"
"      <menuitem name='FileConnectMenu' action='FileConnect'/>"
"      <separator/>"
"      <menuitem name='FileNewMenu' action='FileNew'/>"
"      <menuitem name='FileEditMenu' action='FileEdit'/>"
"      <menuitem name='FileDeleteMenu' action='FileDelete'/>"
"      <separator/>"
"      <menuitem name='FilePreferencesMenu' action='FilePreferences'/>"
"      <separator/>"
"      <menuitem name='FileQuitMenu' action='FileQuit'/>"
"    </menu>"
"    <menu name='ViewMenu' action='View'>"
"      <menuitem name='ViewToolbarMenu' action='ViewToolbar'/>"
"      <menuitem name='ViewStatusbarMenu' action='ViewStatusbar'/>"
"      <separator/>"
"      <menuitem name='ViewSmallToolbuttonMenu' action='ViewSmallToolbutton'/>"
"      <separator/>"
"      <menuitem name='ViewFileListMenu' action='ViewFileList'/>"
"      <menuitem name='ViewFileTreeMenu' action='ViewFileTree'/>"
"    </menu>"
"    <menu name='HelpMenu' action='Help'>"
"      <menuitem name='HelpAboutMenu' action='HelpAbout'/>"
"    </menu>"
"  </menubar>"
"  <toolbar name='ToolBar'>"
"    <toolitem action='FileConnect'/>"
"    <separator/>"
"    <toolitem action='FileNew'/>"
"    <toolitem action='FileEdit'/>"
"    <toolitem action='FileDelete'/>"
"    <separator/>"
"    <toolitem action='FilePreferences'/>"
"    <toolitem action='FileQuit'/>"
"  </toolbar>"
"  <popup name='PopupQuickConnectProtoMenu' action='FileQuickConnectProto'>"
"    <menuitem name='PopupQuickConnectProtoRDPMenu' action='FileQuickConnectProtoRDP'/>"
#ifdef HAVE_LIBVNCCLIENT
"    <menuitem name='PopupQuickConnectProtoVNCMenu' action='FileQuickConnectProtoVNC'/>"
"    <menuitem name='PopupQuickConnectProtoVNCIMenu' action='FileQuickConnectProtoVNCI'/>"
#endif
#ifdef HAVE_LIBSSH
"    <menuitem name='PopupQuickConnectProtoSFTPMenu' action='FileQuickConnectProtoSFTP'/>"
#endif
"  </popup>"
"  <popup name='PopupMenu'>"
"    <menuitem action='FileConnect'/>"
"    <separator/>"
"    <menuitem action='FileEdit'/>"
"    <menuitem action='FileDelete'/>"
"  </popup>"
"</ui>";

static const GtkActionEntry grdc_main_ui_menu_entries[] =
{
    { "File", NULL, N_("_File") },
    { "View", NULL, N_("_View") },
    { "Help", NULL, N_("_Help") },

    { "FileQuickConnect", GTK_STOCK_JUMP_TO, N_("Quick Connect"), "<control>U",
        N_("Open a quick connection"),
        G_CALLBACK (grdc_main_action_file_quick_connect) },

    { "FileQuickConnectProto", NULL, N_("Quick _Connect To") },

    { "FileQuickConnectProtoRDP", NULL, N_("RDP - Windows Terminal Service"), NULL, NULL,
        G_CALLBACK (grdc_main_action_file_quick_connect_rdp) },
    { "FileQuickConnectProtoVNC", NULL, N_("VNC - Virtual Network Computing"), NULL, NULL,
        G_CALLBACK (grdc_main_action_file_quick_connect_vnc) },
    { "FileQuickConnectProtoVNCI", NULL, N_("VNC - Incoming Connection"), NULL, NULL,
        G_CALLBACK (grdc_main_action_file_quick_connect_vnci) },
    { "FileQuickConnectProtoSFTP", NULL, N_("SFTP - Secure File Transfer"), NULL, NULL,
        G_CALLBACK (grdc_main_action_file_quick_connect_sftp) },

    { "FileNew", GTK_STOCK_NEW, NULL, "<control>N",
        N_("Create a new remote desktop file"),
        G_CALLBACK (grdc_main_action_file_new) },

    { "FilePreferences", GTK_STOCK_PREFERENCES, NULL, "<control>P",
        N_("Open the preferences dialog"),
        G_CALLBACK (grdc_main_action_file_preferences) },

    { "FileQuit", GTK_STOCK_QUIT, NULL, "<control>Q",
        N_("Quit Grdc"),
        G_CALLBACK (grdc_main_action_file_quit) },

    { "HelpAbout", GTK_STOCK_ABOUT, NULL, NULL,
        NULL,
        G_CALLBACK (grdc_main_action_help_about) }
};

static const GtkActionEntry grdc_main_ui_file_sensitive_menu_entries[] =
{
    { "FileConnect", GTK_STOCK_CONNECT, NULL, "<control>C",
        N_("Open the connection to the selected remote desktop file"),
        G_CALLBACK (grdc_main_action_file_connect) },

    { "FileEdit", GTK_STOCK_EDIT, NULL, "<control>E",
        N_("Edit the selected remote desktop file"),
        G_CALLBACK (grdc_main_action_file_edit) },

    { "FileDelete", GTK_STOCK_DELETE, NULL, "<control>D",
        N_("Delete the selected remote desktop file"),
        G_CALLBACK (grdc_main_action_file_delete) }
};

static const GtkToggleActionEntry grdc_main_ui_toggle_menu_entries[] =
{
    { "ViewToolbar", NULL, N_("Toolbar"), NULL, NULL,
        G_CALLBACK (grdc_main_action_view_toolbar), TRUE },

    { "ViewStatusbar", NULL, N_("Statusbar"), NULL, NULL,
        G_CALLBACK (grdc_main_action_view_statusbar), TRUE },

    { "ViewSmallToolbutton", NULL, N_("Small Toolbar Buttons"), NULL, NULL,
        G_CALLBACK (grdc_main_action_view_small_toolbutton), FALSE }
};

static const GtkRadioActionEntry grdc_main_ui_view_file_mode_entries[] =
{
    { "ViewFileList", NULL, N_("List View"), NULL, NULL, GRDC_VIEW_FILE_LIST },
    { "ViewFileTree", NULL, N_("Tree View"), NULL, NULL, GRDC_VIEW_FILE_TREE }
};

static gboolean
grdc_main_file_list_on_button_press (GtkWidget *widget, GdkEventButton *event, GrdcMain *grdcmain)
{
    GtkWidget *popup;

    if (event->button == 3)
    {
        popup = gtk_ui_manager_get_widget (grdcmain->uimanager, "/PopupMenu");
        if (popup)
        {
            gtk_menu_popup (GTK_MENU (popup), NULL, NULL, NULL, NULL, event->button, event->time);
        }
    }
    else if (event->type == GDK_2BUTTON_PRESS && grdcmain->selected_filename)
    {
        switch (grdc_pref.default_action)
        {
        case GRDC_ACTION_EDIT:
            grdc_main_action_file_edit (NULL, grdcmain);
            break;
        case GRDC_ACTION_CONNECT:
        default:
            grdc_main_action_file_connect (NULL, grdcmain);
            break;
        }
    }
    return FALSE;
}

static void
grdc_main_ui_set_icon (GtkUIManager *uimanager, const gchar *path, GrdcPixmapType type)
{
    GtkWidget *widget;
    GtkWidget *image;
    gint image_size;
    GdkPixbuf *pixbuf;

    widget = gtk_ui_manager_get_widget (uimanager, path);
    if (widget)
    {
        gtk_icon_size_lookup (GTK_ICON_SIZE_MENU, &image_size, NULL);
        pixbuf = grdc_pixmap (type, image_size);
        image = gtk_image_new_from_pixbuf (pixbuf);
        g_object_unref (pixbuf);
        gtk_widget_show (image);
        if (GTK_IS_IMAGE_MENU_ITEM (widget))
        {
            gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (widget), image);
        }
    }
}

void
grdc_main_open (void)
{
    GrdcMain *grdcmain;
    GtkWidget *vbox;
    GtkWidget *menubar;
    GtkUIManager *uimanager;
    GtkActionGroup *action_group;
    GtkWidget *scrolledwindow;
    GtkWidget *tree;
    GtkToolItem *toolitem;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;
    GError *error;

    grdcmain = g_new (GrdcMain, 1);

    grdcmain->initialized = FALSE;
    grdcmain->selected_filename = NULL;
    grdcmain->selected_name = NULL;
    grdcmain->selected_iter = NULL;

    /* Create main window */
    grdcmain->main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    g_signal_connect (G_OBJECT (grdcmain->main_window), "delete-event", G_CALLBACK (grdc_main_delete_event), grdcmain);
    g_signal_connect (G_OBJECT (grdcmain->main_window), "destroy", G_CALLBACK (grdc_main_destroy), grdcmain);
    gtk_container_set_border_width (GTK_CONTAINER (grdcmain->main_window), 0);
    gtk_window_set_title (GTK_WINDOW (grdcmain->main_window), _("Remote Desktop Client"));
    gtk_window_set_default_size (GTK_WINDOW (grdcmain->main_window), grdc_pref.main_width, grdc_pref.main_height);
    gtk_window_set_position (GTK_WINDOW (grdcmain->main_window), GTK_WIN_POS_CENTER);
    gtk_window_set_icon (GTK_WINDOW (grdcmain->main_window), GRDC_PIXMAP_LOGO);
    if (grdc_pref.main_maximize)
    {
        gtk_window_maximize (GTK_WINDOW (grdcmain->main_window));
    }

    /* Create the main container */
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (grdcmain->main_window), vbox);
    gtk_widget_show (vbox);

    /* Create the menubar and toolbar */
    uimanager = gtk_ui_manager_new ();
    grdcmain->uimanager = uimanager;

    action_group = gtk_action_group_new ("GrdcMainActions");
    gtk_action_group_set_translation_domain (action_group, NULL);
    gtk_action_group_add_actions (action_group, grdc_main_ui_menu_entries,
        G_N_ELEMENTS (grdc_main_ui_menu_entries), grdcmain);
    gtk_action_group_add_toggle_actions (action_group, grdc_main_ui_toggle_menu_entries,
        G_N_ELEMENTS (grdc_main_ui_toggle_menu_entries), grdcmain);
    gtk_action_group_add_radio_actions (action_group, grdc_main_ui_view_file_mode_entries,
        G_N_ELEMENTS (grdc_main_ui_view_file_mode_entries),
        grdc_pref.view_file_mode, G_CALLBACK (grdc_main_action_view_file_mode), grdcmain);

    gtk_ui_manager_insert_action_group (uimanager, action_group, 0);
    g_object_unref (action_group);
    grdcmain->main_group = action_group;

    action_group = gtk_action_group_new ("GrdcMainFileSensitiveActions");
    gtk_action_group_set_translation_domain (action_group, NULL);
    gtk_action_group_add_actions (action_group, grdc_main_ui_file_sensitive_menu_entries,
        G_N_ELEMENTS (grdc_main_ui_file_sensitive_menu_entries), grdcmain);

    gtk_ui_manager_insert_action_group (uimanager, action_group, 0);
    g_object_unref (action_group);
    grdcmain->file_sensitive_group = action_group;

    error = NULL;
    gtk_ui_manager_add_ui_from_string (uimanager, grdc_main_ui_xml, -1, &error);
    if (error)
    {
        g_message ("building menus failed: %s", error->message);
        g_error_free (error);
    }

    menubar = gtk_ui_manager_get_widget (uimanager, "/MenuBar");
    gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 0);

    grdcmain->toolbar = gtk_ui_manager_get_widget (uimanager, "/ToolBar");
    gtk_box_pack_start (GTK_BOX (vbox), grdcmain->toolbar, FALSE, FALSE, 0);

    /* Customized menu items and toolbar items which can't be built through standard ui manager */
    toolitem = gtk_menu_tool_button_new_from_stock (GTK_STOCK_JUMP_TO);
    gtk_widget_show (GTK_WIDGET (toolitem));
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Quick Connect"));
    gtk_toolbar_insert (GTK_TOOLBAR (grdcmain->toolbar), toolitem, 0);
    g_signal_connect (G_OBJECT (toolitem), "clicked",
        G_CALLBACK (grdc_main_action_file_quick_connect), grdcmain);

    gtk_menu_tool_button_set_menu (GTK_MENU_TOOL_BUTTON (toolitem),
        gtk_ui_manager_get_widget (uimanager, "/PopupQuickConnectProtoMenu"));

    grdc_main_ui_set_icon (uimanager,
        "/MenuBar/FileMenu/FileQuickConnectProtoMenu/FileQuickConnectProtoRDPMenu",
        GRDC_PIXMAP_TYPE_RDP);
    grdc_main_ui_set_icon (uimanager,
        "/PopupQuickConnectProtoMenu/PopupQuickConnectProtoRDPMenu",
        GRDC_PIXMAP_TYPE_RDP);

    grdc_main_ui_set_icon (uimanager,
        "/MenuBar/FileMenu/FileQuickConnectProtoMenu/FileQuickConnectProtoVNCMenu",
        GRDC_PIXMAP_TYPE_VNC);
    grdc_main_ui_set_icon (uimanager,
        "/PopupQuickConnectProtoMenu/PopupQuickConnectProtoVNCMenu",
        GRDC_PIXMAP_TYPE_VNC);

    grdc_main_ui_set_icon (uimanager,
        "/MenuBar/FileMenu/FileQuickConnectProtoMenu/FileQuickConnectProtoVNCIMenu",
        GRDC_PIXMAP_TYPE_VNC);
    grdc_main_ui_set_icon (uimanager,
        "/PopupQuickConnectProtoMenu/PopupQuickConnectProtoVNCIMenu",
        GRDC_PIXMAP_TYPE_VNC);

    grdc_main_ui_set_icon (uimanager,
        "/MenuBar/FileMenu/FileQuickConnectProtoMenu/FileQuickConnectProtoSFTPMenu",
        GRDC_PIXMAP_TYPE_SSH);
    grdc_main_ui_set_icon (uimanager,
        "/PopupQuickConnectProtoMenu/PopupQuickConnectProtoSFTPMenu",
        GRDC_PIXMAP_TYPE_SSH);

    gtk_window_add_accel_group (GTK_WINDOW (grdcmain->main_window), gtk_ui_manager_get_accel_group (uimanager));

    gtk_action_group_set_sensitive (grdcmain->file_sensitive_group, FALSE);

    /* Create the scrolled window for the file list */
    scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
        GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start (GTK_BOX (vbox), scrolledwindow, TRUE, TRUE, 0);

    /* Create the grdc file list */
    tree = gtk_tree_view_new ();

    column = gtk_tree_view_column_new ();
    gtk_tree_view_column_set_title (column, _("Name"));
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_column_set_sort_column_id (column, NAME_COLUMN);
    renderer = gtk_cell_renderer_pixbuf_new ();
    gtk_tree_view_column_pack_start (column, renderer, FALSE);
    gtk_tree_view_column_add_attribute (column, renderer, "pixbuf", PROTOCOL_COLUMN);
    renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_column_pack_start (column, renderer, FALSE);
    gtk_tree_view_column_add_attribute (column, renderer, "text", NAME_COLUMN);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (_("Group"),
        renderer, "text", GROUP_COLUMN, NULL);
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_column_set_sort_column_id (column, GROUP_COLUMN);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);
    grdcmain->group_column = column;

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (_("Server"),
        renderer, "text", SERVER_COLUMN, NULL);
    gtk_tree_view_column_set_resizable (column, TRUE);
    gtk_tree_view_column_set_sort_column_id (column, SERVER_COLUMN);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

    gtk_container_add (GTK_CONTAINER (scrolledwindow), tree);
    gtk_widget_show (tree);

    gtk_tree_selection_set_select_function (
        gtk_tree_view_get_selection (GTK_TREE_VIEW (tree)),
        grdc_main_selection_func, grdcmain, NULL);
    g_signal_connect (G_OBJECT (tree), "button-press-event",
        G_CALLBACK (grdc_main_file_list_on_button_press), grdcmain);

    grdcmain->file_list = tree;

    /* Create statusbar */
    grdcmain->statusbar = gtk_statusbar_new ();
    gtk_box_pack_start (GTK_BOX (vbox), grdcmain->statusbar, FALSE, FALSE, 0);
    gtk_widget_show (grdcmain->statusbar);

    /* Prepare the data */
    grdc_main_load_files (grdcmain);

    /* Load the preferences */
    if (grdc_pref.hide_toolbar)
    {
        gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (
            gtk_action_group_get_action (grdcmain->main_group, "ViewToolbar")), FALSE);
    }
    if (grdc_pref.hide_statusbar)
    {
        gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (
            gtk_action_group_get_action (grdcmain->main_group, "ViewStatusbar")), FALSE);
    }
    if (grdc_pref.small_toolbutton)
    {
        gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (
            gtk_action_group_get_action (grdcmain->main_group, "ViewSmallToolbutton")), TRUE);
    }

    /* Show the main window */
    gtk_widget_show (grdcmain->main_window);

    grdcmain->initialized = TRUE;
}

