/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <glib/gi18n.h>
#include <stdlib.h>
#include "config.h"
#include "grdcpublic.h"
#include "grdcfile.h"
#include "grdcinitdialog.h"
#include "grdcplug.h"
#include "grdcplug_rdp.h"
#include "grdcplug_vnc.h"
#include "grdcpref.h"
#include "grdcscrolledviewport.h"
#include "grdcsftpwindow.h"
#include "grdcscaler.h"
#include "grdcconnectionwindow.h"

#define MOTION_TIME 100

/* One process can only have one scale option popup window at a time... */
static GtkWidget *scale_option_window = NULL;

typedef struct _GrdcConnectionWindow
{
    GtkWidget *window;
    GrdcFile *grdc_file;

    GtkWidget *floating_toolbar;
    /* To avoid strange event-loop */
    guint32 floating_toolbar_motion_time;

    /* Containers for GrdcPlug sub-classes: GrdcPlug->alignment->viewport...->window */
    GtkWidget *grdc_plug;
    GtkWidget *viewport;
    GtkWidget *alignment;

    GtkWidget *toolbar;
    GtkWidget *scrolled_window;

    /* Scrolled Viewport widget */
    GtkWidget *scrolled_viewport;

    /* Toolitems that need to be handled */
    GtkToolItem *toolitem_scale;
    GtkToolItem *toolitem_preferences;
    GtkToolItem *toolitem_tools;
    GtkWidget *scale_option_button;

    gboolean connected;

    gboolean quit_when_close;
    gboolean sticky;
} GrdcConnectionWindow;

void grdc_connection_window_create_scrolled (GrdcConnectionWindow *cnnwin);
void grdc_connection_window_create_fullscreen (GrdcConnectionWindow *cnnwin);
void grdc_connection_window_create_scrolled_fullscreen (GrdcConnectionWindow *cnnwin);
void grdc_connection_window_create_viewport_fullscreen (GrdcConnectionWindow *cnnwin);

static GPtrArray *grdc_connection_window_array = NULL;

static void
grdc_connection_window_register (GrdcConnectionWindow *cnnwin)
{
    if (grdc_connection_window_array == NULL)
    {
        grdc_connection_window_array = g_ptr_array_new ();
    }
    g_ptr_array_add (grdc_connection_window_array, cnnwin);
}

static void
grdc_connection_window_unregister (GrdcConnectionWindow *cnnwin)
{
    if (grdc_connection_window_array)
    {
        g_ptr_array_remove (grdc_connection_window_array, cnnwin);
    }
}

static void
grdc_connection_window_disconnect (GrdcConnectionWindow *cnnwin)
{
    /* Notify the GrdcPlug to disconnect, but not to close the window here.
       The window will be destroyed in GrdcPlug "disconnect" signal */
    grdc_plug_close_connection (GRDC_PLUG (cnnwin->grdc_plug));
}

static void
grdc_connection_window_keyboard_grab (GrdcConnectionWindow *cnnwin)
{
    if (cnnwin->grdc_file->keyboard_grab)
    {
        gdk_keyboard_grab (cnnwin->window->window, TRUE, GDK_CURRENT_TIME);
    }
    else
    {
        gdk_keyboard_ungrab (GDK_CURRENT_TIME);
    }
}

static gboolean
grdc_connection_window_delete_event (GtkWidget *widget, GdkEvent *event, gpointer data)
{
    grdc_connection_window_disconnect ((GrdcConnectionWindow*) data);
    return TRUE;
}

static void
grdc_connection_window_destroy (GtkWidget *widget, GrdcConnectionWindow* cnnwin)
{
    if (!cnnwin->connected)
    {
        grdc_file_free (cnnwin->grdc_file);
        if (cnnwin->floating_toolbar != NULL)
        {
            gtk_widget_destroy (cnnwin->floating_toolbar);
            cnnwin->floating_toolbar = NULL;
        }
        grdc_connection_window_unregister (cnnwin);
        g_free (cnnwin);
        if (cnnwin->quit_when_close)
        {
            gtk_main_quit ();
        }
    }
}

static void
grdc_connection_window_floating_toolbar_hide (GrdcConnectionWindow *cnnwin)
{
    GtkRequisition req;
    gint x;

    if (cnnwin->floating_toolbar == NULL) return;

    gtk_widget_size_request (cnnwin->floating_toolbar, &req);
    gtk_window_get_position (GTK_WINDOW (cnnwin->floating_toolbar), &x, NULL);
    gtk_window_move (GTK_WINDOW (cnnwin->floating_toolbar), x, 2 - req.height);
    if (grdc_pref.invisible_toolbar)
    {
        gtk_window_set_opacity (GTK_WINDOW (cnnwin->floating_toolbar), 0.0);
    }
    cnnwin->floating_toolbar_motion_time = gtk_get_current_event_time ();
}

static void
grdc_connection_window_get_desktop_size (GrdcConnectionWindow* cnnwin, gint *width, gint *height)
{
    gboolean scale;

    scale = gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON (cnnwin->toolitem_scale));
    *width = GRDC_PLUG (cnnwin->grdc_plug)->width;
    if (scale && cnnwin->grdc_file->hscale > 0) *width = (*width) * cnnwin->grdc_file->hscale / 100;
    *height = GRDC_PLUG (cnnwin->grdc_plug)->height;
    if (scale && cnnwin->grdc_file->vscale > 0) *height = (*height) * cnnwin->grdc_file->vscale / 100;
}

static gboolean
grdc_connection_window_toolbar_autofit_restore (GrdcConnectionWindow* cnnwin)
{
    gint width, height;

    if (cnnwin->connected && cnnwin->scrolled_window)
    {
        grdc_connection_window_get_desktop_size (cnnwin, &width, &height);
        gtk_window_resize (GTK_WINDOW (cnnwin->window), MAX (1, width),
            MAX (1, height + cnnwin->toolbar->allocation.height));
        gtk_container_check_resize (GTK_CONTAINER (cnnwin->window));
    }
    if (cnnwin->scrolled_window)
    {
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (cnnwin->scrolled_window),
            GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    }
    return FALSE;
}

static void
grdc_connection_window_toolbar_autofit (GtkWidget *widget, GrdcConnectionWindow* cnnwin)
{
    if (cnnwin->scrolled_window)
    {
        if ((gdk_window_get_state (cnnwin->window->window) & GDK_WINDOW_STATE_MAXIMIZED) != 0)
        {
            gtk_window_unmaximize (GTK_WINDOW (cnnwin->window));
        }

        /* It's tricky to make the toolbars disappear automatically, while keeping scrollable.
           Please tell me if you know a better way to do this */
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (cnnwin->scrolled_window),
            GTK_POLICY_NEVER, GTK_POLICY_NEVER);

        gtk_main_iteration ();

        g_timeout_add (200, (GSourceFunc) grdc_connection_window_toolbar_autofit_restore, cnnwin);
    }

}

static void
grdc_connection_window_toolbar_scrolled (GtkWidget *widget, GrdcConnectionWindow* cnnwin)
{
    grdc_connection_window_create_scrolled (cnnwin);
}

static void
grdc_connection_window_toolbar_scrolled_fullscreen (GtkWidget *widget, GrdcConnectionWindow* cnnwin)
{
    grdc_connection_window_create_scrolled_fullscreen (cnnwin);
}

static void
grdc_connection_window_toolbar_viewport_fullscreen (GtkWidget *widget, GrdcConnectionWindow* cnnwin)
{
    grdc_connection_window_create_viewport_fullscreen (cnnwin);
}

static void
grdc_connection_window_update_alignment (GrdcConnectionWindow* cnnwin)
{
    GrdcPlug *gp = GRDC_PLUG (cnnwin->grdc_plug);
    gboolean scale;
    gint width, height;

    scale = gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON (cnnwin->toolitem_scale));
    if (scale && cnnwin->grdc_file->aspectscale && cnnwin->grdc_file->hscale == 0)
    {
        width = cnnwin->alignment->allocation.width;
        height = cnnwin->alignment->allocation.height;
        if (width > 1 && height > 1)
        {
            if (gp->width * height >= width * gp->height)
            {
                gtk_alignment_set (GTK_ALIGNMENT (cnnwin->alignment), 0.5, 0.5,
                    1.0,
                    (gfloat) (gp->height * width) / (gfloat) gp->width / (gfloat) height);
            }
            else
            {
                gtk_alignment_set (GTK_ALIGNMENT (cnnwin->alignment), 0.5, 0.5,
                    (gfloat) (gp->width * height) / (gfloat) gp->height / (gfloat) width,
                    1.0);
            }
        }
    }
    else
    {
        gtk_alignment_set (GTK_ALIGNMENT (cnnwin->alignment), 0.5, 0.5,
            ((scale && cnnwin->grdc_file->hscale == 0) ? 1.0 : 0.0),
            ((scale && cnnwin->grdc_file->vscale == 0) ? 1.0 : 0.0));
    }
}

static void
grdc_connection_window_toolbar_scaled_mode (GtkWidget *widget, GrdcConnectionWindow* cnnwin)
{
    gboolean scale;

    grdc_connection_window_update_alignment (cnnwin);

    scale = gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON (widget));
    gtk_widget_set_sensitive (GTK_WIDGET (cnnwin->scale_option_button), scale);
    grdc_plug_call_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_SCALE,
        (scale ? GINT_TO_POINTER (1) : NULL));
}

static void
grdc_connection_window_scale_option_on_scaled (GtkWidget *widget, GrdcConnectionWindow* cnnwin)
{
    cnnwin->grdc_file->hscale = GRDC_SCALER (widget)->hscale;
    cnnwin->grdc_file->vscale = GRDC_SCALER (widget)->vscale;
    cnnwin->grdc_file->aspectscale = GRDC_SCALER (widget)->aspectscale;
    grdc_connection_window_update_alignment (cnnwin);
    grdc_plug_call_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_SCALE,
        GINT_TO_POINTER (1));
}

static void
grdc_connection_window_scale_option_popdown (GrdcConnectionWindow* cnnwin)
{
    cnnwin->sticky = FALSE;

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (cnnwin->scale_option_button), FALSE);

    gdk_keyboard_ungrab (GDK_CURRENT_TIME);
    gdk_pointer_ungrab (GDK_CURRENT_TIME);
    if (scale_option_window)
    {
        gtk_grab_remove (scale_option_window);
        gtk_widget_destroy (scale_option_window);
        scale_option_window = NULL;
    }
    grdc_connection_window_floating_toolbar_hide (cnnwin);
}

static gboolean
grdc_connection_window_trap_on_button (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    return TRUE;
}

static gboolean
grdc_connection_window_scale_option_on_key (GtkWidget *widget, GdkEventKey *event, GrdcConnectionWindow* cnnwin)
{
    switch (event->keyval)
    {
    case GDK_Escape:
        grdc_connection_window_scale_option_popdown (cnnwin);
        return TRUE;
    }
    return FALSE;
}

static gboolean
grdc_connection_window_scale_option_on_button (GtkWidget *widget, GdkEventButton *event, GrdcConnectionWindow* cnnwin)
{
    grdc_connection_window_scale_option_popdown (cnnwin);
    return TRUE;
}

static void
grdc_connection_window_toolbar_scale_option (GtkWidget *widget, GrdcConnectionWindow* cnnwin)
{
    GtkWidget *window;
    GtkWidget *eventbox;
    GtkWidget *frame;
    GtkWidget *scaler;
    gint x, y;
    gboolean pushin;

    if (scale_option_window)
    {
        grdc_connection_window_scale_option_popdown (cnnwin);
    }
    else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
    {
        window = gtk_window_new (GTK_WINDOW_POPUP);
        gtk_container_set_border_width (GTK_CONTAINER (window), 0);

        /* Use an event-box to trap all button clicks events before sending up to the window */
        eventbox = gtk_event_box_new ();
        gtk_widget_show (eventbox);
        gtk_container_add (GTK_CONTAINER (window), eventbox);
        gtk_widget_add_events (eventbox, GDK_BUTTON_PRESS_MASK);
        g_signal_connect (G_OBJECT (eventbox), "button-press-event",
            G_CALLBACK (grdc_connection_window_trap_on_button), NULL);

        frame = gtk_frame_new (NULL);
        gtk_widget_show (frame);
        gtk_container_set_border_width (GTK_CONTAINER (frame), 0);
        gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
        gtk_container_add (GTK_CONTAINER (eventbox), frame);

        scaler = grdc_scaler_new ();
        gtk_widget_show (scaler);
        gtk_widget_set_size_request (scaler, 250, -1);
        gtk_container_set_border_width (GTK_CONTAINER (scaler), 4);
        grdc_scaler_set (GRDC_SCALER (scaler), cnnwin->grdc_file->hscale, cnnwin->grdc_file->vscale,
            cnnwin->grdc_file->aspectscale);
        grdc_scaler_set_draw_value (GRDC_SCALER (scaler), FALSE);
        gtk_container_add (GTK_CONTAINER (frame), scaler);
        g_signal_connect (G_OBJECT (scaler), "scaled",
            G_CALLBACK (grdc_connection_window_scale_option_on_scaled), cnnwin);

        g_signal_connect (G_OBJECT (window), "key-press-event",
            G_CALLBACK (grdc_connection_window_scale_option_on_key), cnnwin);
        g_signal_connect (G_OBJECT (window), "button-press-event",
            G_CALLBACK (grdc_connection_window_scale_option_on_button), cnnwin);

        gtk_widget_realize (window);
        grdc_public_popup_position (NULL, &x, &y, &pushin, cnnwin->toolitem_scale);
        gtk_window_move (GTK_WINDOW (window), x, y);
        gtk_widget_show (window);

        gtk_grab_add (window);
        gdk_pointer_grab (window->window, TRUE,
            GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK,
            NULL, NULL, GDK_CURRENT_TIME);
        gdk_keyboard_grab (window->window, TRUE, GDK_CURRENT_TIME);

        scale_option_window = window;
        cnnwin->sticky = TRUE;
    }
}

static void
grdc_connection_window_toolbar_preferences_popdown (GtkWidget *widget, GrdcConnectionWindow *cnnwin)
{
    cnnwin->sticky = FALSE;

    gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON (cnnwin->toolitem_preferences), FALSE);
    grdc_connection_window_floating_toolbar_hide (cnnwin);
}

static void
grdc_connection_window_toolbar_tools_popdown (GtkWidget *widget, GrdcConnectionWindow *cnnwin)
{
    cnnwin->sticky = FALSE;

    gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON (cnnwin->toolitem_tools), FALSE);
    grdc_connection_window_floating_toolbar_hide (cnnwin);
}

static void
grdc_connection_window_call_plug_feature_radio (GtkMenuItem *menuitem, GrdcConnectionWindow *cnnwin)
{
    GrdcPlugFeature type;
    gpointer value;
    gboolean changed;

    if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (menuitem)))
    {
        type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (menuitem), "feature-type"));
        value = g_object_get_data (G_OBJECT (menuitem), "feature-value");

        grdc_plug_call_feature (GRDC_PLUG (cnnwin->grdc_plug), type, value);

        if (grdc_pref.save_view_mode)
        {
            changed = FALSE;
            switch (type)
            {
                case GRDC_PLUG_FEATURE_PREF_QUALITY:
                    cnnwin->grdc_file->quality = GPOINTER_TO_INT (value);
                    changed = TRUE;
                    break;
                default:
                    break;
            }
            if (changed) grdc_file_save (cnnwin->grdc_file);
        }
    }
}

static void
grdc_connection_window_call_plug_feature_check (GtkMenuItem *menuitem, GrdcConnectionWindow *cnnwin)
{
    GrdcPlugFeature type;
    gboolean value;

    type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (menuitem), "feature-type"));
    value = gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (menuitem));
    grdc_plug_call_feature (GRDC_PLUG (cnnwin->grdc_plug), type, (value ? GINT_TO_POINTER (1) : NULL));
}

static void
grdc_connection_window_call_plug_feature_activate (GtkMenuItem *menuitem, GrdcConnectionWindow *cnnwin)
{
    GrdcPlugFeature type;

    type = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (menuitem), "feature-type"));
    grdc_plug_call_feature (GRDC_PLUG (cnnwin->grdc_plug), type, NULL);
}

static void
grdc_connection_window_toolbar_preferences (GtkWidget *widget, GrdcConnectionWindow *cnnwin)
{
    GtkWidget *menu;
    GtkWidget *menuitem;
    GSList *group;
    gint i;
    gboolean firstgroup;

    if (!gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON (widget))) return;

    cnnwin->sticky = TRUE;

    firstgroup = TRUE;

    menu = gtk_menu_new ();

    if (grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_PREF_QUALITY))
    {
        group = NULL;
        for (i = 0; quality_list[i]; i += 2)
        {
            menuitem = gtk_radio_menu_item_new_with_label (group, _(quality_list[i + 1]));
            group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (menuitem));
            gtk_widget_show (menuitem);
            gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);

            g_object_set_data (G_OBJECT (menuitem), "feature-type", GINT_TO_POINTER (GRDC_PLUG_FEATURE_PREF_QUALITY));
            g_object_set_data (G_OBJECT (menuitem), "feature-value", GINT_TO_POINTER (atoi (quality_list[i])));

            if (atoi (quality_list[i]) == cnnwin->grdc_file->quality)
            {
                gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem), TRUE);
            }

            g_signal_connect (G_OBJECT (menuitem), "toggled",
                G_CALLBACK (grdc_connection_window_call_plug_feature_radio), cnnwin);
        }
        firstgroup = FALSE;
    }

    if (grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_PREF_VIEWONLY) ||
        grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_PREF_DISABLESERVERINPUT))
    {
        if (!firstgroup)
        {
            menuitem = gtk_separator_menu_item_new ();
            gtk_widget_show (menuitem);
            gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
        }

        if (grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_PREF_VIEWONLY))
        {
            menuitem = gtk_check_menu_item_new_with_label (_("View Only"));
            gtk_widget_show (menuitem);
            gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);

            g_object_set_data (G_OBJECT (menuitem), "feature-type", GINT_TO_POINTER (GRDC_PLUG_FEATURE_PREF_VIEWONLY));

            gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem), cnnwin->grdc_file->viewonly);

            g_signal_connect (G_OBJECT (menuitem), "toggled",
                G_CALLBACK (grdc_connection_window_call_plug_feature_check), cnnwin);
        }

        if (grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_PREF_DISABLESERVERINPUT))
        {
            menuitem = gtk_check_menu_item_new_with_label (_("Disable Server Input"));
            gtk_widget_show (menuitem);
            gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);

            g_object_set_data (G_OBJECT (menuitem), "feature-type",
                GINT_TO_POINTER (GRDC_PLUG_FEATURE_PREF_DISABLESERVERINPUT));

            gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem), cnnwin->grdc_file->disableserverinput);

            g_signal_connect (G_OBJECT (menuitem), "toggled",
                G_CALLBACK (grdc_connection_window_call_plug_feature_check), cnnwin);
        }

        firstgroup = FALSE;
    }

    g_signal_connect (G_OBJECT (menu), "deactivate",
        G_CALLBACK (grdc_connection_window_toolbar_preferences_popdown), cnnwin);

    gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
        grdc_public_popup_position, widget,
        0, gtk_get_current_event_time());
}

static void
grdc_connection_window_toolbar_tools (GtkWidget *widget, GrdcConnectionWindow *cnnwin)
{
    GtkWidget *menu;
    GtkWidget *menuitem;
    GtkWidget *image;

    if (!gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON (widget))) return;

    cnnwin->sticky = TRUE;

    menu = gtk_menu_new ();

    /* Chat */
    menuitem = gtk_image_menu_item_new_with_label (_("Open Chat..."));
    gtk_widget_show (menuitem);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);

    image = gtk_image_new_from_icon_name ("face-smile", GTK_ICON_SIZE_MENU);
    gtk_widget_show (image);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menuitem), image);

    g_object_set_data (G_OBJECT (menuitem), "feature-type", GINT_TO_POINTER (GRDC_PLUG_FEATURE_TOOL_CHAT));

    g_signal_connect (G_OBJECT (menuitem), "activate",
        G_CALLBACK (grdc_connection_window_call_plug_feature_activate), cnnwin);
    if (!grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_TOOL_CHAT))
    {
        gtk_widget_set_sensitive (menuitem, FALSE);
    }

    /* SFTP */
    menuitem = gtk_image_menu_item_new_with_label (_("Open Secure File Transfer..."));
    gtk_widget_show (menuitem);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);

    image = gtk_image_new_from_icon_name ("folder-remote", GTK_ICON_SIZE_MENU);
    gtk_widget_show (image);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menuitem), image);

    g_object_set_data (G_OBJECT (menuitem), "feature-type", GINT_TO_POINTER (GRDC_PLUG_FEATURE_TOOL_SFTP));

    g_signal_connect (G_OBJECT (menuitem), "activate",
        G_CALLBACK (grdc_connection_window_call_plug_feature_activate), cnnwin);
    if (!grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_TOOL_SFTP))
    {
        gtk_widget_set_sensitive (menuitem, FALSE);
    }

    /* SSH Terminal */
    menuitem = gtk_image_menu_item_new_with_label (_("Open Secure Shell in New Terminal..."));
    gtk_widget_show (menuitem);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);

    image = gtk_image_new_from_icon_name ("utilities-terminal", GTK_ICON_SIZE_MENU);
    gtk_widget_show (image);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menuitem), image);

    g_object_set_data (G_OBJECT (menuitem), "feature-type", GINT_TO_POINTER (GRDC_PLUG_FEATURE_TOOL_SSHTERM));

    g_signal_connect (G_OBJECT (menuitem), "activate",
        G_CALLBACK (grdc_connection_window_call_plug_feature_activate), cnnwin);
    if (!grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_TOOL_SSHTERM))
    {
        gtk_widget_set_sensitive (menuitem, FALSE);
    }

    g_signal_connect (G_OBJECT (menu), "deactivate",
        G_CALLBACK (grdc_connection_window_toolbar_tools_popdown), cnnwin);

    gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
        grdc_public_popup_position, widget,
        0, gtk_get_current_event_time());
}

static void
grdc_connection_window_toolbar_minimize (GtkWidget *widget, GrdcConnectionWindow *cnnwin)
{
    grdc_connection_window_floating_toolbar_hide (cnnwin);
    gtk_window_iconify (GTK_WINDOW (cnnwin->window));
}

static void
grdc_connection_window_toolbar_disconnect (GtkWidget *widget, GrdcConnectionWindow* cnnwin)
{
    grdc_connection_window_disconnect (cnnwin);
}

static void
grdc_connection_window_toolbar_grab (GtkWidget *widget, GrdcConnectionWindow *cnnwin)
{
    cnnwin->grdc_file->keyboard_grab = gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON (widget));
    grdc_connection_window_keyboard_grab (cnnwin);
}

static GtkWidget*
grdc_connection_window_create_toolbar (GrdcConnectionWindow *cnnwin, gint mode)
{
    GtkWidget *toolbar;
    GtkToolItem *toolitem;
    GtkWidget *widget;
    GtkWidget *arrow;

    toolbar = gtk_toolbar_new ();
    gtk_widget_show (toolbar);
    gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), GTK_TOOLBAR_ICONS);
    gtk_toolbar_set_show_arrow (GTK_TOOLBAR (toolbar), FALSE);
    if (grdc_pref.small_toolbutton)
    {
        gtk_toolbar_set_icon_size (GTK_TOOLBAR (toolbar), GTK_ICON_SIZE_MENU);
    }

    toolitem = gtk_separator_tool_item_new ();
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));

    if (mode == SCROLLED_WINDOW_MODE)
    {
        toolitem = gtk_tool_button_new_from_stock (GTK_STOCK_ZOOM_100);
        gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Auto-Fit Window"));
        gtk_tool_item_set_tooltip_text (toolitem, _("Resize the window to fit in remote resolution"));
        g_signal_connect (G_OBJECT (toolitem), "clicked",
            G_CALLBACK(grdc_connection_window_toolbar_autofit), cnnwin);
    }
    else
    {
        toolitem = gtk_tool_button_new_from_stock (GTK_STOCK_LEAVE_FULLSCREEN);
        gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Scrolled Window"));
        gtk_tool_item_set_tooltip_text (toolitem, _("Toggle scrolled window mode"));
        g_signal_connect (G_OBJECT (toolitem), "clicked",
            G_CALLBACK(grdc_connection_window_toolbar_scrolled), cnnwin);
    }
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));

    toolitem = gtk_tool_button_new_from_stock (GTK_STOCK_FULLSCREEN);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Scrolled Fullscreen"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Toggle scrolled fullscreen mode"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    g_signal_connect (G_OBJECT (toolitem), "clicked",
        G_CALLBACK(grdc_connection_window_toolbar_scrolled_fullscreen), cnnwin);
    if (mode == FULLSCREEN_MODE || mode == SCROLLED_FULLSCREEN_MODE)
    {
        gtk_widget_set_sensitive (GTK_WIDGET (toolitem), FALSE);
    }

    toolitem = gtk_tool_button_new_from_stock (GTK_STOCK_ZOOM_IN);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Viewport Fullscreen"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Toggle viewport fullscreen mode"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    g_signal_connect (G_OBJECT (toolitem), "clicked",
        G_CALLBACK(grdc_connection_window_toolbar_viewport_fullscreen), cnnwin);
    if (mode == FULLSCREEN_MODE || mode == VIEWPORT_FULLSCREEN_MODE)
    {
        gtk_widget_set_sensitive (GTK_WIDGET (toolitem), FALSE);
    }

    toolitem = gtk_toggle_tool_button_new_from_stock (GTK_STOCK_ZOOM_FIT);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Scaled Mode"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Toggle scaled mode"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON (toolitem), GRDC_PLUG (cnnwin->grdc_plug)->scale);
    g_signal_connect (G_OBJECT (toolitem), "toggled",
        G_CALLBACK(grdc_connection_window_toolbar_scaled_mode), cnnwin);
    if (!grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_SCALE))
    {
        gtk_widget_set_sensitive (GTK_WIDGET (toolitem), FALSE);
    }
    cnnwin->toolitem_scale = toolitem;

    /* We need a toggle tool button with a popup arrow; and the popup is a window not a menu.
       GTK+ support neither of them. We need some tricks here... */
    toolitem = gtk_tool_item_new ();
    gtk_widget_show (GTK_WIDGET (toolitem));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);

    widget = gtk_toggle_button_new ();
    gtk_widget_show (widget);
    gtk_container_set_border_width (GTK_CONTAINER (widget), 0);
    gtk_button_set_relief (GTK_BUTTON (widget), GTK_RELIEF_NONE);
    gtk_button_set_focus_on_click (GTK_BUTTON (widget), FALSE);
    gtk_container_add (GTK_CONTAINER (toolitem), widget);

    arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_NONE);
    gtk_widget_show (arrow);
    gtk_container_add (GTK_CONTAINER (widget), arrow);

    g_signal_connect (G_OBJECT (widget), "toggled",
        G_CALLBACK(grdc_connection_window_toolbar_scale_option), cnnwin);
    if (!grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_SCALE) ||
        !GRDC_PLUG (cnnwin->grdc_plug)->scale)
    {
        gtk_widget_set_sensitive (GTK_WIDGET (widget), FALSE);
    }
    cnnwin->scale_option_button = widget;

    toolitem = gtk_separator_tool_item_new ();
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));

    toolitem = gtk_toggle_tool_button_new ();
    gtk_tool_button_set_icon_name (GTK_TOOL_BUTTON (toolitem), "input-keyboard");
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Grab Keyboard"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Grab all keyboard events"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    gtk_toggle_tool_button_set_active (GTK_TOGGLE_TOOL_BUTTON (toolitem), cnnwin->grdc_file->keyboard_grab);
    g_signal_connect (G_OBJECT (toolitem), "toggled",
        G_CALLBACK(grdc_connection_window_toolbar_grab), cnnwin);

    toolitem = gtk_toggle_tool_button_new_from_stock (GTK_STOCK_PREFERENCES);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Preferences"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Preferences"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    if (grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_PREF))
    {
        g_signal_connect (G_OBJECT (toolitem), "toggled",
            G_CALLBACK(grdc_connection_window_toolbar_preferences), cnnwin);
    }
    else
    {
        gtk_widget_set_sensitive (GTK_WIDGET (toolitem), FALSE);
    }
    cnnwin->toolitem_preferences = toolitem;

    toolitem = gtk_toggle_tool_button_new_from_stock (GTK_STOCK_EXECUTE);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Tools"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Tools"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    if (grdc_plug_query_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_TOOL))
    {
        g_signal_connect (G_OBJECT (toolitem), "toggled",
            G_CALLBACK(grdc_connection_window_toolbar_tools), cnnwin);
    }
    else
    {
        gtk_widget_set_sensitive (GTK_WIDGET (toolitem), FALSE);
    }
    cnnwin->toolitem_tools = toolitem;

    toolitem = gtk_separator_tool_item_new ();
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));

    toolitem = gtk_tool_button_new_from_stock (GTK_STOCK_GOTO_BOTTOM);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Minimize"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Minimize Window"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    g_signal_connect (G_OBJECT (toolitem), "clicked",
        G_CALLBACK(grdc_connection_window_toolbar_minimize), cnnwin);

    toolitem = gtk_tool_button_new_from_stock (GTK_STOCK_DISCONNECT);
    gtk_tool_button_set_label (GTK_TOOL_BUTTON (toolitem), _("Disconnect"));
    gtk_tool_item_set_tooltip_text (toolitem, _("Disconnect"));
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));
    g_signal_connect (G_OBJECT (toolitem), "clicked",
        G_CALLBACK(grdc_connection_window_toolbar_disconnect), cnnwin);

    toolitem = gtk_separator_tool_item_new ();
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    gtk_widget_show (GTK_WIDGET (toolitem));

    return toolbar;
}

static void
grdc_connection_window_update_toolbar_opacity (GrdcConnectionWindow *cnnwin)
{
    gtk_window_set_opacity (GTK_WINDOW (cnnwin->floating_toolbar),
        (1.0 - TOOLBAR_OPACITY_MIN) / ((gdouble) TOOLBAR_OPACITY_LEVEL)
        * ((gdouble) (TOOLBAR_OPACITY_LEVEL - cnnwin->grdc_file->toolbar_opacity))
        + TOOLBAR_OPACITY_MIN);
}

static gboolean
grdc_connection_window_toolbar_enter (
    GtkWidget *widget,
    GdkEventCrossing *event,
    GrdcConnectionWindow *cnnwin)
{
    gint x;

    /* I hardcoded a 100ms timeout here, so that the toolbar won't show up immediately after it hides */
    if (gtk_get_current_event_time () - cnnwin->floating_toolbar_motion_time < MOTION_TIME) return TRUE;

    gtk_window_get_position (GTK_WINDOW (cnnwin->floating_toolbar), &x, NULL);
    gtk_window_move (GTK_WINDOW (cnnwin->floating_toolbar), x, 0);
    grdc_connection_window_update_toolbar_opacity (cnnwin);
    return TRUE;
}

static gboolean
grdc_connection_window_toolbar_leave (
    GtkWidget *widget,
    GdkEventCrossing *event,
    GrdcConnectionWindow *cnnwin)
{
    if (!cnnwin->sticky && event->mode == GDK_CROSSING_NORMAL)
    {
        grdc_connection_window_floating_toolbar_hide (cnnwin);
        return TRUE;
    }
    return FALSE;
}

static gboolean
grdc_connection_window_focus_in (GtkWidget *widget, GdkEventFocus *event, GrdcConnectionWindow *cnnwin)
{
    if (cnnwin->floating_toolbar)
    {
        gtk_widget_show (cnnwin->floating_toolbar);
    }
    return FALSE;
}

static gboolean
grdc_connection_window_focus_out (GtkWidget *widget, GdkEventFocus *event, GrdcConnectionWindow *cnnwin)
{
    if (!cnnwin->sticky && cnnwin->floating_toolbar)
    {
        gtk_widget_hide (cnnwin->floating_toolbar);
    }
    if (cnnwin->scrolled_viewport)
    {
        grdc_scrolled_viewport_remove_motion (GRDC_SCROLLED_VIEWPORT (cnnwin->scrolled_viewport));
    }
    grdc_plug_call_feature (GRDC_PLUG (cnnwin->grdc_plug), GRDC_PLUG_FEATURE_UNFOCUS, NULL);
    return FALSE;
}

static gboolean
grdc_connection_window_on_enter (
    GtkWidget *widget,
    GdkEventCrossing *event,
    GrdcConnectionWindow *cnnwin)
{
    if (event->detail == GDK_NOTIFY_VIRTUAL ||
        event->detail == GDK_NOTIFY_NONLINEAR ||
        event->detail == GDK_NOTIFY_NONLINEAR_VIRTUAL)
    {
        grdc_connection_window_keyboard_grab (cnnwin);
    }
    return FALSE;
}

static gboolean
grdc_connection_window_on_leave (
    GtkWidget *widget,
    GdkEventCrossing *event,
    GrdcConnectionWindow *cnnwin)
{
    if (event->detail == GDK_NOTIFY_VIRTUAL ||
        event->detail == GDK_NOTIFY_NONLINEAR ||
        event->detail == GDK_NOTIFY_NONLINEAR_VIRTUAL)
    {
        gdk_keyboard_ungrab (GDK_CURRENT_TIME);
    }
    return FALSE;
}

static gboolean
grdc_connection_window_toolbar_scroll (GtkWidget *widget, GdkEventScroll *event, GrdcConnectionWindow *cnnwin)
{
    switch (event->direction)
    {
    case GDK_SCROLL_UP:
        if (cnnwin->grdc_file->toolbar_opacity > 0)
        {
            cnnwin->grdc_file->toolbar_opacity--;
            grdc_connection_window_update_toolbar_opacity (cnnwin);
            return TRUE;
        }
        break;
    case GDK_SCROLL_DOWN:
        if (cnnwin->grdc_file->toolbar_opacity < TOOLBAR_OPACITY_LEVEL)
        {
            cnnwin->grdc_file->toolbar_opacity++;
            grdc_connection_window_update_toolbar_opacity (cnnwin);
            return TRUE;
        }
        break;
    default:
        break;
    }
    return FALSE;
}

static void
grdc_connection_window_alignment_on_allocate (GtkWidget *widget, GtkAllocation *allocation, GrdcConnectionWindow *cnnwin)
{
    grdc_connection_window_update_alignment (cnnwin);
}

static gboolean
grdc_connection_window_on_configure (GtkWidget *widget, GdkEventConfigure *event, GrdcConnectionWindow *cnnwin)
{
    GtkRequisition req;
    GdkScreen *screen;

    if (cnnwin->window && cnnwin->window->window && cnnwin->grdc_file->viewmode == SCROLLED_WINDOW_MODE)
    {
        /* Here we store the window state in real-time */
        if ((gdk_window_get_state (cnnwin->window->window) & GDK_WINDOW_STATE_MAXIMIZED) == 0)
        {
            gtk_window_get_size (GTK_WINDOW (cnnwin->window),
                &cnnwin->grdc_file->window_width, &cnnwin->grdc_file->window_height);
            cnnwin->grdc_file->window_maximize = FALSE;
        }
        else
        {
            cnnwin->grdc_file->window_maximize = TRUE;
        }
    }

    if (cnnwin->floating_toolbar)
    {
        screen = gdk_screen_get_default ();
        gtk_widget_size_request (cnnwin->floating_toolbar, &req);
        gtk_window_move (GTK_WINDOW (cnnwin->floating_toolbar),
            (gdk_screen_get_width (screen) - req.width) / 2, 2 - req.height);
    }
    
    if (cnnwin->scrolled_viewport)
    {
        /* Notify window of change so that scroll border can be hidden or shown if needed */
        grdc_plug_emit_signal (GRDC_PLUG (cnnwin->grdc_plug), "desktop-resize");
    }
    return FALSE;
}

static void
grdc_connection_window_create_floating_toolbar (GrdcConnectionWindow *cnnwin, gint mode)
{
    GtkWidget *window;
    GtkWidget *vbox;
    GtkWidget *widget;
    GtkWidget *eventbox;
    GtkRequisition req;
    GdkScreen *screen;

    screen = gdk_screen_get_default ();

    if (cnnwin->floating_toolbar != NULL)
    {
        gtk_widget_destroy (cnnwin->floating_toolbar);
        cnnwin->floating_toolbar = NULL;
    }

    /* This has to be a popup window to become visible in fullscreen mode */
    window = gtk_window_new (GTK_WINDOW_POPUP);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gtk_container_add (GTK_CONTAINER (window), vbox);

    widget = grdc_connection_window_create_toolbar (cnnwin, mode);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 0);

    /* An event box is required to wrap the label to avoid infinite "leave-enter" event loop */
    eventbox = gtk_event_box_new ();
    gtk_widget_show (eventbox);
    gtk_box_pack_start (GTK_BOX (vbox), eventbox, FALSE, FALSE, 0);
    widget = gtk_label_new (cnnwin->grdc_file->name);
    gtk_label_set_max_width_chars (GTK_LABEL (widget), 50);
    gtk_widget_show (widget);
    gtk_container_add (GTK_CONTAINER (eventbox), widget);

    gtk_widget_size_request (window, &req);
    gtk_window_move (GTK_WINDOW (window), (gdk_screen_get_width (screen) - req.width) / 2, 2 - req.height);
    if (grdc_pref.invisible_toolbar)
    {
        gtk_window_set_opacity (GTK_WINDOW (window), 0.0);
    }

    cnnwin->floating_toolbar = window;

    g_signal_connect (G_OBJECT (window), "enter-notify-event", 
        G_CALLBACK (grdc_connection_window_toolbar_enter), cnnwin);
    g_signal_connect (G_OBJECT (window), "leave-notify-event", 
        G_CALLBACK (grdc_connection_window_toolbar_leave), cnnwin);
    g_signal_connect (G_OBJECT (window), "scroll-event", 
        G_CALLBACK (grdc_connection_window_toolbar_scroll), cnnwin);

    if (cnnwin->connected) gtk_widget_show (window);
}

static GtkWidget*
grdc_connection_window_create_main (GrdcConnectionWindow *cnnwin)
{
    GtkWidget *window;

    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (window), cnnwin->grdc_file->name);
    gtk_window_set_default_size (GTK_WINDOW (window),
        cnnwin->grdc_file->window_width, cnnwin->grdc_file->window_height);
    gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
    gtk_container_set_border_width (GTK_CONTAINER (window), 0);

    gtk_window_set_icon (GTK_WINDOW (window), grdc_file_get_icon (cnnwin->grdc_file, 0));

    g_signal_connect (G_OBJECT (window), "delete-event",
        G_CALLBACK (grdc_connection_window_delete_event), cnnwin);
    g_signal_connect (G_OBJECT (window), "destroy",
        G_CALLBACK (grdc_connection_window_destroy), cnnwin);

    g_signal_connect (G_OBJECT (window), "focus-in-event",
        G_CALLBACK (grdc_connection_window_focus_in), cnnwin);
    g_signal_connect (G_OBJECT (window), "focus-out-event",
        G_CALLBACK (grdc_connection_window_focus_out), cnnwin);

    g_signal_connect (G_OBJECT (window), "enter-notify-event",
        G_CALLBACK (grdc_connection_window_on_enter), cnnwin);
    g_signal_connect (G_OBJECT (window), "leave-notify-event",
        G_CALLBACK (grdc_connection_window_on_leave), cnnwin);

    g_signal_connect (G_OBJECT (window), "configure_event",
        G_CALLBACK (grdc_connection_window_on_configure), cnnwin);

    cnnwin->scrolled_window = NULL;
    cnnwin->scrolled_viewport = NULL;
    cnnwin->toolbar = NULL;

    return window;
}

/* Create a scrolled window container */
void
grdc_connection_window_create_scrolled (GrdcConnectionWindow *cnnwin)
{
    GtkWidget *window;
    GtkWidget *vbox;
    GtkWidget *toolbar;
    GtkWidget *scrolledwindow;

    if (grdc_pref.save_view_mode && cnnwin->grdc_file->viewmode != AUTO_MODE)
    {
        cnnwin->grdc_file->viewmode = SCROLLED_WINDOW_MODE;
    }

    if (cnnwin->floating_toolbar != NULL)
    {
        gtk_widget_destroy (cnnwin->floating_toolbar);
        cnnwin->floating_toolbar = NULL;
    }

    window = grdc_connection_window_create_main (cnnwin);
    gtk_widget_realize (window);

    /* Create the vbox container */
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gtk_container_add (GTK_CONTAINER (window), vbox);

    /* Create the toolbar */
    toolbar = grdc_connection_window_create_toolbar (cnnwin, SCROLLED_WINDOW_MODE);
    gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);

    /* Create the scrolled window */
    scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
        GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start (GTK_BOX (vbox), scrolledwindow, TRUE, TRUE, 0);

    /* Add the viewport into the scrolled window */
    if (cnnwin->window == NULL)
    {
        gtk_container_add (GTK_CONTAINER (scrolledwindow), cnnwin->viewport);
    }
    else
    {
        gtk_widget_reparent (cnnwin->viewport, scrolledwindow);
        gtk_widget_destroy (cnnwin->window);
    }

    gtk_container_set_focus_chain (GTK_CONTAINER (vbox), g_list_append (NULL, scrolledwindow));

    cnnwin->scrolled_window = scrolledwindow;
    cnnwin->toolbar = toolbar;
    cnnwin->window = window;

    if (cnnwin->connected)
    {
        if (cnnwin->grdc_file->window_maximize)
        {
            gtk_window_maximize (GTK_WINDOW (window));
        }
        gtk_widget_show (window);
    }
}

void
grdc_connection_window_create_fullscreen (GrdcConnectionWindow *cnnwin)
{
    GtkWidget *window;

    window = grdc_connection_window_create_main (cnnwin);
    gtk_widget_realize (window);
    if (cnnwin->window == NULL)
    {
        gtk_container_add (GTK_CONTAINER (window), cnnwin->viewport);
    }
    else
    {
        gtk_widget_reparent (cnnwin->viewport, window);
        gtk_widget_destroy (cnnwin->window);
    }

    cnnwin->window = window;

    /* Create the floating toolbar */
    grdc_connection_window_create_floating_toolbar (cnnwin, FULLSCREEN_MODE);

    gtk_window_fullscreen (GTK_WINDOW (window));
    if (cnnwin->connected) gtk_widget_show (window);
}

static gboolean
grdc_connection_window_scrolled_required (GrdcConnectionWindow *cnnwin)
{
    GdkScreen *screen;
    gint screen_width, screen_height;
    gint server_width, server_height;

    gtk_widget_get_size_request (cnnwin->grdc_plug, &server_width, &server_height);

    if (server_width <= 0 || server_height <= 0) return TRUE;

    screen = gdk_screen_get_default ();
    screen_width = gdk_screen_get_width (screen);
    screen_height = gdk_screen_get_height (screen);

    if (screen_width >= server_width && screen_height >= server_height) return FALSE;

    return TRUE;
}

/* Create a scrolled fullscreen container */
void
grdc_connection_window_create_scrolled_fullscreen (GrdcConnectionWindow *cnnwin)
{
    GtkWidget *window;
    GtkWidget *scrolledwindow;

    if (grdc_pref.save_view_mode) cnnwin->grdc_file->viewmode = SCROLLED_FULLSCREEN_MODE;

    if (!grdc_connection_window_scrolled_required (cnnwin))
    {
        grdc_connection_window_create_fullscreen (cnnwin);
        return;
    }

    window = grdc_connection_window_create_main (cnnwin);
    gtk_widget_realize (window);

    /* Create the scrolled window */
    scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow),
        GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_container_set_border_width (GTK_CONTAINER (scrolledwindow), 0);
    gtk_container_add (GTK_CONTAINER (window), scrolledwindow);

    /* Link the containers to the main window and the viewport */
    if (cnnwin->window == NULL)
    {
        gtk_container_add (GTK_CONTAINER (scrolledwindow), cnnwin->viewport);
    }
    else
    {
        gtk_widget_reparent (cnnwin->viewport, scrolledwindow);
        gtk_widget_destroy (cnnwin->window);
    }

    cnnwin->window = window;

    /* Create the floating toolbar */
    grdc_connection_window_create_floating_toolbar (cnnwin, SCROLLED_FULLSCREEN_MODE);

    gtk_window_fullscreen (GTK_WINDOW (window));
    if (cnnwin->connected) gtk_widget_show (window);
}

/* Create a viewport fullscreen container */
void
grdc_connection_window_create_viewport_fullscreen (GrdcConnectionWindow *cnnwin)
{
    GtkWidget *window;
    GdkColor color;
    GtkWidget *gsv;

    if (grdc_pref.save_view_mode) cnnwin->grdc_file->viewmode = VIEWPORT_FULLSCREEN_MODE;

    if (!grdc_connection_window_scrolled_required (cnnwin))
    {
        grdc_connection_window_create_fullscreen (cnnwin);
        return;
    }

    window = grdc_connection_window_create_main (cnnwin);
    gtk_widget_realize (window);
    gdk_color_parse ("black", &color);
    gtk_widget_modify_bg (window, GTK_STATE_NORMAL, &color);
    gtk_container_set_border_width (GTK_CONTAINER (window), 1);

    gsv = grdc_scrolled_viewport_new ();
    gtk_widget_show (gsv);
    gtk_container_add (GTK_CONTAINER (window), gsv);

    /* Create the eventbox to trap events */
    if (cnnwin->window == NULL)
    {
        gtk_container_add (GTK_CONTAINER (gsv), cnnwin->viewport);
    }
    else
    {
        gtk_widget_reparent (cnnwin->viewport, gsv);
        gtk_widget_destroy (cnnwin->window);
    }

    cnnwin->scrolled_viewport = gsv;
    cnnwin->window = window;

    /* Create the floating toolbar */
    grdc_connection_window_create_floating_toolbar (cnnwin, VIEWPORT_FULLSCREEN_MODE);

    gtk_window_fullscreen (GTK_WINDOW (window));
    if (cnnwin->connected) gtk_widget_show (window);
}

static void
grdc_connection_window_on_connect (GrdcPlug *gp, GrdcConnectionWindow *cnnwin)
{
    GdkScreen *screen;

    screen = gdk_screen_get_default ();

    grdc_connection_window_register (cnnwin);
    cnnwin->connected = TRUE;

    /* Remember recent list for quick connect */
    if (cnnwin->grdc_file->filename == NULL)
    {
        grdc_pref_add_recent (cnnwin->grdc_file->protocol, cnnwin->grdc_file->server);
    }

    /* If "Use client resolution" is selected, we switch to fullscreen by default */
    if (cnnwin->grdc_file->viewmode != SCROLLED_WINDOW_MODE &&
        cnnwin->grdc_file->resolution && g_strcmp0 (cnnwin->grdc_file->resolution, "AUTO") == 0 &&
        !grdc_connection_window_scrolled_required (cnnwin))
    {
        grdc_connection_window_create_fullscreen (cnnwin);
    }
    /* Protocols like RDP sets the resolution before connecting. We can set the default window size here */
    else if (cnnwin->grdc_file->viewmode == AUTO_MODE && g_strcmp0 (cnnwin->grdc_file->protocol, "RDP") == 0)
    {
        if (cnnwin->grdc_file->resolution_width >= gdk_screen_get_width (screen) ||
            cnnwin->grdc_file->resolution_height >= gdk_screen_get_height (screen))
        {
            gtk_window_maximize (GTK_WINDOW (cnnwin->window));
            cnnwin->grdc_file->window_maximize = TRUE;
        }
        else
        {
            grdc_connection_window_toolbar_autofit (NULL, cnnwin);
            cnnwin->grdc_file->window_maximize = FALSE;
        }
    }
    else if (cnnwin->grdc_file->window_maximize)
    {
        gtk_window_maximize (GTK_WINDOW (cnnwin->window));
    }

    gtk_widget_show (cnnwin->window);
    if (cnnwin->floating_toolbar)
    {
        gtk_widget_show (cnnwin->floating_toolbar);
    }
}

static void
grdc_connection_window_on_disconnect (GrdcPlug *gp, GrdcConnectionWindow *cnnwin)
{
    GtkWidget *dialog;

    cnnwin->connected = FALSE;

   if (scale_option_window)
   {
       gtk_widget_destroy (scale_option_window);
       scale_option_window = NULL;
   }

    if (grdc_pref.save_view_mode)
    {
        cnnwin->grdc_file->scale = GRDC_PLUG (cnnwin->grdc_plug)->scale;
        grdc_file_save (cnnwin->grdc_file);
    }

    if (GRDC_PLUG (cnnwin->grdc_plug)->has_error)
    {
        dialog = gtk_message_dialog_new (NULL,
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
            GRDC_PLUG (cnnwin->grdc_plug)->error_message, NULL);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
    }

    gtk_widget_destroy (cnnwin->window);
}

static void
grdc_connection_window_on_desktop_resize (GrdcPlug *gp, GrdcConnectionWindow *cnnwin)
{
    gint client_width, client_height;
    gint server_width, server_height;
    gboolean scroll_required;
    GdkScreen *screen;

    grdc_connection_window_get_desktop_size (cnnwin, &server_width, &server_height);
    screen = gdk_screen_get_default ();

    if (cnnwin->scrolled_viewport)
    {
        gtk_window_get_size (GTK_WINDOW (cnnwin->window), &client_width, &client_height);

        scroll_required = (server_width > client_width || server_height > client_height);
        gtk_container_set_border_width (GTK_CONTAINER (cnnwin->window), scroll_required ? 1 : 0);
    }
    else if (cnnwin->scrolled_window && cnnwin->grdc_file->viewmode == AUTO_MODE)
    {
        if (server_width >= gdk_screen_get_width (screen) ||
            server_height >= gdk_screen_get_height (screen))
        {
            gtk_window_maximize (GTK_WINDOW (cnnwin->window));
            cnnwin->grdc_file->window_maximize = TRUE;
        }
        else
        {
            grdc_connection_window_toolbar_autofit (NULL, cnnwin);
            cnnwin->grdc_file->window_maximize = FALSE;
        }
    }
}

gboolean
grdc_connection_window_open (GtkWindow *parent, const gchar *filename, gboolean quit_when_close)
{
    GrdcFile *grdcfile;
    GtkWidget *dialog;

    grdcfile = grdc_file_load (filename);
    if (grdcfile)
    {
        grdc_connection_window_open_struct (parent, grdcfile, quit_when_close);
        return TRUE;
    }
    else
    {
        dialog = gtk_message_dialog_new (parent, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
            _("File %s not found."), filename);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        return FALSE;
    }
}

void
grdc_connection_window_open_struct (GtkWindow *parent, GrdcFile *grdcfile, gboolean quit_when_close)
{
    GrdcConnectionWindow *cnnwin;
    GdkColor color;

    if (g_strcmp0 (grdcfile->protocol, "SFTP") == 0)
    {
#ifdef HAVE_LIBSSH
        grdc_sftp_window_open (parent, grdcfile, quit_when_close);
#endif
        return;
    }

    grdc_file_update_screen_resolution (grdcfile);

    cnnwin = g_new (GrdcConnectionWindow, 1);

    cnnwin->grdc_file = grdcfile;
    cnnwin->window = NULL;
    cnnwin->floating_toolbar = NULL;
    cnnwin->floating_toolbar_motion_time = 0;
    cnnwin->connected = FALSE;
    cnnwin->quit_when_close = quit_when_close;
    cnnwin->sticky = FALSE;
    cnnwin->scrolled_window = NULL;
    cnnwin->scrolled_viewport = NULL;
    cnnwin->toolbar = NULL;

    /* Create the GrdcPlug */
    if (g_strcmp0 (grdcfile->protocol, "RDP") == 0)
    {
        cnnwin->grdc_plug = grdc_plug_rdp_new ();
    }
    else if (strncmp (grdcfile->protocol, "VNC", 3) == 0)
    {
        cnnwin->grdc_plug = grdc_plug_vnc_new (grdcfile->scale);
    }
    else
    {
        g_print ("Fatal: Protocol %s not supported\n", grdcfile->protocol);
        g_free (cnnwin);
        return;
    }

    gtk_widget_show (cnnwin->grdc_plug);
    g_signal_connect (G_OBJECT (cnnwin->grdc_plug), "connect",
        G_CALLBACK (grdc_connection_window_on_connect), cnnwin);
    g_signal_connect (G_OBJECT (cnnwin->grdc_plug), "disconnect",
        G_CALLBACK (grdc_connection_window_on_disconnect), cnnwin);
    g_signal_connect (G_OBJECT (cnnwin->grdc_plug), "desktop-resize",
        G_CALLBACK (grdc_connection_window_on_desktop_resize), cnnwin);

    /* Create the alignment to make the GrdcPlug centered */
    cnnwin->alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
    gtk_widget_show (cnnwin->alignment);
    gtk_container_set_border_width (GTK_CONTAINER (cnnwin->alignment), 0);
    gtk_container_add (GTK_CONTAINER (cnnwin->alignment), cnnwin->grdc_plug);
    g_signal_connect (G_OBJECT (cnnwin->alignment), "size-allocate",
        G_CALLBACK (grdc_connection_window_alignment_on_allocate), cnnwin);

    /* Create the viewport to make the GrdcPlug scrollable */
    cnnwin->viewport = gtk_viewport_new (NULL, NULL);
    gtk_widget_show (cnnwin->viewport);
    gdk_color_parse ("black", &color);
    gtk_widget_modify_bg (cnnwin->viewport, GTK_STATE_NORMAL, &color);
    gtk_container_set_border_width (GTK_CONTAINER (cnnwin->viewport), 0);
    gtk_viewport_set_shadow_type (GTK_VIEWPORT (cnnwin->viewport), GTK_SHADOW_NONE);
    gtk_container_add (GTK_CONTAINER (cnnwin->viewport), cnnwin->alignment);

    if (!grdc_pref.save_view_mode) cnnwin->grdc_file->viewmode = grdc_pref.default_mode;

    switch (cnnwin->grdc_file->viewmode)
    {
        case SCROLLED_FULLSCREEN_MODE:
            grdc_connection_window_create_scrolled_fullscreen (cnnwin);
            break;

        case VIEWPORT_FULLSCREEN_MODE:
            grdc_connection_window_create_viewport_fullscreen (cnnwin);
            break;

        case SCROLLED_WINDOW_MODE:
        default:
            grdc_connection_window_create_scrolled (cnnwin);
            break;
    }

    gtk_widget_realize (cnnwin->window);

    if (!grdc_plug_open_connection (GRDC_PLUG (cnnwin->grdc_plug), parent, grdcfile))
    {
        gtk_widget_destroy (cnnwin->window);
    }
}

gint
grdc_connection_window_count (void)
{
    if (grdc_connection_window_array == NULL) return 0;
    return grdc_connection_window_array->len;
}

void
grdc_connection_window_quit_all (void)
{
    gpointer *pdata;
    gint i, len;

    if (grdc_connection_window_array == NULL) return;

    /* We need to duplicate the array, because the elements in the original array will get removed while killing */
    len = grdc_connection_window_count ();
    pdata = g_memdup (grdc_connection_window_array->pdata, sizeof(gpointer) * len);

    for (i = 0; i < len; i++)
    {
        grdc_connection_window_disconnect ((GrdcConnectionWindow*) pdata[i]);
        gtk_main_iteration ();
    }
    g_free (pdata);
}

