/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gstdio.h>
#include <glib/gi18n.h>
#include <signal.h>
#include "config.h"
#ifdef HAVE_PTHREAD
#include <pthread.h>
#endif
#include "grdcpublic.h"
#include "grdcfile.h"
#include "grdcinitdialog.h"
#include "grdcplug.h"
#include "grdcplug_vnc.h"

G_DEFINE_TYPE (GrdcPlugVnc, grdc_plug_vnc, GRDC_TYPE_PLUG)

#ifdef HAVE_LIBVNCCLIENT

/***************************** LibVNCClient related codes *********************************/
#include <rfb/rfbclient.h>

static void
grdc_plug_vnc_update_quality (rfbClient *cl, const gchar *quality_string)
{
    gint quality;

    quality = (gint) g_ascii_strtoll (quality_string, NULL, 0);

    switch (quality)
    {
    case 9:
        cl->appData.useBGR233 = 0;
        cl->appData.encodingsString = "copyrect hextile raw";
        cl->appData.compressLevel = 0;
        cl->appData.qualityLevel = 9;
        break;
    case 2:
        cl->appData.useBGR233 = 0;
        cl->appData.encodingsString = "tight zrle ultra copyrect hextile zlib corre rre raw";
        cl->appData.compressLevel = 3;
        cl->appData.qualityLevel = 7;
        break;
    case 1:
        cl->appData.useBGR233 = 0;
        cl->appData.encodingsString = "tight zrle ultra copyrect hextile zlib corre rre raw";
        cl->appData.compressLevel = 5;
        cl->appData.qualityLevel = 5;
        break;
    case 0:
    default:
        cl->appData.useBGR233 = 1;
        cl->appData.encodingsString = "tight zrle ultra copyrect hextile zlib corre rre raw";
        cl->appData.compressLevel = 9;
        cl->appData.qualityLevel = 1;
        break;
    }

    SetFormatAndEncodings(cl);
}

static rfbBool
grdc_plug_vnc_rfb_allocfb (rfbClient *cl)
{
    GrdcPlugVnc *gp_vnc;
    gint width, height, depth, size;

    gp_vnc = GRDC_PLUG_VNC (rfbClientGetClientData (cl, NULL));

    width = cl->width;
    height = cl->height;
    depth = cl->format.bitsPerPixel;
    size = width * height * (depth / 8);

    if (gp_vnc->rgb_buffer) g_free (gp_vnc->rgb_buffer);
    gp_vnc->rgb_buffer = (guchar*) g_malloc (size);
    memset(gp_vnc->rgb_buffer, 0, size);

    cl->frameBuffer = gp_vnc->rgb_buffer;

    cl->format.redShift = 0;
    cl->format.greenShift = 8;
    cl->format.blueShift = 16;
    cl->format.redMax = 0xff;
    cl->format.greenMax = 0xff;
    cl->format.blueMax = 0xff;

    cl->appData.useRemoteCursor = (gp_vnc->grdc_file->showcursor ? FALSE : TRUE);

    grdc_plug_vnc_update_quality (cl, gp_vnc->grdc_file->quality);

    return TRUE;
}

static void
grdc_plug_vnc_rfb_updatefb (rfbClient* cl, int x, int y, int w, int h)
{
    GrdcPlugVnc *gp_vnc;

    gp_vnc = GRDC_PLUG_VNC (rfbClientGetClientData (cl, NULL));

    THREADS_ENTER
    gtk_widget_queue_draw_area (GTK_WIDGET (gp_vnc), x, y, w, h);
    THREADS_LEAVE
}

static void
grdc_plug_vnc_rfb_cuttext (rfbClient* cl, const char *text, int textlen)
{
    THREADS_ENTER
    gtk_clipboard_set_text (gtk_clipboard_get (GDK_SELECTION_CLIPBOARD), text, textlen);
    THREADS_LEAVE
}

static char*
grdc_plug_vnc_rfb_password (rfbClient *cl)
{
    GrdcPlugVnc *gp_vnc;
    GtkWidget *dialog;
    gint ret;
    gchar *pwd = NULL;

    gp_vnc = GRDC_PLUG_VNC (rfbClientGetClientData (cl, NULL));
    gp_vnc->auth_called = TRUE;

    dialog = GRDC_PLUG (gp_vnc)->init_dialog;

    if (gp_vnc->grdc_file->password && gp_vnc->grdc_file->password[0] != '\0')
    {
        pwd = g_strdup (gp_vnc->grdc_file->password);
    }
    else
    {
        THREADS_ENTER
        ret = grdc_init_dialog_authpwd (GRDC_INIT_DIALOG (dialog));
        THREADS_LEAVE

        if (ret == GTK_RESPONSE_OK)
        {
            pwd = g_strdup (GRDC_INIT_DIALOG (dialog)->password);
        }
        else
        {
            gp_vnc->connected = FALSE;
        }
    }
    return pwd;
}

/* Translate known VNC messages. It's for intltool only, not for gcc */
#ifdef __DO_NOT_COMPILE_ME__
N_("Unable to connect to VNC server")
N_("Couldn't convert '%s' to host address")
N_("VNC connection failed: %s")
N_("Your connection has been rejected.")
#endif
/* TODO: We only store the last message at this moment. */
static gchar vnc_error[MAX_ERROR_LENGTH + 1];
static void
grdc_plug_vnc_rfb_output(const char *format, ...)
{
    va_list args;
    va_start(args, format);
    gchar *f, *p;

    /* eliminate the last \n */
    f = g_strdup (format);
    if (f[strlen (f) - 1] == '\n') f[strlen (f) - 1] = '\0';

/*g_printf("%s,len=%i\n", f, strlen(f));*/
    if (g_strcmp0 (f, "VNC connection failed: %s") == 0)
    {
        p = va_arg (args, gchar*);
/*g_printf("(param)%s,len=%i\n", p, strlen(p));*/
        g_snprintf (vnc_error, MAX_ERROR_LENGTH, _(f), _(p));
    }
    else
    {
        g_vsnprintf (vnc_error, MAX_ERROR_LENGTH, _(f), args);
    }

    g_free (f);

    va_end(args);
}

static gboolean
grdc_plug_vnc_main_loop (GrdcPlugVnc *gp_vnc)
{
    gint ret;
    rfbClient *cl;

    if (!gp_vnc->connected)
    {
        gp_vnc->running = FALSE;
        return FALSE;
    }

    cl = (rfbClient*) gp_vnc->client;

/* We need to wait for a while in a thread, to avoid 100% CPU usage in a loop without sleep */
#ifdef HAVE_PTHREAD
    ret = WaitForMessage (cl, 100);
#else
    ret = WaitForMessage (cl, 0);
#endif
    if (ret < 0)
    {
        IDLE_ADD ((GSourceFunc) grdc_plug_close_connection, gp_vnc);
        gp_vnc->running = FALSE;
        return FALSE;
    }
    if (ret == 0) return TRUE;

    if (!HandleRFBServerMessage (cl))
    {
        IDLE_ADD ((GSourceFunc) grdc_plug_close_connection, gp_vnc);
        gp_vnc->running = FALSE;
        return FALSE;
    }

    return TRUE;
}

static gboolean
grdc_plug_vnc_main (GrdcPlugVnc *gp_vnc)
{
    rfbClient *cl = NULL;
    gchar *host, *pos;

    gp_vnc->running = TRUE;

    rfbClientLog = grdc_plug_vnc_rfb_output;
    rfbClientErr = grdc_plug_vnc_rfb_output;

    while (gp_vnc->connected)
    {
        gp_vnc->auth_called = FALSE;

        cl = rfbGetClient(8, 3, 4);
        cl->MallocFrameBuffer = grdc_plug_vnc_rfb_allocfb;
        cl->canHandleNewFBSize = TRUE;
        cl->GetPassword = grdc_plug_vnc_rfb_password;
        cl->GotFrameBufferUpdate = grdc_plug_vnc_rfb_updatefb;
        cl->GotXCutText = grdc_plug_vnc_rfb_cuttext;
        rfbClientSetClientData (cl, NULL, gp_vnc);

        host = g_strdup (gp_vnc->grdc_file->server);
        pos = g_strrstr (host, ":");
        if (pos)
        {
            *pos++ = '\0';
            cl->serverPort = MAX (0, (gint) g_ascii_strtoll (pos, NULL, 0));
            /* Support short-form (:0, :1) */
            if (cl->serverPort < 100) cl->serverPort += 5900;
        }
        else
        {
            cl->serverPort = 5900;
        }
        cl->serverHost = host;

        if (rfbInitClient (cl, NULL, NULL)) break;

        /* If the authentication is not called, it has to be a fatel error and must quit */
        if (!gp_vnc->auth_called)
        {
            gp_vnc->connected = FALSE;
            break;
        }

        /* Otherwise, it's a password error. Try to clear saved password if any */
        if (gp_vnc->grdc_file->password) gp_vnc->grdc_file->password[0] = '\0';
    }

    if (!gp_vnc->connected)
    {
        if (cl && !gp_vnc->auth_called)
        {
            g_snprintf (GRDC_PLUG (gp_vnc)->error_message, MAX_ERROR_LENGTH, "%s", vnc_error);
            GRDC_PLUG (gp_vnc)->has_error = TRUE;
        }
        gp_vnc->running = FALSE;

        IDLE_ADD ((GSourceFunc) grdc_plug_close_connection, gp_vnc);

        return FALSE;
    }

    /* Save the password if it's requested to do so */
    if (GRDC_INIT_DIALOG (GRDC_PLUG (gp_vnc)->init_dialog)->save_password)
    {
        if (gp_vnc->grdc_file->password) g_free (gp_vnc->grdc_file->password);
        gp_vnc->grdc_file->password = g_strdup (GRDC_INIT_DIALOG (GRDC_PLUG (gp_vnc)->init_dialog)->password);
        grdc_file_save (gp_vnc->grdc_file);
    }

    gp_vnc->client = cl;
    gp_vnc->width = cl->width;
    gp_vnc->height = cl->height;

    THREADS_ENTER
    gtk_widget_set_size_request (GTK_WIDGET (gp_vnc), cl->width, cl->height);
    gtk_drawing_area_size (GTK_DRAWING_AREA (gp_vnc->drawing_area), cl->width, cl->height);
    THREADS_LEAVE

    grdc_plug_emit_signal (GRDC_PLUG (gp_vnc), "connect");

    if (gp_vnc->thread)
    {
        while (grdc_plug_vnc_main_loop (gp_vnc)) { }
        gp_vnc->running = FALSE;
    }
    else
    {
        IDLE_ADD ((GSourceFunc) grdc_plug_vnc_main_loop, gp_vnc);
    }

    return FALSE;
}

#ifdef HAVE_PTHREAD
static gpointer
grdc_plug_vnc_main_thread (gpointer data)
{
    pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL);

    CANCEL_ASYNC
    grdc_plug_vnc_main (GRDC_PLUG_VNC (data));
    return NULL;
}
#endif

static gboolean
grdc_plug_vnc_on_motion (GtkWidget *widget, GdkEventMotion *event, GrdcPlugVnc *gp_vnc)
{
    if (!gp_vnc->connected || !gp_vnc->client) return FALSE;
    if (gp_vnc->grdc_file->viewonly) return FALSE;

    SendPointerEvent ((rfbClient*) gp_vnc->client, (gint) event->x, (gint) event->y, gp_vnc->button_mask);
    return TRUE;
}

static gboolean
grdc_plug_vnc_on_button (GtkWidget *widget, GdkEventButton *event, GrdcPlugVnc *gp_vnc)
{
    gint mask;

    if (!gp_vnc->connected || !gp_vnc->client) return FALSE;
    if (gp_vnc->grdc_file->viewonly) return FALSE;

    /* We only accept 3 buttons */
    if (event->button < 1 || event->button > 3) return FALSE;
    /* We bypass 2button-press and 3button-press events */
    if (event->type != GDK_BUTTON_PRESS && event->type != GDK_BUTTON_RELEASE) return TRUE;

    mask = (1 << (event->button - 1));
    gp_vnc->button_mask = (event->type == GDK_BUTTON_PRESS ?
        (gp_vnc->button_mask | mask) :
        (gp_vnc->button_mask & (0xff - mask)));
    SendPointerEvent ((rfbClient*) gp_vnc->client, (gint) event->x, (gint) event->y, gp_vnc->button_mask);
    return TRUE;
}

static gboolean
grdc_plug_vnc_on_scroll (GtkWidget *widget, GdkEventScroll *event, GrdcPlugVnc *gp_vnc)
{
    gint mask;

    if (!gp_vnc->connected || !gp_vnc->client) return FALSE;
    if (gp_vnc->grdc_file->viewonly) return FALSE;

    switch (event->direction)
    {
    case GDK_SCROLL_UP:
        mask = (1 << 3);
        break;
    case GDK_SCROLL_DOWN:
        mask = (1 << 4);
        break;
    case GDK_SCROLL_LEFT:
        mask = (1 << 5);
        break;
    case GDK_SCROLL_RIGHT:
        mask = (1 << 6);
        break;
    default:
        return FALSE;
    }

    SendPointerEvent ((rfbClient*) gp_vnc->client, (gint) event->x, (gint) event->y, mask | gp_vnc->button_mask);
    SendPointerEvent ((rfbClient*) gp_vnc->client, (gint) event->x, (gint) event->y, gp_vnc->button_mask);

    return TRUE;
}

static gboolean
grdc_plug_vnc_on_key (GtkWidget *widget, GdkEventKey *event, GrdcPlugVnc *gp_vnc)
{
    if (!gp_vnc->connected || !gp_vnc->client) return FALSE;
    if (gp_vnc->grdc_file->viewonly) return FALSE;

    /* Lucky! GDK uses the same XKeySym as rfbClient. I haven't yet found any key (or key-combo) not working */
    SendKeyEvent ((rfbClient*) gp_vnc->client, event->keyval, (event->type == GDK_KEY_PRESS ? TRUE : FALSE));
    return TRUE;
}

static void
grdc_plug_vnc_on_cuttext_request (GtkClipboard *clipboard, const gchar *text, GrdcPlugVnc *gp_vnc)
{
    if (text)
    {
        SendClientCutText ((rfbClient*) gp_vnc->client, (char*) text, strlen(text));
    }
}

static void
grdc_plug_vnc_on_cuttext (GtkClipboard *clipboard, GdkEvent *event, GrdcPlugVnc *gp_vnc)
{
    if (!gp_vnc->connected || !gp_vnc->client) return;
    if (gp_vnc->grdc_file->viewonly) return;

    gtk_clipboard_request_text (clipboard, (GtkClipboardTextReceivedFunc) grdc_plug_vnc_on_cuttext_request, gp_vnc);
}

/******************************************************************************************/

static gboolean
grdc_plug_vnc_open_connection (GrdcPlug *gp, GrdcFile *grdcfile)
{
    GrdcPlugVnc *gp_vnc = GRDC_PLUG_VNC (gp);

    gp_vnc->connected = TRUE;
    gp_vnc->grdc_file = grdcfile;

    g_signal_connect (G_OBJECT (gp_vnc->drawing_area), "motion-notify-event",
        G_CALLBACK (grdc_plug_vnc_on_motion), gp_vnc);
    g_signal_connect (G_OBJECT (gp_vnc->drawing_area), "button-press-event",
        G_CALLBACK (grdc_plug_vnc_on_button), gp_vnc);
    g_signal_connect (G_OBJECT (gp_vnc->drawing_area), "button-release-event",
        G_CALLBACK (grdc_plug_vnc_on_button), gp_vnc);
    g_signal_connect (G_OBJECT (gp_vnc->drawing_area), "scroll-event",
        G_CALLBACK (grdc_plug_vnc_on_scroll), gp_vnc);
    g_signal_connect (G_OBJECT (gp_vnc->drawing_area), "key-press-event",
        G_CALLBACK (grdc_plug_vnc_on_key), gp_vnc);
    g_signal_connect (G_OBJECT (gp_vnc->drawing_area), "key-release-event",
        G_CALLBACK (grdc_plug_vnc_on_key), gp_vnc);

    g_signal_connect (G_OBJECT (gtk_clipboard_get (GDK_SELECTION_CLIPBOARD)),
        "owner-change", G_CALLBACK (grdc_plug_vnc_on_cuttext), gp_vnc);

#ifdef HAVE_PTHREAD
    if (pthread_create (&gp_vnc->thread, NULL, grdc_plug_vnc_main_thread, gp_vnc))
    {
        /* I don't think this will ever happen... */
        g_print ("Failed to initialize pthread. Falling back to non-thread mode...\n");
        g_timeout_add (0, (GSourceFunc) grdc_plug_vnc_main, gp);
        gp_vnc->thread = 0;
    }
#else
    g_timeout_add (0, (GSourceFunc) grdc_plug_vnc_main, gp);
#endif

    return TRUE;
}

static gboolean
grdc_plug_vnc_close_connection_timeout (GrdcPlugVnc *gp_vnc)
{
    /* wait until the running attribute is set to false by the VNC thread */
    if (gp_vnc->running) return TRUE;

    if (gp_vnc->rgb_buffer)
    {
        g_free (gp_vnc->rgb_buffer);
        gp_vnc->rgb_buffer = NULL;
    }
    if (gp_vnc->client)
    {
        rfbClientCleanup((rfbClient*) gp_vnc->client);
        gp_vnc->client = NULL;
    }

    grdc_plug_emit_signal (GRDC_PLUG (gp_vnc), "disconnect");

    return FALSE;
}

static gboolean
grdc_plug_vnc_close_connection (GrdcPlug *gp)
{
    GrdcPlugVnc *gp_vnc = GRDC_PLUG_VNC (gp);

    if (gp_vnc->closed) return FALSE;
    gp_vnc->closed = TRUE;
    gp_vnc->connected = FALSE;

#ifdef HAVE_PTHREAD
    if (gp_vnc->thread)
    {
        pthread_cancel (gp_vnc->thread);
        pthread_join (gp_vnc->thread, NULL);
        gp_vnc->running = FALSE;
        grdc_plug_vnc_close_connection_timeout (gp_vnc);
    }
    else
    {
        g_timeout_add (200, (GSourceFunc) grdc_plug_vnc_close_connection_timeout, gp);
    }
#else
    g_timeout_add (200, (GSourceFunc) grdc_plug_vnc_close_connection_timeout, gp);
#endif

    return FALSE;
}

gboolean
grdc_plug_vnc_query_feature (GrdcPlug *gp, GrdcPlugFeature feature)
{
    switch (feature)
    {
        case GRDC_PLUG_FEATURE_PREF:
        case GRDC_PLUG_FEATURE_PREF_QUALITY:
        case GRDC_PLUG_FEATURE_PREF_VIEWONLY:
            return TRUE;
        default:
            return FALSE;
    }
}

void
grdc_plug_vnc_call_feature (GrdcPlug *gp, GrdcPlugFeature feature, gpointer data)
{
    switch (feature)
    {
        case GRDC_PLUG_FEATURE_PREF_QUALITY:
            grdc_plug_vnc_update_quality ((rfbClient*) (GRDC_PLUG_VNC (gp)->client), (const gchar*) data);
            break;
        case GRDC_PLUG_FEATURE_PREF_VIEWONLY:
            GRDC_PLUG_VNC (gp)->grdc_file->viewonly = (data != NULL);
            break;
        default:
            break;
    }
}

#else /* HAVE_LIBVNCCLIENT */

static gboolean
grdc_plug_vnc_open_connection (GrdcPlug *gp, GrdcFile *grdcfile)
{
    GtkWidget *dialog;
    /* This should never happen because if no VNC support users are not able to select VNC in preference dialog */
    dialog = gtk_message_dialog_new (NULL,
        GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
        "VNC not supported");
    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
    return FALSE;
}

static void
grdc_plug_vnc_close_connection (GrdcPlug *gp)
{
}

gboolean
grdc_plug_vnc_query_feature (GrdcPlug *gp, GrdcPlugFeature feature)
{
    return FALSE;
}

void
grdc_plug_vnc_call_feature (GrdcPlug *gp, GrdcPlugFeature feature, gpointer data)
{
}

#endif /* HAVE_LIBVNCCLIENT */

static gboolean
grdc_plug_vnc_on_expose (GtkWidget *widget, GdkEventExpose *event, GrdcPlugVnc *gp_vnc)
{
    if (!gp_vnc->rgb_buffer) return FALSE;

    /* widget == gp_vnc->drawing_area */
    /* this is a little tricky. It "moves" the rgb_buffer pointer to (x,y) as top-left corner,
       and keeps the same rowstride. This is an effective way to "clip" the rgb_buffer for gdk */
    gdk_draw_rgb_32_image (widget->window, widget->style->white_gc,
        event->area.x, event->area.y, event->area.width, event->area.height,
        GDK_RGB_DITHER_MAX,
        gp_vnc->rgb_buffer + (event->area.y * gp_vnc->width * 4 + event->area.x * 4),
        gp_vnc->width * 4);
    return TRUE;
}

static void
grdc_plug_vnc_class_init (GrdcPlugVncClass *klass)
{
    klass->parent_class.open_connection = grdc_plug_vnc_open_connection;
    klass->parent_class.close_connection = grdc_plug_vnc_close_connection;
    klass->parent_class.query_feature = grdc_plug_vnc_query_feature;
    klass->parent_class.call_feature = grdc_plug_vnc_call_feature;
}

static void
grdc_plug_vnc_destroy (GtkWidget *widget, gpointer data)
{
}

static void
grdc_plug_vnc_init (GrdcPlugVnc *gp_vnc)
{
    gp_vnc->drawing_area = gtk_drawing_area_new ();
    gtk_widget_show (gp_vnc->drawing_area);
    gtk_fixed_put (GTK_FIXED (gp_vnc), gp_vnc->drawing_area, 0, 0);

    gtk_widget_add_events (gp_vnc->drawing_area, GDK_POINTER_MOTION_MASK | GDK_BUTTON_PRESS_MASK
        | GDK_BUTTON_RELEASE_MASK | GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK);
    GTK_WIDGET_SET_FLAGS (gp_vnc->drawing_area, GTK_CAN_FOCUS);

    g_signal_connect (G_OBJECT (gp_vnc->drawing_area), "expose_event", G_CALLBACK (grdc_plug_vnc_on_expose), gp_vnc);
    g_signal_connect (G_OBJECT (gp_vnc), "destroy", G_CALLBACK (grdc_plug_vnc_destroy), NULL);

    gp_vnc->connected = FALSE;
    gp_vnc->running = FALSE;
    gp_vnc->auth_called = FALSE;
    gp_vnc->closed = FALSE;
    gp_vnc->grdc_file = NULL;
    gp_vnc->width = 0;
    gp_vnc->height = 0;
    gp_vnc->rgb_buffer = NULL;
    gp_vnc->client = NULL;
    gp_vnc->button_mask = 0;
    gp_vnc->thread = 0;
}

GtkWidget*
grdc_plug_vnc_new (void)
{
    return GTK_WIDGET (g_object_new (GRDC_TYPE_PLUG_VNC, NULL));
}

