/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <string.h>
#include <stdarg.h>
#include "grdcpixmaps.h"
#include "grdcappletmenuitem.h"

G_DEFINE_TYPE (GrdcAppletMenuItem, grdc_applet_menu_item, GTK_TYPE_IMAGE_MENU_ITEM)

#define IS_EMPTY(s) ((!s)||(s[0]==0))

static void
grdc_applet_menu_item_destroy (GrdcAppletMenuItem *item, gpointer data)
{
    g_free (item->filename);
    g_free (item->name);
    g_free (item->group);
    g_free (item->protocol);
    g_free (item->server);
}

static void
grdc_applet_menu_item_class_init (GrdcAppletMenuItemClass *klass)
{
}

static void
grdc_applet_menu_item_init (GrdcAppletMenuItem *item)
{
    item->filename = NULL;
    item->name = NULL;
    item->group = NULL;
    item->protocol = NULL;
    item->server = NULL;
    item->ssh_enabled = FALSE;
    g_signal_connect (G_OBJECT (item), "destroy", G_CALLBACK (grdc_applet_menu_item_destroy), NULL);
}

GtkWidget*
grdc_applet_menu_item_new (GrdcAppletMenuItemType item_type, ...)
{
    va_list ap;
    GrdcAppletMenuItem *item;
    GKeyFile *gkeyfile;
    gint icon_width;
    GdkPixbuf *pixbuf;
    GtkWidget *widget;

    va_start(ap, item_type);

    item = GRDC_APPLET_MENU_ITEM (g_object_new (GRDC_TYPE_APPLET_MENU_ITEM, NULL));

    item->item_type = item_type;

    gtk_icon_size_lookup (GTK_ICON_SIZE_MENU, &icon_width, NULL);

    switch (item_type)
    {
    case GRDC_APPLET_MENU_ITEM_FILE:
        item->filename = g_strdup (va_arg(ap, const gchar *));
        /* Load the file */
        gkeyfile = g_key_file_new ();
        if (!g_key_file_load_from_file (gkeyfile, item->filename, G_KEY_FILE_NONE, NULL)) return NULL;

        item->name = g_key_file_get_string (gkeyfile, "grdc", "name", NULL);
        item->group = g_key_file_get_string (gkeyfile, "grdc", "group", NULL);
        item->protocol = g_key_file_get_string (gkeyfile, "grdc", "protocol", NULL);
        item->server = g_key_file_get_string (gkeyfile, "grdc", "server", NULL);
        item->ssh_enabled = g_key_file_get_boolean (gkeyfile, "grdc", "ssh_enabled", NULL);

        g_key_file_free (gkeyfile);
        break;

    case GRDC_APPLET_MENU_ITEM_DISCOVERED:
        item->name = g_strdup (va_arg(ap, const gchar *));
        item->group = g_strdup (_("Discovered"));
        item->protocol = g_strdup ("VNC");
        break;

    case GRDC_APPLET_MENU_ITEM_QUICK:
        item->name = g_strdup (_("Quick Connect"));
        break;
    }

    va_end(ap);

    /* Create the label */
    widget = gtk_label_new (item->name);
    gtk_widget_show (widget);
    gtk_misc_set_alignment (GTK_MISC (widget), 0.0, 0.5);
    gtk_container_add (GTK_CONTAINER (item), widget);

    /* Create the image */
    if (item_type == GRDC_APPLET_MENU_ITEM_FILE || item_type == GRDC_APPLET_MENU_ITEM_DISCOVERED)
    {
        if (g_strcmp0 (item->protocol, "SFTP") == 0)
        {
            pixbuf = grdc_pixmap (GRDC_PIXMAP_TYPE_SSH, icon_width);
        }
        else
        {
            pixbuf = grdc_pixmap ((strncmp (item->protocol, "VNC", 3) == 0 ?
                (item->ssh_enabled ? GRDC_PIXMAP_TYPE_VNC_SSH : GRDC_PIXMAP_TYPE_VNC) :
                (item->ssh_enabled ? GRDC_PIXMAP_TYPE_RDP_SSH : GRDC_PIXMAP_TYPE_RDP)), icon_width);
        }
        widget = gtk_image_new_from_pixbuf (pixbuf);
        g_object_unref (pixbuf);
    }
    else
    {
        widget = gtk_image_new_from_stock (GTK_STOCK_JUMP_TO, GTK_ICON_SIZE_MENU);
    }
    gtk_widget_show (widget);
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), widget);

    if (item->server)
    {
        gtk_widget_set_tooltip_text (GTK_WIDGET (item), item->server);
    }

    return GTK_WIDGET (item);
}

gint grdc_applet_menu_item_compare (gconstpointer a, gconstpointer b, gpointer user_data)
{
    gchar *groups, *groupa, *groupb;
    GrdcAppletMenuItem *itema, *itemb;
    gint cmp;

    /* Passed in parameters are pointers to pointers */
    itema = GRDC_APPLET_MENU_ITEM (*((void**)a));
    itemb = GRDC_APPLET_MENU_ITEM (*((void**)b));

    /* Put ungrouped items to the last */
    if (IS_EMPTY (itema->group) && !IS_EMPTY (itemb->group)) return 1;
    if (!IS_EMPTY (itema->group) && IS_EMPTY (itemb->group)) return -1;
    if (!IS_EMPTY (itema->group) && !IS_EMPTY (itemb->group))
    {
        cmp = g_strcmp0 (itema->group, itemb->group);
        if (cmp != 0)
        {
            groups = (gchar*) user_data;
            groupa = g_strrstr (groups, itema->group);
            groupb = g_strrstr (groups, itemb->group);

            if (groupa != NULL && groupb != NULL)
            {
                if (groupa < groupb) return -1;
                if (groupa > groupb) return 1;
                return cmp;
            }
            else
            {
                if (groupa == NULL && groupb != NULL) return 1;
                if (groupa != NULL && groupb == NULL) return -1;
                return cmp;
            }
        }
    }

    return g_strcmp0 (itema->name, itemb->name);
}

