// -*- c++ -*-
//------------------------------------------------------------------------------
//                              CardView.cpp
//------------------------------------------------------------------------------
// $Id: CardView.cpp,v 1.33 2006/03/19 15:39:44 vlg Exp $
//
//  Copyright (c) 2004-2005 by Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//
// Date: Feb 15, 2004
//------------------------------------------------------------------------------

#include <gtkmm/image.h>
#include <gtkmm/label.h>
#include <gtkmm/box.h>
#include <gtkmm/alignment.h>
#include <gtkmm/button.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/table.h>
#include <gtkmm/messagedialog.h>

#include "Granule.h"
#include "MainWindow.h"
#include "Config.h"
#include "CardView.h"
#include "DeckView.h"
#include "Card.h"
#include "CardRef.h"
#include "EditControls.h"
#include "Intern.h"

CardView::
CardView (VCard* card_) : 
#ifdef IS_HILDON
	Hildon::AppView ("Card View"),
#endif
	m_card (card_),
	m_cancel_button (NULL),
	m_add_n_close_button (NULL),
	m_add_button    (NULL),
	m_help_button   (NULL),
	m_front_text    (NULL),
	m_back_text     (NULL),
	m_example_text  (NULL),
	m_edit_controls (NULL),
	m_syntax_error  (false),
	m_keep_open     (false),
	m_help_dialog   (*this)
{
	trace_with_mask("CardView::CardView(VCard&)",GUITRACE);

	Gtk::Image* image1;
	Gtk::Image* image2;
	Gtk::Image* image3;
	Gtk::Image* image4;
	Gtk::Image* image5;

	Gtk::HBox* hbox1;
	Gtk::HBox* hbox2;
	Gtk::HBox* hbox3;
	Gtk::HBox* hbox4;
	Gtk::HBox* hbox5;

	Gtk::Label* label1;
	Gtk::Label* label2;
	Gtk::Label* label3;
	Gtk::Label* label4;
	Gtk::Label* label5;

	Gtk::Alignment* alignment1;
	Gtk::Alignment* alignment2;
	Gtk::Alignment* alignment3;
	Gtk::Alignment* alignment4;
	Gtk::Alignment* alignment5;
	Gtk::Alignment* alignment6;
	Gtk::Alignment* alignment7;
	Gtk::Alignment* alignment8;

	Gtk::ScrolledWindow* front_scrollwin;
	Gtk::ScrolledWindow* backtext_scrollwin;
	Gtk::ScrolledWindow* example_scrollwin;

	Gtk::Label* back_label;
	Gtk::Label* example_label;
	Gtk::Label* front_label;

	Gtk::VBox* vbox_frontlabel;
	Gtk::VBox* vbox_backlabel;
	Gtk::VBox* vbox_examplelabel;

	Gtk::Table* elems_table;
	Gtk::VBox*  vbox_all;

#ifdef IS_HILDON
	set_size_request (672, 396);
#else
	set_icon_from_file (GRAPPDATADIR "/pixmaps/cardview_32x32.png");

	Gdk::Rectangle geom = CONFIG->get_card_view_geometry ();
	set_size_request (geom.get_width (), geom.get_height ());

	set_title ("Card View");
	set_resizable (true);
	set_modal (true);
	set_transient_for (*MAINWIN);
	property_window_position ().set_value (Gtk::WIN_POS_CENTER);
#endif

	/** <Clear> button
	 */
	image1 = Gtk::manage (new Gtk::Image (Gtk::StockID ("gtk-clear"), 
										  Gtk::IconSize (4)));
	label1     = Gtk::manage (new Gtk::Label (_("Clear")));
	hbox1      = Gtk::manage (new Gtk::HBox (false, 2));
	alignment1 = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));

	/** <Cancel> Button
	 */
	image2     = Gtk::manage (new Gtk::Image (Gtk::StockID ("gtk-cancel"), 
											  Gtk::IconSize (4)));
	label2          = Gtk::manage (new Gtk::Label (_("Cancel")));
	hbox2           = Gtk::manage (new Gtk::HBox (false, 2));
	alignment2      = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));
	m_cancel_button = Gtk::manage (new Gtk::Button ());

	/** <Add> Button
	 */
	image3      = Gtk::manage (new Gtk::Image (Gtk::StockID ("gtk-add"), 
												   Gtk::IconSize (4)));
	label3      = Gtk::manage (new Gtk::Label (_("Add")));
	hbox3       = Gtk::manage (new Gtk::HBox (false, 2));
	alignment3  = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));
	m_add_button = Gtk::manage (new Gtk::Button ());

	/** <Add'n'Close> Button
	 */
	image4      = Gtk::manage (new Gtk::Image (Gtk::StockID ("gtk-apply"), 
												   Gtk::IconSize (4)));
	label4      = Gtk::manage (new Gtk::Label (_("Ok")));
	hbox4       = Gtk::manage (new Gtk::HBox (false, 2));
	alignment4  = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));
	m_add_n_close_button = Gtk::manage (new Gtk::Button ());

	/** <Help> Button
	 */
	image5      = Gtk::manage (new Gtk::Image (Gtk::StockID ("gtk-help"), 
												   Gtk::IconSize (4)));
	label5      = Gtk::manage (new Gtk::Label (_("Help")));
	hbox5       = Gtk::manage (new Gtk::HBox (false, 2));
	alignment8  = Gtk::manage (new Gtk::Alignment (0.5, 0.5, 0, 0));
	m_help_button = Gtk::manage (new Gtk::Button ());

	/** Text
	 */
	m_front_text       = Gtk::manage (new Gtk::TextView);
	front_scrollwin    = Gtk::manage (new Gtk::ScrolledWindow);
	front_label        = Gtk::manage (new Gtk::Label (_("Front : ")));
	alignment6         = Gtk::manage (new Gtk::Alignment (0.84, 0.5, 0, 0));
	vbox_frontlabel    = Gtk::manage (new Gtk::VBox (false, 0));

	m_back_text        = Gtk::manage (new Gtk::TextView);
	backtext_scrollwin = Gtk::manage (new Gtk::ScrolledWindow);
	back_label         = Gtk::manage (new Gtk::Label (_("Back :")));
	alignment5         = Gtk::manage (new Gtk::Alignment (0.76, 0.5, 0, 0));
	vbox_backlabel     = Gtk::manage (new Gtk::VBox (false, 0));

	m_example_text     = Gtk::manage (new Gtk::TextView);
	example_scrollwin  = Gtk::manage (new Gtk::ScrolledWindow);
	example_label      = Gtk::manage (new Gtk::Label (_("Example : ")));
	alignment7         = Gtk::manage (new Gtk::Alignment (0.84, 0.5, 0, 0));
	vbox_examplelabel  = Gtk::manage (new Gtk::VBox (false, 0));

	elems_table = Gtk::manage (new Gtk::Table (2, 2, false));
	vbox_all = Gtk::manage (new Gtk::VBox (false, 0));

	m_front_text  ->modify_font (CONFIG->get_example_font  ());
	m_back_text   ->modify_font (CONFIG->get_example_font  ());
	m_example_text->modify_font (CONFIG->get_example_font  ());

	image1->set_alignment  (0.5,0.5);
	image1->set_padding    (0,0);
	label1->set_alignment  (0.5,0.5);
	label1->set_padding    (0,0);
	label1->set_justify    (Gtk::JUSTIFY_LEFT);
	label1->set_line_wrap  (false);
	label1->set_use_markup (false);
	label1->set_selectable (false);

	hbox1->pack_start (*image1, Gtk::PACK_SHRINK, 0);
	hbox1->pack_start (*label1, Gtk::PACK_SHRINK, 0);
	alignment1->add (*hbox1);

	image2->set_alignment  (0.5,0.5);
	image2->set_padding    (0,0);
	label2->set_alignment  (0.5,0.5);
	label2->set_padding    (0,0);
	label2->set_justify    (Gtk::JUSTIFY_LEFT);
	label2->set_line_wrap  (false);
	label2->set_use_markup (false);
	label2->set_selectable (false);
	hbox2->pack_start (*image2, Gtk::PACK_SHRINK, 0);
	hbox2->pack_start (*label2, Gtk::PACK_SHRINK, 0);
	alignment2->add (*hbox2);

	m_cancel_button->set_flags (Gtk::CAN_FOCUS);
	m_cancel_button->set_relief (Gtk::RELIEF_NORMAL);
	m_cancel_button->add (*alignment2);

	image3->set_alignment  (0.5,0.5);
	image3->set_padding    (0,0);
	label3->set_alignment  (0.5,0.5);
	label3->set_padding    (0,0);
	label3->set_justify    (Gtk::JUSTIFY_LEFT);
	label3->set_line_wrap  (false);
	label3->set_use_markup (false);
	label3->set_selectable (false);
	hbox3->pack_start (*image3, Gtk::PACK_SHRINK, 0);
	hbox3->pack_start (*label3, Gtk::PACK_SHRINK, 0);
	alignment3->add (*hbox3);

	m_add_button->set_flags (Gtk::CAN_FOCUS);
	m_add_button->set_relief (Gtk::RELIEF_NORMAL);
	m_add_button->add (*alignment3);

	image4->set_alignment  (0.5,0.5);
	image4->set_padding    (0,0);
	label4->set_alignment  (0.5,0.5);
	label4->set_padding    (0,0);
	label4->set_justify    (Gtk::JUSTIFY_LEFT);
	label4->set_line_wrap  (false);
	label4->set_use_markup (false);
	label4->set_selectable (false);
	hbox4->pack_start (*image4, Gtk::PACK_SHRINK, 0);
	hbox4->pack_start (*label4, Gtk::PACK_SHRINK, 0);
	alignment4->add (*hbox4);

	m_add_n_close_button->set_flags (Gtk::CAN_FOCUS);
	m_add_n_close_button->set_relief (Gtk::RELIEF_NORMAL);
	m_add_n_close_button->add (*alignment4);

	image5->set_alignment  (0.5,0.5);
	image5->set_padding    (0,0);
	label5->set_alignment  (0.5,0.5);
	label5->set_padding    (0,0);
	label5->set_justify    (Gtk::JUSTIFY_LEFT);
	label5->set_line_wrap  (false);
	label5->set_use_markup (false);
	label5->set_selectable (false);
	hbox5->pack_start (*image5, Gtk::PACK_SHRINK, 0);
	hbox5->pack_start (*label5, Gtk::PACK_SHRINK, 0);
	alignment8->add (*hbox5);

	m_help_button->unset_flags (Gtk::CAN_FOCUS); // Take help off the chain
	m_help_button->set_relief (Gtk::RELIEF_NORMAL);
	m_help_button->add (*alignment8);

	// Back Text
	m_back_text->set_flags (Gtk::CAN_FOCUS);
	m_back_text->set_border_width (3);
	m_back_text->set_editable (true);
	m_back_text->set_cursor_visible (true);
	m_back_text->set_pixels_above_lines (3);
	m_back_text->set_pixels_below_lines (0);
	m_back_text->set_pixels_inside_wrap (0);
	m_back_text->set_left_margin  (4);
	m_back_text->set_right_margin (4);
	m_back_text->set_indent (0);
	m_back_text->set_wrap_mode (Gtk::WRAP_WORD);
	m_back_text->set_justification (Gtk::JUSTIFY_LEFT);
	m_back_text->set_accepts_tab (false);
	m_back_text->get_buffer ()->set_text (_("Fill up the back of the card"));

	backtext_scrollwin->set_flags (Gtk::CAN_FOCUS);
	backtext_scrollwin->set_shadow_type (Gtk::SHADOW_NONE);
	backtext_scrollwin->set_policy (Gtk::POLICY_ALWAYS, Gtk::POLICY_ALWAYS);
	backtext_scrollwin->
		property_window_placement ().set_value (Gtk::CORNER_TOP_LEFT);
	backtext_scrollwin->add (*m_back_text);

	back_label->set_alignment (0.5,0.5);
	back_label->set_padding (0,0);
	back_label->set_justify (Gtk::JUSTIFY_LEFT);
	back_label->set_line_wrap (false);
	back_label->set_use_markup (false);
	back_label->set_selectable (false);
	alignment5->add (*back_label);
	vbox_backlabel->pack_start (*alignment5, Gtk::PACK_SHRINK, 0);

	// Example Text
	m_example_text->set_flags (Gtk::CAN_FOCUS);
	m_example_text->set_border_width (3);
	m_example_text->set_editable (true);
	m_example_text->set_cursor_visible (true);
	m_example_text->set_pixels_above_lines (3);
	m_example_text->set_pixels_below_lines (0);
	m_example_text->set_pixels_inside_wrap (0);
	m_example_text->set_left_margin  (4);
	m_example_text->set_right_margin (4);
	m_example_text->set_indent (0);
	m_example_text->set_wrap_mode (Gtk::WRAP_WORD);
	m_example_text->set_justification (Gtk::JUSTIFY_LEFT);
	m_example_text->set_accepts_tab (false);
	m_example_text->get_buffer ()->set_text (_("Fill up an example"));

	example_scrollwin->set_flags (Gtk::CAN_FOCUS);
	example_scrollwin->set_shadow_type (Gtk::SHADOW_NONE);
	example_scrollwin->set_policy (Gtk::POLICY_ALWAYS, Gtk::POLICY_ALWAYS);
	example_scrollwin->
		property_window_placement ().set_value (Gtk::CORNER_TOP_LEFT);
	example_scrollwin->add (*m_example_text);

	example_label->set_alignment (0.5,0.5);
	example_label->set_padding (0,0);
	example_label->set_justify (Gtk::JUSTIFY_RIGHT);
	example_label->set_line_wrap (false);
	example_label->set_use_markup (false);
	example_label->set_selectable (false);
	vbox_examplelabel->pack_start (*example_label, Gtk::PACK_SHRINK, 0);

	/** Add EditControls
	 */
	m_edit_controls = Gtk::manage(new EditControls(m_front_text->get_buffer()));
	vbox_examplelabel->pack_end (*m_edit_controls, Gtk::PACK_SHRINK, 0);
	
	m_front_text->signal_grab_focus ().connect (
		bind<Gtk::TextView*> (mem_fun (*m_edit_controls, 
									   &EditControls::on_focus_changed),
							  m_front_text));

	m_back_text->signal_grab_focus ().connect (
		bind<Gtk::TextView*> (mem_fun (*m_edit_controls, 
									   &EditControls::on_focus_changed),
							  m_back_text));

	m_example_text->signal_grab_focus ().connect (
		bind<Gtk::TextView*> (mem_fun (*m_edit_controls, 
									   &EditControls::on_focus_changed),
							  m_example_text));

#ifdef IS_HILDON
	/** Trap 'on virtual keyboard close' event to restore
	 *  original size of the dialog
	 */
	HILDONAPPWIN->signal_im_close ().connect (
		mem_fun (*this, &CardView::hildon_im_close_cb));
#endif

	/** Front Text
	 */
	m_front_text->set_flags (Gtk::CAN_FOCUS);
	m_front_text->set_border_width (3);
	m_front_text->set_editable (true);
	m_front_text->set_cursor_visible (true);
	m_front_text->set_pixels_above_lines (3);
	m_front_text->set_pixels_below_lines (0);
	m_front_text->set_pixels_inside_wrap (0);
	m_front_text->set_left_margin  (4);
	m_front_text->set_right_margin (4);
	m_front_text->set_indent (0);
	m_front_text->set_wrap_mode (Gtk::WRAP_WORD);
	m_front_text->set_justification (Gtk::JUSTIFY_LEFT);
	m_front_text->set_accepts_tab (false);
	m_front_text->get_buffer ()->set_text (_("Fill up the front of the card"));

	front_scrollwin->set_flags (Gtk::CAN_FOCUS);
	front_scrollwin->set_shadow_type (Gtk::SHADOW_NONE);
	front_scrollwin->set_policy (Gtk::POLICY_ALWAYS, Gtk::POLICY_ALWAYS);
	front_scrollwin->
		property_window_placement ().set_value (Gtk::CORNER_TOP_LEFT);
	front_scrollwin->add (*m_front_text);

	front_label->set_alignment (0.5,0.5);
	front_label->set_padding (0,0);
	front_label->set_justify (Gtk::JUSTIFY_LEFT);
	front_label->set_line_wrap (false);
	front_label->set_use_markup (false);
	front_label->set_selectable (false);
	alignment6->add (*front_label);
	vbox_frontlabel->pack_start (*alignment6, Gtk::PACK_SHRINK, 0);
	
	/**
	 *  Table holds elements together:
	 *
	 *  |----------------+------------------|
	 *  | vbox(label)    |  scrollwin(text) |
	 *  |----------------+------------------|
	 */
	elems_table->set_border_width (4);
	elems_table->set_row_spacings (8);
	elems_table->set_col_spacings (2);
	elems_table->set_name ("CardViewElemsTable");

	elems_table->attach (*vbox_frontlabel, 0, 1, 0, 1, 
						Gtk::FILL, Gtk::FILL, 0, 0);

	elems_table->attach (*front_scrollwin, 1, 2, 0, 1, 
						Gtk::EXPAND|Gtk::FILL, Gtk::FILL, 0, 0);

	elems_table->attach (*vbox_backlabel, 0, 1, 1, 2, 
						Gtk::FILL, Gtk::FILL, 0, 0);

	elems_table->attach (*backtext_scrollwin, 1, 2, 1, 2, 
						Gtk::EXPAND|Gtk::FILL, Gtk::FILL, 0, 0);

	elems_table->attach (*vbox_examplelabel, 0, 1, 2, 3, 
						Gtk::FILL, Gtk::EXPAND|Gtk::FILL, 0, 0);

	elems_table->attach (*example_scrollwin, 1, 2, 2, 3, 
						Gtk::FILL, Gtk::FILL, 0, 0);

	elems_table->attach (*alignment7, 0, 1, 3, 4, 
						 Gtk::FILL, Gtk::FILL, 0, 0);

#ifdef IS_HILDON
    /** Maemo has a narrow screen and thus requires
	 *  smart packing.
	 */
	Gtk::ScrolledWindow* scrollw = Gtk::manage (new Gtk::ScrolledWindow);
	scrollw->set_flags (Gtk::CAN_FOCUS);
	scrollw->set_shadow_type (Gtk::SHADOW_NONE);
	scrollw->set_policy (Gtk::POLICY_AUTOMATIC, Gtk::POLICY_ALWAYS);
	scrollw->property_window_placement ().set_value (Gtk::CORNER_TOP_LEFT);

	scrollw->add (*elems_table);
	vbox_all->pack_start (*scrollw);
#else                                      // Regular desktop 
	vbox_all->pack_start (*elems_table);
#endif

	/** Create and pack buttons
	 */
	Gtk::HBox* hbbox = Gtk::manage(new Gtk::HBox(false, 0));
	Gtk::HSeparator* hseparator = manage (new Gtk::HSeparator);
	Gtk::HButtonBox*  hbuttonbox = manage (new Gtk::HButtonBox);
    hbuttonbox->set_homogeneous ();
    hbuttonbox->set_spacing (8);
	hbuttonbox->set_layout (Gtk::BUTTONBOX_END);
	hbuttonbox->set_name ("CardView_CtrlButtonBox");

	hbuttonbox->pack_end   (*m_cancel_button,      false, false, 1);
	hbuttonbox->pack_end   (*m_add_button,         false, false, 1);
	hbuttonbox->pack_end   (*m_add_n_close_button, false, false, 1);

	hbbox->pack_start (*m_help_button, Gtk::PACK_SHRINK, 5);
	hbbox->pack_start (*hbuttonbox);

	vbox_all->pack_start (*hseparator, Gtk::PACK_SHRINK, 3);
	vbox_all->pack_start (*hbbox, Gtk::PACK_SHRINK, 5);

	add (*vbox_all);

	/** Setup signals
	 */
	m_help_button->signal_clicked ().connect (
		sigc::mem_fun (*this, &CardView::on_help_clicked));

	m_cancel_button->signal_clicked ().connect (
		sigc::mem_fun (*this, &CardView::on_cancel_clicked));

	m_add_button->signal_clicked ().connect (
		sigc::mem_fun (*this, &CardView::on_add_clicked));

	m_add_n_close_button->signal_clicked ().connect (
		sigc::mem_fun (*this, &CardView::on_addnclose_clicked));

    signal_size_allocate ().connect (
		sigc::mem_fun (*this, &CardView::size_allocate_cb));

	/** Fill up the form
	 */
	if (m_card != NULL) {
		m_front_text  ->get_buffer ()->set_text (m_card->get_question ());
		m_back_text   ->get_buffer ()->set_text (m_card->get_answer   ());
		m_example_text->get_buffer ()->set_text (m_card->get_example  ());
	}

	/** Set Widget names - very helpful with debugging!
	 */
	m_front_text->set_name    ("FrontText");
	m_back_text ->set_name    ("BackText");
	m_example_text->set_name  ("ExampleText");
	m_help_button->set_name   ("HelpButton");
	m_cancel_button->set_name ("CancelButton");
	m_add_button->set_name    ("AddButton");
	m_add_n_close_button->set_name ("AddNCloseButton");

	vbox_all->set_name ("CardView_VBoxAll");

	show_all_children ();
}

//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//                              Callbacks
//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

/**
 * Called by the following methods to edit an existing card:
 *
 *    - DeckPlayer::on_vcard_edit_clicked ()
 *    - DeckView::on_edit_clicked ()
 */
int
CardView::
run (bool disable_cancel_)
{
	trace_with_mask("CardView::run(bool)",GUITRACE);

	m_syntax_error = false;
	m_add_button->set_sensitive (false);
	if (disable_cancel_) {
		m_cancel_button->set_sensitive (false);
	}
	take_snapshot ();

#ifdef IS_HILDON
	GRANULE->register_appview (this);
	Gtk::Main::run (*HILDONAPPWIN);
#else
	Gtk::Main::run (*this);
#endif

	return (m_syntax_error ? Gtk::RESPONSE_REJECT : m_response);
}

/**
 * Handling the very first card is different from the rest.
 * The first card is created by the DeckView.
 * The rest of the cards are created here.
 *
 * Called by:
 *    
 *   - DeckView::on_add_clicked ()
 *
 * Possible values returned by run():
 *    <Cancel> - Gtk::RESPONSE_CANCEL
 *    <Add>    - Gtk::RESPONSE_APPLY
 *    <Ok>     - Gtk::RESPONSE_OK
 */
int
CardView::
run (DeckView& deckview_)
{
	trace_with_mask("CardView::run(DeckView&)",GUITRACE);

	int ret = 0;
	m_syntax_error = false;
	take_snapshot ();

#ifdef IS_HILDON
	GRANULE->register_appview (this);
	Gtk::Main::run (*HILDONAPPWIN);
#else
	set_transient_for (deckview_);
	Gtk::Main::run (*this);
#endif

	DL ((GRAPP,"Processing syntax_error and response...\n"));
	if (m_syntax_error) {
		DL ((GRAPP,"Caught a syntax error (RESPONSE_REJECT)\n"));
		delete m_card;
		m_response = Gtk::RESPONSE_REJECT;
		goto done;
	}

	if (m_response == Gtk::RESPONSE_CANCEL) {
		DL ((GRAPP,"Dialog cancelled (RESPONSE_CANCEL)\n"));
		delete m_card;
		goto done;
	}

	deckview_.add_new_card (dynamic_cast<Card*>(m_card));

	while (m_keep_open) 
	{
		DL ((GRAPP,"Looping ...\n"));
		m_card = new Card;

#ifdef IS_HILDON
		GRANULE->register_appview (this);
		Gtk::Main::run (*HILDONAPPWIN);
#else
		Gtk::Main::run (*this);
#endif
		if (m_syntax_error) {
			delete m_card;
			m_response = Gtk::RESPONSE_REJECT;
			break;
		}
		if (m_response == Gtk::RESPONSE_CANCEL) {
			delete m_card;
			break;
		}
		deckview_.add_new_card (dynamic_cast<Card*>(m_card));
	}

  done:
	return (m_response);
}

void
CardView::
on_cancel_clicked  ()
{
	trace_with_mask("CardView::on_cancel_clicked",GUITRACE);
	
	m_response = Gtk::RESPONSE_CANCEL;

#ifdef IS_HILDON
	GRANULE->unregister_appview (this);
#endif

	DL ((GRAPP,"Call Main::quit()\n"));
	Gtk::Main::quit ();
}

/** Fend off clicking Esc key on the keyboard.
	This often happens when I want to enter ~ (tilda) key, and instead
	press Esc. The effect is to close the editing window which leads to
	the loss of all data already entered. The only way to close dialog
	is to click on Cancel button.

	@param event_ Event data structure.
*/
bool
CardView::
on_key_press_event (GdkEventKey* event_)
{
	trace_with_mask("CardView::on_key_press_event",GEOM);
	DL((GEOM,"key_pressed = %d\n", event_->keyval));

	/** Trap and disable ESC key event.
		The values for all keysyms are found 
		in $prefix/gtk-2.0/gdk/gdkkeysyms.h
	*/
	if (event_->keyval == GDK_Escape) { 
		return false;
	}

    /** Keep processing keystroke
	 */
#ifdef IS_HILDON
	return Bin::on_key_press_event (event_);	
#else
	return Window::on_key_press_event (event_);
#endif
}


void
CardView::
on_addnclose_clicked  ()
{
	trace_with_mask("CardView::on_addnclose_clicked",GUITRACE);
	
	m_response = Gtk::RESPONSE_OK;

	if (!Granule::check_markup (m_front_text->get_buffer ()->get_text ())) {
		Gtk::MessageDialog err1 (
			_("FRONT field contains illegal markup tag(s)."),
			false, Gtk::MESSAGE_ERROR);
		err1.run ();
		m_syntax_error = true;
		return;
	}

	if (!Granule::check_markup (m_back_text->get_buffer ()->get_text ())) {
		Gtk::MessageDialog err2 (
			_("BACK field contains illegal markup tag(s)."),
			false, Gtk::MESSAGE_ERROR);
		err2.run ();
		m_syntax_error = true;
		return;
	}

	if (!Granule::check_markup (m_example_text->get_buffer ()->get_text ())) {
		Gtk::MessageDialog err2 (
			_("EXAMPLE field contains illegal markup tag(s)."),
			false, Gtk::MESSAGE_ERROR);
		err2.run ();
		m_syntax_error = true;
		return;
	}

	/** Transfer data from the widget to the card
	 */
	if (m_card != NULL && text_modified ()) {
		m_card->set_question (m_front_text  ->get_buffer ()->get_text ());
		m_card->set_answer   (m_back_text   ->get_buffer ()->get_text ());
		m_card->set_example  (m_example_text->get_buffer ()->get_text ());
	}
	m_keep_open = false;
	m_front_text->grab_focus ();

#ifdef IS_HILDON
	GRANULE->unregister_appview (this);

	DL ((GRAPP,"Call Main::quit()\n"));
	Gtk::Main::quit ();
#else	
	Gtk::Main::quit ();
#endif
}

void
CardView::
take_snapshot ()
{
	trace_with_mask("CardView::take_snapshot",GUITRACE);

	m_snapshot_front   = m_front_text  ->get_buffer ()->get_text ();
	m_snapshot_back    = m_back_text   ->get_buffer ()->get_text ();
	m_snapshot_example = m_example_text->get_buffer ()->get_text ();
}

bool
CardView::
text_modified ()
{
	return (m_snapshot_front   != m_front_text  ->get_buffer ()->get_text () ||
			m_snapshot_back    != m_back_text   ->get_buffer ()->get_text () ||
			m_snapshot_example != m_example_text->get_buffer ()->get_text ());
}

void
CardView::
on_add_clicked  ()
{
	trace_with_mask("CardView::on_add_clicked",GUITRACE);

	on_addnclose_clicked ();
	clear_dialog ();
	m_keep_open = true;
	m_response = Gtk::RESPONSE_APPLY;
}

void
CardView::
on_help_clicked  ()
{
	trace_with_mask("CardView::on_help_clicked",GUITRACE);

	m_help_dialog.show ();
}

void
CardView::
clear_dialog ()
{
	m_front_text  ->get_buffer ()->set_text ("");
	m_back_text   ->get_buffer ()->set_text ("");
	m_example_text->get_buffer ()->set_text ("");

	m_snapshot_front   = "";
	m_snapshot_back    = "";
	m_snapshot_example = "";
}

void
CardView::
size_allocate_cb (Gtk::Allocation& allocation_)
{
    trace_with_mask("CardView::size_allocate_cb",GEOM);

	CONFIG->set_card_view_geometry (allocation_.get_width (),
									allocation_.get_height ());
}

#ifdef IS_HILDON
void
CardView::
hildon_im_close_cb ()
{
    trace_with_mask("CardView(hildon)::hildon_im_close_cb",GUITRACE);

	set_size_request (672, 396);
}
#endif
