#include "config.h"

#include "gql++/driver-manager.h"
#include "gql++/driver.h"
#include "gql++/exception.h"

namespace GQL
{

using namespace std;

DriverManager::DriverManager(Yehia::PluginManager& pm, const string& area) 
    : pm_(pm), area_name_(area), auto_load_(true)
{
  Yehia::PluginNode::iterator it = pm_.plugins().find(area_name_ + ".drivers");
  if (it == pm.plugins().end())
    drivers_ = new Yehia::PluginNode("drivers");
  else
    drivers_ = new Yehia::PluginNode(it);
}

DriverManager::~DriverManager()
{
  delete drivers_;
}

Connection *DriverManager::get_connection(const string& url,
                                          const string& name,
                                          const string& passwd)
{
  Connection *conn;
  Yehia::PluginNode::iterator i;
  map<string, string> properties;
  string proto;
  string sub_proto;
  Driver *driver = 0;
  
  properties["password"] = passwd;
  properties["user"] = name;

  if (auto_load_)
  {
    string::size_type prot_end = url.find(':');
    string::size_type sub_prot_end = url.find(':', prot_end + 1);

    if (prot_end != string::npos && sub_prot_end != string::npos)
      sub_proto = url.substr(prot_end + 1, sub_prot_end - prot_end - 1);

     // Try to load the driver and register it
    register_driver(sub_proto);
  }
  
  conn = 0;
  for (i = drivers_->begin(); i != drivers_->end() && !conn; ++i)
  {
    if ((*i).plugin() && (driver = dynamic_cast<Driver *>((*i).plugin())) &&
        driver->accept_url(url))
    {
      conn = driver->connect(url, properties);
      break;
    }
  }
  
  if (!conn)
    throw SQLException("No suitable driver");
    
  return conn;
}

bool DriverManager::register_driver(const string& id)
{
  Driver *driver = get_driver(id);
  
  if (!driver)
    driver = dynamic_cast<Driver *>(
            pm_.load_plugin(area_name_ + ".drivers." + id));
  
  return driver != 0;
}

void DriverManager::deregister_driver(const string& id)
{
  Yehia::PluginNode::iterator it = drivers_->find(id);

  if (it == drivers_->end())
    throw SQLException(id + " not registered");

  pm_.release_plugin(area_name_ + ".drivers." + id);
}

list<string> DriverManager::get_registered_drivers() const
{
  list<string> result;
  Yehia::PluginNode::const_iterator it;

  for (it = drivers_->begin(); it != drivers_->end(); ++it)
    if ((*it).plugin())
      result.push_back((*it).name());
  
  return result;
}

Driver *DriverManager::get_driver(const string& id)
{
  Yehia::PluginNode::iterator it = drivers_->find(id);
  
  if (it == pm_.plugins().end())
    return 0;

  return dynamic_cast<Driver *>((*it).plugin());
}  

}

