#include "config.h"

#include "connection.h"
#include "statement.h"
#include "database-metadata.h"

#include "gql++/exception.h"

namespace GQL
{

namespace SQLite
{

using namespace std;
using SigC::manage;

SQLiteConnection::SQLiteConnection(const map<string, string>& info,
                                   const string& db,
                                   const string& url,
                                   SQLiteDriver *driver)
{
  char *errmsg = NULL;
  
  if (info.find("user") == info.end())
    throw SQLException("The user property is missing. It is manadatory");
  if (info.find("password") == info.end())
    throw SQLException("The password property is missing. It is manadatory");
  
  auto_commit_ = true;
  driver_ = driver;
  url_ = url;
  database_ = db;
  metadata_ = 0;

  conn_ = sqlite_open(db.c_str(), 0, &errmsg);

  if (conn_ == NULL)
  {
    string msg = string("Connection failed: ");
    if (errmsg != NULL)
    {
      msg += errmsg;
      free(errmsg);
    }
    else
      msg += "unknown reason";
    
    throw SQLException(msg);
  }

  exec_sql("PRAGMA empty_result_callbacks = ON");
  
  driver_->reference();
  
}

SQLiteConnection::~SQLiteConnection()
{
  if (!auto_commit_)
    commit();
  
  sqlite_close(conn_);
  
  driver_->unreference();
  
  if (metadata_)
    metadata_->unreference();
}

Statement *SQLiteConnection::create_statement()
{
  return manage(new SQLiteStatement(this));
}

PreparedStatement *SQLiteConnection::prepare_statement(const string& url)
{
  return 0;
}

CallableStatement * SQLiteConnection::prepare_call(const string& url)
{
  return 0;
}

void SQLiteConnection::set_auto_commit(bool auto_commit)
{
  if (auto_commit_ == auto_commit)
    return;

  if (auto_commit == false)
  {
    // SQLite is always in autocommit mode, one has to "BEGIN" a
    // transaction explicitly
    exec_sql("BEGIN TRANSACTION");
  }
  
  auto_commit_ = auto_commit;
}

void SQLiteConnection::commit()
{
  exec_sql("COMMIT");
  if (auto_commit_ == false)
    exec_sql("BEGIN TRANSACTION");
}

void SQLiteConnection::rollback()
{
  exec_sql("ROLLBACK");
  if (auto_commit_ == false)
    exec_sql("BEGIN TRANSACTION");
}

string SQLiteConnection::get_catalog() const
{
  string empty;
  
  return empty;
}

string SQLiteConnection::native_sql(const string& sql) const
{
  return sql;
}

DatabaseMetaData *SQLiteConnection::get_meta_data()
{
  if (!metadata_)
  {
    metadata_ = manage(new SQLiteDatabaseMetaData(this));
    metadata_->reference();
  }
  return metadata_;
}

SQLObject *SQLiteConnection::create_object()
{
  return manage(new BasicSQLObject());
}

namespace
{

struct CbData
{
    ModResultSet *result;
    SQLiteConnection *connection;
};

int sqlite_callback(void *data, int ncols, char **row, char **col_names)
{
  CbData *cbdata = (CbData *)data;
  
  if (cbdata->result == 0)
  {
    cbdata->result = manage(new ModResultSet(cbdata->connection, ncols));
    ModResultSetMetaData *mdata = manage(new ModResultSetMetaData(ncols));
    for (int i = 0; i < ncols; i++)
      mdata->set_column_name(i, col_names[i]);
    cbdata->result->set_meta_data(mdata);
  }
  if (row)
    cbdata->result->append(row);

  return 0;
}

}

ResultSet *SQLiteConnection::exec_sql(const string& sql)
{
  CbData cbdata;
  char *errmsg;
  int status;
  
  cbdata.result = 0;
  cbdata.connection = this;
  
  status = sqlite_exec(conn_, sql.c_str(), sqlite_callback, &cbdata, &errmsg);
  if (status != SQLITE_OK)
  {
    if (errmsg)
    {
      string error = errmsg;
      throw SQLException(error);
    }
    else
      throw SQLException("unknown error during statement excecution");
  }
  
  return cbdata.result;
}

}
}
