/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2003 Bert Vermeulen
    Copyright (C) 2002-2003 Peter Stamfest <peter@stamfest.at>

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: dn-browse.c,v 1.9 2003/11/02 07:20:43 stamfest Exp $ */


#include <glib.h>
#include <gtk/gtk.h>
#include <string.h>
#include <errno.h>		/* errno */
#include <stdio.h>		/* FILE */
#include <stdlib.h>		/* free - MUST get rid of malloc/free */

#include "config.h"

#include "common.h"
#include "dn-browse.h"
#include "ref-browse.h"


#include "input.h"		/* new_from_entry */
#include "search.h"		/* fill_out_search */
#include "template.h"		/* struct gq_template */
#include "formfill.h"		/* formlist_from_entry */

#include "tinput.h"		/* formfill_from_template */
#include "browse-dnd.h"		/* copy_entry et al */

#include "configfile.h"		/* config */
#include "errorchain.h"
#include "util.h"
#include "encode.h"
#include "i18n.h"
#include "utf8-compat.h"

#include "browse-export.h"

static void tree_row_search_below(GtkMenuItem *menuitem, struct tab *tab)
{
     GtkCTree *ctree;
     GtkCTreeNode *node;
     browse_entry *e;
     struct ldapserver *server;
     struct tab *search_tab;

     ctree = BROWSETAB(tab)->ctreeroot;
     node = BROWSETAB(tab)->tree_row_popped_up;
     e = (browse_entry *) gtk_ctree_node_get_row_data(ctree, node);

     assert(IS_DN_ENTRY(e));

     server = server_from_node(ctree, node);

     if (e == NULL || server == NULL)
	  return;

     search_tab = get_last_of_mode(SEARCH_MODE);
     if (!search_tab) {
	  new_modetab(&mainwin, SEARCH_MODE);
	  search_tab = get_last_of_mode(SEARCH_MODE);
     }

     if (search_tab) {
#if GTK_MAJOR >= 2
	  fill_out_search(search_tab, server, ((dn_browse_entry *)e)->dn);
#else
	  char *dn_dec = decoded_string(((dn_browse_entry *)e)->dn);
	  fill_out_search(search_tab, server, dn_dec);
	  if (dn_dec)
	       free(dn_dec);
#endif
     }
}

static GtkCTreeNode *ref_browse_single_add(const char *uri,
					   GtkCTree *ctree,
					   GtkCTreeNode *node)
{
     const char *labels[] = { NULL, NULL };
     char *dummy[] = { "dummy", NULL };
     ref_browse_entry *new_entry;
     GtkCTreeNode *new_item, *added = NULL;

     new_entry = (ref_browse_entry *) new_ref_browse_entry(uri);
     
     labels[0] = uri;

     added = gtk_ctree_insert_node(ctree,
				   node, NULL,
				   (char**) labels,  /* bug in the GTK2 API: should be const */
				   0,
				   NULL, NULL, NULL, NULL,
				   FALSE, FALSE);
     
     gtk_ctree_node_set_row_data_full(ctree,
				      added,
				      new_entry,
				      (GtkDestroyNotify) destroy_browse_entry);
     
     /* add dummy node */
     new_item = gtk_ctree_insert_node(ctree,
				      added, NULL,
				      dummy, 
				      0,
				      NULL, NULL, NULL, NULL,
				      TRUE, FALSE);

     return added;
}


static void browse_new_from_entry_callback(GtkMenuItem *widget,
					   dn_browse_entry *entry)
{
     if (IS_DN_ENTRY(entry)) {
	  char *dn = entry->dn;
	  int error_context = 
	       error_new_context(_("Creating new entry from existing entry"),
				 GTK_WIDGET(widget));
	  
	  struct ldapserver *server =
	       (struct ldapserver *) gtk_object_get_data(GTK_OBJECT(widget),
							 "server");
	  
	  new_from_entry(error_context, server, dn);

	  error_flush(error_context);
     }
}

static void browse_new_from_template_callback(GtkWidget *widget,
					      struct gq_template *template)
{
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     GList *formlist;
     struct ldapserver *server;
     struct inputform *iform;
     dn_browse_entry *entry;
     int error_context;

     server = (struct ldapserver *) gtk_object_get_data(GTK_OBJECT(widget),
							"server");
     entry = (dn_browse_entry *) gtk_object_get_data(GTK_OBJECT(widget),
						     "entry");
     if (!IS_DN_ENTRY(entry)) return;
     
     error_context = 
	  error_new_context(_("Creating now entry from template"), 
			    widget);
     
     
     iform = new_inputform();
     iform->dn = NULL;
     iform->server = server;
     ldapserver_ref(server);

     formlist = formfill_from_template(error_context, server, template);
     if(formlist) {
	  iform->formlist = formlist;
	  if (entry && entry->dn) {
	       /* don't need the RDN of the current entry */
	       char *newdn = g_malloc(strlen(entry->dn) + 2);
	       newdn[0] = ',';
	       newdn[1] = 0;
	       strcat(newdn, entry->dn);
	       iform->dn = newdn;
	  }

	  create_form_window(iform);
	  create_form_content(iform);

	  build_inputform(error_context, iform);
     } else {
	  free_inputform(iform);
     }

     error_flush(error_context);
#endif /* HAVE_LDAP_STR2OBJECTCLASS */
}

static void dump_subtree(GtkWidget *widget, struct tab *tab)
{
     GtkCTree *ctree;
     GtkCTreeNode *node;
     browse_entry *e;
     struct ldapserver *server;
     GList *to_export = NULL;
     struct dn_on_server *dos;
     int error_context;

     ctree = BROWSETAB(tab)->ctreeroot;
     node = BROWSETAB(tab)->tree_row_popped_up;
     e = (browse_entry *) gtk_ctree_node_get_row_data(ctree, node);

     assert(IS_DN_ENTRY(e));

     server = server_from_node(ctree, node);

     if (e == NULL || server == NULL)
	  return;

     error_context = error_new_context(_("Exporting entry to LDIF"),
				       tab->win->mainwin);

     dos = new_dn_on_server(((dn_browse_entry *)e)->dn, server);
     dos->flags = LDAP_SCOPE_SUBTREE; /* default is LDAP_SCOPE_BASE */
     to_export = g_list_append(to_export, dos);

     export_many(error_context, tab->win->mainwin, to_export);

     error_flush(error_context);
}

static void delete_browse_entry(GtkWidget *widget, struct tab *tab)
{
     GtkCTree *ctree;
     GtkCTreeNode *node;
     struct ldapserver *server;
     dn_browse_entry *entry;
     int do_delete;

     ctree = BROWSETAB(tab)->ctreeroot;
     node = BROWSETAB(tab)->selected_ctree_node;

     entry = (dn_browse_entry *) gtk_ctree_node_get_row_data(ctree, node);
     if (entry == NULL)
	  return;

     if ((server = server_from_node(ctree, node)) == NULL)
	  return;

     do_delete = 0;

     gtk_clist_freeze(GTK_CLIST(ctree));
     
     if (!entry->seen) {
	  /* toggle expansion twice to fire the expand callback */
	  gtk_ctree_toggle_expansion(ctree, node);
	  gtk_ctree_toggle_expansion(ctree, node);
     }

     if (entry->leaf) {
	  /* item is a leaf node */
	  do_delete = 1;
     } else {
	  /* maybe delete everything in the subtree as well?
	     should do another LDAP_SCOPE_SUBTREE search after
	     each batch of deletes, in case the server is limiting
	     the number of entries returned per search. This could
	     get hairy...

	     For now, just pop up a dialog box with a warning
	  */

	  do_delete = 
	       question_popup(_("Warning"),
			      _("This entry has a subtree!\n"
				"Do you want to delete every entry under it as well?"));
     }


     if (do_delete) {
	  int ctx = error_new_context(_("Deleting entry/subtree"), 
				      GTK_WIDGET(ctree));
	  if (delete_entry_full(ctx, server, entry->dn, TRUE)) {
	       browse_entry *p_entry;
	       GtkCTreeNode *parent = GTK_CTREE_ROW(node)->parent;

	       gtk_ctree_remove_node(ctree,
				     node);

	       /* the only thing left to do is to refresh the parent
                  node in order to get the leaf flag of that entry
                  right again */
	       p_entry = (browse_entry *) gtk_ctree_node_get_row_data(ctree, parent);
	       if (p_entry) {
		    assert(p_entry->base_methods);
		    if (p_entry->base_methods->refresh)
			 p_entry->base_methods->refresh(p_entry, ctx, ctree,
							parent, tab);
	       }
	  }
	  error_flush(ctx);
     }
     gtk_clist_thaw(GTK_CLIST(ctree));
}









/* 
 * Destructor for dn_browse_entry objects
 */
static void destroy_dn_browse_entry(browse_entry *e) 
{
     dn_browse_entry *entry;

     if (!e) return;

     assert(IS_DN_ENTRY(e));
     entry = DN_BROWSE_ENTRY(e);

     if (entry->dn) g_free(entry->dn);
     free(entry);
}


static void dn_browse_entry_expand(browse_entry *be,
				   int error_context,
				   GtkCTree *ctree,
				   GtkCTreeNode *node,
				   struct tab *tab)
{
     LDAP *ld = NULL;
     LDAPMessage *res = NULL, *e;
     struct ldapserver *server = NULL;
     int msg, rc, num_children, update_counter, err;
     char message[MAX_DN_LEN + 21];
     char *dummy[] = { "dummy", NULL };
     char *ref[] = { "ref", NULL };
     char *c, **refs;
     dn_browse_entry *entry;

     LDAPControl ct;
     LDAPControl *ctrls[2] = { NULL, NULL } ;

     assert(IS_DN_ENTRY(be));
     entry = DN_BROWSE_ENTRY(be);

     if (!entry->seen) {
	  server = server_from_node(ctree, node);
/*  	  printf("server=%08lx host=%s dn=%s\n", (long) server, */
/*  		 server->ldaphost, */
/*  		 entry->dn); */
	  
	  gtk_clist_freeze(GTK_CLIST(ctree));
	  
	  while (GTK_CTREE_ROW(node)->children) {
	       gtk_ctree_remove_node(ctree, GTK_CTREE_ROW(node)->children);
	  }

	  if( (ld = open_connection(error_context, server)) == NULL) {
	       gtk_clist_thaw(GTK_CLIST(ctree));
	       return;
	  }

#if HAVE_LDAP_CLIENT_CACHE
	  if (entry->uncache) {
	       ldap_uncache_entry(ld, entry->dn);
	       entry->uncache = FALSE;
	  }
#endif

#if GTK_MAJOR < 2
	  statusbar_msg(_("Onelevel search on %s"), 
			c = decoded_string(entry->dn));

	  if (c) free(c);
#else
	  statusbar_msg(_("Onelevel search on %s"), entry->dn);
#endif
	  
	  ct.ldctl_oid		= LDAP_CONTROL_MANAGEDSAIT;
	  ct.ldctl_value.bv_val	= NULL;
	  ct.ldctl_value.bv_len	= 0;
	  ct.ldctl_iscritical	= 1;
		    
	  ctrls[0] = &ct;
	  
	  /* check if this is a referral object */

	  rc = ldap_search_ext(ld, entry->dn,
			       LDAP_SCOPE_BASE, 
			       "(objectClass=referral)", ref, 0,
			       ctrls,		/* serverctrls */
			       NULL,		/* clientctrls */
			       NULL,		/* timeout */
			       LDAP_NO_LIMIT,	/* sizelimit */
			       &msg);

	  /* FIXME: THIS IS NOT CORRECT */
/* 	  if (rc == -1) { */
/* 	       statusbar_msg(_("Searching for '%1$s': %2$s"),  */
/* 			     entry->dn, */
/* 			     ldap_err2string(msg)); */
/* 	       close_connection(server, FALSE); */
/* 	       gtk_clist_thaw(GTK_CLIST(ctree)); */
	       
/* 	       error_flush(context); */
/* 	       return; */
	       
/* 	  } */

	  while((rc = ldap_result(ld, msg, 0,
				  NULL, &res)) == LDAP_RES_SEARCH_ENTRY) {
	       for(e = ldap_first_entry(ld, res) ; e != NULL ;
		   e = ldap_next_entry(ld, e)) {
		    char **vals = ldap_get_values(ld, e, "ref");
		    int i;
		    
		    if (vals == NULL) continue;
		    
		    for(i = 0; vals[i]; i++) {
			 entry->is_ref = TRUE; /* now we know for sure */
			 ref_browse_single_add(vals[i], ctree, node);
		    }
		    
		    if (vals)  /* redundant, but... */
			 ldap_value_free(vals);
	       }
	       if (res) ldap_msgfree(res);
	       res = NULL;
	  }
	  
	  if (res) ldap_msgfree(res);
	  res = NULL;

	  if (entry->is_ref) {
	       entry->seen = TRUE;
	       statusbar_msg(_("Showing referrals"));
	       gtk_clist_thaw(GTK_CLIST(ctree));
	       close_connection(server, FALSE);
	       return;
	  }





	  rc = ldap_search_ext(ld, entry->dn,
			       LDAP_SCOPE_ONELEVEL, 
			       "(objectClass=*)", dummy, 1,
			       ctrls,		/* serverctrls */
			       NULL,		/* clientctrls */
			       NULL,		/* timeout */
			       LDAP_NO_LIMIT,	/* sizelimit */
			       &msg);

	  /* FIXME */
/* 	  if(rc != LDAP_SUCCESS) { */
/* 	       statusbar_msg(_("Error while searching below '%1$s': %2$s"), */
/* 			     entry->dn, */
/* 			     ldap_err2string(msg)); */
/* 	       close_connection(server, FALSE); */
/* 	       gtk_clist_thaw(GTK_CLIST(ctree)); */

/* 	       error_flush(context); */
/* 	       return; */
/* 	  } */
	  
	  num_children = update_counter = 0;

	  while( (rc = ldap_result(ld, msg, 0,
				   NULL, &res)) == LDAP_RES_SEARCH_ENTRY) {
	       for(e = ldap_first_entry(ld, res) ; e != NULL ;
		   e = ldap_next_entry(ld, e)) {

		    char *dn = ldap_get_dn(ld, e);
		    dn_browse_single_add(dn, ctree, node);
		    if (dn) ldap_memfree(dn);

		    num_children++;
		    update_counter++;
		    if(update_counter >= 100) {
			 statusbar_msg(ngettext("One entry found (running)",
						"%d entries found (running)",
						num_children), num_children);

			 update_counter = 0;
		    }
	       }
	       ldap_msgfree(res);
	  }
	  entry->leaf = (num_children == 0);

	  snprintf(message, sizeof(message),
		   ngettext("One entry found (finished)",
			    "%d entries found (finished)", num_children), 
		   num_children);

	  ldap_get_option(ld, LDAP_OPT_ERROR_NUMBER, &rc);

	  if (rc == LDAP_SERVER_DOWN) {
	       server->server_down++;
	       gtk_clist_thaw(GTK_CLIST(ctree));
	       goto done;
	  }
	  
	  if (res) {
	       rc = ldap_parse_result(ld, res,
				      &err, &c, NULL, &refs, NULL, 0);
	  }

	  if (rc != LDAP_SUCCESS) {	
	       /* FIXME: better error message (but what is the exact cause?)*/
	       error_push(error_context, ldap_err2string(rc));
	       push_ldap_addl_error(ld, error_context);

	       if (rc == LDAP_SERVER_DOWN) {
		    server->server_down++;
	       }
	  } else {
	       if (err == LDAP_SIZELIMIT_EXCEEDED) {
		    int l = strlen(message);
		    snprintf(message + l, sizeof(message) - l, 
			     " - %s", _("size limit exceeded"));
	       } else if (err == LDAP_TIMELIMIT_EXCEEDED) {
		    int l = strlen(message);
		    snprintf(message + l, sizeof(message) - l, 
			     " - %s", _("time limit exceeded"));
	       } else if (err != LDAP_SUCCESS) {
		    error_push(error_context, ldap_err2string(err));
		    push_ldap_addl_error(ld, error_context);
		    if (c && strlen(c)) {
			 error_push(error_context, _("Matched DN: %s"), c);
		    }
		    if (refs) {
			 int i;
			 for (i = 0 ; refs[i] ; i++) {
			      error_push(error_context,
					 _("Referral to: %s"), refs[i]);
			 }
		    }
	       }
	  }

	  statusbar_msg(message);

	  gtk_clist_thaw(GTK_CLIST(ctree));

	  entry->seen = TRUE;
     }


     /* XXX the code that sets this is #if0'ed, so this is dead code...
     if (g_hash_table_lookup(hash, "expand-all")) {
	  GtkCTreeNode *n;
	  gtk_clist_freeze(GTK_CLIST(ctree));
	  for (n = GTK_CTREE_ROW(node)->children ; n ;
	       n = GTK_CTREE_NODE_NEXT(n)) {
	       gtk_ctree_expand(ctree, n);
	  }
	  gtk_clist_thaw(GTK_CLIST(ctree));
     }
     */

 done:
     if (res) ldap_msgfree(res);
     if (server && ld) close_connection(server, FALSE);
}

static void browse_edit_from_entry(browse_entry *e,
				   int error_context,
				   GtkCTree *ctreeroot,
				   GtkCTreeNode *ctreenode,
				   struct tab *tab)
{
     GList *oldlist, *newlist, *tmplist;
     GtkWidget *pane2_scrwin, *pane2_vbox;
     struct ldapserver *server;
     struct inputform *iform;
     char *dn;
/*      int hidden = 0; */
     dn_browse_entry *entry;

     assert(IS_DN_ENTRY(e));
     entry = DN_BROWSE_ENTRY(e);

     if (ctreenode == NULL)
	  return;

     if( (server = server_from_node(ctreeroot, ctreenode)) == NULL)
	  return;

     dn = entry->dn;
     record_path(tab, (browse_entry *) entry, ctreeroot, ctreenode);
     
     ctreeroot = BROWSETAB(tab)->ctreeroot;
#if 0
     /* delete old struct inputform (if any) */
     iform = BROWSETAB(tab)->inputform;
     if(iform) {
	  /* but first get current hide status */
	  hidden = iform->hide_status;

	  inputform_free(iform);
     }
#endif 
     iform = new_inputform();
     BROWSETAB(tab)->inputform = iform;

     iform->server = server;
     ldapserver_ref(server);

     iform->edit = 1;

     /* pass on old "hide" status */
     iform->hide_status = BROWSETAB(tab)->hidden;

     tmplist = NULL;
     oldlist = formlist_from_entry(error_context, server, dn, 0);
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     oldlist = add_schema_attrs(error_context, server, oldlist);
#endif

     if(oldlist) {
	  iform->oldlist = oldlist;
	  newlist = dup_formlist(oldlist);
	  iform->formlist = newlist;
	  iform->olddn = g_strdup(dn);
	  iform->dn = g_strdup(dn);

	  if (ctreeroot) {
	       iform->ctreeroot = ctreeroot;
	       iform->ctree_refresh = GTK_CTREE_ROW(ctreenode)->parent;
	  }

	  /* XXX should free etc first */
	  pane2_scrwin = BROWSETAB(tab)->pane2_scrwin;
	  gtk_container_remove(GTK_CONTAINER(pane2_scrwin),
			       GTK_BIN(pane2_scrwin)->child);

	  pane2_vbox = gtk_vbox_new(FALSE, 2);
	  iform->target_vbox = pane2_vbox;
	  BROWSETAB(tab)->pane2_vbox = pane2_vbox;

	  gtk_widget_show(pane2_vbox);
	  gtk_widget_set_parent_window(pane2_vbox, (mainwin.mainwin->window));
	  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin), pane2_vbox);

	  create_form_content(iform);
	  build_inputform(error_context, iform);
     } else {
	  inputform_free(iform);
	  BROWSETAB(tab)->inputform = NULL;
     }

}

static void dn_browse_entry_refresh(browse_entry *entry,
				    int error_context,
				    GtkCTree *ctree,
				    GtkCTreeNode *node,
				    struct tab *tab)
{
     assert(IS_DN_ENTRY(entry));

     refresh_subtree(error_context, ctree, node);
     DN_BROWSE_ENTRY(entry)->base_methods->select(entry, 
						  error_context, 
						  ctree, node, tab);
}

static char* dn_browse_entry_get_name(browse_entry *entry,
				      gboolean long_form)
{
     char **exploded_dn;
     char *g;
#if GTK_MAJOR < 2
     char *l;
#endif

     assert(IS_DN_ENTRY(entry));

     if (long_form) {
	  return g_strdup(DN_BROWSE_ENTRY(entry)->dn);
     } else {
	  exploded_dn = gq_ldap_explode_dn(DN_BROWSE_ENTRY(entry)->dn, FALSE);
	  
#if GTK_MAJOR >= 2
	  g = g_strdup(exploded_dn[0]);
#else
	  l = decoded_string(exploded_dn[0]);

	  /* impedance match -> malloc to g_malloc */
	  g = g_strdup(l);
	  free(l);
#endif
	  gq_exploded_free(exploded_dn);

	  return g;
     }
}

static void dn_browse_entry_popup(browse_entry *entry, GtkWidget *menu, 
				  GtkWidget *ctreeroot,
				  GtkCTreeNode *ctree_node,
				  struct tab *tab) 
{
     GtkWidget *menu_item, *submenu;
     struct ldapserver *server;
     int is_dn;
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     GList *templatelist;
     struct gq_template *template;
#endif

     assert(IS_DN_ENTRY(entry));

     is_dn = IS_DN_ENTRY(entry);
     
     if ((server = server_from_node(GTK_CTREE(ctreeroot), ctree_node)) == NULL)
	  return;
     
     /* New submenu */
     menu_item = gtk_menu_item_new_with_label(_("New"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     submenu = gtk_menu_new();
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item), submenu);
     gtk_widget_show(menu_item);

#ifdef HAVE_LDAP_STR2OBJECTCLASS

     templatelist = config->templates;
     while(templatelist) {
	  template = (struct gq_template *) templatelist->data;
	  menu_item = gtk_menu_item_new_with_label(template->name);
	  gtk_object_set_data_full(GTK_OBJECT(menu_item), "server",
				   server, (GtkDestroyNotify) ldapserver_unref);
	  ldapserver_ref(server);

	  gtk_object_set_data(GTK_OBJECT(menu_item), "entry", entry);
	  gtk_menu_append(GTK_MENU(submenu), menu_item);
	  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			     GTK_SIGNAL_FUNC(browse_new_from_template_callback),
			     (gpointer) template);
	  gtk_widget_show(menu_item);

	  templatelist = templatelist->next;
     }

#endif

     menu_item = gtk_menu_item_new_with_label(_("Use current entry"));
     gtk_object_set_data(GTK_OBJECT(menu_item), "server", server);
     gtk_menu_append(GTK_MENU(submenu), menu_item);
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			GTK_SIGNAL_FUNC(browse_new_from_entry_callback),
			(gpointer) entry);
     gtk_widget_show(menu_item);

#if 0
     /* Expand all */
     menu_item = gtk_menu_item_new_with_label(_("Expand all"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_show(menu_item);
	  
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			GTK_SIGNAL_FUNC(tree_row_expand_all),
			(gpointer) tab);
#endif

#if 0
     /* moved to server_browse_entry only. Just a clean-up of the
	menu, no technical reasons, just policy. */

     /* Close connection */
     menu_item = gtk_menu_item_new_with_label(_("Close Connection"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_show(menu_item);
	  
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			GTK_SIGNAL_FUNC(tree_row_close_connection),
			(gpointer) tab);

#endif

     /* Export to LDIF */
     menu_item = gtk_menu_item_new_with_label(_("Export to LDIF"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			GTK_SIGNAL_FUNC(dump_subtree),
			(gpointer) tab);
     gtk_widget_show(menu_item);

     menu_item = gtk_menu_item_new();
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_show(menu_item);

     /* Search below */
     menu_item = gtk_menu_item_new_with_label(_("Search below"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			GTK_SIGNAL_FUNC(tree_row_search_below),
			(gpointer) tab);
     gtk_widget_show(menu_item);

     menu_item = gtk_menu_item_new();
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_show(menu_item);


#ifdef BROWSER_DND
     /* Copy */
     menu_item = gtk_menu_item_new_with_label(_("Copy"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			(GtkSignalFunc) copy_entry,
			(gpointer) tab);

     if (!is_dn) {
	  gtk_widget_set_sensitive(menu_item, FALSE);
     }

     gtk_widget_show(menu_item);

     /* Copy all */
     menu_item = gtk_menu_item_new_with_label(_("Copy all"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			(GtkSignalFunc) copy_entry_all,
			(gpointer) tab);

     if (!is_dn) {
	  gtk_widget_set_sensitive(menu_item, FALSE);
     }
     gtk_widget_show(menu_item);

     /* Paste */
     menu_item = gtk_menu_item_new_with_label(_("Paste"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			(GtkSignalFunc) paste_entry,
			(gpointer) tab);

     if (!is_dn) {
	  gtk_widget_set_sensitive(menu_item, FALSE);
     }
     gtk_widget_show(menu_item);

     menu_item = gtk_menu_item_new();
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_show(menu_item);
#endif
     /* Delete */
     menu_item = gtk_menu_item_new_with_label(_("Delete"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			(GtkSignalFunc) delete_browse_entry,
			(gpointer) tab);

     if (!is_dn) {
	  gtk_widget_set_sensitive(menu_item, FALSE);
     }

     gtk_widget_show(menu_item);

}

static struct browse_entry_vtab dn_vtab = {
     destroy_dn_browse_entry,		/* destroy */
     dn_browse_entry_expand,		/* expand */
     browse_edit_from_entry,		/* select */
     dn_browse_entry_refresh,		/* refresh */
     dn_browse_entry_get_name,		/* get_name */
     dn_browse_entry_popup,		/* popup */
};


/* 
 * Constructor for dn_browse_entry objects taking the dn
 */
browse_entry *new_dn_browse_entry(const char *dn) 
{
     dn_browse_entry *e;
     e = g_malloc0(sizeof(dn_browse_entry));

     e->type = DN_BROWSE_ENTRY_ID;
     e->base_methods = &dn_vtab;

     if (dn != NULL) e->dn = g_strdup(dn);
     e->seen = FALSE;
     e->leaf = FALSE;

     return (browse_entry *) e;
}



/* 
   Local Variables:
   c-basic-offset: 5
   End:
*/
