#include "kifuFile.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaIOError.h"
#include "osl/record/csaString.h"
#include "osl/record/kisen.h"
#include "osl/record/kakinoki.h"
#include "osl/record/ki2.h"
#include "osl/record/ki2IOError.h"
#include "osl/record/usi.h"

#include <boost/scoped_ptr.hpp>

#include <qfileinfo.h>
#include <qdir.h>
#include <qstringlist.h>
#include <qtextcodec.h>
#include <qregexp.h>
#include <QDateTime>
#include <QTextStream>

#include <sstream>

void KifuFile::getRecordData(const osl::record::Record &record)
{
  moves.clear();
  time.clear();
  searchInfo.clear();
    
  osl::stl::vector<std::string> rawComments;
  record.getMoves(moves, time, rawComments, searchInfo);
  QTextCodec *codec = QTextCodec::codecForName("EUC-JP");
  const std::string black = record.getPlayer(osl::BLACK);
  const std::string white = record.getPlayer(osl::WHITE);
  playerName[playerToIndex(osl::BLACK)] = codec->toUnicode(black.c_str(), black.length());
  playerName[playerToIndex(osl::WHITE)] = codec->toUnicode(white.c_str(), white.length());
  std::string s = record.getInitialComment();
  if (! s.empty()) 
  {
    std::istringstream is(s);
    std::string line;
    while (std::getline(is, line))
      initial_comment.push_back(codec->toUnicode(line.c_str(),
						 line.size()));
  }
  for (size_t i = 0; i < rawComments.size(); i++)
  {
    const std::string& comment = rawComments[i];
    comments.push_back(codec->toUnicode(comment.c_str(),
					comment.length()));
  }

  boost::gregorian::date date = record.getDate();
  if (! date.is_special())
    year = date.year();
}



CsaFile::CsaFile(QString f)
  : KifuFile(f), loaded(false)
{
}

bool CsaFile::loadData()
{
  if (!loaded)
  {
    boost::scoped_ptr<osl::record::csa::CsaFile> csaFile;
    try
    {
      csaFile.reset(new osl::record::csa::CsaFile(filename.ascii()));
    }
    catch (osl::CsaIOError&)
    {
      return false;
    }
    state = csaFile->getInitialState();
    getRecordData(csaFile->getRecord());
    QFileInfo fileInfo(filename);
    lastLoadedTime = fileInfo.lastModified().toTime_t();
    loaded = true;
  }
  return true;
}

bool CsaFile::reloadIfChanged()
{
  QFile f(filename);
  QFileInfo fileInfo(f);
  uint last_time = fileInfo.lastModified().toTime_t();
  if (last_time > lastLoadedTime)
    lastLoadedTime = last_time;
  else
    return false;
  if (! f.open(IO_ReadOnly))
    return false;

  QTextStream stream(&f);
  QString content = stream.read();
  content.truncate(content.findRev('\n') + 1);
  osl::record::csa::CsaString csa(content.ascii());
  getRecordData(csa.getRecord());
  return true;
}

KifuFile *CsaFile::nextFile() const
{
  QFileInfo fileInfo(filename);
  QDir dir = fileInfo.dirPath(true);

  QString name = fileInfo.fileName();
  QStringList files = dir.entryList("*.csa");

  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    if (files[i] == name && i + 1 < (size_t)files.size())
    {
      return new CsaFile(dir.absPath() + "/" + files[i+1]);
    }
  }
  return NULL;
}

KifuFile *CsaFile::prevFile() const
{
  QFileInfo fileInfo(filename);
  QDir dir = fileInfo.dirPath(true);

  QString name = fileInfo.fileName();
  QStringList files = dir.entryList("*.csa");

  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    if (files[i] == name && i > 0)
    {
      return new CsaFile(dir.absPath() + "/" + files[i-1]);
    }
  }
  return NULL;
}


KakinokiFile::KakinokiFile(QString f)
  : KifuFile(f), loaded(false)
{
}

bool KakinokiFile::loadData()
{
  if (!loaded)
  {
    boost::scoped_ptr<osl::KakinokiFile> file;
    try
    {
      file.reset(new osl::KakinokiFile(filename.ascii()));
    }
    catch (osl::KakinokiIOError&)
    {
      return false;
    }
    state = file->getInitialState();
    getRecordData(file->getRecord());
    loaded = true;
  }
  return true;
}

KifuFile *KakinokiFile::nextFile() const
{
  QFileInfo fileInfo(filename);
  QDir dir = fileInfo.dirPath(true);

  QString name = fileInfo.fileName();
  QStringList files = dir.entryList("*.kif");

  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    if (files[i] == name && i + 1 < (size_t)files.size())
    {
      return new KakinokiFile(dir.absPath() + "/" + files[i+1]);
    }
  }
  return NULL;
}

KifuFile *KakinokiFile::prevFile() const
{
  QFileInfo fileInfo(filename);
  QDir dir = fileInfo.dirPath(true);

  QString name = fileInfo.fileName();
  QStringList files = dir.entryList("*.kif");

  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    if (files[i] == name && i > 0)
    {
      return new KakinokiFile(dir.absPath() + "/" + files[i-1]);
    }
  }
  return NULL;
}


Ki2File::Ki2File(QString f)
  : KifuFile(f), loaded(false)
{
}

bool Ki2File::loadData()
{
  if (!loaded)
  {
    boost::scoped_ptr<osl::Ki2File> file;
    try
    {
      file.reset(new osl::Ki2File(filename.ascii()));
    }
    catch (osl::Ki2IOError&)
    {
      return false;
    }
    state = file->getInitialState();
    getRecordData(file->getRecord());
    loaded = true;
  }
  return true;
}

KifuFile *Ki2File::nextFile() const
{
  QFileInfo fileInfo(filename);
  QDir dir = fileInfo.dirPath(true);

  QString name = fileInfo.fileName();
  QStringList files = dir.entryList("*.kif");

  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    if (files[i] == name && i + 1 < (size_t)files.size())
    {
      return new Ki2File(dir.absPath() + "/" + files[i+1]);
    }
  }
  return NULL;
}

KifuFile *Ki2File::prevFile() const
{
  QFileInfo fileInfo(filename);
  QDir dir = fileInfo.dirPath(true);

  QString name = fileInfo.fileName();
  QStringList files = dir.entryList("*.kif");

  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    if (files[i] == name && i > 0)
    {
      return new Ki2File(dir.absPath() + "/" + files[i-1]);
    }
  }
  return NULL;
}


KisenFile::KisenFile(QString f, int i)
  : KifuFile(f), index(i), loaded(false)
{
}

bool KisenFile::loadData()
{
  if (!loaded)
  {
    try
    {
      if (filename.endsWith(".kpf"))
      {
	osl::record::KisenPlusFile kisenPlusFile(filename.ascii());
	kisenPlusFile.getMoves(index, moves, time);
	state = kisenPlusFile.getInitialState();
      }
      else
      {
	osl::record::KisenFile kisenFile(filename.ascii());
	moves = kisenFile.getMoves(index);
	time =  osl::stl::vector<int>(moves.size(), 1);
	state = kisenFile.getInitialState();
      }
    }
    catch (osl::CsaIOError&)
    {
      return false;
    }
    QString ipxName(filename);
    ipxName.replace(QRegExp("\\.(kif|kpf)$"), ".ipx");
    osl::record::KisenIpxFile ipx(ipxName.ascii());
    QTextCodec *codec = QTextCodec::codecForName("euc-jp");
    std::string black = ipx.getPlayer(index, osl::BLACK)
      + ipx.getTitle(index, osl::BLACK);
    std::string white = ipx.getPlayer(index, osl::WHITE)
      + ipx.getTitle(index, osl::WHITE);
    playerName[playerToIndex(osl::BLACK)] = codec->toUnicode(black.c_str(), black.length());
    playerName[playerToIndex(osl::WHITE)] = codec->toUnicode(white.c_str(), white.length());
    boost::gregorian::date date = ipx.getStartDate(index);
    if (! date.is_special())
      year = date.year();
  }
  return true;
}

KifuFile *KisenFile::nextFile() const
{
  try
  {
    if (filename.endsWith(".kpf"))
    {
      osl::record::KisenPlusFile kisenPlusFile(filename.ascii());
      if (index < (int)kisenPlusFile.size() - 1)
      {
	return new KisenFile(filename, index + 1);
      }
    }
    else
    {
      osl::record::KisenFile kisenFile(filename.ascii());
      if (index < (int)kisenFile.size() - 1)
      {
	return new KisenFile(filename, index + 1);
      }
    }
  }
  catch (osl::CsaIOError&)
  {
  }
  return NULL;
}

KifuFile *KisenFile::prevFile() const
{
  try
  {
    if (filename.endsWith(".kpf"))
    {
      osl::record::KisenPlusFile kisenPlusFile(filename.ascii());
      if (index > 0)
      {
	return new KisenFile(filename, index - 1);
      }
    }
    else
    {
      osl::record::KisenFile kisenFile(filename.ascii());
      if (index > 0)
      {
	return new KisenFile(filename, index - 1);
      }
    }
  }
  catch (osl::CsaIOError&)
  {
  }
  return NULL;
}


UsiFile::UsiFile(QString f)
  : KifuFile(f), loaded(false)
{
}

bool UsiFile::loadData()
{
  if (!loaded)
  {
    boost::scoped_ptr<osl::UsiFile> file;
    try
    {
      file.reset(new osl::UsiFile(filename.ascii()));
    }
    catch (osl::record::usi::ParseError&)
    {
      return false;
    }
    state = file->getInitialState();
    getRecordData(file->getRecord());
    loaded = true;
  }
  return true;
}

KifuFile *UsiFile::nextFile() const
{
  QFileInfo fileInfo(filename);
  QDir dir = fileInfo.dirPath(true);

  QString name = fileInfo.fileName();
  QStringList files = dir.entryList("*.usi");

  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    if (files[i] == name && i + 1 < (size_t)files.size())
    {
      return new UsiFile(dir.absPath() + "/" + files[i+1]);
    }
  }
  return NULL;
}

KifuFile *UsiFile::prevFile() const
{
  QFileInfo fileInfo(filename);
  QDir dir = fileInfo.dirPath(true);

  QString name = fileInfo.fileName();
  QStringList files = dir.entryList("*.usi");

  for (size_t i = 0; i < (size_t)files.size(); i++)
  {
    if (files[i] == name && i > 0)
    {
      return new UsiFile(dir.absPath() + "/" + files[i-1]);
    }
  }
  return NULL;
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
