;;; Copyright (C) 2011 Team GPS.
;;; 
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;; 
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;; 
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

(ns twitter_clj.usi_converter
  (:import (java.util.concurrent LinkedBlockingQueue))
  (:require [clojure.string :as str]
            [clojure.java.io :as io]
            [clojure.contrib.command-line :as cmd]
            [clojure.contrib.logging :as log]))

;; ==================================================
;; Global variables
;; ==================================================

(def options (atom {}))

(def row-positions
  {"1" "a", "2" "b", "3" "c", "4" "d", "5" "e", "6" "f", "7" "g", "8" "h", "9" "i",
   "一" "a", "二" "b", "三" "c", "四" "d", "五" "e", "六" "f", "七" "g", "八" "h", "九" "i",
   1 "a", 2 "b", 3 "c", 4 "d", 5 "e", 6 "f", 7 "g", 8 "h", 9 "i"})
   
(def column-positions
  {"1" "1", "2" "2", "3" "3", "4" "4", "5" "5", "6" "6", "7" "7", "8" "8", "9" "9",
   "１" "1", "２" "2", "３" "3", "４" "4", "５" "5", "６" "6", "７" "7", "８" "8", "９" "9",
   1 "1", 2 "2", 3 "3", 4 "4", 5 "5", 6 "6", 7 "7", 8 "8", 9 "9"})

(def pieces
 {"歩" "p", "香" "l", "桂" "n", "銀" "s", "金" "g", 
  "角" "b", "馬" "+b", "飛" "r", "龍" "+r", "竜" "+r"})

(def moves-queue
  (LinkedBlockingQueue.))


;; ==================================================
;; Functions
;; ==================================================

(defn put-queue
  [str]
  (log/info (format "Pushing the queue: %s" str))
  (.put moves-queue str))


(defn parse-line
  [move-str from last-move]
  {:pre [(pos? (count move-str))]
   :post [(<= 4 (count %))]}
  (let [move (cond
               (= \同 (first move-str))
                 (str (get column-positions (str (subs from 0 1)))
                      (get row-positions    (str (subs from 1 2)))
                      (subs last-move 2 4))
               :else
                 (str (get column-positions (str (subs from 0 1)))
                      (get row-positions    (str (subs from 1 2)))
                      (get column-positions (str (subs move-str 0 1)))
                      (get row-positions    (str (subs move-str 1 2)))))]
   (cond
     (.contains move-str "不成")
       move
     (re-find #"成$" move-str)
       (str move "+")
     :else
       move)))


(defn parse-line-drop
  [move-str]
  {:pre [(.contains move-str "打")]}
  (str (str/upper-case (get pieces (subs move-str 2 3)))
       "*"
       (get column-positions (str (subs move-str 0 1)))
       (get row-positions    (str (subs move-str 1 2)))))


(defn parse-moves
  [lines]
  (loop [current-move 1
         lines lines
         ret []]
    (if-not (seq lines)
      ret ;; base
      (let [line (str/trim (first lines))]
        (cond
          (re-find #"^\*" line) ;; comment
            (recur current-move (rest lines) ret)
          (re-find #"^(\d+) (.*?)\((\d\d)\)" line)
            (let [[_0 _1 move-str from] (re-find #"^(\d+) (.*?)\((\d\d)\)" line)
                  move (parse-line move-str from (last ret))]
              (recur (inc current-move) (rest lines) (conj ret move)))
          (re-find #"^(\d+) (.*?打)" line)
            (let [[_0 _1 move-str] (re-find #"^(\d+) (.*?打)" line)
                  move (parse-line-drop move-str)]
              (recur (inc current-move) (rest lines) (conj ret move)))
          (re-find #"^(\d+) 投了" line)
            (conj ret "resign") ;; base
        :else
          (recur current-move (rest lines) ret))))))


(defn start-main-loop
  [nstart nend]
  (letfn [(main-loop [nstart nend]
            (let [contents (slurp (:file @options) :encoding "Shift_JIS")
                  moves (parse-moves (str/split-lines contents))]
              (log/debug (format "Read %d/%d moves..." (count moves) nstart))
              (cond
                (<= nend (count moves))
                  nil ;; base
                (< (count moves) nstart)
                  (let [sec 10]
                    (log/debug (format "Sleeping for %d seconds..." sec))
                    (Thread/sleep (* sec 1000))
                    (recur nstart nend))
                :else
                  (let [lines (for [i (range 0 (count (filter #(not= "resign" %) moves)))]
                                (format "position startpos moves %s"
                                        (str/join " " (take (inc i) moves))))]
                    (dorun (map put-queue (drop (dec nstart) lines)))
                    (if-not (= "resign" (last moves))
                      (recur (inc (count moves)) nend)
                      (put-queue "resign"))))))] ;; base
    (if-not (.exists (io/as-file (:file @options)))
      (do
        (log/fatal "Kifu file not found")
        (throw (java.io.FileNotFoundException. (:file @options))))
      (let [thread (Thread. #(main-loop nstart nend))]
        (.start thread)
        thread))))
       

(defn -main [& args]
  (cmd/with-command-line args
    "Convert a kifu file to usi moves"
    [[end    "ending moves" 350]
     [file f "kifu file" ""]
     [start  "starting moves" 35]
     remaining]
    (swap! options assoc :end   (Integer. end))
    (swap! options assoc :file  file)
    (swap! options assoc :start (Integer. start))
    (let [thread (start-main-loop (:start @options) (:end @options))]
      (loop []
        (let [s (.take moves-queue)]
          (println s)
          (if (= "resign" s)
            (.join thread) ;; base
            (recur)))))))

