#include "quiescenceTree.h"

#include "osl/search/simpleHashRecord.h"
#include "osl/search/simpleHashTable.h"
#include "osl/move_generator/legalMoves.h"
#include "osl/apply_move/applyMove.h"

#include "gpsshogi/gui/util.h"

#include <qmessagebox.h>
#if QT_VERSION >= 0x040000
#include <Q3PopupMenu>
#include <Q3Action>
#else
#include <qpopupmenu.h>
#include <qaction.h>
#define Q3PopupMenu QPopupMenu
#define Q3Action QAction
#endif
#include <sstream>

static bool hasChildAfterMove(const osl::SimpleState& state,
			      const osl::search::SimpleHashTable& table,
			      const osl::HashKey& child_key,
			      const osl::Move move)
{
  osl::NumEffectState child_state(state);
  osl::ApplyMoveOfTurn::doMove(child_state, move);
  osl::MoveVector moves;
  osl::LegalMoves::generate(child_state, moves);
  for (size_t i=0; i<moves.size(); ++i)
    if (table.find(child_key.newHashWithMove(moves[i])))
      return true;
  return false;
}

class QuiescenceItem : public MoveTreeItem
{
public:
  QuiescenceItem(QuiescenceTree *parent, osl::Move m,
		 bool bestMove,
		 const osl::search::SimpleHashRecord *record,
		 const osl::hash::HashKey& key,
		 const osl::search::SimpleHashTable *table)
    : MoveTreeItem(parent, m),
      bestMove(bestMove), record(record), key(key), table(table),
      parent_item(0), tree(parent)
  {
    setUpText(m, record);
  }
  QuiescenceItem(QuiescenceItem *parent, osl::Move m,
		 bool bestMove,
		 const osl::search::SimpleHashRecord *record,
		 const osl::hash::HashKey& key,
		 const osl::search::SimpleHashTable *table)
    : MoveTreeItem(parent, m),
      bestMove(bestMove), record(record), key(key), table(table),
      parent_item(parent), tree(parent->tree)
  {
    setUpText(m, record);
  }
  void paintCell(QPainter *p, const QColorGroup &cg,
		 int column, int width, int align);
  void setOpen(bool);
  bool isBestMove() const {
    return bestMove;
  }
  const osl::search::SimpleHashRecord *getRecord() const {
    return record;
  }

  void getMovesToCurrent(osl::MoveVector& moves) 
  {
    QuiescenceItem *i = this;
    while (i) {
      moves.push_back(i->getMove());
      i = i->parent_item;
    }
    std::reverse(moves.begin(), moves.end());
  }
private:
  void setUpText(osl::Move, const osl::search::SimpleHashRecord *record);
  const char *staticValueType();
  QString staticValue();

  bool bestMove;
  const osl::search::SimpleHashRecord *record;
  const osl::hash::HashKey key;
  const osl::search::SimpleHashTable *table;

  QuiescenceItem *parent_item;
  QuiescenceTree *tree;
};

void QuiescenceItem::paintCell(QPainter *p, const QColorGroup &cg,
			       int column, int width, int align)
{
  if (column == 0)
  {
    QColorGroup colorGroup(cg);
    QColor bestColor(colorGroup.text());
    if (bestMove)
    {
      bestColor =  QColor("blue");
    }
    colorGroup.setColor(QColorGroup::Text, bestColor);
    MoveTreeItem::paintCell(p, colorGroup, column, width, align);
  }
  else
    MoveTreeItem::paintCell(p, cg, column, width, align);
}

void QuiescenceItem::setUpText(osl::Move move,
			       const osl::search::SimpleHashRecord *record)
{
  int column = 0;
  setText(column++, (move.isInvalid() ? "ROOT"
		     : gpsshogi::gui::Util::moveToString(move)));
  if (!record)
    return;
  setText(column++, record->qrecord.lowerDepth() >= 0 ?
	  QString("%1").arg(record->qrecord.lowerBound()) : "*");
  setText(column++, record->qrecord.upperDepth() >= 0 ?
	  QString("%1").arg(record->qrecord.upperBound()) : "*");
  setText(column++, QString("%1").arg(record->qrecord.lowerDepth()));
  setText(column++, QString("%1").arg(record->qrecord.upperDepth()));
  setText(column++, staticValueType());
  setText(column++, staticValue());
}

const char *QuiescenceItem::
staticValueType()
{
  if (! record)
    return "*";
  return osl::search::QuiescenceRecord::
    toString(record->qrecord.staticValueType());
}

QString QuiescenceItem::
staticValue()
{
  if (! record)
    return "*";
  return (record->qrecord.hasStaticValue()
	  ? QString("%1").arg(record->qrecord.staticValue()) 
	  : "*");
}

void QuiescenceItem::setOpen(bool open)
{
  if (open && depth() >= 32)
  {
    setExpandable(false);
    return;
  }

  if (open && ! childCount() && record)
  {
    listView()->setUpdatesEnabled(false);

    const osl::search::QuiescenceRecord& qrecord = record->qrecord;
    osl::MoveVector history, moves;
    getMovesToCurrent(history);
    osl::NumEffectState state(tree->initialState());
    for (size_t i=0; i<history.size(); ++i)
      osl::ApplyMoveOfTurn::doMove(state, history[i]);
    osl::move_generator::LegalMoves::generate(state, moves);
  
    const osl::HashKey key(state);

    for (size_t i = 0; i < moves.size(); i++)
    {
      QuiescenceItem *item;
      const osl::Move move = moves[i];
      const osl::hash::HashKey newKey = key.newHashWithMove(move);
      const osl::search::SimpleHashRecord *newRecord = table->find(newKey);

      if (! newRecord) 
	continue;
      
      item = new QuiescenceItem(this, move,
				qrecord.bestMove() == move,
				newRecord,
				newKey,
				table);
      item->setExpandable(hasChildAfterMove(state, *table, 
					    newKey, move));
    }
    listView()->setUpdatesEnabled(true);
  }
  Q3ListViewItem::setOpen(open);
}


QuiescenceTree::QuiescenceTree(QWidget *parent, const char *name)
  : MoveTree(parent, name)
{
  addColumn("Move");
  addColumn("Lower Bound");
  addColumn("Upper Bound");
  addColumn("Lower Limit");
  addColumn("Upper Limit");
  addColumn("Static Value Type");
  addColumn("Static Value");

  for (int i = 1; i < columns(); i++)
  {
    setColumnAlignment(i, Qt::AlignRight);
    setColumnWidth(i, columnWidth(i) / 2);
  }
}

void QuiescenceTree::showRecord(const osl::hash::HashKey& key, 
				const osl::SimpleState& state,
				const osl::search::SimpleHashTable *table)
{
  clear();
  initial_state = state;
  const osl::search::SimpleHashRecord *record = table->find(key);

  addItem(0, 0, key, initial_state, table, record);
  // Root
  if (record)
  {
    new QuiescenceItem(this,
		       osl::Move::INVALID(),
		       false,
		       record,
		       key,
		       table); // dummy key
  }
}

void QuiescenceTree::addItem(int depth, QuiescenceItem *parent,
			     const osl::hash::HashKey& key,
			     const osl::SimpleState& state,
			     const osl::search::SimpleHashTable *table,
			     const osl::search::SimpleHashRecord *record)
{
  if (depth > osl::search::QSearchTraits::MaxDepth)
    return;
  if (!record)
    return;

  const osl::search::QuiescenceRecord& qrecord = record->qrecord;
  osl::MoveVector moves;
  osl::NumEffectState initial(initial_state);
  osl::LegalMoves::generate(initial, moves);

  for (size_t i = 0; i < moves.size(); i++)
  {
    QuiescenceItem *item;
    const osl::Move move = moves[i];
    const osl::hash::HashKey newKey = key.newHashWithMove(move);
    const osl::search::SimpleHashRecord *newRecord = table->find(newKey);
    if (! newRecord)
      continue;
    if (parent)
      item = new QuiescenceItem(parent, move,
				qrecord.bestMove() == move,
				newRecord,
				newKey,
				table);
    else
      item = new QuiescenceItem(this, move,
				qrecord.bestMove() == move,
				newRecord,
				newKey,
				table);
    item->setExpandable(hasChildAfterMove(state, *table, 
					  newKey, move));
  }
}

void QuiescenceTree::showContextMenu(Q3PopupMenu * contextMenu)
{
  contextMenu->insertSeparator();
  Q3Action *showRecordAction = new Q3Action("Dump Record Data", 0,
					    contextMenu);
  showRecordAction->addTo(contextMenu);
  connect(showRecordAction, SIGNAL(activated()),
	  this, SLOT(showRecord()));
}

void QuiescenceTree::showRecord()
{
  QuiescenceItem *item = (QuiescenceItem *) selectedItem();
  if (!item || !item->getRecord())
    return;

  std::ostringstream oss(std::ostringstream::out);
  const osl::search::QuiescenceRecord& record = item->getRecord()->qrecord;
  record.dump(oss);
  const std::string &record_string = oss.str();
  QMessageBox::information(this, "QuiescenceRecord Dump",
			   QString(record_string.c_str()), QMessageBox::Ok);
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
