#include "networkViewer.h"
#include "moveList.h"

#include "gpsshogi/gui/util.h"
#include "gpsshogi/gui/board.h"

#include "osl/apply_move/applyMove.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/record/csaIOError.h"
#include "osl/apply_move/applyMove.h"

#include <qmessagebox.h>
#include <qlayout.h>
#include <qstring.h>
#include <qlayout.h>
#include <iostream>

using gpsshogi::gui::Util;
/* ======================================================================
 * NetworkViewer
 *  the viewer, with the board on the left and chat screen on the right
 */
NetworkViewer::
NetworkViewer(QWidget *parent, const char *name)
  : BoardTabChild(parent, name), index(0)
{
  board->setManualMovement(false);

  moveList = new MoveList(this);
  QHBoxLayout *mainLayout = new QHBoxLayout(this);
  mainLayout->addWidget(moveList);
  mainLayout->addWidget(board);

  networkClient = new NetworkClient(this);
  QObject::connect(networkClient, SIGNAL(gotBoard(QString)),
		   SLOT(receive_board(QString)));
  QObject::connect(networkClient, SIGNAL(gotLastMove(QString)),
		   SLOT(receive_last_move(QString)));
  QObject::connect(networkClient, SIGNAL(chatReceived()),
		   this, SIGNAL(chatReceived()));

  QVBoxLayout *sideLayout = new QVBoxLayout(mainLayout);

  sideLayout->addWidget(networkClient);
}

void
NetworkViewer::
setHostname(QString str)
{
  networkClient->setHostname(str);
}

void
NetworkViewer::
setUsername(QString str)
{
  networkClient->setUsername(str);
}

void
NetworkViewer::
setPassword(QString str)
{
  networkClient->setPassword(str);
}

void
NetworkViewer::
connect()
{
  networkClient->openConnection();
}

void
NetworkViewer::
receive_board(QString content)
{
  try {
    osl::record::csa::CsaString csa(content.ascii());
    osl::stl::vector<osl::Move> moves;
    osl::stl::vector<int> consumed_time;
    csa.getRecord().getMoves(moves, consumed_time);

    initialState = csa.getInitialState();
    board->setState(csa.getInitialState());
    moveList->setState(initialState, moves, consumed_time, osl::vector<QString>());
  } catch (osl::CsaIOError& e) {
    std::cerr << "receive_board " << e.what() << "\n" << content.ascii() << "\n";
  }
}

void
NetworkViewer::
receive_last_move(QString content)
{
  osl::Move move = osl::record::csa::strToMove(content.ascii(), board->getState());
  moveList->pushMove(move);
  toLastState();
}

void NetworkViewer::forward()
{
  if (index > (int)moveList->numMoves())
    index++;

  updateState();
}

void NetworkViewer::backward()
{
  if (index > 0)
    index--;

  updateState();
}

void NetworkViewer::updateState()
{
  osl::state::SimpleState state(initialState);
  for (int i = 0; i < index; i++)
  {
    osl::ApplyMoveOfTurn::doMove(state, moveList->getMove(i));
  }
  board->setState(state);
}

void NetworkViewer::toLastState()
{
  index = moveList->numMoves();
  updateState();
}

int NetworkViewer::moveCount() const
{
  return index;
}

osl::state::SimpleState NetworkViewer::
getStateAndMovesToCurrent(osl::stl::vector<osl::Move> &moves)
{
  moves.reserve(index);
  for (int i = 0; i < index; i++)
    moves.push_back(moveList->getMove(i));

  return initialState;
}

void NetworkViewer::paintEvent(QPaintEvent *event)
{
  emit painted();
  BoardTabChild::paintEvent(event);
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
