/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*----------------------------------------------------------------------
 
  gpiv - Graphic program for Particle Image Velocimetry, based on gtk/gnome
          libraries.

   Copyright (C) 2002 Gerber van der Graaf

   This file is part of gpiv.

   Gpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

----------------------------------------------------------------------*/

/*
 * utility functions for gpiv
 * $Log: utils.c,v $
 * Revision 1.15  2006/01/29 15:36:56  gerber
 * repaired a few annoying bugs
 *
 * Revision 1.13  2005/03/03 16:21:09  gerber
 * version number
 *
 * Revision 1.12  2005/02/26 09:43:31  gerber
 * parameter flags (parameter_logic) defined as gboolean
 *
 * Revision 1.11  2005/02/26 09:17:14  gerber
 * structured of interrogate function by using gpiv_piv_isiadapt
 *
 * Revision 1.10  2005/01/19 15:53:43  gerber
 * Initiation of Data Acquisition (DAC); trigerring of lasers and camera
 * by using RTAI and Realtime Linux, recording images from IEEE1394
 * (Firewire) IIDC compliant camera's
 *
 * Revision 1.9  2004/10/15 19:24:05  gerber
 * GPIV_ and Gpiv prefix to defines and structure names of libgpiv
 *
 * Revision 1.8  2004/06/14 21:19:23  gerber
 * Image depth up to 16 bits.
 * Improvement "single int" and "drag int" in Eval tab.
 * Viewer's pop-up menu.
 * Adaption for gpiv_matrix_* and gpiv_vector_*.
 * Resizing console.
 * See Changelog for further info.
 *
 * Revision 1.7  2003/09/04 13:31:55  gerber
 * init of printing (unfinished)
 *
 * Revision 1.6  2003/09/01 11:17:15  gerber
 * improved monitoring of interrogation process
 *
 * Revision 1.5  2003/08/22 15:24:53  gerber
 * interactive spatial scaling
 *
 * Revision 1.4  2003/07/31 11:43:26  gerber
 * display images in gnome canvas (HOERAreset)
 *
 * Revision 1.3  2003/07/25 15:40:24  gerber
 * removed/disabled setting of correlation in Eval tab, Correlation type in Image info tab
 *
 * Revision 1.2  2003/06/27 13:47:26  gerber
 * display ruler, line/point evaluation
 *
 * Revision 1.1.1.1  2003/06/17 17:10:52  gerber
 * Imported gpiv
 *
 */

#include "gpiv_gui.h"
#include "utils.h"
#include "display.h"
#include "dialog_interface.h"
#include "console_interface.h"


char *
month_name(GDateMonth month)
/*--------------------------------------------------------------------
 * returns the month name
 */
{
    gchar *name = NULL;

    if (month == G_DATE_JANUARY) {
        name = g_strdup ( _("January"));
    } else if (month == G_DATE_FEBRUARY) {
        name = g_strdup ( _("February"));
    } else if (month == G_DATE_MARCH) {
        name = g_strdup ( _("March"));
    } else if (month == G_DATE_APRIL) {
        name = g_strdup ( _("April"));
    } else if (month == G_DATE_MAY) {
        name = g_strdup ( _("May"));
    } else if (month == G_DATE_JUNE) {
        name = g_strdup ( _("June"));
    } else if (month == G_DATE_JULY) {
        name = g_strdup ( _("July"));
    } else if (month == G_DATE_AUGUST) {
        name = g_strdup ( _("August"));
    } else if (month == G_DATE_SEPTEMBER) {
        name = g_strdup ( _("September"));
    } else if (month == G_DATE_OCTOBER) {
        name = g_strdup ( _("October"));
    } else if (month == G_DATE_NOVEMBER) {
        name = g_strdup ( _("November"));
    } else if (month == G_DATE_DECEMBER) {
        name = g_strdup ( _("December"));
    } else if (month == G_DATE_BAD_MONTH) {
        name = g_strdup ( _("Mad month"));
    } else {
        name = g_strdup ( _("month_name: Unvalid month"));
    }

    return name;
}
void 
free_all_mems(/* GpivConsole *gpiv */)
/*--------------------------------------------------------------------
 * Frees all allocated memory data from the application
 */
{
    gint i = 0;

    for (i = 1; i < gpiv_var.number_fnames_last; i++) {
        g_free(gpiv_var.fn_last[i]);
    }
}



void
push_list_lastfnames(gchar *fname
                     )
/*--------------------------------------------------------------------
 * Pushes fname on gpiv_var fn_last[0]
 * Eventually removes fn_last[MAX_LIST]
 */
{
    gchar fname_nr[GPIV_MAX_CHARS];
    gint i;

    gnome_config_push_prefix("/gpiv/RuntimeVariables/");
    if (gpiv_var.number_fnames_last < MAX_LIST) {
        gpiv_var.number_fnames_last++;
        gnome_config_set_int("number_fnames_last", 
                             gpiv_var.number_fnames_last);
    }

    for (i = gpiv_var.number_fnames_last - 1; i > 0; i--) {
         gpiv_var.fn_last[i] = g_strdup(gpiv_var.fn_last[i - 1]);
    }

    gpiv_var.fn_last[0] = g_strdup(fname);
    for (i = 0; i < gpiv_var.number_fnames_last; i++) {
         g_snprintf(fname_nr, GPIV_MAX_CHARS,"fname_last_%d", i);
         gnome_config_set_string(fname_nr, gpiv_var.fn_last[i]);
    }
    if (verbose)
        g_message("push_list_lastfnames: fn_last[0]=%s", gpiv_var.fn_last[0]);

    gnome_config_pop_prefix();
    gnome_config_sync();
}



GnomeCanvasItem
**alloc_gci_matrix(long nr,
		   long nc)
/*-----------------------------------------------------------------------------
 * Allocates 2-dimensional array for GnomeCanvasItem
 */
{
  long i;
  GnomeCanvasItem **item;


/*
 * allocate pointers to rows
 */
    item = (GnomeCanvasItem **) g_malloc((size_t)((nr - 1) *
                               sizeof(GnomeCanvasItem*)));
    if (!item) gpiv_error("%s: allocation failure 1 in g_malloc()", 
		       "alloc_gci_matrix");
    item += 1;
    
/*
 * allocate rows and set pointers to them
 */
    item[0] = (GnomeCanvasItem *) g_malloc((size_t)((nr * nc - 1) *
                                   sizeof(GnomeCanvasItem)));
    if (!item[0]) gpiv_error("%s: allocation failure 2 in g_malloc()", 
			    "alloc_gci_matrix");
    item[0] += 1;
    
    for(i = 1; i <= nr; i++) item[i] = item[i-1] + 1;
    
/*
 * return pointer to array of pointers to rows
 */
    return item;
}



void 
free_gci_matrix(GnomeCanvasItem **item,
		long nr, 
		long nc)
/*-----------------------------------------------------------------------------
 * Frees 2-dimensional array for GnomeCanvasItem
 */
{
  assert(item[0] != NULL);

   free((char*) (item[0] - 1));
   free((char*) (item - 1));
}



void 
free_all_bufmems(Display * disp)
/*-----------------------------------------------------------------------------
 * Frees all dynamic memory of Buffer structure
 */
{
    free_img_bufmems(disp);
    free_eval_bufmems(disp);
    free_valid_bufmems(disp);
    free_post_bufmems(disp);
}



void 
free_img_bufmems(Display * disp)
/*-----------------------------------------------------------------------------
 * Frees all dynamic memory of Buffer structure of img
 */
{
/*     char *foo = '\0'; */
/*     g_snprintf(disp->fname_base, GPIV_MAX_CHARS, "");  */

    if (disp->img.exist_img) {
        gdk_pixbuf_unref (disp->img.pixbuf1);
        if (disp->img.rgbbuf_img1 != NULL) {
            g_free(disp->img.rgbbuf_img1);
            disp->img.rgbbuf_img1 = NULL;
        }
        gdk_pixbuf_unref (disp->img.pixbuf2);
        if (disp->img.img1 != NULL) {
            gpiv_free_img(disp->img.img1, image_par);
            disp->img.img1 = NULL;
        }
        destroy_img(disp);
        
        if (image_par.x_corr == 1) {
            if (disp->img.rgbbuf_img2 != NULL) {
                g_free(disp->img.rgbbuf_img2);
                disp->img.rgbbuf_img2 = NULL;
            }
            if (disp->img.img2 != NULL) {
                gpiv_free_img(disp->img.img2, image_par);
                disp->img.img2 = NULL;
            }
        }
    }
    
    destroy_background(disp);
    if (disp->intreg.exist_int) 
        destroy_all_intregs(disp);
}



void 
free_eval_bufmems(Display * disp)
/*-----------------------------------------------------------------------------
 * Frees all dynamic memory of Buffer structure of evaluation processing
 */
{
    if (disp->gpd.exist_piv) {
        destroy_all_vectors(&disp->gpd);
        gpiv_free_pivdata(&disp->gpd.piv_data);
        disp->gpd.exist_piv = FALSE;
    }

    if (disp->gpd.scaled_piv) {
        gpiv_free_pivdata(&disp->gpd.piv_data_scaled);
        disp->gpd.scaled_piv = FALSE;
    }

    disp->gpd.averaged_piv = FALSE;
}



void 
free_valid_bufmems(Display * disp)
/*-----------------------------------------------------------------------------
 * Frees all dynamic memory of Buffer structure of validation processing
 */
{
    if (disp->gpd.exist_histo) {
        gpiv_free_bindata(&disp->gpd.klass);
        disp->gpd.exist_valid = FALSE;
    }
}



void 
free_post_bufmems(Display * disp)
/*-----------------------------------------------------------------------------
 * Frees all dynamic memory of Buffer structure of post processing
 */
{

    if (disp->gpd.exist_vor) {
        destroy_all_scalars(disp/* ->gpd.vor_data */, GPIV_VORTICITY);
        gpiv_free_scdata(&disp->gpd.vor_data);
        disp->gpd.exist_vor = FALSE;
        disp->gpd.piv_post_par.diff_type_logic = FALSE;
        disp->gpd.piv_post_par.operator_vorstra_logic = FALSE;
    }

   if (disp->gpd.exist_vor_scaled) {
        gpiv_free_scdata(&disp->gpd.vor_data_scaled);
        disp->gpd.exist_vor_scaled = FALSE;
    }

    if (disp->gpd.exist_sstrain) {
        destroy_all_scalars(/* & */disp/* ->gpd.sstrain_data */, GPIV_S_STRAIN);
        gpiv_free_scdata (&disp->gpd.sstrain_data);
        disp->gpd.exist_sstrain = FALSE;
        disp->gpd.piv_post_par.diff_type_logic = FALSE;
        disp->gpd.piv_post_par.operator_vorstra_logic = FALSE;
    }
        
    if (disp->gpd.exist_sstrain_scaled) {
        gpiv_free_scdata (&disp->gpd.sstrain_data_scaled);
        disp->gpd.exist_sstrain_scaled = FALSE;
    }
        
    if (disp->gpd.exist_nstrain) {
        destroy_all_scalars(/* & */disp/* ->gpd.nstrain_data */, GPIV_N_STRAIN);
        gpiv_free_scdata (&disp->gpd.nstrain_data);
        disp->gpd.exist_nstrain = FALSE;
        disp->gpd.piv_post_par.diff_type_logic = FALSE;
        disp->gpd.piv_post_par.operator_vorstra_logic = FALSE;
    }

    if (disp->gpd.exist_nstrain_scaled) {
        gpiv_free_scdata (&disp->gpd.nstrain_data_scaled);
        disp->gpd.exist_nstrain_scaled = FALSE;
    }

}



void
cp_parameters(Par par_src,
              Par * par_dest,
              gboolean force,
              gboolean print_par,
              gboolean img_par
              )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Copies gpiv parameters from src to dest
 *
 * INPUTS:
 *      par_src:                source piv parameters
 *      force:                  flag to force the copying, even if destination
 *                              already exists
 *      print_par:              verbose output. Not implemented, yet.
 *
 * OUTPUTS:
 *      par_dest:               destination piv parameters
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/
{
    if (force 
        || ( par_src.show_tooltips_logic 
             && !par_dest->show_tooltips_logic)) {
        par_dest->show_tooltips = par_src.show_tooltips;
        par_dest->show_tooltips_logic = TRUE;
    }
    
    if (force 
        || ( par_src.view_tabulator_logic 
             && !par_dest->view_tabulator_logic)) {
        par_dest->view_tabulator = par_src.view_tabulator;
        par_dest->view_tabulator_logic = TRUE;
    }
    
    if (force 
        || ( par_src.view_gpivbuttons_logic 
             && !par_dest->view_gpivbuttons_logic)) {
         par_dest->view_gpivbuttons = par_src.view_gpivbuttons;
        par_dest->view_gpivbuttons_logic = TRUE;
    }
    
    if (force 
        || (  par_src.nbins_logic 
             && !par_dest->nbins_logic)) {
        par_dest->nbins = par_src.nbins;
        par_dest->nbins_logic = TRUE;
    }
#ifndef DISABLE_DAC
    if (force 
        || ( par_src.process_dac_logic 
             && !par_dest->process_dac_logic)) {
        par_dest->process_dac = par_src.process_dac;
        par_dest->process_dac_logic = TRUE;
    }
#endif /* DISABLE_DAC */
    
    if (force 
        || ( par_src.process_piv_logic 
             && !par_dest->process_piv_logic)) {
        par_dest->process_piv = par_src.process_piv;
        par_dest->process_piv_logic = TRUE;
    }
    
    if (force 
        || ( par_src.process_gradient_logic 
             && !par_dest->process_gradient_logic)) {
        par_dest->process_gradient = par_src.process_gradient;
        par_dest->process_gradient_logic = TRUE;
    }
    
    if (force 
        || ( par_src.process_resstats_logic 
             && !par_dest->process_resstats_logic)) {
        par_src.process_resstats = par_dest->process_resstats;
        par_dest->process_resstats_logic = TRUE;
    }
    
    if (force 
        || ( par_src.process_errvec_logic 
             && !par_dest->process_errvec_logic)) {
        par_dest->process_errvec = par_src.process_errvec;
        par_dest->process_errvec_logic = TRUE;
    }
    
    if (force 
        || ( par_src.process_peaklock_logic 
             && !par_dest->process_peaklock_logic)) {
        par_dest->process_peaklock = par_src.process_peaklock;
        par_dest->process_peaklock_logic = TRUE;
    }
    
    if (force 
        || ( par_src.process_scale_logic 
             && !par_dest->process_scale_logic)) {
        par_dest->process_scale = par_src.process_scale;
        par_dest->process_scale_logic = TRUE;
    }
    
    if (force 
        || ( par_src.process_average_logic 
             && !par_dest->process_average_logic)) {
        par_dest->process_average = par_src.process_average;
        par_dest->process_average_logic = TRUE;
    }
    
    if (force 
        || ( par_src.process_subtract_logic 
             && !par_dest->process_subtract_logic)) {
        par_dest->process_subtract = par_src.process_subtract;
        par_dest->process_subtract_logic = TRUE;
    }
    
    if (force 
        || ( par_src.process_vorstra_logic 
             && !par_dest->process_vorstra_logic)) {
        par_dest->process_vorstra = par_src.process_vorstra;
        par_dest->process_vorstra_logic = TRUE;
    }
    
    if (force 
        || ( par_src.vector_scale_logic 
             && !par_dest->vector_scale_logic)) {
        par_dest->vector_scale = par_src.vector_scale;
        par_dest->vector_scale_logic = TRUE;
    }
    
    if (force 
        || (  par_src.vector_color_logic 
             && !par_dest->vector_color_logic)) {
        par_dest->vector_color = par_src.vector_color;
        par_dest->vector_color_logic = TRUE;
    }
    
    if (force 
        || ( par_src.zoom_index_logic 
             && !par_dest->zoom_index_logic)) {
        par_dest->zoom_index = par_src.zoom_index;
        par_dest->zoom_index_logic = TRUE;
    }
    
    if (force 
        || ( par_src.background_logic 
             && !par_dest->background_logic)) {
        par_dest->background = par_src.background;
        par_dest->background_logic = TRUE;
    }
    
    if (force 
        || ( par_src.display_img1_logic 
             && !par_dest->display_img1_logic)) {
        par_dest->display_img1 = par_src.display_img1;
        par_dest->display_img1_logic = TRUE;
    }
    
    if (force 
        || ( par_src.display_img2_logic 
             && !par_dest->display_img2_logic)) {
        par_dest->display_img2 = par_src.display_img2;
        par_dest->display_img2_logic = TRUE;
    }
    
    if (force 
        || ( par_src.display_intregs_logic 
             && !par_dest->display_intregs_logic)) {
        par_dest->display_intregs = par_src.display_intregs;
        par_dest->display_intregs_logic = TRUE;
    }
    
    if (force 
        || ( par_src.display_piv_logic 
             && !par_dest->display_piv_logic)) {
        par_dest->display_piv = par_src.display_piv;
        par_dest->display_piv_logic = TRUE;
    }
    
    if (force 
        || ( par_src.display_none_logic 
             && !par_dest->display_none_logic)) {
        par_dest->display_none = par_src.display_none;
        par_dest->display_none_logic = TRUE;
    }
    
    if (force 
        || ( par_src.display_vor_logic 
             && ! par_dest->display_vor_logic)) {
        par_dest->display_vor = par_src.display_vor;
        par_dest->display_vor_logic = TRUE;
    }
    
    if (force 
        || ( par_src.display_sstrain_logic 
        && !par_dest->display_sstrain_logic)) {
        par_dest->display_sstrain = par_src.display_sstrain;
        par_dest->display_sstrain_logic = TRUE;
    }

    if (force 
        || ( par_src.display_nstrain_logic 
             && !par_dest->display_nstrain_logic)) {
        par_dest->display_nstrain = par_src.display_nstrain;
        par_dest->display_nstrain_logic = TRUE;
    }
    
    if (force 
        || ( par_src.hdf_logic 
             && !par_dest->hdf_logic)) {
        par_dest->hdf = par_src.hdf;
        par_dest->hdf_logic = TRUE;
    }
    
    if (force 
        || ( par_src.hdf_img_logic 
             && !par_dest->hdf_img_logic)) {
        par_dest->hdf_img = par_src.hdf_img;
        par_dest->hdf_img_logic = TRUE;
    }
    
    if (force 
        || ( par_src.x_corr_logic 
             && !par_dest->x_corr_logic)) {
        par_dest->x_corr = par_src.x_corr;
        par_dest->x_corr_logic = TRUE;
    }
    
    if (img_par) {
        if (force 
            || ( par_src.img_width_logic 
                 && !par_dest->img_width_logic)) {
            par_dest->img_width = par_src.img_width;
            par_dest->img_width_logic = TRUE;
        }
        
        if (force 
            || ( par_src.img_height_logic 
                 && !par_dest->img_height_logic)) {
            par_dest->img_height = par_src.img_height;
            par_dest->img_height_logic = TRUE;
        }
        
        if (force 
        || ( par_src.img_depth_logic 
             && !par_dest->img_depth_logic)) {
            par_dest->img_depth = par_src.img_depth;
            par_dest->img_depth_logic = TRUE;
        }
    }
}



/*
 * gtk routines
 */


void
on_widget_leave(GtkContainer * container,
		GtkDirectionType direction, gpointer user_data)
/*-----------------------------------------------------------------------------
 * Resets text in application bar to default after poiner leaves 
 * container object
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(container), "gpiv");
    gnome_appbar_push(GNOME_APPBAR(gpiv->appbar), msg_default);
}



void 
sensitive(GpivConsole *gpiv,
          enum WidgetSet wi_set, 
          gboolean sense)
/*-----------------------------------------------------------------------------
 * Changes the sensitivity of a set of witgets
 */
{

#ifndef DISABLE_DAC
    Dac *dac = gpiv->dac;
    Trig *trig =  dac->trig;
    Cam *cam = dac->cam;
/*
 * Setting sensitivity of Dac witgets
 */
    if (wi_set == DAC) {
	gtk_widget_set_sensitive(gpiv->tablabel_dac, sense);
	gtk_widget_set_sensitive(gpiv->dac->label_title, sense);
	gtk_widget_set_sensitive(gpiv->dac->label_fname, sense);
	gtk_widget_set_sensitive(gpiv->dac->entry_fname, sense);
    }


    if (wi_set == DAC_TRIG) {
	gtk_widget_set_sensitive(trig->frame_trigger, sense);
	gtk_widget_set_sensitive(trig->radiobutton_mouse_3, sense);
	gtk_widget_set_sensitive(trig->radiobutton_mouse_4, sense);
	gtk_widget_set_sensitive(trig->radiobutton_mouse_5, sense);
	gtk_widget_set_sensitive(trig->radiobutton_mouse_6, sense);
	gtk_widget_set_sensitive(trig->label_trigger_dt, sense);
	gtk_widget_set_sensitive(trig->spinbutton_trigger_dt, sense);
	gtk_widget_set_sensitive(trig->label_trigger_incrdt, sense);
	gtk_widget_set_sensitive(trig->spinbutton_trigger_incrdt, sense);
	gtk_widget_set_sensitive(trig->label_trigger_cap, sense);
	gtk_widget_set_sensitive(trig->spinbutton_trigger_cap, sense);
	gtk_widget_set_sensitive(trig->button_trigger_start, sense);
	gtk_widget_set_sensitive(trig->button_trigger_stop, sense);
    }

    if (wi_set == DAC_TIMING) {
	gtk_widget_set_sensitive(trig->frame_trigger, sense);
	gtk_widget_set_sensitive(trig->radiobutton_mouse_1, sense);
	gtk_widget_set_sensitive(trig->radiobutton_mouse_2, sense);
	gtk_widget_set_sensitive(trig->label_trigger_nf, sense);
	gtk_widget_set_sensitive(trig->spinbutton_trigger_nf, sense);
    }


    if (wi_set == DAC_CAM) {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
                                     (gpiv->button_toolbar_dac), 
                                     gpiv_par.process_dac);
	gtk_widget_set_sensitive(gpiv->button_toolbar_dac, sense);
	gtk_widget_set_sensitive(cam->frame_cam, sense);
	gtk_widget_set_sensitive(cam->camera_select, sense);
	gtk_widget_set_sensitive(cam->camera_select_menu, sense);
	gtk_widget_set_sensitive(cam->menu_item, sense);
	gtk_widget_set_sensitive(cam->label_temp, sense);
	gtk_widget_set_sensitive(cam->label_label_temp, sense);
	gtk_widget_set_sensitive(cam->button_rec_start, sense);
	gtk_widget_set_sensitive(cam->button_rec_stop, sense);
    }
#endif /* DISABLE_DAC */

/*
 * Setting sensitivity of Image witgets
 */
    if (wi_set == IMG) {

/* 	gtk_widget_set_sensitive(gpiv->imgh->label_name, sense); */
/* 	gtk_widget_set_sensitive(gpiv->imgh->entry_name, sense); */
/* 	gtk_widget_set_sensitive(gpiv->imgh->spinbutton_ncols, sense); */

	gtk_widget_set_sensitive(gpiv->imgh->label_colpos, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_sscale, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_ncols, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_nrows, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_depth, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_rowpos, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_tscale, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_imgtitle, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_crdate, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_location, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_author, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_software, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_source, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_usertext, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_warning, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_disclaimer, sense);
	gtk_widget_set_sensitive(gpiv->imgh->label_comment, sense);
	gtk_widget_set_sensitive(gpiv->imgh->spinbutton_colpos, sense);
	gtk_widget_set_sensitive(gpiv->imgh->spinbutton_sscale, sense);
	gtk_widget_set_sensitive(gpiv->imgh->spinbutton_rowpos, sense);
	gtk_widget_set_sensitive(gpiv->imgh->spinbutton_tscale, sense);
	gtk_widget_set_sensitive(gpiv->imgh->entry_imgtitle, sense);
	gtk_widget_set_sensitive(gpiv->imgh->entry_crdate, sense);
	gtk_widget_set_sensitive(gpiv->imgh->entry_location, sense);
	gtk_widget_set_sensitive(gpiv->imgh->entry_author, sense);
	gtk_widget_set_sensitive(gpiv->imgh->entry_software, sense);
	gtk_widget_set_sensitive(gpiv->imgh->entry_source, sense);
	gtk_widget_set_sensitive(gpiv->imgh->entry_usertext, sense);
	gtk_widget_set_sensitive(gpiv->imgh->entry_warning, sense);
	gtk_widget_set_sensitive(gpiv->imgh->entry_disclaimer, sense);
	gtk_widget_set_sensitive(gpiv->imgh->entry_comment, sense);
    }


/*
 * Setting sensitivity of Image Processing witgets
 */
    if (wi_set == IMGPROC) {
    }


/*
 * Setting sensitivity of PIV witgets
 */
    if (wi_set == EVAL) {
	gtk_widget_set_sensitive(gpiv->piveval->spinbutton_colstart, sense);
 	gtk_widget_set_sensitive(gpiv->piveval->spinbutton_colend, sense);
	gtk_widget_set_sensitive(gpiv->piveval->spinbutton_preshiftcol, sense);
	gtk_widget_set_sensitive(gpiv->piveval->spinbutton_rowstart, sense);
	gtk_widget_set_sensitive(gpiv->piveval->spinbutton_rowend, sense);
	gtk_widget_set_sensitive(gpiv->piveval->spinbutton_preshiftrow, sense);

/* 	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intsize1_1, sense); */
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intsize1_2, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intsize1_3, sense);

	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intsize1_4, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intsize1_5, sense);

/* 	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intsize2_1, sense); */
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intsize2_2, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intsize2_3, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intsize2_4, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intsize2_5, sense);

	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intshift_1, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intshift_2, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intshift_3, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intshift_4, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_intshift_5, sense);

	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_fit_none, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_fit_gauss, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_fit_power, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_fit_gravity, sense);

	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_mouse_1, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_mouse_2, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_mouse_3, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_mouse_4, sense);

	gtk_widget_set_sensitive(gpiv->piveval->checkbutton_monitor, sense);

	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_peak_1, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_peak_2, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_peak_3, sense);

	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_weightkernel, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_zerooff, sense);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_centraldiff, sense);

	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_cross_1, FALSE);
	gtk_widget_set_sensitive(gpiv->piveval->radiobutton_cross_2, FALSE);

	gtk_widget_set_sensitive(gpiv->piveval->button, sense);
    }


/*
 * Setting sensitivity of a slection of PIV witgets
 */
    if (wi_set == INTREGS) {
/* 	if (piv_eval_par.int_size_1 >= 16) { */
/* 	    gtk_widget_set_sensitive(GTK_WIDGET */
/* 				     (gpiv->piveval->radiobutton_intsize2_1), sense); */
	    if (piv_eval_par.int_size_1 >= 32) {
		gtk_widget_set_sensitive(GTK_WIDGET
					 (gpiv->piveval->radiobutton_intsize2_2),
					 sense);
		if (piv_eval_par.int_size_1 >= 64) {
		    gtk_widget_set_sensitive(GTK_WIDGET
					     (gpiv->piveval->radiobutton_intsize2_3),
					     sense);
		    if (piv_eval_par.int_size_1 >= 128) {
			gtk_widget_set_sensitive(GTK_WIDGET
						 (gpiv->piveval->radiobutton_intsize2_4),
						 sense);
		    }
		}
	    }
/* 	} */
    }



/*
 * Setting sensitivity of Validation witgets
 */
    if (wi_set == VALID) {
	gtk_widget_set_sensitive(gpiv->pivvalid->radiobutton_errvec_residu_snr, sense);
	gtk_widget_set_sensitive(gpiv->pivvalid->radiobutton_errvec_residu_median, sense);
	gtk_widget_set_sensitive(gpiv->pivvalid->radiobutton_errvec_residu_normmedian, sense);
	gtk_widget_set_sensitive(gpiv->pivvalid->radiobutton_disable_1, sense);
	gtk_widget_set_sensitive(gpiv->pivvalid->label_errvec_res, sense);
	gtk_widget_set_sensitive(gpiv->pivvalid->spinbutton_errvec_res, sense);
	gtk_widget_set_sensitive(gpiv->pivvalid->checkbutton_errvec_disres, sense);
        gtk_widget_set_sensitive(gpiv->pivvalid->button_errvec_resstats, sense);

	gtk_widget_set_sensitive(gpiv->pivvalid->radiobutton_errvec_subst_1, sense);
/* 	gtk_widget_set_sensitive(gpiv->pivvalid->radiobutton_errvec_subst_2, sense); */
	gtk_widget_set_sensitive(gpiv->pivvalid->radiobutton_errvec_subst_3, sense);
	gtk_widget_set_sensitive(gpiv->pivvalid->button_errvec, sense);

	gtk_widget_set_sensitive(gpiv->pivvalid->spinbutton_histo_bins, sense);
	gtk_widget_set_sensitive(gpiv->pivvalid->button_peaklck, sense);
	gtk_widget_set_sensitive(gpiv->pivvalid->button_uhisto, sense);
	gtk_widget_set_sensitive(gpiv->pivvalid->button_vhisto, sense);
    }


/*
 * Setting sensitivity of Post processing witgets
 */
    if (wi_set == POST) {
	gtk_widget_set_sensitive(gpiv->pivpost->radiobutton_vorstra_output_1, sense);
	gtk_widget_set_sensitive(gpiv->pivpost->radiobutton_vorstra_output_2, sense);
	gtk_widget_set_sensitive(gpiv->pivpost->radiobutton_vorstra_output_3, sense);

	gtk_widget_set_sensitive(gpiv->pivpost->radiobutton_vorstra_diffscheme_1,
				 sense);
	gtk_widget_set_sensitive(gpiv->pivpost->radiobutton_vorstra_diffscheme_2,
				 sense);
	gtk_widget_set_sensitive(gpiv->pivpost->radiobutton_vorstra_diffscheme_3,
				 sense);
	gtk_widget_set_sensitive(gpiv->pivpost->radiobutton_vorstra_diffscheme_4,
				 sense);
	gtk_widget_set_sensitive(gpiv->pivpost->button_vorstra, sense);
    }

}



void 
destroy(GtkWidget * widget, 
        gpointer data)
/*-----------------------------------------------------------------------------
 * Destroys widget
 */
{
    gtk_widget_destroy(widget);
}



void message_gpiv(gchar * msg, ...)
/*-----------------------------------------------------------------------------
 * Message box with warning
 */
{
    /* GtkWidget */ GtkDialog *gtk_message_wi;
    va_list args;
    char local_msg[3 * GPIV_MAX_CHARS];

    va_start (args, msg);
    vsnprintf (local_msg, 3 * GPIV_MAX_CHARS, msg, args);
    gtk_message_wi = create_message_dialog(local_msg);
    gtk_widget_show(GTK_WIDGET(gtk_message_wi));
/*     gtk_widget_show_all (gtk_message_wi); */
    va_end(args);
}



void warning_gpiv(gchar * msg, ...)
/*-----------------------------------------------------------------------------
 * Message box with warning
 */
{
    GtkDialog *gtk_warning_wi;
    va_list args;
    char local_msg[3 * GPIV_MAX_CHARS];

    va_start (args, msg);
    vsnprintf (local_msg, 3 * GPIV_MAX_CHARS, msg, args);
    gtk_warning_wi = create_message_dialog(local_msg);
    gtk_widget_show(GTK_WIDGET(gtk_warning_wi));
    va_end(args);
}



void error_gpiv(gchar * msg, ...)
/*-----------------------------------------------------------------------------
 * Message box with error
 */
{
    GtkDialog *gtk_error_wi;
    va_list args;
    char local_msg[2 * GPIV_MAX_CHARS];

    va_start(args, msg);
    vsnprintf (local_msg, 2 * GPIV_MAX_CHARS, msg, args);
    gtk_error_wi = create_error_dialog(local_msg);
    gtk_widget_show(GTK_WIDGET(gtk_error_wi));
    va_end(args);
}
