/* $Revision: 1.3 $ */
/*
 *  gpgp: Gnome/GTK Front for PGP
 *  Copyright (C) 1998  Max Valianskiy
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../config.h"
#include "../version.h"

#include <assert.h>
#include <gnome.h>

#include "gpgp.h"
#include "pgman.h"

enum 
{
    GNOME_PGP_ERROR_SIGNAL,
    GNOME_PGP_CANCEL_SIGNAL,
    LAST_SIGNAL
};

struct signpack
{
    GtkWidget* dialog;
    GnomePgpKeyring* sec;
    GnomePgpSign* signbox;
};

struct finish_pack
{
    int needpip;
    int pip[2]; 
    GnomePgpSign *data;
};

struct ok_pack
{
    GtkWidget* page;
    GtkWidget* box;
};

static void gnome_pgp_sign_class_init(GnomePgpSignClass *klass);
static void gnome_pgp_sign_init(GnomePgpSign *ttt);
static void sign_cb (GtkWidget *widget, GnomePgpSign *data);
static void enc_cb (GtkWidget *widget, GnomePgpSign *data);
static void sign_cancel_cb (GtkWidget *widget, struct signpack *data);
static void sign_ok_cb (GtkWidget *widget, struct signpack *data);
static void cancel_cb (GtkWidget *widget, gpointer data);
static void go_cb (GtkWidget *widget, GnomePgpSign* data);
static gint toggle_pgppass_cb (GtkWidget *widget, GnomePgpSign* data);
static void enc_ok_cb (GtkWidget *widget, struct signpack *data);
static void okbox_ok_cb (GtkWidget *widget, struct ok_pack* data);
static void next_time_cb(GtkToggleButton* widget, void* data);
static void toggle_sign_cb (GtkWidget *widget, GnomePgpSign* data);
static void toggle_enc_cb (GtkWidget *widget, GnomePgpSign* data);
static void toggle_clr_cb (GtkWidget *widget, GnomePgpSign* data);
static void sign_finish_cb(struct finish_pack* pack2);

static gint gnome_pgp_sign_signals[LAST_SIGNAL] = { 0 };
static int pgppass_ignore = FALSE;

guint gnome_pgp_sign_get_type()
{
    static guint gnome_pgp_sign_type = 0;

    if (!gnome_pgp_sign_type)
    {
        GtkTypeInfo gnome_pgp_sign_info =
        {
            "GnomePgpSign",
            sizeof (GnomePgpSign),
            sizeof (GnomePgpSignClass),
            (GtkClassInitFunc) gnome_pgp_sign_class_init,
            (GtkObjectInitFunc) gnome_pgp_sign_init,
            (GtkArgSetFunc) NULL,
            (GtkArgGetFunc) NULL
        };
        gnome_pgp_sign_type = gtk_type_unique(
            gtk_window_get_type (), &gnome_pgp_sign_info);
    }
    return gnome_pgp_sign_type;
}

static void gnome_pgp_sign_class_init (GnomePgpSignClass *class)
{
    GtkObjectClass *object_class;

    object_class = (GtkObjectClass*) class;
    gnome_pgp_sign_signals[GNOME_PGP_ERROR_SIGNAL] =
        gtk_signal_new(
            "error",
            GTK_RUN_FIRST,
            object_class->type,
            GTK_SIGNAL_OFFSET (GnomePgpSignClass,error),
            gtk_signal_default_marshaller, GTK_TYPE_NONE, 0);
    gnome_pgp_sign_signals[GNOME_PGP_CANCEL_SIGNAL] =
        gtk_signal_new(
            "cancel",
            GTK_RUN_FIRST,
            object_class->type,
            GTK_SIGNAL_OFFSET (GnomePgpSignClass,cancel),
            gtk_signal_default_marshaller, GTK_TYPE_NONE, 0);
    gtk_object_class_add_signals(
        object_class,
        gnome_pgp_sign_signals,
        LAST_SIGNAL);
    class->error = NULL;
}

static void gnome_pgp_sign_init (GnomePgpSign *page)
{
    GtkWidget* box;
    GtkWidget* box1;
    GtkWidget* box2;
    GtkWidget* box3;
    GtkWidget* box4;
    GtkWidget* box5;
    GtkWidget *buttonbox2;
    GtkWidget *bok, *bcancel;
    GtkWidget *frame1, *frame2, *frame3;

    gtk_window_set_title(GTK_WINDOW(page),_("Sign and Encrypt"));

    box = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(page),box);
    gtk_widget_show(box);

    frame1 = gtk_frame_new(_("Sign"));
    gtk_box_pack_start(GTK_BOX(box), frame1, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show(frame1); 

    box2 = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(frame1),box2);
    gtk_widget_show(box2);

    box3 = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(GTK_BOX(box2), box3, TRUE, TRUE, GNOME_PAD_SMALL);
    gtk_widget_show(box3);

    page->sign = GTK_CHECK_BUTTON(gtk_check_button_new_with_label(_("Sign")));
    gtk_box_pack_start(
        GTK_BOX(box3),
        GTK_WIDGET(page->sign),
        TRUE,
        TRUE,
        GNOME_PAD_SMALL);
    gtk_signal_connect(
        GTK_OBJECT (page->sign),
        "toggled",
        GTK_SIGNAL_FUNC (toggle_sign_cb),
        page);
    gtk_widget_show(GTK_WIDGET(page->sign)); 

    page->sec = GNOME_ENTRY(gnome_entry_new("seckey"));
    gtk_box_pack_start(
        GTK_BOX(box3),
        GTK_WIDGET(page->sec),
        TRUE,
        TRUE,
        GNOME_PAD_SMALL);
    gtk_entry_set_text(
        GTK_ENTRY(gnome_entry_gtk_entry(page->sec)),
        gpgp_options.defkey);
    gtk_widget_show(GTK_WIDGET(page->sec));

    page->bsign=GTK_BUTTON(gtk_button_new_with_label(_("Choose key")));
    gtk_box_pack_start(
        GTK_BOX(box3),
        GTK_WIDGET(page->bsign),
        FALSE,
        FALSE,
        GNOME_PAD_SMALL);
    gtk_signal_connect(
        GTK_OBJECT(GTK_WIDGET(page->bsign)),
        "clicked",
        GTK_SIGNAL_FUNC(sign_cb),
        page);
    gtk_widget_show(GTK_WIDGET(page->bsign)); 
 
    /* password box */
    box1 = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(
        GTK_BOX(box2),
        box1,
        FALSE,
        FALSE,
        GNOME_PAD_SMALL);
    gtk_widget_show(box1);
 
    page->enable = GTK_CHECK_BUTTON(
        gtk_check_button_new_with_label(_("Use PGPPASS")));
    gtk_box_pack_start(
        GTK_BOX(box1),
        GTK_WIDGET(page->enable),
        TRUE,
        TRUE,
        GNOME_PAD_SMALL);
    gtk_signal_connect(
        GTK_OBJECT (page->enable),
        "toggled",
        GTK_SIGNAL_FUNC(toggle_pgppass_cb),
        page);
    gtk_widget_set_sensitive(GTK_WIDGET(page->enable), FALSE);
    gtk_widget_show(GTK_WIDGET(page->enable)); 
  
    page->pass = GTK_ENTRY(gtk_entry_new());
    gtk_entry_set_visibility(page->pass, FALSE);
    gtk_box_pack_start(
        GTK_BOX(box1),
        GTK_WIDGET(page->pass),
        TRUE,
        TRUE,
        GNOME_PAD_SMALL);
    gtk_widget_show(GTK_WIDGET(page->pass));

    /* Encrypt */
    frame2 = gtk_frame_new(_("Encrypt"));
    gtk_box_pack_start(GTK_BOX(box), frame2, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show(frame2); 

    box4 = gtk_hbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(frame2),box4);
    gtk_widget_show(box4);

    page->enc = GTK_CHECK_BUTTON(gtk_check_button_new_with_label(_("Encrypt")));
    gtk_box_pack_start(
        GTK_BOX(box4),
        GTK_WIDGET(page->enc),
        TRUE,
        TRUE,
        GNOME_PAD_SMALL);
    gtk_signal_connect(
        GTK_OBJECT (page->enc),
        "toggled",
        GTK_SIGNAL_FUNC (toggle_enc_cb),
        page);
    gtk_widget_show(GTK_WIDGET(page->enc)); 
  
    page->pub = GNOME_ENTRY(gnome_entry_new("pubkey"));
    gtk_box_pack_start(
        GTK_BOX(box4),
        GTK_WIDGET(page->pub),
        TRUE,
        TRUE,
        GNOME_PAD_SMALL);
    gtk_widget_show(GTK_WIDGET(page->pub));

    page->benc=GTK_BUTTON(gtk_button_new_with_label(_("Choose key")));
    gtk_box_pack_start(GTK_BOX(box4),GTK_WIDGET(page->benc), FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_signal_connect (GTK_OBJECT (GTK_WIDGET(page->benc)), "clicked",
                        GTK_SIGNAL_FUNC (enc_cb),
                        page);
    gtk_widget_show(GTK_WIDGET(page->benc)); 
 
    /* options */
    frame3 = gtk_frame_new(_("Encrypt"));
    gtk_box_pack_start(GTK_BOX(box), frame3, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show(frame3); 

    box5 = gtk_hbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(frame3),box5);
    gtk_widget_show(box5);

    page->textmode = GTK_CHECK_BUTTON(
        gtk_check_button_new_with_label(_("Text mode")));
    gtk_box_pack_start(
        GTK_BOX(box5),
        GTK_WIDGET(page->textmode),
        TRUE, TRUE, GNOME_PAD_SMALL);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(page->textmode),1);
    gtk_widget_show(GTK_WIDGET(page->textmode)); 
 
    page->armor = GTK_CHECK_BUTTON(
        gtk_check_button_new_with_label(_("Ascii Armor")));
    gtk_box_pack_start(
        GTK_BOX(box5), GTK_WIDGET(page->armor),
        TRUE, TRUE, GNOME_PAD_SMALL);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(page->armor),1);
    gtk_widget_show( GTK_WIDGET(page->armor)); 
 
    page->clearsig = GTK_CHECK_BUTTON(
        gtk_check_button_new_with_label(_("Clear Signature")));
    gtk_box_pack_start(
        GTK_BOX(box5), GTK_WIDGET(page->clearsig),
        TRUE, TRUE, GNOME_PAD_SMALL);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(page->clearsig),0);
    gtk_signal_connect (GTK_OBJECT (page->clearsig), "toggled",
                        GTK_SIGNAL_FUNC (toggle_clr_cb),
                        page);
    gtk_widget_show( GTK_WIDGET(page->clearsig)); 
  
    /* buttonbox2 */
    buttonbox2 = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start(
        GTK_BOX(box), buttonbox2, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show(buttonbox2);
 
    /* buttons */
    bok=gnome_stock_button(GNOME_STOCK_BUTTON_OK);
    gtk_box_pack_start(
        GTK_BOX(buttonbox2),bok, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_signal_connect (GTK_OBJECT (bok), "clicked",
                        GTK_SIGNAL_FUNC (go_cb),
                        page);
    gtk_widget_show(bok); 

    bcancel=gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL);
    gtk_box_pack_start(
        GTK_BOX(buttonbox2),bcancel, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_signal_connect (GTK_OBJECT (bcancel), "clicked",
                        GTK_SIGNAL_FUNC (cancel_cb),
                        page);
    gtk_widget_show(bcancel); 

    toggle_sign_cb(GTK_WIDGET(page->sign), page);
    toggle_enc_cb(GTK_WIDGET(page->sign), page);
}

/*
 * gnome_pgp_sign_new()
 *
 * Procedure initiates signing/encryption action.
 *
 * Parameters:
 * buf          - data buffer to sign/encrypt
 * bufsize      - size of that data buffer
 * filename     - file name where the data comes from
 *
 * History:
 * 19-May-99 tftp       Added several assert() macros.
 */
GtkWidget *gnome_pgp_sign_new(char* buf, int bufsize, char* filename)
{
    GnomePgpSign* page;

    assert(buf != NULL);
    assert(bufsize >= 0);

    page = gtk_type_new (gnome_pgp_sign_get_type ());
 
    page->buf = buf;
    page->bufsize = bufsize;
    page->filename = filename;
    return GTK_WIDGET (page);
}

static void sign_cb(GtkWidget *widget, GnomePgpSign *data)
{
    GtkWidget* window;
    GtkWidget* box;
    GtkWidget* buttonbox;
    GtkWidget* secret;
    GtkWidget *bok, *bcancel;
    struct signpack* signpack = g_malloc (sizeof(struct signpack));

    window = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_title(GTK_WINDOW(window),_("Choose key"));
    gtk_widget_set_usize(window, 500, 150);
 
    box = gtk_vbox_new(FALSE,0);
    gtk_container_add( GTK_CONTAINER(window), box);
    gtk_widget_show(box);

    secret = gnome_pgp_keyring_new();
    gnome_pgp_keyring_select(GNOME_PGP_KEYRING(secret), gpgp_options.defsec);
    gtk_box_pack_start(GTK_BOX(box),secret, TRUE, TRUE, GNOME_PAD_SMALL);
    gtk_widget_show(secret);

    buttonbox = gtk_hbutton_box_new();
    gtk_box_pack_start(GTK_BOX(box),buttonbox, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show(buttonbox);

    /* Signpack */
    signpack->dialog = window;
    signpack->sec = GNOME_PGP_KEYRING(secret);
    signpack->signbox = data;

    /* Buttons */
    bok=gnome_stock_button(GNOME_STOCK_BUTTON_OK);
    gtk_box_pack_start(GTK_BOX(buttonbox),bok, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_signal_connect (GTK_OBJECT (bok), "clicked",
                        GTK_SIGNAL_FUNC (sign_ok_cb),
                        signpack);
    gtk_widget_show(bok); 

    bcancel=gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL);
    gtk_box_pack_start(GTK_BOX(buttonbox),bcancel, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_signal_connect (GTK_OBJECT (bcancel), "clicked",
                        GTK_SIGNAL_FUNC (sign_cancel_cb),
                        signpack);
    gtk_widget_show(bcancel); 

    gtk_widget_show(window);
}

static void cancel_cb (GtkWidget *widget, gpointer data)
{
    gtk_signal_emit(
        GTK_OBJECT(data),
        gnome_pgp_sign_signals[GNOME_PGP_CANCEL_SIGNAL]);
    free(GNOME_PGP_SIGN(data)->obuf);
    GNOME_PGP_SIGN(data)->obuf=NULL;
    GNOME_PGP_SIGN(data)->obufsize=0;
    gtk_widget_destroy(GTK_WIDGET(data));
}

static void enc_cb (GtkWidget *widget, GnomePgpSign *data)
{
    GtkWidget* window;
    GtkWidget* box;
    GtkWidget* buttonbox;
    GtkWidget* secret;
    GtkWidget *bok, *bcancel;
    struct signpack* signpack = g_malloc (sizeof(struct signpack));

    window = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_title(GTK_WINDOW(window),_("Choose key"));
    gtk_widget_set_usize(window, 500, 350);
 
    box = gtk_vbox_new(FALSE,0);
    gtk_container_add( GTK_CONTAINER(window), box);
    gtk_widget_show(box);

    secret = gnome_pgp_keyring_new();
    gnome_pgp_keyring_select(GNOME_PGP_KEYRING(secret), gpgp_options.defpub);
    gtk_box_pack_start(GTK_BOX(box),secret, TRUE, TRUE, GNOME_PAD_SMALL);
    gtk_widget_show(secret);

    buttonbox = gtk_hbutton_box_new();
    gtk_box_pack_start(GTK_BOX(box),buttonbox, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_widget_show(buttonbox);

    /* signpack */
    signpack->dialog = window;
    signpack->sec = GNOME_PGP_KEYRING(secret);
    signpack->signbox = data;

    /* buttons */
    bok=gnome_stock_button(GNOME_STOCK_BUTTON_OK);
    gtk_box_pack_start(GTK_BOX(buttonbox),bok, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_signal_connect (GTK_OBJECT (bok), "clicked",
                        GTK_SIGNAL_FUNC (enc_ok_cb),
                        signpack);
    gtk_widget_show(bok); 

    bcancel=gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL);
    gtk_box_pack_start(GTK_BOX(buttonbox),bcancel, FALSE, FALSE, GNOME_PAD_SMALL);
    gtk_signal_connect (GTK_OBJECT (bcancel), "clicked",
                        GTK_SIGNAL_FUNC (sign_cancel_cb),
                        signpack);
    gtk_widget_show(bcancel); 

    gtk_widget_show(window);
}

static void sign_cancel_cb (GtkWidget *widget, struct signpack *data)
{
    gtk_widget_destroy(data->dialog);
    free(data);
}

static void sign_ok_cb (GtkWidget *widget, struct signpack *data)
{
    char* text;

    gtk_clist_get_text( 
        GTK_CLIST(data->sec->keylist),
        GPOINTER_TO_INT(GTK_CLIST(data->sec->keylist)->selection->data),
        2,
        &text);
    gtk_entry_set_text(GTK_ENTRY(
        gnome_entry_gtk_entry(data->signbox->sec)), text);
    sign_cancel_cb(widget, data);
}

static void enc_ok_cb (GtkWidget *widget, struct signpack *data)
{
    char* text;

    gtk_clist_get_text( 
        GTK_CLIST(data->sec->keylist),
        GPOINTER_TO_INT(GTK_CLIST(data->sec->keylist)->selection->data),
        2,
        &text);
    gtk_entry_set_text(
        GTK_ENTRY(gnome_entry_gtk_entry(data->signbox->pub)), text);
    sign_cancel_cb(widget, data);
}

static int gpgp_MessageBox_cb(
    GnomeDialog *dialog, gint button_number, gpointer data)
{
    gnome_dialog_close(dialog);
    return FALSE;
}

void gpgp_MessageBox(const gchar *type, const gchar *text)
{
    GtkWidget * label;
    GtkWidget * dialog;

    if (type == NULL)
        type = _("Error");

    label  = gtk_label_new(text);
    assert(label != NULL);

    dialog = gnome_dialog_new(type, GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox),
                       label, TRUE, TRUE, GNOME_PAD);
    gtk_signal_connect(GTK_OBJECT(dialog), "clicked",
                       GTK_SIGNAL_FUNC(gpgp_MessageBox_cb),
                       NULL);
    gtk_widget_show(label);
    gtk_widget_show(dialog);
}

/*
 * go_cb()
 *
 * Procedure handles "OK" button in the sign/encrypt dialog.
 *
 * History:
 * 19-May-99 tftp       Added checks, warnings about missing data.
 */
static void go_cb (GtkWidget *widget, GnomePgpSign *data)
{
    struct finish_pack* pack=g_malloc(sizeof(struct finish_pack));
    char optline[1000]="";
    int do_armor, do_sign, do_encrypt, do_textmode, do_clearsign;
 
    assert(widget != NULL);
    assert(data != NULL);

    /* Collect selected options in GTK-specific manner */

    do_sign = GTK_TOGGLE_BUTTON(data->sign)->active;
    do_encrypt = GTK_TOGGLE_BUTTON(data->enc)->active;
    do_textmode = GTK_TOGGLE_BUTTON(data->textmode)->active;
    do_armor = GTK_TOGGLE_BUTTON(data->armor)->active;
    do_clearsign = GTK_TOGGLE_BUTTON(data->clearsig)->active;

    pack->needpip=0;
    pack->data=data;

    if (do_sign)
    {
        int len, passlen;
        char *pass, *seckey;

        pass = gtk_entry_get_text(data->pass);
        passlen = strlen(pass);
        if (passlen <= 0)
        {
            gpgp_MessageBox(NULL,_("No passphrase specified!"));
            return;
        }

        seckey=gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(data->sec)));
        len = strlen(seckey);
        if (len <= 0)
        {
            gpgp_MessageBox(NULL,_("No secret key specified!"));
            return;
        }

        /* Create, load pipe with passphrase data */
        pipe(pack->pip);
        pack->needpip=1;
        write(pack->pip[1], pass, passlen);
        close(pack->pip[1]);

        len = strlen(optline);
        g_snprintf(
            optline+len, sizeof(optline)-len,
            " --sign -u \"=%s\" --passphrase-fd %d",
            seckey, pack->pip[0]);

        /*
         * Optionally we specify file name. We do not really need it
         * for data retrieval because data has been already placed
         * into memory buffer. The file name may be used by OpenPGP
         * to restore original file name upon decryption. If we got
         * data off the clipboard (integrated editor) then we got no
         * file name (NULL) and have to skip that option.
         */
        if (data->filename != NULL)
        {
            strcat(optline, " --set-filename ");
            strcat(optline, data->filename);
        }
    }

    if (do_encrypt)
    {
        int len;
        char *pubkey;

        pubkey=gtk_entry_get_text(GTK_ENTRY(gnome_entry_gtk_entry(data->pub)));
        len = strlen(pubkey);
        if (len <= 0)
        {
            gpgp_MessageBox(NULL,_("No public key specified!"));
            return;
        }
        strcat(optline, " --encrypt -r \"=");
        strcat(optline, pubkey);
        strcat(optline, "\"");
    }

    if (!(do_sign || do_encrypt)) /* store - neither sign nor encrypt */
        strcat(optline, " --store");
 
    if (do_textmode)
        strcat(optline, " --textmode");

    if (do_armor)
        strcat(optline, " --armor");
 
    if (do_clearsign)
        strcat(optline, " --clearsign");
 
    gtk_widget_hide(GTK_WIDGET(data));
#if 0
    {
        char *tmp = (char *) malloc(data->bufsize+100);
        assert(tmp != NULL);
        sprintf(tmp,"%d:[%.*s]\n",data->bufsize,data->bufsize,data->buf);
        gpgp_MessageBox(_("Debug"),tmp);
        free(tmp);
    }
#endif
    pgp_sign_action(
        optline,
        data->buf,
        data->bufsize,
        &data->obuf,
        &data->obufsize,
        (PgmanActionFunc) sign_finish_cb,
        pack);
}

static void sign_finish_cb(struct finish_pack* pack2)
{
    GtkWidget* okbox;
    struct ok_pack* pack;
    GtkWidget* check;
  
    if (pack2->needpip)
    {
        close(pack2->pip[0]);
        close(pack2->pip[1]);
        pack2->needpip=0;
    }

    if (pgman_error)
    {
        pack2->data->error_info=pgman_error;
        gtk_signal_emit(
            GTK_OBJECT(pack2->data),
            gnome_pgp_sign_signals[GNOME_PGP_ERROR_SIGNAL]);
        g_free(pack2);
        return;
    }

    if (gpgp_options.sign_verbose)
    {
        pack = g_malloc(sizeof(struct ok_pack));
        okbox = gnome_message_box_new(
            _("Operation complete"),
            GNOME_MESSAGE_BOX_INFO,
            GNOME_STOCK_BUTTON_OK,
            NULL);
        pack->box  = okbox;
        pack->page = GTK_WIDGET(pack2->data);
        gnome_dialog_button_connect(
            GNOME_DIALOG(okbox),
            0,
            GTK_SIGNAL_FUNC(okbox_ok_cb),
            pack);
        check = gtk_check_button_new_with_label(_("Show next time"));
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(check),TRUE);
        gtk_signal_connect(
            GTK_OBJECT(check),
            "toggled",
            GTK_SIGNAL_FUNC(next_time_cb),
            NULL);
        gtk_box_pack_start(
            GTK_BOX(GNOME_DIALOG(okbox)->vbox),
            check,
            FALSE,
            TRUE,
            GNOME_PAD_SMALL);
        gtk_widget_show(check);
        gtk_widget_show(okbox);
    }
    else
        gtk_widget_destroy(GTK_WIDGET(pack2->data));
    g_free(pack2);
}

static void next_time_cb(GtkToggleButton* widget, void* data)
{
    assert(widget != NULL);
    gpgp_options.sign_verbose = widget->active;
    gnome_config_set_bool("gpgp/Options/Sign Verbose", gpgp_options.sign_verbose);
    gnome_config_sync();
}

static void okbox_ok_cb (GtkWidget *widget, struct ok_pack* data)
{
    if (data != NULL)
    {
        gtk_widget_destroy(data->page);
        gtk_widget_destroy(data->box);
        g_free(data);
    }
}

static gint toggle_pgppass_cb (GtkWidget *widget, GnomePgpSign* data)
{
    const char *pass = "";
    int passphrase_enabled = TRUE;

    if (pgppass_ignore)
        return TRUE;
    if(GTK_TOGGLE_BUTTON(data->enable)->active)
    {
        const char *cp = getenv(ENV_PGPPASS);
        int pgppass_valid = FALSE;
        if (cp != NULL)
        {
            if (strlen(cp) > 0)
            {
                pass = cp;
                passphrase_enabled = FALSE;
                pgppass_valid = TRUE;
            }
        }
        if (!pgppass_valid)
        {
            GtkWidget *window;

            window = gnome_message_box_new(
                _("Environment variable PGPPASS is not defined or empty"),
                GNOME_MESSAGE_BOX_WARNING,
                GNOME_STOCK_BUTTON_OK, NULL );
            gtk_widget_show(window);

            /* Do not try to access PGPPASS later in this session */
            pgppass_ignore = TRUE;
            gtk_toggle_button_set_state(
                GTK_TOGGLE_BUTTON(data->enable),
                FALSE);
            gtk_widget_set_sensitive(GTK_WIDGET(data->enable), FALSE);
            return TRUE;
        }
    }
    gtk_entry_set_text ( data->pass, pass);
    gtk_entry_set_editable( data->pass, passphrase_enabled);
    gtk_widget_set_sensitive ( GTK_WIDGET(data->pass), passphrase_enabled);
    return TRUE;
}

/*
 * This is the handler for "Sign" checkbox. We enable/disable
 * signing-specific controls here. If signing becomes enabled
 * then we also reevaluate the state of PGPPASS checkbox.
 */
static void toggle_sign_cb (GtkWidget *widget, GnomePgpSign* data)
{
    int is_sign;
    is_sign = GTK_TOGGLE_BUTTON(data->sign)->active;
    gtk_widget_set_sensitive(GTK_WIDGET(data->sec), is_sign);
    gtk_widget_set_sensitive(GTK_WIDGET(data->pass), is_sign);
    gtk_widget_set_sensitive(
        GTK_WIDGET(data->enable),
        is_sign && !pgppass_ignore && (getenv(ENV_PGPPASS) != NULL));
    gtk_widget_set_sensitive(GTK_WIDGET(data->bsign), is_sign);
    if (is_sign)
        toggle_pgppass_cb(widget, data);
}

/*
 * This is the handler for "Encrypt" checkbox. We enable/disable
 * encryption-specific controls here.
 */
static void toggle_enc_cb (GtkWidget *widget, GnomePgpSign* data)
{
    int is_enc;
    is_enc = GTK_TOGGLE_BUTTON(data->enc)->active;
    gtk_widget_set_sensitive(GTK_WIDGET(data->pub), is_enc);
    gtk_widget_set_sensitive(GTK_WIDGET(data->benc), is_enc);
}

static void toggle_clr_cb (GtkWidget *widget, GnomePgpSign* data)
{
    if ( GTK_TOGGLE_BUTTON(data->clearsig)->active )
    {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(data->enc),0);
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(data->sign),1);
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(data->armor),1);
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(data->textmode),1);
     
        gtk_widget_set_sensitive ( GTK_WIDGET(data->enc), FALSE);     
        gtk_widget_set_sensitive ( GTK_WIDGET(data->sign), FALSE);    
        gtk_widget_set_sensitive ( GTK_WIDGET(data->armor), FALSE);   
        gtk_widget_set_sensitive ( GTK_WIDGET(data->textmode), FALSE);  
      
        toggle_enc_cb(widget, data);
        toggle_sign_cb(widget, data);
    }
    else
    {
        gtk_widget_set_sensitive ( GTK_WIDGET(data->enc), TRUE );     
        gtk_widget_set_sensitive ( GTK_WIDGET(data->sign), TRUE );    
        gtk_widget_set_sensitive ( GTK_WIDGET(data->armor), TRUE );   
        gtk_widget_set_sensitive ( GTK_WIDGET(data->textmode), TRUE );  
    }
}
