/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib/gi18n.h>

#include "assert-macros.h"

#include "gpass/configuration.h"
#include "gpass/entry-factory.h"

/***********************************************************
 *
 * initialize/terminate,  setup/teardown
 *
 ***********************************************************/
static void
initialize(void)
{
    g_type_init();
}

static void
terminate(void)
{
}

static void
setup(void)
{
}

static void
teardown(void)
{
    gpass_configuration_finalize();
}

/***********************************************************
 *
 * test case
 *
 ***********************************************************/
static GPassEntryClass *
create_general_password_class(void)
{
    GPassEntryFactory *factory;
    GPassEntryFactoryCursor *cursor;
    GPassEntryClass *entry_class;
    GError *error;
    
    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);
    cursor = gpass_entry_factory_create_cursor(factory);
    error = gpass_entry_factory_cursor_seek(cursor, "general");
    ASSERT_NULL(error);
    g_object_get(cursor, "entry_class", &entry_class, NULL);
    g_object_unref(cursor);
    g_object_unref(factory);
    return entry_class;
}

START_TEST(test_s_nick)
{
    GPassEntryClass *entry_class = create_general_password_class();
    
    ASSERT_EQUAL_STRING(_("General"), gpass_entry_class_nick(entry_class));
}
END_TEST

START_TEST(test_s_icon_id)
{
    GPassEntryClass *entry_class = create_general_password_class();
    
    ASSERT_EQUAL_STRING("gnome-stock-authentication",
                        gpass_entry_class_icon_id(entry_class));
}
END_TEST

START_TEST(test_s_default_launcher)
{
    GPassEntryClass *entry_class = create_general_password_class();
    
    ASSERT_EQUAL_STRING("gnome-open @hostname@",
                        gpass_entry_class_default_launcher(entry_class));
}
END_TEST

START_TEST(test_s_can_have_child)
{
    GPassEntryClass *entry_class = create_general_password_class();
    
    ASSERT_FALSE(gpass_entry_class_can_have_child(entry_class));
}
END_TEST

START_TEST(test_s_attributes)
{
    GPassEntryClass *entry_class = create_general_password_class();
    GPassAttributeList *attributes = gpass_entry_class_attributes(entry_class);
    GPassAttribute *attr;
    GPassAttributeType type;
    const gchar *name;

    attr = gpass_attribute_list_lookup(attributes, "name");
    g_object_get(attr, "type", &type, "name", &name, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_STRING, type);
    ASSERT_EQUAL_STRING("name", name);

    attr = gpass_attribute_list_lookup(attributes, "description");
    g_object_get(attr, "type", &type, "name", &name, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_TEXT, type);
    ASSERT_EQUAL_STRING("description", name);

    attr = gpass_attribute_list_lookup(attributes, "creation-time");
    g_object_get(attr, "type", &type, "name", &name, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_TIME, type);
    ASSERT_EQUAL_STRING("creation-time", name);

    attr = gpass_attribute_list_lookup(attributes, "modification-time");
    g_object_get(attr, "type", &type, "name", &name, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_TIME, type);
    ASSERT_EQUAL_STRING("modification-time", name);

    attr = gpass_attribute_list_lookup(attributes, "expiration");
    g_object_get(attr, "type", &type, "name", &name, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_BOOLEAN, type);
    ASSERT_EQUAL_STRING("expiration", name);

    attr = gpass_attribute_list_lookup(attributes, "expiration-time");
    g_object_get(attr, "type", &type, "name", &name, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_TIME, type);
    ASSERT_EQUAL_STRING("expiration-time", name);

    attr = gpass_attribute_list_lookup(attributes, "username");
    g_object_get(attr, "type", &type, "name", &name, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_STRING, type);
    ASSERT_EQUAL_STRING("username", name);

    attr = gpass_attribute_list_lookup(attributes, "password");
    g_object_get(attr, "type", &type, "name", &name, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_PASSWORD, type);
    ASSERT_EQUAL_STRING("password", name);

    attr = gpass_attribute_list_lookup(attributes, "hostname");
    g_object_get(attr, "type", &type, "name", &name, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_URL, type);
    ASSERT_EQUAL_STRING("hostname", name);

    ASSERT_EQUAL_INT(9, gpass_attribute_list_length(attributes));
    
    g_object_unref(attributes);
}
END_TEST

static GPassEntry *
create_general_password(void)
{
    GPassEntryFactory *factory;
    GPassEntry *entry;
    GError *error;
    
    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);
    error = gpass_entry_factory_create_entry(factory, "general", &entry);
    ASSERT_NULL(error);
    g_object_unref(factory);
    return entry;
}

START_TEST(test_get)
{
    GPassEntry *entry = create_general_password();
    const gchar *str;
    time_t t;
    gboolean b;

    g_object_set(entry,
                 "name", "general password 1",
                 "description", "The general password No.1",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlab.jp/gpass/",
                 "username", "gpass",
                 "password", "qwerty", NULL);

    g_object_get(entry, "name", &str, NULL);
    ASSERT_EQUAL_STRING("general password 1", str);
    g_object_get(entry, "description", &str, NULL);
    ASSERT_EQUAL_STRING("The general password No.1", str);
    g_object_get(entry, "type", &str, NULL);
    ASSERT_EQUAL_STRING("general", str);
    g_object_get(entry, "creation-time", &t, NULL);
    ASSERT_EQUAL_INT(10000, t);
    g_object_get(entry, "modification-time", &t, NULL);
    ASSERT_EQUAL_INT(20000, t);
    g_object_get(entry, "expiration", &b, NULL);
    ASSERT_TRUE(b);
    g_object_get(entry, "expiration-time", &t, NULL);
    ASSERT_EQUAL_INT(30000, t);
    g_object_get(entry, "hostname", &str, NULL);
    ASSERT_EQUAL_STRING("http://projects.netlab.jp/gpass/", str);
    g_object_get(entry, "username", &str, NULL);
    ASSERT_EQUAL_STRING("gpass", str);
    g_object_get(entry, "password", &str, NULL);
    ASSERT_EQUAL_STRING("qwerty", str);

    g_object_unref(entry);
}
END_TEST

START_TEST(test_get__default)
{
    GPassEntry *entry = create_general_password();
    const gchar *str;
    time_t t;
    gboolean b;

    g_object_get(entry, "name", &str, NULL);
    ASSERT_EQUAL_STRING("", str);
    g_object_get(entry, "description", &str, NULL);
    ASSERT_EQUAL_STRING("", str);
    g_object_get(entry, "type", &str, NULL);
    ASSERT_EQUAL_STRING("general", str);
    g_object_get(entry, "creation-time", &t, NULL);
    ASSERT_EQUAL_INT(0, t);
    g_object_get(entry, "modification-time", &t, NULL);
    ASSERT_EQUAL_INT(0, t);
    g_object_get(entry, "expiration", &b, NULL);
    ASSERT_FALSE(b);
    g_object_get(entry, "expiration-time", &t, NULL);
    ASSERT_EQUAL_INT(0, t);
    g_object_get(entry, "hostname", &str, NULL);
    ASSERT_EQUAL_STRING("", str);
    g_object_get(entry, "username", &str, NULL);
    ASSERT_EQUAL_STRING("", str);
    g_object_get(entry, "password", &str, NULL);
    ASSERT_EQUAL_STRING("", str);

    g_object_unref(entry);
}
END_TEST

START_TEST(test_get__id)
{
    GPassEntry *entry = create_general_password();
    const gchar *str;

    g_object_get(entry, "id", &str, NULL);
    ASSERT_EQUAL_STRING("", str);

    g_object_set(entry,
                 "name", "general password 1",
                 "description", "The general password No.1",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlab.jp/gpass/",
                 "username", "gpass",
                 "password", "qwerty", NULL);
    g_object_get(entry, "id", &str, NULL);
    ASSERT_EQUAL_STRING("gpass", str);
    
    g_object_unref(entry);
}
END_TEST

START_TEST(test_set)
{
    GPassEntry *entry = create_general_password();
    const gchar *str;

    g_object_set(entry, "name", "general password 1", NULL);
    g_object_get(entry, "name", &str, NULL);
    ASSERT_EQUAL_STRING("general password 1", str);
    
    g_object_set(entry, "name", "entry", NULL);
    g_object_get(entry, "name", &str, NULL);
    ASSERT_EQUAL_STRING("entry", str);

    g_object_set(entry, "hostname", "http://projects.netlab.jp/gpass/", NULL);
    g_object_get(entry, "hostname", &str, NULL);
    ASSERT_EQUAL_STRING("http://projects.netlab.jp/gpass/", str);

    g_object_set(entry, "hostname", NULL, NULL);
    g_object_get(entry, "hostname", &str, NULL);
    ASSERT_NULL(str);

    g_object_set(entry, "hostname", "projects.netlab.jp", NULL);
    g_object_get(entry, "hostname", &str, NULL);
    ASSERT_EQUAL_STRING("projects.netlab.jp", str);

    g_object_unref(entry);
}
END_TEST

START_TEST(test_set_attributes)
{
    GPassEntry *entry = create_general_password();
    GPassEntryClass *entry_class = create_general_password_class();
    GPassAttributeList *attributes = gpass_entry_class_attributes(entry_class);
    GPassAttribute *attr;
    const gchar *str;
    time_t t;
    gboolean b;

    attr = gpass_attribute_list_lookup(attributes, "name");
    gpass_attribute_set(attr, "general password 1");
    attr = gpass_attribute_list_lookup(attributes, "description");
    gpass_attribute_set(attr, "The general password No.1");
    attr = gpass_attribute_list_lookup(attributes, "creation-time");
    gpass_attribute_set(attr, 10000);
    attr = gpass_attribute_list_lookup(attributes, "modification-time");
    gpass_attribute_set(attr, 20000);
    attr = gpass_attribute_list_lookup(attributes, "expiration");
    gpass_attribute_set(attr, TRUE);
    attr = gpass_attribute_list_lookup(attributes, "expiration-time");
    gpass_attribute_set(attr, 30000);
    attr = gpass_attribute_list_lookup(attributes, "username");
    gpass_attribute_set(attr, "gpass");
    attr = gpass_attribute_list_lookup(attributes, "password");
    gpass_attribute_set(attr, "qwerty");
    attr = gpass_attribute_list_lookup(attributes, "hostname");
    gpass_attribute_set(attr, "http://projects.netlab.jp/gpass/");
    
    gpass_entry_set_attributes(entry, attributes);
    g_object_unref(attributes);

    g_object_get(entry, "name", &str, NULL);
    ASSERT_EQUAL_STRING("general password 1", str);
    g_object_get(entry, "description", &str, NULL);
    ASSERT_EQUAL_STRING("The general password No.1", str);
    g_object_get(entry, "type", &str, NULL);
    ASSERT_EQUAL_STRING("general", str);
    g_object_get(entry, "creation-time", &t, NULL);
    ASSERT_EQUAL_INT(10000, t);
    g_object_get(entry, "modification-time", &t, NULL);
    ASSERT_EQUAL_INT(20000, t);
    g_object_get(entry, "expiration", &b, NULL);
    ASSERT_TRUE(b);
    g_object_get(entry, "expiration-time", &t, NULL);
    ASSERT_EQUAL_INT(30000, t);
    g_object_get(entry, "username", &str, NULL);
    ASSERT_EQUAL_STRING("gpass", str);
    g_object_get(entry, "password", &str, NULL);
    ASSERT_EQUAL_STRING("qwerty", str);
    g_object_get(entry, "hostname", &str, NULL);
    ASSERT_EQUAL_STRING("http://projects.netlab.jp/gpass/", str);

    g_object_unref(entry);
}
END_TEST

START_TEST(test_get_attributes)
{
    GPassEntry *entry = create_general_password();
    GPassEntryClass *entry_class = create_general_password_class();
    GPassAttributeList *attributes = gpass_entry_class_attributes(entry_class);
    GPassAttribute *attr;
    const gchar *s;
    time_t t;
    gboolean b;

    g_object_set(entry,
                 "name", "general password 1",
                 "description", "The general password No.1",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlab.jp/gpass/",
                 "username", "gpass",
                 "password", "qwerty", NULL);
    gpass_entry_get_attributes(entry, attributes);

    attr = gpass_attribute_list_lookup(attributes, "name");
    gpass_attribute_get(attr, &s);
    ASSERT_EQUAL_STRING("general password 1", s);
    
    attr = gpass_attribute_list_lookup(attributes, "description");
    gpass_attribute_get(attr, &s);
    ASSERT_EQUAL_STRING("The general password No.1", s);

    attr = gpass_attribute_list_lookup(attributes, "creation-time");
    gpass_attribute_get(attr, &t);
    ASSERT_EQUAL_INT(10000, t);

    attr = gpass_attribute_list_lookup(attributes, "modification-time");
    gpass_attribute_get(attr, &t);
    ASSERT_EQUAL_INT(20000, t);

    attr = gpass_attribute_list_lookup(attributes, "expiration");
    gpass_attribute_get(attr, &b);
    ASSERT_TRUE(b);

    attr = gpass_attribute_list_lookup(attributes, "expiration-time");
    gpass_attribute_get(attr, &t);
    ASSERT_EQUAL_INT(30000, t);

    attr = gpass_attribute_list_lookup(attributes, "username");
    gpass_attribute_get(attr, &s);
    ASSERT_EQUAL_STRING("gpass", s);

    attr = gpass_attribute_list_lookup(attributes, "password");
    gpass_attribute_get(attr, &s);
    ASSERT_EQUAL_STRING("qwerty", s);

    attr = gpass_attribute_list_lookup(attributes, "hostname");
    gpass_attribute_get(attr, &s);
    ASSERT_EQUAL_STRING("http://projects.netlab.jp/gpass/", s);

    g_object_unref(attributes);
    g_object_unref(entry);
}
END_TEST

START_TEST(test_format)
{
    GPassEntry *entry = create_general_password();
    GString *result = g_string_new(NULL);

    g_object_set(entry,
                 "name", "general password 1",
                 "description", "The general password No.1",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlab.jp/gpass/",
                 "username", "gpass",
                 "password", "qwerty", NULL);
    
    gpass_entry_format(entry, "gnome-open @hostname@", &result);
    ASSERT_EQUAL_STRING("gnome-open http://projects.netlab.jp/gpass/",
                        result->str);

    gpass_entry_format(entry, "@foo@ @name@ @bar@", &result);
    ASSERT_EQUAL_STRING("@foo@ general password 1 @bar@", result->str);

    gpass_entry_format(entry, "@creation-time@ \\@name\\@", &result);
    ASSERT_EQUAL_STRING("10000 \\@name\\@", result->str);

    gpass_entry_format(entry, "@expiration@ @name", &result);
    ASSERT_EQUAL_STRING("TRUE @name", result->str);

    g_string_free(result, TRUE);
    g_object_unref(entry);
}
END_TEST

START_TEST(test_include)
{
    GPassEntry *entry = create_general_password();

    g_object_set(entry,
                 "name", "general password 1",
                 "description", "The general password No.1",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlab.jp/gpass/",
                 "username", "gpass",
                 "password", "qwerty", NULL);
    ASSERT_TRUE(gpass_entry_include(entry, "general"));
    ASSERT_TRUE(gpass_entry_include(entry, "al pas"));
    ASSERT_TRUE(gpass_entry_include(entry, "The"));
    ASSERT_TRUE(gpass_entry_include(entry, "gpass"));
    ASSERT_TRUE(gpass_entry_include(entry, "gpass/"));
    
    ASSERT_FALSE(gpass_entry_include(entry, "abcdefg"));
    ASSERT_FALSE(gpass_entry_include(entry, "gpassgpass"));
    ASSERT_FALSE(gpass_entry_include(entry, "qwerty"));
    
    g_object_unref(entry);
}
END_TEST

START_TEST(test_equal)
{
    GPassEntry *entry1 = create_general_password();
    GPassEntry *entry2 = create_general_password();

    g_object_set(entry1,
                 "name", "general password 1",
                 "description", "The general password No.1",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlab.jp/gpass/",
                 "username", "gpass",
                 "password", "qwerty", NULL);
    ASSERT_FALSE(gpass_entry_equal(entry1, entry2));
    
    g_object_set(entry2,
                 "name", "general password 1",
                 "description", "The general password No.1",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlab.jp/gpass/",
                 "username", "gpass",
                 "password", "qwerty", NULL);
    ASSERT_TRUE(gpass_entry_equal(entry1, entry2));

    g_object_set(entry2, "name", "general entry 1", NULL);
    ASSERT_FALSE(gpass_entry_equal(entry1, entry2));
    g_object_set(entry2, "name", "general password 1", NULL);

    g_object_set(entry2, "creation-time", 50, NULL);
    ASSERT_FALSE(gpass_entry_equal(entry1, entry2));
    g_object_set(entry2, "creation-time", 10000, NULL);

    g_object_set(entry2, "expiration", FALSE, NULL);
    ASSERT_FALSE(gpass_entry_equal(entry1, entry2));

    g_object_set(entry1, "expiration", FALSE, NULL);
    g_object_set(entry2, "expiration-time", 20000, NULL);
    ASSERT_TRUE(gpass_entry_equal(entry1, entry2));

    g_object_set(entry2, "username", "gpassgpass", NULL);
    ASSERT_FALSE(gpass_entry_equal(entry1, entry2));
    g_object_set(entry2, "username", "gpass", NULL);
    
    g_object_unref(entry1);
    g_object_unref(entry2);
}
END_TEST

/***********************************************************
 *
 * suite / main
 *
 ***********************************************************/
static Suite *
test_suite(void)
{
    Suite *s = suite_create("GPassGeneralPassword");
    TCase *tc;
    
    tc = tcase_create("functions");
    suite_add_tcase(s, tc);
    tcase_add_checked_fixture(tc, setup, teardown);
    
    tcase_add_test(tc, test_s_nick);
    tcase_add_test(tc, test_s_icon_id);
    tcase_add_test(tc, test_s_default_launcher);
    tcase_add_test(tc, test_s_can_have_child);
    tcase_add_test(tc, test_s_attributes);
    tcase_add_test(tc, test_get);
    tcase_add_test(tc, test_get__default);
    tcase_add_test(tc, test_get__id);
    tcase_add_test(tc, test_set);
    tcase_add_test(tc, test_set_attributes);
    tcase_add_test(tc, test_get_attributes);
    tcase_add_test(tc, test_format);
    tcase_add_test(tc, test_include);
    tcase_add_test(tc, test_equal);
    return s;
}

int
main(int argc, char *argv[])
{
    Suite *s;
    SRunner *sr;
    int nf;

    initialize();
    
    s = test_suite();
    sr = srunner_create(s);
    srunner_run_all(sr, CK_ENV);
    nf = srunner_ntests_failed(sr);
    srunner_free(sr);
    
    terminate();
    return (nf == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}
