/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include "helper.h"
#include "gpass/password.h"
#include "password-generator.h"

static GPassViewClass *parent_class = NULL;

static void
gpass_password_generator_instance_init(GTypeInstance *instance,
                                       gpointer g_class)
{
    GPassPasswordGenerator *self = GPASS_PASSWORD_GENERATOR(instance);

    self->password = g_string_new("");
    self->generate_type_class =
        g_type_class_ref(gpass_password_generate_type_get_type());
}

enum {
    PROP_0,
    PROP_PASSWORD,
};

static void
gpass_password_generator_get_property(GObject *object, guint prop_id,
                                      GValue *value, GParamSpec *pspec)
{
    GPassPasswordGenerator *self = GPASS_PASSWORD_GENERATOR(object);
    
    switch (prop_id) {
    case PROP_PASSWORD:
        g_value_set_static_string(value, self->password->str);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static GError *
gpass_password_generator_instance_loaded_glade_xml(GPassView *view)
{
    GPassPasswordGenerator *self = GPASS_PASSWORD_GENERATOR(view);
    GEnumValue *value;
    GtkWidget *widget;
    gint i;

    view->first_widget = glade_xml_get_widget(view->xml, "generate");
    widget = glade_xml_get_widget(view->xml, "type");
    for (i = 0; i < self->generate_type_class->n_values; i++) {
        value = self->generate_type_class->values + i;
        gtk_combo_box_append_text(GTK_COMBO_BOX(widget), _(value->value_nick));
    }
    gtk_combo_box_set_active(GTK_COMBO_BOX(widget), 0);
    return parent_class->loaded_glade_xml(view);
}

static void
gpass_password_generator_instance_finalize(GObject *object)
{
    GPassPasswordGenerator *self = GPASS_PASSWORD_GENERATOR(object);

    g_string_free(self->password, TRUE);
    g_type_class_unref(self->generate_type_class);
    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
gpass_password_generator_class_init(gpointer g_class, gpointer g_class_data)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(g_class);
    GPassViewClass *view_class = GPASS_VIEW_CLASS(g_class);
    
    parent_class = g_type_class_peek_parent(g_class);
    gobject_class->get_property = gpass_password_generator_get_property;
    gobject_class->finalize = gpass_password_generator_instance_finalize;
    view_class->loaded_glade_xml =
        gpass_password_generator_instance_loaded_glade_xml;

    g_object_class_install_property
        (gobject_class, PROP_PASSWORD,
         g_param_spec_string("password", _("Password"),
                             _("The generated password"),
                             NULL, G_PARAM_READABLE));
}

GType
gpass_password_generator_get_type(void)
{
    static GType type = 0;
    
    if (type == 0) {
        static const GTypeInfo info = {
            sizeof(GPassPasswordGeneratorClass),
            NULL,
            NULL,
            gpass_password_generator_class_init,
            NULL,
            NULL,
            sizeof(GPassPasswordGenerator),
            0,
            gpass_password_generator_instance_init
        };
        
        type = g_type_register_static(GPASS_TYPE_VIEW,
                                      "GPassPasswordGenerator", &info, 0);
    }
    return type;
}

/***********************************************************
 *
 * Signal handlers
 *
 ***********************************************************/
void
gpass_password_generator_on_use_flags_toggled(GtkWidget *widget,
                                              gpointer user_data)
{
    GPassPasswordGenerator *self;
    GtkWidget *toggle_button = widget;
    
    gpass_view_self_from_widget(widget, (gpointer **) &self);
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "use_lowercase");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
        return;
    }
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "use_uppercase");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
        return;
    }
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "use_number");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
        return;
    }
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "use_special");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
        return;
    }
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle_button), TRUE);
}

void
gpass_password_generator_on_generate_clicked(GtkWidget *widget,
                                             gpointer user_data)
{
    GPassPasswordGenerator *self;
    GPassPasswordGenerateType type;
    gint use_flags;
    gint length;
    gint index;
    GEnumValue *value;
    GError *error;
    
    gpass_view_self_from_widget(widget, (gpointer **) &self);
    
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "type");
    index = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
    value = self->generate_type_class->values + index;
    type = value->value;
    
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "length");
    length = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
    
    use_flags = 0;
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "use_lowercase");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
        use_flags |= GPASS_PASSWORD_GENERATE_USE_LOWERCASE;
    }
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "use_uppercase");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
        use_flags |= GPASS_PASSWORD_GENERATE_USE_UPPERCASE;
    }
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "use_number");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
        use_flags |= GPASS_PASSWORD_GENERATE_USE_NUMBER;
    }
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "use_special");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
        use_flags |= GPASS_PASSWORD_GENERATE_USE_SPECIAL;
    }
    
    error = gpass_password_generate(type, use_flags, length, &self->password);
    if (error != NULL) {
        gpass_helper_error(error);
    }
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "password");
    gtk_entry_set_text(GTK_ENTRY(widget), self->password->str);
}

void
gpass_password_generator_on_response(GtkWidget *widget, gint response_id,
                                     gpointer user_data)
{
    GPassPasswordGenerator *self;
    
    gpass_view_self_from_widget(widget, (gpointer **) &self);
    switch (response_id) {
    case GTK_RESPONSE_OK:
        GPASS_VIEW(self)->result = GPASS_VIEW_RESULT_SUCCEED;
        break;
    default:
        GPASS_VIEW(self)->result = GPASS_VIEW_RESULT_FAILED;
    }
    gpass_view_shutdown_main_loop(GPASS_VIEW(self));
}
