/*
    Copyright (C) 2009 Andrew Caudwell (acaudwell@gmail.com)

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version
    3 of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef RDIRNODE_H
#define RDIRNODE_H

#include "core/sdlapp.h"
#include "core/bounds.h"
#include "core/quadtree.h"
#include "core/pi.h"

#include "gource_settings.h"

#include "spline.h"
#include "file.h"

#include <list>
#include <set>

class RFile;

class RDirNode : public QuadItem {
    std::string abspath;

    std::string path_token;
    int         path_token_offset;

    RDirNode* parent;
    std::list<RDirNode*> children;
    std::list<RFile*> files;

    std::map<RDirNode*, SplineEdge> splines;

    vec4f col;

    vec2f spos;

    vec2f projected_pos;
    vec2f projected_spos;

    vec2f pos;
    vec2f vel;
    vec2f accel, prev_accel;

    float dir_area;

    bool visible;
    bool position_initialized;

    float since_node_visible;
    float since_last_file_change;
    float since_last_node_change;

    float file_area;
    float dir_radius;
    float dir_radius_sqrt;
    float parent_radius;

    int depth;

    int visible_count;

    vec3f screenpos;
    vec2f node_normal;

    void calcRadius();
    void calcColour();

    std::string commonPathPrefix(const std::string& str) const;

    void changePath(const std::string & abspath);

    void calcProjectedPos();

    void setInitialPosition();

    void drawEdge(RDirNode* child) const;
    void updateSpline(float dt);
    void move(float dt);

    vec2f calcFileDest(int layer_no, int file_no);
    void updateFilePositions();

    void adjustPath();
    void drawDirName(const FXFont& dirfont) const;
public:
    RDirNode(RDirNode* parent, const std::string & abspath);
    ~RDirNode();

    void printFiles();
    
    bool empty() const;

    bool isAnchor(RDirNode* node) const;

    RDirNode* getRoot();

    void fileUpdated(bool userInitiated);
    void nodeUpdated(bool userInitiated);

    void addVisible();
    bool isVisible();

    float getArea() const;

    int totalDirCount() const;
    int totalFileCount() const;

    int getTokenOffset() const;

    int dirCount() const;
    int fileCount() const;
    int visibleFileCount() const;
    bool noDirs() const;
    bool noFiles() const;

    bool prefixedBy(const std::string & path) const;

    const std::string & getPath() const;

    const vec2f & getNodeNormal() const;

    bool isParent(RDirNode* node) const;

    bool addFile(RFile* f);
    bool removeFile(RFile* f);

    int getDepth() const;

    const std::list<RDirNode*> & getChildren() const;

    void updateQuadItemBounds();

    float getParentRadius() const;
    float getRadius() const;
    float getRadiusSqrt() const;

    vec3f averageFileColour() const;

    const vec4f & getColour() const;

    RDirNode* getParent() const;

    const vec2f & getPos() const;

    void calcEdges();

    const vec2f & getProjectedPos() const;
    const vec2f & getSPos() const;

    void setPos(const vec2f & pos);

    void rotate(float s, float c);

    void setParent(RDirNode* parent);

    float distanceToParent() const;

    void addNode(RDirNode* node);

    void debug(int indent=0) const;

    void applyForceDir(RDirNode* dir);
    void applyForces(QuadTree &quadtree);

    void logic(float dt);

    void drawEdges(float dt) const;
    void drawEdgeShadows(float dt) const;

    void drawBloom(const Frustum & frustum, float dt);

    void drawShadows(const Frustum & frustum, float dt) const;
    void drawFiles(const Frustum & frustum, float dt) const;
    void drawSimple(const Frustum & frustum, float dt) const;

    void calcScreenPos();

    void drawNames(const FXFont & dirfont, const Frustum & frustum);

    void nodeCount() const;
};

class DirForceFunctor : public VisitFunctor<QuadItem>{
  private:
    RDirNode * this_dir;
    std::set<RDirNode*> seen;
    size_t loopCount;

  public:
    DirForceFunctor(RDirNode * dir) : this_dir(dir), seen(), loopCount(0){}
    int getLoopCount() const{ return loopCount; }
    void operator()(QuadItem * item){

        std::set<RDirNode*>::iterator seentest;
        RDirNode* d = (RDirNode*) (item);

        if(d==this_dir) return;
        if(d==this_dir->getParent()) return;
        if(d->getParent() == this_dir) return;
        if(this_dir->isParent(d)) return;
        if(d->isParent(this_dir)) return;

        if(d->node_count != 1) {
            if((seentest = seen.find(d)) != seen.end())
                return;

            seen.insert(d);
        }

        this_dir->applyForceDir(d);

        loopCount++;

    }

};

extern int gGourceDirNodeInnerLoops;
extern int gGourceFileInnerLoops;

extern float gGourcePointSize;
extern bool  gGourceNodeDebug;
extern bool  gGourceGravity;
extern float gGourceForceGravity;

extern std::map<std::string, RDirNode*> gGourceDirMap;

#endif
