<?php
/**
 * Gollem external API interface.
 *
 * This file defines Gollem's external API interface. Other
 * applications can interact with Gollem through this API.
 *
 * $Horde: gollem/lib/api.php,v 1.14 2005/06/01 14:28:46 jan Exp $
 *
 * @author  Amith Varghese (amith@xalan.com)
 * @author  Michael Slusarz (slusarz@curecanti.org)
 * @package Gollem
 */

$_services['perms'] = array(
    'args' => array(),
    'type' => '{urn:horde}stringArray');

$_services['selectlistLink'] = array(
    'args' => array('link_text' => 'string', 'link_style' => 'string', 'formid' => 'string', 'icon' => 'boolean', 'selectid' => 'string'),
    'type' => 'string');

$_services['selectlistResults'] = array(
    'args' => array('selectid' => 'string'),
    'type' => 'array');

$_services['returnFromSelectlist'] = array(
    'args' => array('selectid' => 'string', 'index' => 'string'),
    'type' => 'string');

$_services['setSelectList'] = array(
    'args' => array('selectid' => 'string', 'files' => 'array'),
    'type' => 'string');

function _gollem_perms()
{
    static $perms = array();
    if (!empty($perms)) {
        return $perms;
    }

    @define('GOLLEM_BASE', dirname(__FILE__) . '/..');
    require GOLLEM_BASE . '/config/backends.php';

    $perms['tree']['gollem']['backends'] = false;
    $perms['title']['gollem:backends'] = _("Backends");

    // Run through every backend.
    foreach ($backends as $backend => $curBackend) {
        $perms['tree']['gollem']['backends'][$backend] = false;
        $perms['title']['gollem:backends:' . $backend] = $curBackend['name'];
    }

    return $perms;
}

/**
 * Creates a link to the gollem file selection window.
 * The file section window will return a cache ID value which should be used
 * (along with the selectListResults and returnFromSelectList functions
 * belong) to obtain the data from a list of selected files.
 *
 * THERE MUST be a form field named 'selectlist_selectid' in the calling
 * form. This field will be populated with the selection ID when the user
 * completes file selection.
 *
 * THERE MUST be a form parameter named 'actionID' in the calling form.
 * This form will be populated with the value 'selectlist_process' when
 * the user completes file selection.  The calling form will be submitted
 * after the window closes (i.e. the calling form must process the
 * 'selectlist_process' actionID).
 *
 * @param string $link_text   The text to use in the link.
 * @param string $link_style  The style to use for the link.
 * @param string $formid      The formid of the calling script.
 * @param boolean $icon       Create the link with an icon instead of text?
 * @param string $selectid    Selection ID.
 *
 * @return string  The URL string.
 */
function _gollem_selectlistLink($link_text, $link_style, $formid,
                                $icon = false, $selectid = '')
{
    Horde::addScriptFile('open_selectlist_win.js', 'gollem');
    $link = Horde::link('', $link_text, $link_style, '', 'open_selectlist_win(\'' . $formid . '\', \'' . $selectid . '\'); return false;');
    if ($icon) {
        $link_text = Horde::img('gollem.png', $link_text, 'valign="middle"');
    }
    return $link . $link_text . '</a>';
}

/**
 * Return the list of files selected by the user for a given selection ID.
 *
 * @param string $selectid  The selection ID.
 *
 * @param array  An array with each file entry stored in its own array, with
 *               the key as the directory name and the value as the filename.
 */
function _gollem_selectlistResults($selectid)
{
    if (!isset($_SESSION['gollem']['selectlist'][$selectid]['files'])) {
        return null;
    } else {
        $list = array();
        foreach ($_SESSION['gollem']['selectlist'][$selectid]['files'] as $val) {
            list($dir, $filename) = explode('|', $val);
            $list[] = array($dir => $filename);
        }
        return $list;
    }
}

/**
 * Returns the data for a given selection ID and index.
 *
 * @param string $selectid  The selection ID.
 * @param integer $index    The index of the file data to return.
 *
 * @return string  The file data.
 */
function _gollem_returnFromSelectlist($selectid, $index)
{
    @define('GOLLEM_BASE', dirname(__FILE__) . '/..');
    require_once GOLLEM_BASE . '/lib/base.php';

    if (!isset($_SESSION['gollem']['selectlist'][$selectid]['files'][$index])) {
        return null;
    }

    list($dir, $filename) = explode('|', $_SESSION['gollem']['selectlist'][$selectid]['files'][$index]);
    return $GLOBALS['gollem_vfs']->read($dir, $filename);
}

/**
 * Set the files selected for a given selection ID
 *
 * @param string $selectid  The selection ID to use.
 * @param array $files      An array with each file entry stored in its own
 *                          array, with the key as the directory name and the
 *                          value as the filename.
 *
 * @return string  The selection ID.
 */
function _gollem_setSelectlist($selectid = '', $files = array())
{
    @define('GOLLEM_BASE', dirname(__FILE__) . '/..');
    require_once GOLLEM_BASE . '/lib/base.php';

    if (empty($selectid)) {
        $selectid = substr(base_convert(microtime() . mt_rand(), 10, 36), -16);
    }

    if (count($files) > 0) {
        $list = array();
        foreach ($files as $file) {
            $list[] = key($file) . '|' . current($file);
        }
        $_SESSION['gollem']['selectlist'][$selectid]['files'] = $list;
    }

    return $selectid;
}
