/*-
 * Copyright (c) 2001 Jordan DeLong
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef PAGER_H
#define PAGER_H

#include "plugutil.h"

/* used by pager_t's for list of client windows */
typedef struct paged {
	client_t	*client;	/* client we represent */
	Window		win;		/* our mini-representation window */
	int		last_width;
	int		last_height;	/* saved geometry; used for pager_winscale */

	TAILQ_ENTRY(paged) p_list;
} paged_t;

/* a pager */
typedef struct {
	client_t	*client;	/* this pager's client structure */
	desktop_t	*desktop;	/* desktop we represent */
	Window		win;		/* the pager window */
	int		spacewidth;	/* width of each workspace in pager coords */
	int		spaceheight;	/* height of each workspace */

	TAILQ_HEAD(pagedlist, paged) paged_list;
} pager_t;

/* per screen structure */
struct pagerscr {
	pager_t		**pagers;	/* pagers: 1 per desktop on the scr */
	GC		drawgc;		/* graphics context for this scr */
	int		pager_count;	/* number of pagers */

	/* images for each screen */
	image_t		*sel_img;	/* selected workspace img */
	image_t		*nonsel_img;	/* nonselected workspaces img */

	/* colors for each screen */
	Pixel		nonsel_color;	/* for nonselected workspaces */
	Pixel		sel_color;	/* for the selected workspace */
	Pixel		grid_color;	/* for the grid */
	Pixel		pagedwin_color;	/* for paged_t windows */
	Pixel		pagedbdr_color;	/* for paged_t window borders */
	Pixel		pagedfocwin_color;	/* color for focused paged_ts */
	Pixel		pagedfocbdr_color;	/* focused border color */
};

extern struct pagerscr *pagerscr;
extern double pager_ratio;
extern dgroup_t *pager_dgroup;
extern int pager_parentrel;
extern int pager_drawgrid;
extern int pager_nomove;
extern int pager_pagedbdrwidth;
extern int pager_stacklayer;
extern pixmap_t *pager_selpixmap;
extern pixmap_t *pager_nonselpixmap;
extern pixmap_t *pager_backpixmap;
extern int pager_backscale;
extern pixmap_t *pager_winpixmap;
extern int pager_winscale;
extern pixmap_t *pager_focwinpixmap;
extern int pager_focwinscale;
extern XContext pager_context;
extern XContext paged_context;
extern paged_t *paged_focused;

int pager_init();
pager_t *pager_create(screen_t *screen, desktop_t *desktop, int usexy,
	int x, int y);
void pager_delete(pager_t *pager);
void pager_addpaged(pager_t *pager, client_t *client);
void pager_rmpaged(pager_t *pager, paged_t *paged, client_t *client);
void pager_movepaged(pager_t *pager, paged_t *paged, pager_t *newpager,
	int doreparent);
void pager_sizepaged(pager_t *pager, paged_t *paged);
void pager_raisepaged(paged_t *paged, client_t *lowest);
void pager_focuspaged(paged_t *paged);
void pager_click(pager_t *pager, int x, int y);
void pager_expose(pager_t *pager, GC gc, XExposeEvent *e);
void pager_drag(pager_t *origpager, paged_t *paged, XButtonEvent *ev);

#endif
