package db

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"

	"github.com/cheggaaa/pb"
	"github.com/go-redis/redis"
	c "github.com/kotakanbe/go-cve-dictionary/config"
	"github.com/kotakanbe/go-cve-dictionary/jvn"
	log "github.com/kotakanbe/go-cve-dictionary/log"
	"github.com/kotakanbe/go-cve-dictionary/models"
	"github.com/kotakanbe/go-cve-dictionary/nvd"
)

/**
# Redis Data Structure

- HASH
  ┌───┬────────────┬──────────┬──────────┬─────────────────────────────────┐
  │NO │    HASH    │  FIELD   │  VALUE   │             PURPOSE             │
  └───┴────────────┴──────────┴──────────┴─────────────────────────────────┘
  ┌───┬────────────┬──────────┬──────────┬─────────────────────────────────┐
  │ 1 │ CVE#$CVEID │NVD or JVN│ $CVEJSON │     TO GET CVEJSON BY CVEID     │
  ├───┼────────────┼──────────┼──────────┼─────────────────────────────────┤
  │ 2 │  CVE#CPE   │ $CPENAME │ $CPEJSON │    TO GET CPEJSON BY CPENAME    │
  └───┴────────────┴──────────┴──────────┴─────────────────────────────────┘

- ZINDE  X
  ┌───┬────────────┬──────────┬──────────┬─────────────────────────────────┐
  │NO │    KEY     │  SCORE   │  MEMBER  │             PURPOSE             │
  └───┴────────────┴──────────┴──────────┴─────────────────────────────────┘
  ┌───┬────────────┬──────────┬──────────┬─────────────────────────────────┐
  │ 3 │CVE#$CPENAME│    0     │  $CVEID  │TO GET RELATED []CVEID BY CPENAME│
  ├───┼────────────┼──────────┼──────────┼─────────────────────────────────┤
  │ 4 │CVE#CPENAME │    0     │ $CPENAME │   TO GET ALL CPENAME QUICKLY    │
  └───┴────────────┴──────────┴──────────┴─────────────────────────────────┘
**/

const (
	dialectRedis  = "redis"
	hashKeyPrefix = "CVE#"
)

// RedisDriver is Driver for Redis
type RedisDriver struct {
	name string
	conn *redis.Client
}

// Name return db name
func (r *RedisDriver) Name() string {
	return r.name
}

// NewRedis return Redis driver
func NewRedis(dbType, dbpath string, debugSQL bool) (driver *RedisDriver, err error) {
	driver = &RedisDriver{
		name: dbType,
	}
	log.Debugf("Opening DB (%s).", driver.Name())
	if err = driver.OpenDB(dbType, dbpath, debugSQL); err != nil {
		return
	}

	return
}

// OpenDB opens Database
func (r *RedisDriver) OpenDB(dbType, dbPath string, debugSQL bool) (err error) {
	var option *redis.Options
	if option, err = redis.ParseURL(dbPath); err != nil {
		log.Error(err)
		return fmt.Errorf("Failed to Parse Redis URL. dbpath: %s, err: %s", dbPath, err)
	}
	r.conn = redis.NewClient(option)
	if err = r.conn.Ping().Err(); err != nil {
		return fmt.Errorf("Failed to open DB. dbtype: %s, dbpath: %s, err: %s", dbType, dbPath, err)
	}
	return nil
}

// CloseDB close Database
func (r *RedisDriver) CloseDB() (err error) {
	if err = r.conn.Close(); err != nil {
		log.Errorf("Failed to close DB. Type: %s. err: %s", r.name, err)
		return
	}
	return
}

// Get Select Cve information from DB.
func (r *RedisDriver) Get(cveID string) *models.CveDetail {
	// Avoid null slice being null in JSON
	emptyCveDetail := models.CveDetail{
		Nvd: models.Nvd{
			References: []models.Reference{},
			Cpes:       []models.Cpe{},
		},
		Jvn: models.Jvn{
			References: []models.Reference{},
			Cpes:       []models.Cpe{},
		},
	}

	var result *redis.StringStringMapCmd
	if result = r.conn.HGetAll(hashKeyPrefix + cveID); result.Err() != nil {
		log.Error(result.Err())
		return &emptyCveDetail
	}

	jvn := models.Jvn{
		References: []models.Reference{},
		Cpes:       []models.Cpe{},
	}

	var err error
	if j, ok := result.Val()["Jvn"]; ok {
		if err = json.Unmarshal([]byte(j), &jvn); err != nil {
			log.Errorf("Failed to Unmarshal json. err : %s", err)
			return &emptyCveDetail
		}
	}

	nvd := models.Nvd{
		References: []models.Reference{},
		Cpes:       []models.Cpe{},
	}
	if j, ok := result.Val()["Nvd"]; ok {
		if err = json.Unmarshal([]byte(j), &nvd); err != nil {
			log.Errorf("Failed to Unmarshal json. err : %s", err)
			return &emptyCveDetail
		}
	}

	cveDetail := &models.CveDetail{
		CveID: cveID,
		Nvd:   nvd,
		Jvn:   jvn,
	}

	return cveDetail
}

// GetMulti Select Cves information from DB.
func (r *RedisDriver) GetMulti(cveIDs []string) (cveDetails map[string]*models.CveDetail) {
	var err error
	cveDetails = map[string]*models.CveDetail{}
	pipe := r.conn.Pipeline()
	rs := map[string]*redis.StringStringMapCmd{}
	for _, cveID := range cveIDs {
		rs[cveID] = pipe.HGetAll(hashKeyPrefix + cveID)
	}
	if _, err = pipe.Exec(); err != nil {
		if err != redis.Nil {
			log.Errorf("Failed to get multi cve json. err : %s", err)
			return cveDetails
		}
	}

	for cveID, result := range rs {
		jvn := models.Jvn{
			References: []models.Reference{},
			Cpes:       []models.Cpe{},
		}
		if j, ok := result.Val()["Jvn"]; ok {
			if err = json.Unmarshal([]byte(j), &jvn); err != nil {
				log.Errorf("Failed to Unmarshal json. err : %s", err)
			}
		}

		nvd := models.Nvd{
			References: []models.Reference{},
			Cpes:       []models.Cpe{},
		}
		if j, ok := result.Val()["Nvd"]; ok {
			if err = json.Unmarshal([]byte(j), &nvd); err != nil {
				log.Errorf("Failed to Unmarshal json. err : %s", err)
			}
		}

		cveDetail := &models.CveDetail{
			CveID: cveID,
			Nvd:   nvd,
			Jvn:   jvn,
		}
		cveDetails[cveID] = cveDetail
	}
	return cveDetails
}

// GetByCpeName Select Cve information from DB.
func (r *RedisDriver) GetByCpeName(cpeName string) (details []*models.CveDetail) {
	var result *redis.StringSliceCmd
	if result = r.conn.ZRange(hashKeyPrefix+cpeName, 0, -1); result.Err() != nil {
		log.Error(result.Err())
		return details
	}

	for _, v := range result.Val() {
		details = append(details, r.Get(v))
	}
	return
}

// InsertJvn insert items fetched from JVN.
func (r *RedisDriver) InsertJvn(items []jvn.Item) error {
	log.Info("Inserting fetched CVEs...")

	cves := convertJvn(items)
	if err := r.insertIntoJvn(cves); err != nil {
		return err
	}
	return nil
}

// InsertIntoJvn inserts Cve Information into DB
func (r *RedisDriver) insertIntoJvn(cves []models.CveDetail) error {
	var err error
	var refreshedJvns []string
	bar := pb.New(len(cves))
	if c.Conf.Quiet {
		bar.Output = ioutil.Discard
	} else {
		bar.Output = os.Stderr
	}
	bar.Start()

	for chunked := range chunkSlice(cves, 10) {
		var pipe redis.Pipeliner
		pipe = r.conn.Pipeline()
		for _, c := range chunked {
			bar.Increment()

			var jj []byte
			c.Jvn.CveID = c.CveID
			if jj, err = json.Marshal(c.Jvn); err != nil {
				return fmt.Errorf("Failed to marshal json. err: %s", err)
			}
			refreshedJvns = append(refreshedJvns, c.CveID)
			if result := pipe.HSet(hashKeyPrefix+c.CveID, "Jvn", string(jj)); result.Err() != nil {
				return fmt.Errorf("Failed to HSet CVE. err: %s", result.Err())
			}

			for _, cpe := range c.Jvn.Cpes {
				var jc []byte
				if jc, err = json.Marshal(cpe); err != nil {
					return fmt.Errorf("Failed to marshal json. err: %s", err)
				}

				if result := pipe.HSet(hashKeyPrefix+"Cpe", cpe.CpeName, jc); result.Err() != nil {
					return fmt.Errorf("Failed to HSet cpe. err: %s", result.Err())
				}
				if result := pipe.ZAdd(
					hashKeyPrefix+cpe.CpeName,
					redis.Z{Score: 0, Member: c.CveID},
				); result.Err() != nil {
					return fmt.Errorf("Failed to ZAdd cpe name. err: %s", result.Err())
				}
				if result := pipe.ZAdd(
					hashKeyPrefix+"CpeName",
					redis.Z{Score: 0, Member: cpe.CpeName},
				); result.Err() != nil {
					return fmt.Errorf("Failed to ZAdd cpe. err: %s", result.Err())
				}
			}
		}
		if _, err = pipe.Exec(); err != nil {
			return fmt.Errorf("Failed to exec pipeline. err: %s", err)
		}
	}
	bar.Finish()
	log.Infof("Refreshed %d Jvns.", len(refreshedJvns))
	//  log.Debugf("%v", refreshedJvns)
	return nil
}

// CountNvd count nvd table
func (r *RedisDriver) CountNvd() (int, error) {
	var result *redis.StringSliceCmd
	if result = r.conn.Keys(hashKeyPrefix + "CVE*"); result.Err() != nil {
		return 0, result.Err()
	}
	return len(result.Val()), nil
}

// InsertNvd inserts CveInformation into DB
func (r *RedisDriver) InsertNvd(entries []nvd.Entry) error {
	log.Info("Inserting CVEs...")

	cves := convertNvd(entries)
	if err := r.insertIntoNvd(cves); err != nil {
		return err
	}
	return nil
}

// insertIntoNvd inserts CveInformation into DB
func (r *RedisDriver) insertIntoNvd(cves []models.CveDetail) error {
	var err error
	var refreshedNvds []string
	bar := pb.New(len(cves))
	if c.Conf.Quiet {
		bar.Output = ioutil.Discard
	} else {
		bar.Output = os.Stderr
	}
	bar.Start()

	for chunked := range chunkSlice(cves, 10) {
		var pipe redis.Pipeliner
		pipe = r.conn.Pipeline()
		for _, c := range chunked {
			bar.Increment()

			var jn []byte
			c.Nvd.CveID = c.CveID
			if jn, err = json.Marshal(c.Nvd); err != nil {
				return fmt.Errorf("Failed to marshal json. err: %s", err)
			}
			refreshedNvds = append(refreshedNvds, c.CveID)
			if result := pipe.HSet(hashKeyPrefix+c.CveID, "Nvd", string(jn)); result.Err() != nil {
				return fmt.Errorf("Failed to HSet CVE. err: %s", result.Err())
			}

			for _, cpe := range c.Nvd.Cpes {
				var jc []byte
				if jc, err = json.Marshal(cpe); err != nil {
					return fmt.Errorf("Failed to marshal json. err: %s", err)
				}

				if result := pipe.HSet(hashKeyPrefix+"Cpe", cpe.CpeName, jc); result.Err() != nil {
					return fmt.Errorf("Failed to HSet cpe. err: %s", result.Err())
				}
				if result := pipe.ZAdd(
					hashKeyPrefix+cpe.CpeName,
					redis.Z{Score: 0, Member: c.CveID},
				); result.Err() != nil {
					return fmt.Errorf("Failed to ZAdd cpe name. err: %s", result.Err())
				}
				if result := pipe.ZAdd(
					hashKeyPrefix+"CpeName",
					redis.Z{Score: 0, Member: cpe.CpeName},
				); result.Err() != nil {
					return fmt.Errorf("Failed to ZAdd cpe. err: %s", result.Err())
				}
			}
		}
		if _, err = pipe.Exec(); err != nil {
			return fmt.Errorf("Failed to exec pipeline. err: %s", err)
		}
	}
	bar.Finish()

	log.Infof("Refreshed %d Nvds.", len(refreshedNvds))
	//  log.Debugf("%v", refreshedNvds)
	return nil
}
