package commands

import (
	"context"
	"flag"
	"io/ioutil"
	"os"
	"strconv"
	"time"

	"github.com/google/subcommands"
	c "github.com/kotakanbe/go-cve-dictionary/config"
	db "github.com/kotakanbe/go-cve-dictionary/db"
	"github.com/kotakanbe/go-cve-dictionary/jvn"
	log "github.com/kotakanbe/go-cve-dictionary/log"
	util "github.com/kotakanbe/go-cve-dictionary/util"
)

// FetchJvnCmd is Subcommand for fetch JVN information.
type FetchJvnCmd struct {
	debug    bool
	debugSQL bool
	quiet    bool
	logDir   string

	dbpath   string
	dbtype   string
	dumpPath string

	latest bool
	last2Y bool
	years  bool

	httpProxy string
}

// Name return subcommand name
func (*FetchJvnCmd) Name() string { return "fetchjvn" }

// Synopsis return synopsis
func (*FetchJvnCmd) Synopsis() string { return "Fetch Vulnerability dictionary from JVN" }

// Usage return usage
func (*FetchJvnCmd) Usage() string {
	return `fetchjvn:
	fetchjvn
		[-latest]
		[-last2y]
		[-years] 1998 1999 ...
		[-dbpath=$PWD/cve.sqlite3 or connection string]
		[-dbtype=mysql|postgres|sqlite3|redis]
		[-http-proxy=http://192.168.0.1:8080]
		[-debug]
		[-debug-sql]
		[-quiet]
		[-log-dir=/path/to/log]

`
}

// SetFlags set flag
func (p *FetchJvnCmd) SetFlags(f *flag.FlagSet) {
	f.BoolVar(&p.debug, "debug", false, "debug mode")
	f.BoolVar(&p.debugSQL, "debug-sql", false, "SQL debug mode")
	f.BoolVar(&p.quiet, "quiet", false, "quiet mode (no output)")

	defaultLogDir := util.GetDefaultLogDir()
	f.StringVar(&p.logDir, "log-dir", defaultLogDir, "/path/to/log")

	pwd := os.Getenv("PWD")
	f.StringVar(&p.dbpath, "dbpath", pwd+"/cve.sqlite3",
		"/path/to/sqlite3 or SQL connection string")

	f.StringVar(&p.dbtype, "dbtype", "sqlite3",
		"Database type to store data in (sqlite3,  mysql, postgres or redis supported)")

	f.BoolVar(&p.latest, "latest", false,
		"Refresh JVN data for latest.")

	f.BoolVar(&p.last2Y, "last2y", false,
		"Refresh JVN data in the last two years.")

	f.BoolVar(&p.years, "years", false,
		"Refresh JVN data of specific years.")

	f.StringVar(
		&p.httpProxy,
		"http-proxy",
		"",
		"http://proxy-url:port (default: empty)",
	)
}

// Execute execute
func (p *FetchJvnCmd) Execute(_ context.Context, f *flag.FlagSet, _ ...interface{}) subcommands.ExitStatus {
	c.Conf.Quiet = p.quiet
	if c.Conf.Quiet {
		log.Initialize(p.logDir, ioutil.Discard)
	} else {
		log.Initialize(p.logDir, os.Stderr)
	}

	c.Conf.DebugSQL = p.debugSQL
	c.Conf.Debug = p.debug
	if c.Conf.Debug {
		log.SetDebug()
	}

	c.Conf.DBPath = p.dbpath
	c.Conf.DBType = p.dbtype
	//  c.Conf.DumpPath = p.dumpPath
	c.Conf.HTTPProxy = p.httpProxy

	if !c.Conf.Validate() {
		return subcommands.ExitUsageError
	}

	years := []int{}
	thisYear := time.Now().Year()

	switch {
	case p.latest:
		//no op
	case p.last2Y:
		for i := 0; i < 2; i++ {
			years = append(years, thisYear-i)
		}
	case p.years:
		if len(f.Args()) == 0 {
			log.Errorf("Specify years to fetch (from 1998 to %d)", thisYear)
			return subcommands.ExitUsageError
		}
		for _, arg := range f.Args() {
			year, err := strconv.Atoi(arg)
			if err != nil || year < 1998 || time.Now().Year() < year {
				log.Errorf("Specify years to fetch (from 1998 to %d), arg: %s", thisYear, arg)
				return subcommands.ExitUsageError
			}
			found := false
			for _, y := range years {
				if y == year {
					found = true
					break
				}
			}
			if !found {
				years = append(years, year)
			}
		}
	default:
		log.Errorf("specify -latest, -last2y or -years")
		return subcommands.ExitUsageError
	}

	log.Infof("Fetching CVE information from JVN.")
	items, err := jvn.FetchJvn(years)
	if err != nil {
		return subcommands.ExitFailure
	}
	log.Infof("Fetched %d CVEs", len(items))

	var driver db.DB
	if driver, err = db.NewDB(c.Conf.DBType, c.Conf.DBPath, c.Conf.DebugSQL); err != nil {
		log.Error(err)
		return subcommands.ExitFailure
	}
	defer driver.CloseDB()

	log.Infof("Inserting JVN into DB (%s).", driver.Name())
	if err := driver.InsertJvn(items); err != nil {
		log.Fatalf("Failed to insert. dbpath: %s, err: %s", c.Conf.DBPath, err)
		return subcommands.ExitFailure
	}
	return subcommands.ExitSuccess
}
