/*  GNU Robbo
 *  Copyright (C) notes:
 *  An Idea and Atari version: LK Avalon, Janusz Pelc, 1989
 *                 Linux Code: Arkadiusz Lipiec, 2002-2009
 *                                 <arkadiusz.lipiec@gmail.com>
 *                             Thunor 2007-2009
 *                                 <thunorsif@hotmail.com>
 *
 *  GNU Robbo is free software - you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  GNU Robbo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the impled warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with GNU CC; see the file COPYING. If not, write to the
 *  Free Software Foundation, 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 *
 */

#include <SDL/SDL_main.h>
#include "game.h"

/* Defines */
/*
#define DEBUG_MAIN
#define DEBUG_DEMO_MODE
*/
#define DEBUG_DUMP_VM_USAGE

#define MESSAGE_BOX_GETKEY_ID "getkey"
#define MESSAGE_BOX_GETKEY_ERROR_ID "getkeyerror"
#define MESSAGE_BOX_SAVING_ID "saving"
#define MESSAGE_BOX_RESTORE_ID "restore"

#define DELAY_MESSAGE_BOX_GETKEY (5 * game_cycle_limit)			/* ms */
#define DELAY_MESSAGE_BOX_GETKEY_ERROR (1.5 * game_cycle_limit)	/* ms */
#define DELAY_MESSAGE_BOX_SAVING (1.5 * game_cycle_limit)		/* ms */
#define DELAY_MESSAGE_BOX_RESTORE (1.5 * game_cycle_limit)		/* ms */

#define DELAY_DEMO (8 * game_cycle_limit)	/* ms */

#define DEMO_MODE_LENGTH 128
#define DEMO_MODE_DEMO_COUNT 13	/* Equal to number of demos + 1 */
#define DEMO_MODE_RECORD_SLOT DEMO_MODE_DEMO_COUNT - 1

/* Variables */
SDL_TimerID game_timer_id;	/* For timing */
int game_next_cycle;		/* A flag which will be set to TRUE by the timer */
unsigned int next_rand;		/* seed for random function */
int calling_game_mode;		/* So we don't lose the calling state */

/* Function prototypes */
Uint32 game_timer(Uint32 interval, void *param);
void clean_up_before_exit(void);


/***************************************************/
/* Random number choosing **************************/
/***************************************************/

int my_rand(void) {
	next_rand = next_rand * 103515245 + 12345;
	return (next_rand >> 7);
}

/***************************************************/
/*** Random generator init :) **********************/
/***************************************************/ 

void my_srand(unsigned int seed) {
	next_rand += seed;
}

/***************************************************************************
 * Main                                                                    *
 ***************************************************************************/
/* Main loop */

#ifdef __cplusplus
	extern "C"
#endif
int main(int argc, char *argv[]) {
	int getkey_timeout = 0, getkey_count = 0, getkey_state = SDL_RELEASED;
	char temp_default_joystick_name[MAX_JOYSTICK_NAME_LENGTH];
	int quit = 0, count, result, actionid;
	int gua_device, gua_id, gua_state;
	char path_resource_file[100];
	#ifdef DEBUG_DUMP_VM_USAGE
		char vmusage[80];
	#endif
	time_t t;
	
	#ifdef DEBUG_MAIN
		printf("*** Start %s ***\n", __func__);
		printf("size of int: %i\n", sizeof(int));
		printf("SDL_PRESSED=%i\n", SDL_PRESSED);
		printf("SDL_RELEASED=%i\n", SDL_RELEASED);
		printf("*** Stop %s ***\n", __func__);
	#endif

	/* If these aren't NULL when loading a skin then they must be freed first */
	icons = NULL;
	ciphers = NULL;
	alpha = NULL;
	font = NULL;
	icon = NULL;

	/* Set some defaults */
	time(&t);
	my_srand(t);	/* random generator init */
 	game_mode = INTRO_SCREEN;
	game_cycle_delay = 10;
	game_cycle_limit = DEFAULT_GAME_CYCLE_LIMIT;
	game_next_cycle = TRUE;
	cycle_count = 0;
 	introscreenmenuposition = 0;
 	helpscreenpage = 0;
 	helpscreenmenuposition = 1;
 	optionsscreenpage = 0;
 	optionsscreenmenuposition[0] = 0;
 	optionsscreenmenuposition[1] = OPTIONS_MENUPOS_EXIT;
 	optionsscreenmenuposition[2] = 0;
 	optionsscreenmenuposition[3] = 0;
 	optionsscreenmenuposition[4] = 0;
 	optionsscreenmenuposition[5] = 0;
	restart_timeout = -1;
	key_repeat_delay = DEFAULT_KEY_REPEAT_DELAY;
	key_repeat_interval = DEFAULT_KEY_REPEAT_INTERVAL;
	default_joystick = UNDEFINED;
	strcpy(default_joystick_name, "-1");
 	joystick_dead_zone = JOYSTICK_DEAD_ZONE;
	game_mechanics.sensible_bears = MECHANIC_SENSIBLE_BEARS;
	game_mechanics.sensible_questionmarks = MECHANIC_SENSIBLE_QUESTIONMARKS;
	game_mechanics.sensible_solid_lasers = MECHANIC_SENSIBLE_SOLID_LASERS;
	video.xres = video.yres = video.field_size = video.fullscreen = UNDEFINED;
	viewport.max_w = viewport.max_h = viewport.maximise = UNDEFINED;
 	volume = 0;
	
	/* Set-up the default user controls */
	set_default_user_controls(user_controls);
	
	/* Process any command line arguments. These will override any found in any resource file. */
	if (argc > 1) {
		for (count = 1; count < argc; count++) {
			if (!strcmp(argv[count], "--help")) {
							 /*  1234567890123456789012345678901234567890 <- Formatting for small terminal. */
				fprintf(stdout, "GNU Robbo version %s\n", VERSION);
				fprintf(stdout, "Usage: gnurobbo [options]\n");
				fprintf(stdout, "\nOptions:\n");
				fprintf(stdout, "  -XRESxYRES  e.g. -800x480\n");
				fprintf(stdout, "  -ts n       Tile size 16 or 32\n");
				fprintf(stdout, "  -vpx        Maximise the viewport\n");
				fprintf(stdout, "  -f          Run the game fullscreen\n");
				fprintf(stdout, "\nReport bugs to :-\n");
				fprintf(stdout, "http://sourceforge.net/projects/gnurobbo\n");
				return 1;
			} else if (!strcmp(argv[count], "-f")) {
				video.fullscreen = SDL_FULLSCREEN;
			} else if (!strcmp(argv[count], "-vpx")) {
				viewport.maximise = TRUE;
			} else if (!strcmp(argv[count], "-ts")) {
				if (count < argc - 1 &&
					sscanf(argv[count + 1], "%i", &video.field_size) == 1 &&
					(video.field_size == 16 || video.field_size == 32)) {
					count++;
				} else {
					fprintf(stdout, "Invalid tile size: Valid sizes are 16 and 32.\n");
					return 1;
				}
			} else if (sscanf(argv[count], "-%ix%i", &video.xres, &video.yres) == 2) {
				if (video.xres < 240 || video.yres < 240) {
					fprintf(stdout, "Invalid resolution: A minimum of 240x240 is required.\n");
					return 1;
				}
			} else {
				fprintf(stdout, "Unrecognised option %s.\n", argv[count]);
				return 1;
			}
		}
		/* Now we can check that the field_size is valid for the requested resolution if any */ 
		if (video.field_size == 32 &&
			((video.xres != UNDEFINED && video.xres < 480) || (video.yres != UNDEFINED && video.yres < 480))) {
			fprintf(stdout, "Invalid tile size: A size of 32 requires a minimum resolution of 480x480.\n");
			return 1;
		}
	}
	#ifdef DEBUG_MAIN
		printf("*** Start %s ***\n", __func__);
		printf("video.fullscreen=%i\n", video.fullscreen);
		printf("video.xres=%i\n", video.xres);
		printf("video.yres=%i\n", video.yres);
		printf("video.field_size=%i\n", video.field_size);
		printf("viewport.maximise=%i\n", viewport.maximise);
		printf("*** Stop %s ***\n", __func__);
	#endif

	printf("PACKAGE_DATA_DIR is %s\n", PACKAGE_DATA_DIR);

	/* Initialise the level pack list and currently selected pack and level */
	found_pack_count = level_pack_count = 1;
	selected_pack = 0;
	strcpy(level_packs[selected_pack].filename, DEFAULT_LEVEL_PACK);
	strcpy(level_packs[selected_pack].name, "-1");						/* We don't know yet */
	level_packs[selected_pack].last_level = DEFAULT_LEVEL_START;		/* We don't know yet */
	level_packs[selected_pack].level_reached = DEFAULT_LEVEL_START;		/* The player hasn't reached any further than 1 yet */
	level_packs[selected_pack].level_selected = DEFAULT_LEVEL_START;	/* Level 1 is selected for play */
	level_packs[selected_pack].selected = TRUE;							/* Original pack is selected for play */
	
	/* Find all level packs */
	if (find_all_dat_files()) return 1;

 	/* Read values from the level packs */
	read_level_packs();
	
	/* Initialise the skin list and currently selected skin */
	skin_count = 1;
	selected_skin = 0;
	strcpy(skins[selected_skin].foldername, DEFAULT_SKIN);
	
	/* Find all skins */
	if (find_all_skins()) return 1;

 	/* Read values from the skinrc files */
 	read_skinrc_files();

	/* Initialise the locale list and currently selected locale */
	locale_count = 1;
	selected_locale = 0;
	strcpy(locales[selected_locale].foldername, DEFAULT_LOCALE);
	
	/* Find all locales */
	if (find_all_locales()) return 1;

 	/* Read values from the localerc files */
 	read_localerc_files();

 	/* Set-up the resource file path for the required platform and read it now */
	#if defined(PLATFORM_WIN32)
		strcpy(path_resource_file, "");
 	#elif defined(PLATFORM_GP2X)
		strcpy(path_resource_file, "./");
 	#elif defined(PLATFORM_ZAURUS)
		strcpy(path_resource_file, getenv("HOME"));
		strcat(path_resource_file, "/");
 	#elif defined(PLATFORM_PC)
		strcpy(path_resource_file, getenv("HOME"));
		strcat(path_resource_file, "/");
 	#endif
	strcat(path_resource_file, RESOURCE_FILE);
	printf("RESOURCE_FILE is %s\n", path_resource_file);
	result = read_resource_file(path_resource_file);
 	if (result == 1) {
 		/* Write a new rc file */
		save_resource_file(path_resource_file);
 	} else if (result == 2) {
 		/* rc data is corrupt */
		return 1;
 	}

	#ifdef DEBUG_MAIN
		printf("*** Start %s ***\n", __func__);
		printf("skin_count=%i\n", skin_count);
		printf("selected_skin=%i\n", selected_skin);
		for (count = 0; count < skin_count; count++) {
			printf("skins[%i].foldername=%s\n", count, skins[count].foldername);
			printf("skins[%i].name=%s\n", count, skins[count].name);
			printf("skins[%i].author=%s\n", count, skins[count].author);
			printf("skins[%i].about=%s\n", count, skins[count].about);
			printf("skins[%i].background_colour=0x%06X\n", count, skins[count].background_colour);
			printf("skins[%i].version_text_colour=0x%06X\n", count, skins[count].version_text_colour);
			printf("skins[%i].general_text_colour=0x%06X\n", count, skins[count].general_text_colour);
			printf("skins[%i].menu_text_colour=0x%06X\n", count, skins[count].menu_text_colour);
			printf("skins[%i].menu_selected_text_colour=0x%06X\n", count, skins[count].menu_selected_text_colour);
			printf("skins[%i].menu_selected_limit_text_colour=0x%06X\n", count, skins[count].menu_selected_limit_text_colour);
			printf("skins[%i].menu_greyed_text_colour=0x%06X\n", count, skins[count].menu_greyed_text_colour);
			printf("skins[%i].default_controls_text_colour=0x%06X\n", count, skins[count].default_controls_text_colour);
			printf("skins[%i].credits_text_colour=0x%06X\n", count, skins[count].credits_text_colour);
			printf("skins[%i].help_tile_colour=0x%06X\n", count, skins[count].help_tile_colour);
			printf("skins[%i].fade_colour=0x%06X\n", count, skins[count].fade_colour);
			printf("skins[%i].author_text_colour=0x%06X\n", count, skins[count].author_text_colour);
		}
		printf("*** Stop %s ***\n", __func__);
		printf("*** Start %s ***\n", __func__);
		printf("found_pack_count=%i\n", found_pack_count);
		printf("level_pack_count=%i\n", level_pack_count);
		printf("selected_pack=%i\n", selected_pack);
		for (count = 0; count < level_pack_count; count++) {
			printf("level_packs[%i].filename=%s\n", count, level_packs[count].filename);
			printf("level_packs[%i].name=%s\n", count, level_packs[count].name);
			printf("level_packs[%i].last_level=%i\n", count, level_packs[count].last_level);
			printf("level_packs[%i].level_reached=%i\n", count, level_packs[count].level_reached);
			printf("level_packs[%i].level_selected=%i\n", count, level_packs[count].level_selected);
			printf("level_packs[%i].selected=%i\n", count, level_packs[count].selected);
		}
		printf("*** Stop %s ***\n", __func__);
	#endif

	/* Sort level packs here */
	sort_level_packs();

	/* Sort skins here */
	sort_skins();

	/* Sort locales here */
	sort_locales();
	
	/* Initialise SDL */
	if(SDL_Init(SDL_INIT_EVERYTHING)) {
		fprintf(stdout, "Cannot initialise SDL: %s",SDL_GetError());
		return 1;
	}

	atexit(clean_up_before_exit);
	
	/* Initialise the joystick list and attempt to open one */
	if ((joystick_count = get_joystick_list(joystick_list, TRUE))) {
		/* Set default joystick */
		if ((default_joystick = initialise_joystick(default_joystick, default_joystick_name, TRUE)) == UNDEFINED)
			default_joystick = initialise_joystick(UNDEFINED, default_joystick_name, TRUE);
	}
	
	/* Set the video mode */
	if (set_video_mode()) {
		fprintf(stdout, "Cannot initialise screen: %s\n",SDL_GetError());
		exit(1);  
	}
	
	/* Initiate a 10ms timer */
	game_timer_id = SDL_AddTimer(10, game_timer, NULL);

	/* Font initialisation */ 	
   	if(TTF_Init()) {
   		fprintf(stdout, "Cannot initialise SDL_ttf module\n");
   		exit(1);
 	}

	/* Load selected skin */
	load_selected_skin();

	/* Load selected locale */
	load_selected_locale();

	SDL_WM_SetCaption("GNU Robbo", "gnurobbo");
	SDL_WM_SetIcon(icon, NULL);	/* Technically this should be called before SDL_SetVideoMode but it works in Xfce4 */

	show_introscreen(REDRAW_INITIALISE);
	
	demo_mode(DEMO_MODE_INITIALISE, 0);

	#ifdef DEBUG_DUMP_VM_USAGE
		sprintf(vmusage, "cat /proc/%u/status | grep Vm > vmusage.txt", getpid());
		system (vmusage);
	#endif

	while(quit == 0) {
		if(game_mode == GAME_ON) {
		/***************************************************************************
		 * GAME ON                                                                 *
		 ***************************************************************************/

			/* Get none or one user action and quit on SDL_QUIT */
			quit = get_user_action(&actionid, TRUE, &gua_device, &gua_id, &gua_state);

			#ifdef DEBUG_RECORD_DEMO
				if (actionid != UNDEFINED && !demo_mode(DEMO_MODE_IS_ACTIVE, 0) && robbo.moved == 0) demo_mode(DEMO_MODE_RECORD, actionid);
			#endif

			/* The restart timeout counts down to a forced restart such as following a BIG_BOOM */
			if (restart_timeout) {
				restart_timeout--;
			} else if (restart_timeout == 0) {
				restart_timeout--;
				actionid = ACTION_RESTART;
			}

			/* If demo mode is active then substitute the actionid with one from the demo */
			if (demo_mode(DEMO_MODE_IS_ACTIVE, 0)) {
				if (actionid == UNDEFINED) {
					if (robbo.moved == 0) actionid = demo_mode(DEMO_MODE_PLAYBACK, 0);
				} else {
					actionid = ACTION_EXIT;	/* Quit demo mode on any action */
				}
			}

			switch (actionid) {
				case ACTION_UP:
					/* Quit scrolling when Robbo is not teleporting and not in demo mode */
					if (!demo_mode(DEMO_MODE_IS_ACTIVE, 0) && !robbo.teleporting && viewport.cycles_to_dest > 1) viewport.cycles_to_dest = 1;
					/* Attempt to move Robbo to his intended destination */
					if (robbo.alive) move_robbo(0, -1);
					break;
				case ACTION_DOWN:
					/* Quit scrolling when Robbo is not teleporting and not in demo mode */
					if (!demo_mode(DEMO_MODE_IS_ACTIVE, 0) && !robbo.teleporting && viewport.cycles_to_dest > 1) viewport.cycles_to_dest = 1;
					/* Attempt to move Robbo to his intended destination */
					if (robbo.alive) move_robbo(0, 1);
					break;
				case ACTION_LEFT:
					/* Quit scrolling when Robbo is not teleporting and not in demo mode */
					if (!demo_mode(DEMO_MODE_IS_ACTIVE, 0) && !robbo.teleporting && viewport.cycles_to_dest > 1) viewport.cycles_to_dest = 1;
					/* Attempt to move Robbo to his intended destination */
					if (robbo.alive) move_robbo(-1, 0);
					break;
				case ACTION_RIGHT:
					/* Quit scrolling when Robbo is not teleporting and not in demo mode */
					if (!demo_mode(DEMO_MODE_IS_ACTIVE, 0) && !robbo.teleporting && viewport.cycles_to_dest > 1) viewport.cycles_to_dest = 1;
					/* Attempt to move Robbo to his intended destination */
					if (robbo.alive) move_robbo(1, 0);
					break;
				case ACTION_SHOOT_UP:
					if (robbo.alive) shoot_robbo(0, -1);
					break;
				case ACTION_SHOOT_DOWN:
					if (robbo.alive) shoot_robbo(0, 1);
					break;
				case ACTION_SHOOT_LEFT:
					if (robbo.alive) shoot_robbo(-1, 0);
					break;
				case ACTION_SHOOT_RIGHT:
					if (robbo.alive) shoot_robbo(1, 0);
					break;
				case ACTION_SELECT:
					break;
				case ACTION_EXIT:
					/* If demo mode is active then deactivate it */
					demo_mode(DEMO_MODE_DEACTIVATE, 0);
					/* Cancel any active or pending fades */
					restart_timeout = -1;
					show_game_area_fade(REDRAW_INITIALISE, 0);
					game_mode = INTRO_SCREEN;
					show_introscreen(REDRAW_INITIALISE);
					break;
				case ACTION_HELP:
					/* Cancel any active fade */
					show_game_area_fade(REDRAW_INITIALISE, 0);
					calling_game_mode = game_mode;
					game_mode = HELP_SCREEN;
					show_helpscreen(REDRAW_INITIALISE);
					break;
				case ACTION_OPTIONS:
					/* Cancel any active fade */
					show_game_area_fade(REDRAW_INITIALISE, 0);
					calling_game_mode = game_mode;
					game_mode = OPTIONS_SCREEN;
					show_optionsscreen(REDRAW_INITIALISE);
					break;
				case ACTION_RESTART:
					restart_timeout = -1;
					if (level_init()) {
						game_mode = INTRO_SCREEN;
						show_introscreen(REDRAW_INITIALISE);
					} else {
						show_game_area(REDRAW_INITIALISE);
						/* Initialise the fade */
						show_game_area_fade(REDRAW_INITIALISE, 17);
					}
					break;
				case ACTION_PREVIOUS_LEVEL:
					if (level_packs[selected_pack].level_selected > 1) {
						level_packs[selected_pack].level_selected--;
						/* Cancel any active or pending fades */
						restart_timeout = -1;
						show_game_area_fade(REDRAW_INITIALISE, 0);
						if (level_init()) {
							game_mode = INTRO_SCREEN;
							show_introscreen(REDRAW_INITIALISE);
						} else {
							show_game_area(REDRAW_INITIALISE);
						}
					}
					break;
				case ACTION_NEXT_LEVEL:
					if (level_packs[selected_pack].level_selected < level_packs[selected_pack].level_reached) {
						level_packs[selected_pack].level_selected++;
						/* Cancel any active or pending fades */
						restart_timeout = -1;
						show_game_area_fade(REDRAW_INITIALISE, 0);
						if (level_init()) {
							game_mode = INTRO_SCREEN;
							show_introscreen(REDRAW_INITIALISE);
						} else {
							show_game_area(REDRAW_INITIALISE);
						}
					}
					break;
				case ACTION_PREVIOUS_PACK:
					if (selected_pack > 0) {
						level_packs[selected_pack].selected = FALSE;
						selected_pack--;
						level_packs[selected_pack].selected = TRUE;
						/* Cancel any active or pending fades */
						restart_timeout = -1;
						show_game_area_fade(REDRAW_INITIALISE, 0);
						if (level_init()) {
							game_mode = INTRO_SCREEN;
							show_introscreen(REDRAW_INITIALISE);
						} else {
							show_game_area(REDRAW_INITIALISE);
						}
					}
					break;
				case ACTION_NEXT_PACK:
					if (selected_pack < found_pack_count - 1) {
						level_packs[selected_pack].selected = FALSE;
						selected_pack++;
						level_packs[selected_pack].selected = TRUE;
						/* Cancel any active or pending fades */
						restart_timeout = -1;
						show_game_area_fade(REDRAW_INITIALISE, 0);
						if (level_init()) {
							game_mode = INTRO_SCREEN;
							show_introscreen(REDRAW_INITIALISE);
						} else {
							show_game_area(REDRAW_INITIALISE);
						}
					}
					break;
				case ACTION_MODIFIER1:
				case ACTION_MODIFIER2:
					break;
				case ACTION_TOGGLE_FULLSCREEN:
					toggle_fullscreen(&video.fullscreen);
					break;
				case ACTION_HOME:
				case ACTION_END:
					break;
				case ACTION_PAGEUP:
					#ifdef DEBUG_COLOUR_SELECT
						debug_colour_select_component--;	/* R = 0, G = 1, B = 2 */
						if (debug_colour_select_component < 0) debug_colour_select_component = 2;
					#endif
					break;
				case ACTION_PAGEDOWN:
					#ifdef DEBUG_COLOUR_SELECT
						debug_colour_select_component++;	/* R = 0, G = 1, B = 2 */
						if (debug_colour_select_component > 2) debug_colour_select_component = 0;
					#endif
					break;
				case ACTION_VOLUP:
					#ifdef DEBUG_COLOUR_SELECT
						if (debug_colour_select_component == 0) {
							inc_colour_component(&debug_colour_select_r);
						} else if (debug_colour_select_component == 1) {	
							inc_colour_component(&debug_colour_select_g);
						} else if (debug_colour_select_component == 2) {	
							inc_colour_component(&debug_colour_select_b);
						}
						level.colour_override = debug_colour_select_r << 16 | debug_colour_select_g << 8 | debug_colour_select_b;
						show_game_area(REDRAW_INITIALISE);
					#else
						volume_up();
					#endif
					break;
				case ACTION_VOLDOWN:
					#ifdef DEBUG_COLOUR_SELECT
						if (debug_colour_select_component == 0) {
							dec_colour_component(&debug_colour_select_r);
						} else if (debug_colour_select_component == 1) {	
							dec_colour_component(&debug_colour_select_g);
						} else if (debug_colour_select_component == 2) {	
							dec_colour_component(&debug_colour_select_b);
						}
						level.colour_override = debug_colour_select_r << 16 | debug_colour_select_g << 8 | debug_colour_select_b;
						show_game_area(REDRAW_INITIALISE);
					#else
						volume_down();
					#endif
					break;
				default:
					break;
			}
			
			if (game_mode == GAME_ON) {
				update_game();
				show_game_area(REDRAW_ANIMATED);
				#ifdef DEBUG_COLOUR_SELECT
					show_level_colour(debug_colour_select_r, debug_colour_select_g, debug_colour_select_b, debug_colour_select_component);
				#endif
			}
			
		} else if(game_mode == INTRO_SCREEN) {
		/***************************************************************************
		 * INTRO SCREEN                                                            *
		 ***************************************************************************/
			/* Draw the GNU Robbo banner */
			show_introscreen(REDRAW_ANIMATED);

			/* Get none or one user action and quit on SDL_QUIT */
			quit = get_user_action(&actionid, TRUE, &gua_device, &gua_id, &gua_state);
			
			/* Reset the demo mode timeout if a key was pressed */
			if (actionid != UNDEFINED) demo_mode(DEMO_MODE_TIMEOUT_INITIALISE, 0);

			/* Decrement the demo mode timeout which will automatically force
			   demo mode after a predetermined period of time */
			demo_mode(DEMO_MODE_TIMEOUT_DECREMENT, 0);
		
			switch (actionid) {
				case ACTION_UP:
					introscreenmenuposition--;
					if (introscreenmenuposition < 0) introscreenmenuposition = 5;
					show_introscreen(REDRAW_INTERMEDIATE);
					break;
				case ACTION_DOWN:
					introscreenmenuposition++;
					if (introscreenmenuposition > 5) introscreenmenuposition = 0;
					show_introscreen(REDRAW_INTERMEDIATE);
					break;
				case ACTION_LEFT:
					if (introscreenmenuposition == 1) {
						/* Previous level */
						if (level_packs[selected_pack].level_selected > 1) {
							level_packs[selected_pack].level_selected--;
							show_introscreen(REDRAW_INTERMEDIATE);
						}
					} else if (introscreenmenuposition == 2) {
						/* Previous pack */
						if (selected_pack > 0) {
							level_packs[selected_pack].selected = FALSE;
							selected_pack--;
							level_packs[selected_pack].selected = TRUE;
							show_introscreen(REDRAW_INTERMEDIATE);
						}
					}
					break;
				case ACTION_RIGHT:
					if (introscreenmenuposition == 1) {
						/* Next level */
						if (level_packs[selected_pack].level_selected < level_packs[selected_pack].level_reached) {
							level_packs[selected_pack].level_selected++;
							show_introscreen(REDRAW_INTERMEDIATE);
						}
					} else if (introscreenmenuposition == 2) {
						/* Next pack */
						if (selected_pack < found_pack_count - 1) {
							level_packs[selected_pack].selected = FALSE;
							selected_pack++;
							level_packs[selected_pack].selected = TRUE;
							show_introscreen(REDRAW_INTERMEDIATE);
						}
					}
					break;
				case ACTION_SHOOT_UP:
				case ACTION_SHOOT_DOWN:
				case ACTION_SHOOT_LEFT:
				case ACTION_SHOOT_RIGHT:
					break;
				case ACTION_SELECT:
					if (introscreenmenuposition == 0 || introscreenmenuposition == 1 || introscreenmenuposition == 2) {
						/* Game on */
						game_mode = GAME_ON;
						if (level_init()) {
							game_mode = INTRO_SCREEN;
							show_introscreen(REDRAW_INITIALISE);
						} else {
							show_game_area(REDRAW_INITIALISE);
							/* Initialise the fade */
							show_game_area_fade(REDRAW_INITIALISE, 16);
						}
					} else if (introscreenmenuposition == 3) {
						/* Help */
						calling_game_mode = game_mode;
						game_mode = HELP_SCREEN;
						show_helpscreen(REDRAW_INITIALISE);
					} else if (introscreenmenuposition == 4) {
						/* Options */
						calling_game_mode = game_mode;
						game_mode = OPTIONS_SCREEN;
						show_optionsscreen(REDRAW_INITIALISE);
					} else if (introscreenmenuposition == 5) {
						/* Exit */
						quit = 1;
					}
					break;
				case ACTION_EXIT:
					/* quit = 1; */
					introscreenmenuposition = 5;
					show_introscreen(REDRAW_INTERMEDIATE);
					break;
				case ACTION_HELP:
					calling_game_mode = game_mode;
					game_mode = HELP_SCREEN;
					show_helpscreen(REDRAW_INITIALISE);
					break;
				case ACTION_OPTIONS:
					calling_game_mode = game_mode;
					game_mode = OPTIONS_SCREEN;
					show_optionsscreen(REDRAW_INITIALISE);
					break;
				case ACTION_RESTART:
					break;
				case ACTION_PREVIOUS_LEVEL:
					if (level_packs[selected_pack].level_selected > 1) {
						level_packs[selected_pack].level_selected--;
						show_introscreen(REDRAW_INTERMEDIATE);
					}
					break;
				case ACTION_NEXT_LEVEL:
					if (level_packs[selected_pack].level_selected < level_packs[selected_pack].level_reached) {
						level_packs[selected_pack].level_selected++;
						show_introscreen(REDRAW_INTERMEDIATE);
					}
					break;
				case ACTION_PREVIOUS_PACK:
					if (selected_pack > 0) {
						level_packs[selected_pack].selected = FALSE;
						selected_pack--;
						level_packs[selected_pack].selected = TRUE;
						show_introscreen(REDRAW_INTERMEDIATE);
					}
					break;
				case ACTION_NEXT_PACK:
					if (selected_pack < found_pack_count - 1) {
						level_packs[selected_pack].selected = FALSE;
						selected_pack++;
						level_packs[selected_pack].selected = TRUE;
						show_introscreen(REDRAW_INTERMEDIATE);
					}
					break;
				case ACTION_MODIFIER1:
				case ACTION_MODIFIER2:
					break;
				case ACTION_TOGGLE_FULLSCREEN:
					toggle_fullscreen(&video.fullscreen);
					break;
				case ACTION_HOME:
					if (introscreenmenuposition == 1) {
						/* First level */
						level_packs[selected_pack].level_selected = 1;
						show_introscreen(REDRAW_INTERMEDIATE);
					} else if (introscreenmenuposition == 2) {
						/* First pack */
						level_packs[selected_pack].selected = FALSE;
						selected_pack = 0;
						level_packs[selected_pack].selected = TRUE;
						show_introscreen(REDRAW_INTERMEDIATE);
					} else {
						/* Go to top of menu */
						introscreenmenuposition = 0;
						show_introscreen(REDRAW_INTERMEDIATE);
					}
					break;
				case ACTION_END:
					if (introscreenmenuposition == 1) {
						/* Last level */
						level_packs[selected_pack].level_selected = level_packs[selected_pack].level_reached;
						show_introscreen(REDRAW_INTERMEDIATE);
					} else if (introscreenmenuposition == 2) {
						/* Last pack */
						level_packs[selected_pack].selected = FALSE;
						selected_pack = found_pack_count - 1;
						level_packs[selected_pack].selected = TRUE;
						show_introscreen(REDRAW_INTERMEDIATE);
					} else {
						/* Go to bottom of menu */
						introscreenmenuposition = 5;
						show_introscreen(REDRAW_INTERMEDIATE);
					}
					break;
				case ACTION_PAGEUP:
				case ACTION_PAGEDOWN:
					break;
				case ACTION_VOLUP:
					volume_up();
					break;
				case ACTION_VOLDOWN:
					volume_down();
					break;
				default:
					break;
			}
		} else if(game_mode == END_SCREEN) {
		/***************************************************************************
		 * END SCREEN                                                              *
		 ***************************************************************************/
			/* Get none or one user action and quit on SDL_QUIT */
			quit = get_user_action(&actionid, TRUE, &gua_device, &gua_id, &gua_state);
			switch (actionid) {
				case ACTION_UP:
				case ACTION_DOWN:
				case ACTION_LEFT:
				case ACTION_RIGHT:
				case ACTION_SHOOT_UP:
				case ACTION_SHOOT_DOWN:
				case ACTION_SHOOT_LEFT:
				case ACTION_SHOOT_RIGHT:
				case ACTION_SELECT:
					break;
				case ACTION_EXIT:
					game_mode = INTRO_SCREEN;
					show_introscreen(REDRAW_INITIALISE);
					break;
				case ACTION_HELP:
				case ACTION_OPTIONS:
				case ACTION_RESTART:
				case ACTION_PREVIOUS_LEVEL:
				case ACTION_NEXT_LEVEL:
				case ACTION_PREVIOUS_PACK:
				case ACTION_NEXT_PACK:
				case ACTION_MODIFIER1:
				case ACTION_MODIFIER2:
					break;
				case ACTION_TOGGLE_FULLSCREEN:
					toggle_fullscreen(&video.fullscreen);
					break;
				case ACTION_HOME:
				case ACTION_END:
				case ACTION_PAGEUP:
				case ACTION_PAGEDOWN:
					break;
				case ACTION_VOLUP:
					volume_up();
					break;
				case ACTION_VOLDOWN:
					volume_down();
					break;
				default:
					break;
			}
		} else if(game_mode == HELP_SCREEN) {
		/***************************************************************************
		 * HELP SCREEN                                                             *
		 ***************************************************************************/
			/* Draw the Help banner */
			show_helpscreen(REDRAW_ANIMATED);
			
			/* Get none or one user action and quit on SDL_QUIT */
			quit = get_user_action(&actionid, TRUE, &gua_device, &gua_id, &gua_state);
			switch (actionid) {
				case ACTION_UP:
				case ACTION_DOWN:
					break;
				case ACTION_LEFT:
					if (helpscreenmenuposition == 0) {
						if (helpscreenpage == 1 || helpscreenpage == 2 || helpscreenpage == 3) {
							helpscreenpage--;
							helpscreenmenuposition = 2;
							show_helpscreen(REDRAW_EVERYTHING);
						}
					} else if (helpscreenmenuposition == 1) {
						if (helpscreenpage == 1 || helpscreenpage == 2 || helpscreenpage == 3) {
							helpscreenmenuposition = 0;
							show_helpscreen(REDRAW_INTERMEDIATE);
						}
					} else if (helpscreenmenuposition == 2) {
						helpscreenmenuposition = 1;
						show_helpscreen(REDRAW_INTERMEDIATE);
					}
					break;
				case ACTION_RIGHT:
					if (helpscreenmenuposition == 0) {
						helpscreenmenuposition = 1;
						show_helpscreen(REDRAW_INTERMEDIATE);
					} else if (helpscreenmenuposition == 1) {
						if (helpscreenpage == 0 || helpscreenpage == 1 || helpscreenpage == 2) {
							helpscreenmenuposition = 2;
							show_helpscreen(REDRAW_INTERMEDIATE);
						}
					} else if (helpscreenmenuposition == 2) {
						if (helpscreenpage == 0 || helpscreenpage == 1 || helpscreenpage == 2) {
							helpscreenpage++;
							helpscreenmenuposition = 0;
							show_helpscreen(REDRAW_EVERYTHING);
						}
					}
					break;
				case ACTION_SHOOT_UP:
				case ACTION_SHOOT_DOWN:
				case ACTION_SHOOT_LEFT:
				case ACTION_SHOOT_RIGHT:
					break;
				case ACTION_SELECT:
					if (helpscreenmenuposition == 1) {
						game_mode = calling_game_mode;	/* Return to calling state */
						if (game_mode == INTRO_SCREEN) {
							show_introscreen(REDRAW_INITIALISE);
						} else if (game_mode == GAME_ON) {
							show_game_area(REDRAW_INITIALISE);
						}
					}
					break;
				case ACTION_EXIT:
					game_mode = calling_game_mode;	/* Return to calling state */
					if (game_mode == INTRO_SCREEN) {
						show_introscreen(REDRAW_INITIALISE);
					} else if (game_mode == GAME_ON) {
						show_game_area(REDRAW_INITIALISE);
					}
					break;
				case ACTION_HELP:
				case ACTION_OPTIONS:
				case ACTION_RESTART:
				case ACTION_PREVIOUS_LEVEL:
				case ACTION_NEXT_LEVEL:
				case ACTION_PREVIOUS_PACK:
				case ACTION_NEXT_PACK:
				case ACTION_MODIFIER1:
				case ACTION_MODIFIER2:
					break;
				case ACTION_TOGGLE_FULLSCREEN:
					toggle_fullscreen(&video.fullscreen);
					break;
				case ACTION_HOME:
					/* First page */
					if (helpscreenpage != 0) {
						helpscreenpage = 0;
						helpscreenmenuposition = 1;
						show_helpscreen(REDRAW_EVERYTHING);
					}
					break;
				case ACTION_END:
					/* Last page */
					if (helpscreenpage != HELP_SCREEN_PAGES - 1) {
						helpscreenpage = HELP_SCREEN_PAGES - 1;
						helpscreenmenuposition = 1;
						show_helpscreen(REDRAW_EVERYTHING);
					}
					break;
				case ACTION_PAGEUP:
					/* Previous page */
					if (helpscreenpage > 0) {
						helpscreenpage--;
						helpscreenmenuposition = 1;
						show_helpscreen(REDRAW_EVERYTHING);
					}
					break;
				case ACTION_PAGEDOWN:
					/* Next page */
					if (helpscreenpage < HELP_SCREEN_PAGES - 1) {
						helpscreenpage++;
						helpscreenmenuposition = 1;
						show_helpscreen(REDRAW_EVERYTHING);
					}
					break;
				case ACTION_VOLUP:
					volume_up();
					break;
				case ACTION_VOLDOWN:
					volume_down();
					break;
				default:
					break;
			}
		} else if(game_mode == OPTIONS_SCREEN) {
		/***************************************************************************
		 * OPTIONS SCREEN                                                          *
		 ***************************************************************************/
			/* All the screens are laid out like this no matter what's displayed :-
			   --------------------
			   | 0                | Obviously 9 and 12 are missing on the first and
			   | .                | last pages.
			   | .                |
			   | 8                | There are defines for 9 to 12 in game.h
			   |                  |
			   | 9   10    11  12 |
			   | <  Save  Exit  > |
			   -------------------- */
			
			/* Draw the Options banner */
			show_optionsscreen(REDRAW_ANIMATED);
			
			/* Are we waiting for a key/button press to assign to a control? */
			if (getkey_timeout) {
				getkey_timeout--;
				if (getkey_timeout == 0) {
					show_message_box(REDRAW_INITIALISE, MESSAGE_BOX_GETKEY_ERROR_ID, DELAY_MESSAGE_BOX_GETKEY_ERROR, txt_No_input_was_detected, TRUE, 0, 0);
				} else {
					/* Get none or one user action and quit on SDL_QUIT */
					quit = get_user_action(&actionid, FALSE, &gua_device, &gua_id, &gua_state);
					/* Record the pressed/released state changes only */
					if ((gua_state == SDL_PRESSED || gua_state == SDL_RELEASED) && getkey_state != gua_state) {
						getkey_state = gua_state;
						getkey_count++;	/* Count the state changes */
						if (getkey_count == 3) {
							show_message_box(REDRAW_INITIALISE, "", -1, "", 0, 0, 0);	/* Kill the message box */
							getkey_timeout = 0;	/* Kill the timeout */
							/* Store the new control */
							temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].device = gua_device;
							temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].id = gua_id;
							show_optionsscreen(REDRAW_INTERMEDIATE);
						}
					}
				}
			/* No, so resume normal action servicing */
			} else {
				/* Get none or one user action and quit on SDL_QUIT */
				quit = get_user_action(&actionid, TRUE, &gua_device, &gua_id, &gua_state);
				switch (actionid) {
					case ACTION_UP:
						if (optionsscreenpage == 0) {
							if (optionsscreenmenuposition[optionsscreenpage] < 6) {
								optionsscreenmenuposition[optionsscreenpage]--;
								if ((optionsscreenmenuposition[optionsscreenpage] == 1 || optionsscreenmenuposition[optionsscreenpage] == 2) && joystick_count == 0) {
									optionsscreenmenuposition[optionsscreenpage] = 0;
								} else if (optionsscreenmenuposition[optionsscreenpage] < 0) {
									optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_SAVE;
								}
							} else {
								optionsscreenmenuposition[optionsscreenpage] = 5;
							}
							show_optionsscreen(REDRAW_INTERMEDIATE);
						} else if (optionsscreenpage >= 2 && optionsscreenpage <= 4) {
							if (optionsscreenmenuposition[optionsscreenpage] < OPTIONS_MENUPOS_LEFT) {
								optionsscreenmenuposition[optionsscreenpage]--;
								if (optionsscreenmenuposition[optionsscreenpage] < 0) optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_SAVE;
							} else {
								optionsscreenmenuposition[optionsscreenpage] = 8;
							}
							show_optionsscreen(REDRAW_INTERMEDIATE);
						} else if (optionsscreenpage == 5) {
							if (optionsscreenmenuposition[optionsscreenpage] < OPTIONS_MENUPOS_LEFT) {
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_SAVE;
							} else {
								optionsscreenmenuposition[optionsscreenpage] = 0;
							}
							show_optionsscreen(REDRAW_INTERMEDIATE);
						}
						break;
					case ACTION_DOWN:
						if (optionsscreenpage == 0) {
							if (optionsscreenmenuposition[optionsscreenpage] > 5) {
								optionsscreenmenuposition[optionsscreenpage] = 0;
							} else if (optionsscreenmenuposition[optionsscreenpage] < 5) {
								optionsscreenmenuposition[optionsscreenpage]++;
								if ((optionsscreenmenuposition[optionsscreenpage] == 1 || optionsscreenmenuposition[optionsscreenpage] == 2) && joystick_count == 0) optionsscreenmenuposition[optionsscreenpage] = 3;
							} else {
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_SAVE;
							}
							show_optionsscreen(REDRAW_INTERMEDIATE);
						} else if (optionsscreenpage >= 2 && optionsscreenpage <= 4) {
							if (optionsscreenmenuposition[optionsscreenpage] > 8) {
								optionsscreenmenuposition[optionsscreenpage] = 0;
							} else if (optionsscreenmenuposition[optionsscreenpage] < 8) {
								optionsscreenmenuposition[optionsscreenpage]++;
							} else {
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_SAVE;
							}
							show_optionsscreen(REDRAW_INTERMEDIATE);
						} else if (optionsscreenpage == 5) {
							if (optionsscreenmenuposition[optionsscreenpage] > 0) {
								optionsscreenmenuposition[optionsscreenpage] = 0;
							} else {
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_SAVE;
							}
							show_optionsscreen(REDRAW_INTERMEDIATE);
						}
						break;
					case ACTION_LEFT:
						if (optionsscreenpage == 0) {
							if (optionsscreenmenuposition[optionsscreenpage] == 0) {
								/* Increase speed */
								if (temp_game_cycle_limit < GAME_CYCLE_LIMIT_MAX) {
									temp_game_cycle_limit = 100 / (100 / temp_game_cycle_limit - 1);
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == 1) {
								/* Previous joystick */
								if (temp_default_joystick > 0) {
									temp_default_joystick--;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == 2) {
								/* Decrease joystick_dead_zone */
								if (temp_joystick_dead_zone > 1) {
									temp_joystick_dead_zone--;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == 3) {
								/* Decrease key_repeat_delay */
								if (temp_key_repeat_delay > KEY_REPEAT_MIN) {
									temp_key_repeat_delay -= 20;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == 4) {
								/* Decrease key_repeat_interval */
								if (temp_key_repeat_interval > KEY_REPEAT_MIN) {
									temp_key_repeat_interval -= 20;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == 5) {
								/* Decrease selected_locale */
								if (temp_selected_locale > 0) {
									temp_selected_locale--;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] > OPTIONS_MENUPOS_SAVE) {
								optionsscreenmenuposition[optionsscreenpage]--;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							}
						} else if (optionsscreenpage >= 1 && optionsscreenpage <= 4) {
							if (optionsscreenpage >= 2 && optionsscreenpage <= 4 && optionsscreenmenuposition[optionsscreenpage] < OPTIONS_MENUPOS_LEFT) {
								if (optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_NOT_USED1
									&& optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_MODIFIER1
									&& optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_MODIFIER2
									&& temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].device != UNDEFINED) {
									/* Previous modifier */
									if (temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod == ACTION_MODIFIER2) {
										temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod = ACTION_MODIFIER1;
										show_optionsscreen(REDRAW_INTERMEDIATE);
									} else if (temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod == ACTION_MODIFIER1) {
										temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod = UNDEFINED;
										show_optionsscreen(REDRAW_INTERMEDIATE);
									}
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == OPTIONS_MENUPOS_LEFT) {
								/* Previous page */
								optionsscreenpage--;
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_RIGHT;
								show_optionsscreen(REDRAW_EVERYTHING);
							} else if (optionsscreenmenuposition[optionsscreenpage] > OPTIONS_MENUPOS_LEFT) {
								optionsscreenmenuposition[optionsscreenpage]--;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							}
						} else if (optionsscreenpage == 5) {
							if (optionsscreenmenuposition[optionsscreenpage] == 0) {
								/* Decrease selected_skin */
								if (temp_selected_skin > 0) {
									temp_selected_skin--;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == OPTIONS_MENUPOS_LEFT) {
								/* Previous page */
								optionsscreenpage--;
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_RIGHT;
								show_optionsscreen(REDRAW_EVERYTHING);
							} else if (optionsscreenmenuposition[optionsscreenpage] > OPTIONS_MENUPOS_LEFT) {
								optionsscreenmenuposition[optionsscreenpage]--;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							}
						}
						break;
					case ACTION_RIGHT:
						if (optionsscreenpage == 0) {
							if (optionsscreenmenuposition[optionsscreenpage] == 0) {
								/* Decrease speed */
								if (temp_game_cycle_limit > GAME_CYCLE_LIMIT_MIN) {
									temp_game_cycle_limit = 100 / (100 / temp_game_cycle_limit + 1);
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == 1) {
								/* Next joystick */
								if (temp_default_joystick < joystick_count - 1) {
									temp_default_joystick++;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == 2) {
								/* Increase joystick_dead_zone */
								if (temp_joystick_dead_zone < 99) {
									temp_joystick_dead_zone++;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == 3) {
								/* Increase key_repeat_delay */
								if (temp_key_repeat_delay < KEY_REPEAT_MAX) {
									temp_key_repeat_delay += 20;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == 4) {
								/* Increase key_repeat_interval */
								if (temp_key_repeat_interval < KEY_REPEAT_MAX) {
									temp_key_repeat_interval += 20;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == 5) {
								/* Increase selected_locale */
								if (temp_selected_locale < locale_count - 1) {
									temp_selected_locale++;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == OPTIONS_MENUPOS_RIGHT) {
								/* Next page */
								optionsscreenpage++;
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_LEFT;
								show_optionsscreen(REDRAW_EVERYTHING);
							} else if (optionsscreenmenuposition[optionsscreenpage] > 8) {
								optionsscreenmenuposition[optionsscreenpage]++;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							}
						} else if (optionsscreenpage >= 1 && optionsscreenpage <= 4) {
							if (optionsscreenmenuposition[optionsscreenpage] < OPTIONS_MENUPOS_LEFT) {
								if (optionsscreenpage >= 2 && optionsscreenpage <= 4 && optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_NOT_USED1
									&& optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_MODIFIER1
									&& optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_MODIFIER2
									&& temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].device != UNDEFINED) {
									/* Next modifier */
									if (temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod == UNDEFINED) {
										temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod = ACTION_MODIFIER1;
										show_optionsscreen(REDRAW_INTERMEDIATE);
									} else if (temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod == ACTION_MODIFIER1) {
										temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod = ACTION_MODIFIER2;
										show_optionsscreen(REDRAW_INTERMEDIATE);
									}
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] == OPTIONS_MENUPOS_RIGHT) {
								/* Next page */
								optionsscreenpage++;
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_LEFT;
								show_optionsscreen(REDRAW_EVERYTHING);
							} else if (optionsscreenmenuposition[optionsscreenpage] > 8) {
								optionsscreenmenuposition[optionsscreenpage]++;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							}
						} else if (optionsscreenpage == 5) {
							if (optionsscreenmenuposition[optionsscreenpage] == 0) {
								/* Increase selected_skin */
								if (temp_selected_skin < skin_count - 1) {
									temp_selected_skin++;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] > 8) {
								if (optionsscreenmenuposition[optionsscreenpage] < OPTIONS_MENUPOS_EXIT) optionsscreenmenuposition[optionsscreenpage]++;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							}
						}
						break;
					case ACTION_SHOOT_UP:
					case ACTION_SHOOT_DOWN:
					case ACTION_SHOOT_LEFT:
					case ACTION_SHOOT_RIGHT:
						break;
					case ACTION_SELECT:
						if (optionsscreenpage == 4 && optionsscreenmenuposition[optionsscreenpage] == 8) {
							/* Restore default controls */
							set_default_user_controls(temp_user_controls);
							show_optionsscreen(REDRAW_INTERMEDIATE);
							show_message_box(REDRAW_INITIALISE, MESSAGE_BOX_RESTORE_ID, DELAY_MESSAGE_BOX_RESTORE, txt_Default_controls_restored, TRUE, 0, 0);
						} else if (optionsscreenpage >= 2 && optionsscreenpage <= 4 && optionsscreenmenuposition[optionsscreenpage] < OPTIONS_MENUPOS_LEFT) {
							/* Initiate assigning a new key or button to a control */
							getkey_timeout = DELAY_MESSAGE_BOX_GETKEY - 1;	/* Minus one makes the transition smooth if the error box gets shown */
							getkey_state = gua_state;
							getkey_count = 0;
							show_message_box(REDRAW_INITIALISE, MESSAGE_BOX_GETKEY_ID, DELAY_MESSAGE_BOX_GETKEY, txt_Press_something_and_release, TRUE, 0, 0);
						} else if (optionsscreenmenuposition[optionsscreenpage] == OPTIONS_MENUPOS_SAVE) {
							/* Save */
							game_cycle_limit = temp_game_cycle_limit;

							if (joystick_count > 0) {
								if (default_joystick != temp_default_joystick) {
									strcpy(temp_default_joystick_name, "-1");
									if ((result = initialise_joystick(temp_default_joystick, temp_default_joystick_name, TRUE)) != UNDEFINED) {
										default_joystick = result;
										get_joystick_name(default_joystick, default_joystick_name);
									}
								}
								joystick_dead_zone = temp_joystick_dead_zone;
							}
							
							key_repeat_delay = temp_key_repeat_delay;
							key_repeat_interval = temp_key_repeat_interval;

							if (selected_locale != temp_selected_locale) {
								selected_locale = temp_selected_locale;
								load_selected_locale();
								show_optionsscreen(REDRAW_EVERYTHING);
							}

							for (count = 0; count < USER_CONTROLS; count++) {
								user_controls[count].device = temp_user_controls[count].device;
								user_controls[count].id = temp_user_controls[count].id;
								user_controls[count].mod = temp_user_controls[count].mod;
								user_controls[count].state = SDL_RELEASED;	/* Stops controls becoming stuck in a pressed state */
								user_controls[count].cyclesactive = 0;		/* Stops controls becoming stuck in a pressed state */
								user_controls[count].delay = KEY_REPEAT_DELAY;
								user_controls[count].interval = KEY_REPEAT_INTERVAL;
							}

							if (selected_skin != temp_selected_skin) {
								selected_skin = temp_selected_skin;
								load_selected_skin();
								level.colour_override = UNDEFINED;	/* Important ;) */
								read_skin_level_colour_override();
								SDL_WM_SetIcon(icon, NULL);	/* Technically this should be called before SDL_SetVideoMode but it works in Xfce4 */
								show_optionsscreen(REDRAW_EVERYTHING);
							}
							
							show_message_box(REDRAW_INITIALISE, MESSAGE_BOX_SAVING_ID, DELAY_MESSAGE_BOX_SAVING, txt_Changes_saved, TRUE, 0, 0);
						} else if (optionsscreenmenuposition[optionsscreenpage] == OPTIONS_MENUPOS_EXIT) {
							/* Exit */
							game_mode = calling_game_mode;	/* Return to calling state */
							if (game_mode == INTRO_SCREEN) {
								show_introscreen(REDRAW_INITIALISE);
							} else if (game_mode == GAME_ON) {
								show_game_area(REDRAW_INITIALISE);
							}
						}
						break;
					case ACTION_EXIT:
						game_mode = calling_game_mode;	/* Return to calling state */
						if (game_mode == INTRO_SCREEN) {
							show_introscreen(REDRAW_INITIALISE);
						} else if (game_mode == GAME_ON) {
							show_game_area(REDRAW_INITIALISE);
						}
						break;
					case ACTION_HELP:
					case ACTION_OPTIONS:
						break;
					case ACTION_RESTART:
						if (optionsscreenpage >= 2 && optionsscreenpage <= 4) {
							if (optionsscreenmenuposition[optionsscreenpage] < OPTIONS_MENUPOS_LEFT) {
								/* Delete a control */
								if (optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_NOT_USED1
									&& temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].device != UNDEFINED) {
									temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].device = UNDEFINED;
									temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].id = UNDEFINED;
									temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod = UNDEFINED;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							}
						}
						break;
					case ACTION_PREVIOUS_LEVEL:
					case ACTION_NEXT_LEVEL:
					case ACTION_PREVIOUS_PACK:
					case ACTION_NEXT_PACK:
					case ACTION_MODIFIER1:
					case ACTION_MODIFIER2:
						break;
					case ACTION_TOGGLE_FULLSCREEN:
						toggle_fullscreen(&video.fullscreen);
						break;
					case ACTION_HOME:
						if (optionsscreenpage == 0) {
							if (optionsscreenmenuposition[optionsscreenpage] == 0) {
								/* Highest speed */
								temp_game_cycle_limit = GAME_CYCLE_LIMIT_MAX;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] == 1) {
								/* First joystick */
								temp_default_joystick = 0;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] == 2) {
								/* Lowest joystick_dead_zone */
								temp_joystick_dead_zone = 1;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] == 3) {
								/* Lowest key_repeat_delay */
								temp_key_repeat_delay = KEY_REPEAT_MIN;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] == 4) {
								/* Lowest key_repeat_interval */
								temp_key_repeat_interval = KEY_REPEAT_MIN;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] == 5) {
								/* First locale */
								temp_selected_locale = 0;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else {
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_SAVE;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							}
						} else if (optionsscreenpage >= 1 && optionsscreenpage <= 4) {
							if (optionsscreenpage >= 2 && optionsscreenpage <= 4 && optionsscreenmenuposition[optionsscreenpage] < OPTIONS_MENUPOS_LEFT) {
								if (optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_NOT_USED1
									&& optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_MODIFIER1
									&& optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_MODIFIER2
									&& temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].device != UNDEFINED
									&& temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod != UNDEFINED) {
									/* First modifier */
									temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod = UNDEFINED;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] >= OPTIONS_MENUPOS_LEFT) {
								optionsscreenpage = 0;
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_SAVE;
								show_optionsscreen(REDRAW_EVERYTHING);
							}
						} else if (optionsscreenpage == 5) {
							if (optionsscreenmenuposition[optionsscreenpage] == 0) {
								/* First skin */
								temp_selected_skin = 0;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] >= OPTIONS_MENUPOS_LEFT) {
								optionsscreenpage = 0;
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_SAVE;
								show_optionsscreen(REDRAW_EVERYTHING);
							}
						}
						break;
					case ACTION_END:
						if (optionsscreenpage == 0) {
							if (optionsscreenmenuposition[optionsscreenpage] == 0) {
								/* Lowest speed */
								temp_game_cycle_limit = GAME_CYCLE_LIMIT_MIN;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] == 1) {
								/* Last joystick */
								temp_default_joystick = joystick_count - 1;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] == 2) {
								/* Highest joystick_dead_zone */
								temp_joystick_dead_zone = 99;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] == 3) {
								/* Highest key_repeat_delay */
								temp_key_repeat_delay = KEY_REPEAT_MAX;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] == 4) {
								/* Highest key_repeat_interval */
								temp_key_repeat_interval = KEY_REPEAT_MAX;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] == 5) {
								/* Last locale */
								temp_selected_locale = locale_count - 1;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else {
								optionsscreenpage = OPTIONS_SCREEN_PAGES - 1;
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_EXIT;
								show_optionsscreen(REDRAW_EVERYTHING);
							}
						} else if (optionsscreenpage >= 1 && optionsscreenpage <= 4) {
							if (optionsscreenpage >= 2 && optionsscreenpage <= 4 && optionsscreenmenuposition[optionsscreenpage] < OPTIONS_MENUPOS_LEFT) {
								if (optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_NOT_USED1
									&& optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_MODIFIER1
									&& optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9 != ACTION_MODIFIER2
									&& temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].device != UNDEFINED
									&& temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod != ACTION_MODIFIER2) {
									/* Last modifier */
									temp_user_controls[optionsscreenmenuposition[optionsscreenpage] + (optionsscreenpage - 2) * 9].mod = ACTION_MODIFIER2;
									show_optionsscreen(REDRAW_INTERMEDIATE);
								}
							} else if (optionsscreenmenuposition[optionsscreenpage] >= OPTIONS_MENUPOS_LEFT) {
								optionsscreenpage = OPTIONS_SCREEN_PAGES - 1;
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_EXIT;
								show_optionsscreen(REDRAW_EVERYTHING);
							}
						} else if (optionsscreenpage == 5) {
							if (optionsscreenmenuposition[optionsscreenpage] == 0) {
								/* Last skin */
								temp_selected_skin = skin_count - 1;
								show_optionsscreen(REDRAW_INTERMEDIATE);
							} else if (optionsscreenmenuposition[optionsscreenpage] >= OPTIONS_MENUPOS_LEFT) {
								optionsscreenpage = OPTIONS_SCREEN_PAGES - 1;
								optionsscreenmenuposition[optionsscreenpage] = OPTIONS_MENUPOS_EXIT;
								show_optionsscreen(REDRAW_EVERYTHING);
							}
						}
						break;
					case ACTION_PAGEUP:
						/* Previous page */
						if (optionsscreenpage > 0) {
							optionsscreenpage--;
							show_optionsscreen(REDRAW_EVERYTHING);
						}
						break;
					case ACTION_PAGEDOWN:
						/* Next page */
						if (optionsscreenpage < OPTIONS_SCREEN_PAGES - 1) {
							optionsscreenpage++;
							show_optionsscreen(REDRAW_EVERYTHING);
						}
						break;
					case ACTION_VOLUP:
						volume_up();
						break;
					case ACTION_VOLDOWN:
						volume_down();
						break;
					default:
						break;
				}
			}
		}
		
		/* If the fade is active then show it */
		show_game_area_fade(REDRAW_EVERYTHING, 0);

		/* If there's a message box active then show it */
		show_message_box(REDRAW_EVERYTHING, "", 0, "", 0, 0, 0);

		/* Update the screen */
		SDL_Flip(screen);

		/* I think I should explain how the timing works as people may think I'm simply calling SDL_Delay(10)
		   every cycle and then continuing which would be OKish if the workload was always the same.
		   There is an SDL timer that sets the game_next_cycle flag every 20Hz (slow), 25Hz (normal) or 33Hz (fast).
		   This game is far from being CPU intensive especially since only small portions of the screen are being
		   updated when needed, so 90%+ of the time is spent in the while loop waiting for the game_next_cycle
		   flag to be set by the SDL timer. The SDL_Delay(game_cycle_delay) within the while loop is simply passing
		   control to the OS to service other tasks. If there was no delay here then this game would be a CPU hog.
		   I should mention that on my computer I can do SDL_Delay(0) but other computers will need a minimum of
		   10ms, therefore game_cycle_delay is set to 10ms by default but can be set to something else by modifying
		   [game_cycle_delay] in the .gnurobborc file in your home folder. I expect this will never be modified */

		/* Is there a cycle limit in place? */
		if (game_cycle_limit) {
			/* Wait until the timer sets the flag before proceeding */
			while (!game_next_cycle) SDL_Delay(game_cycle_delay);	/* Robbo spends 90%+ of his time in this loop :) */
			game_next_cycle = FALSE;
		}
		cycle_count++;

		if (game_mode != INTRO_SCREEN) demo_mode(DEMO_MODE_TIMEOUT_INITIALISE, 0);	/* Easier to do it once here */
		
	}
	
	#ifdef DEBUG_RECORD_DEMO
		demo_mode(DEMO_MODE_DUMP, 0);
	#endif

	save_resource_file(path_resource_file);
	
	/* atexit() will call clean_up_before_exit() on return */
	return 0;
}

/***************************************************************************
 * Clean Up Before Exit                                                    *
 ***************************************************************************/
/* This is registered with atexit() once SDL is initialised */

void clean_up_before_exit(void) {

	SDL_RemoveTimer(game_timer_id);

	/* WARNING: attempting to close joystick 0 on the GP2X causes a seg fault
	   and it'll probably do the same on other similar devices */
	#if defined(PLATFORM_GP2X)
		if ((joystick) && (SDL_JoystickIndex(joystick)) > 0) SDL_JoystickClose(joystick);
	#elif defined(PLATFORM_ZAURUS)
		if (joystick) SDL_JoystickClose(joystick);
	#elif defined(PLATFORM_PC)
		if (joystick) SDL_JoystickClose(joystick);
	#endif

	/* Free SDL surfaces if they are not NULL */
	if (icons) SDL_FreeSurface(icons);
	if (ciphers) SDL_FreeSurface(ciphers);
	if (alpha) SDL_FreeSurface(alpha);
	if (icon) SDL_FreeSurface(icon);

	/* Free the font if not NULL*/
	if (font) TTF_CloseFont(font);
	
	TTF_Quit();
	SDL_Quit();
}

/***************************************************************************
 * Game Timer                                                              *
 ***************************************************************************/

Uint32 game_timer(Uint32 interval, void *param) {
	static int intervals = 0;

	/* Count the units of 10ms and set a flag which enables the main loop to proceed */
	if (game_cycle_limit) {
		intervals++;
		if (intervals >= 100 / game_cycle_limit) {
			game_next_cycle = TRUE;
			intervals = 0;
		}
	} else {
		intervals = 0;
	}

	return interval;
}

/***************************************************************************
 * Demo Mode                                                               *
 ***************************************************************************/
/* Because of the randomness of certain objects it is advisable to record
   demos of Robbo interacting with simple things such as boxes, doors and
   teleports. Interacting with monsters and guns early on could end the demo
   prematurely so do it towards the end where unpredictability is not an issue.
   
   On entry: function = DEMO_MODE_INITIALISE to initialise at program start
			 function = DEMO_MODE_TIMEOUT_INITIALISE to reset the timeout
			 function = DEMO_MODE_TIMEOUT_DECREMENT to decrement the timeout
			 function = DEMO_MODE_ACTIVATE to activate demo mode
			 function = DEMO_MODE_IS_ACTIVE to report if currently active 
			 function = DEMO_MODE_DEACTIVATE to deactivate demo mode if active
			 function = DEMO_MODE_RECORD_INITIALISE to initialise prior to recording
			 function = DEMO_MODE_RECORD to record a demo with actionid = actionid
			 function = DEMO_MODE_PLAYBACK to playback a demo
			 function = DEMO_MODE_DUMP to dump a recorded demo to the console on program exit
   On exit:	 returns an actionid for DEMO_MODE_PLAYBACK
			 returns TRUE of FALSE for DEMO_MODE_IS_ACTIVATED
			 returns TRUE of FALSE for DEMO_MODE_TIMEOUT_DECREMENT indicating if
				demo mode was automatically activated or not
			 else UNDEFINED (-1)
*/

int demo_mode(int function, int actionid) {
	static int demo_mode_active;								/* It's ON */
	static int demo_mode_timeout;								/* Time to wait before activating demo mode */
	static int demo_mode_frame;									/* Input frame pointer */
	static int *demo_mode_demo_data[DEMO_MODE_DEMO_COUNT];		/* Pointers to playback input data + 1 for the record slot */
	static int demo_mode_demos[DEMO_MODE_DEMO_COUNT - 1][2];	/* Available demos [pack][level] */
	static int demo_mode_demos_index;							/* Index into demo_mode_demos */
	static int demo_mode_original_selected_pack;				/* The current user selected pack which the demo will replace */
	static int demo_mode_original_level_selected;				/* The current user level selected which the demo will replace */
	/* Playback input data: record at any speed, it doesn't make any difference */
	static int demo_mode_demo_data_Original_level1[] = {3, 3, 2, 2, 1, 1, 1, 1, 2, 1, 1, 3, 3, 1, 3, 1, 1, 1, 1, 1, 2, 25, 25, 25, 5, 25, 2, 2, 0, 0, 0, 3, 1, 1, 1, 1, 1, 3, 1, 3, 25, 6, 25, 3, 3, 0, 3, 3, 3, 1, 1, 2, 1, 10};
	static int demo_mode_demo_data_Original_level3[] = {3, 1, 1, 1, 1, 1, 1, 3, 3, 3, 0, 3, 1, 3, 1, 1, 1, 1, 2, 1, 1, 1, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 3, 3, 1, 1, 2, 2, 1, 1, 1, 3, 1, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 3, 10};
	static int demo_mode_demo_data_Original_level6[] = {3, 3, 0, 3, 3, 0, 3, 3, 3, 3, 1, 1, 3, 1, 3, 1, 3, 2, 0, 0, 0, 2, 2, 2, 1, 1, 3, 3, 3, 3, 0, 0, 10};

	static int demo_mode_demo_data_RobboIX_level1[] = {1, 1, 1, 1, 1, 1, 1, 3, 3, 0, 3, 3, 0, 0, 0, 3, 3, 3, 0, 0, 3, 3, 3, 3, 3, 1, 1, 3, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 10};
	static int demo_mode_demo_data_RobboIX_level4[] = {1, 3, 3, 0, 3, 3, 3, 1, 1, 1, 2, 3, 3, 1, 3, 1, 3, 3, 3, 3, 3, 3, 2, 1, 1, 1, 1, 3, 1, 1, 1, 2, 2, 2, 2, 3, 0, 0, 0, 2, 2, 2, 1, 2, 1, 0, 25, 5, 25, 3, 0, 1, 2, 1, 1, 1, 1, 1, 10};
	static int demo_mode_demo_data_RobboIX_level14[] = {2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 25, 5, 25, 3, 3, 5, 25, 3, 3, 5, 25, 3, 3, 5, 25, 3, 3, 25, 5, 25, 3, 3, 25, 5, 25, 3, 3, 25, 5, 25, 2, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 1, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 1, 1, 1, 1, 3, 3, 3, 1, 1, 1, 2, 10};

	static int demo_mode_demo_data_RobboVII_level2[] = {2, 2, 2, 2, 2, 2, 2, 1, 1, 2, 2, 2, 2, 2, 2, 2, 1, 1, 3, 3, 3, 3, 1, 1, 2, 1, 2, 1, 0, 3, 3, 2, 3, 1, 1, 1, 1, 1, 1, 3, 10};
	static int demo_mode_demo_data_RobboVII_level4[] = {2, 2, 2, 1, 1, 1, 1, 3, 3, 3, 0, 2, 25, 7, 25, 0, 25, 5, 25, 2, 2, 2, 1, 25, 7, 25, 1, 3, 3, 1, 1, 0, 0, 0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 2, 1, 1, 2, 0, 2, 2, 2, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 3, 10};
	static int demo_mode_demo_data_RobboVII_level5[] = {3, 1, 1, 1, 2, 2, 2, 2, 2, 2, 1, 1, 1, 2, 2, 2, 2, 25, 6, 25, 0, 0, 0, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 10};

	static int demo_mode_demo_data_RobboVIII_level1[] = {0, 0, 0, 0, 0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 1, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 3, 1, 1, 1, 1, 2, 2, 2, 3, 3, 3, 3, 1, 1, 1, 1, 1, 1, 1, 10};
	static int demo_mode_demo_data_RobboVIII_level2[] = {1, 3, 3, 3, 0, 1, 1, 1, 3, 3, 3, 0, 0, 0, 2, 0, 3, 3, 3, 1, 2, 2, 1, 1, 3, 3, 3, 3, 0, 0, 0, 0, 0, 2, 2, 3, 3, 3, 1, 1, 1, 3, 3, 3, 0, 0, 0, 2, 0, 3, 3, 1, 2, 1, 1, 1, 25, 5, 25, 2, 2, 3, 3, 25, 5, 25, 10};
	static int demo_mode_demo_data_RobboVIII_level6[] = {3, 1, 1, 2, 1, 2, 1, 3, 1, 2, 1, 3, 1, 1, 1, 1, 1, 1, 1, 3, 0, 0, 0, 0, 0, 2, 1, 1, 1, 1, 2, 0, 0, 3, 0, 0, 0, 0, 2, 2, 2, 2, 2, 2, 2, 1, 1, 1, 2, 1, 1, 1, 1, 1, 1, 1, 1, 10};

	static int demo_mode_demo_data_recorded[DEMO_MODE_LENGTH];
	int returnval = UNDEFINED, count;

	if (function == DEMO_MODE_INITIALISE) {
		demo_mode_active = FALSE;
		demo_mode_timeout = DELAY_DEMO;
		demo_mode_frame = 0;
		demo_mode_demos_index = -1;	/* Will be incremented first before use */
		/* Initialise the demo data */
		/* 3 early levels from Original */
		for (count = 0; count < level_pack_count - 1; count++) if (!strcmp(level_packs[count].name, "Original")) break;
		demo_mode_demos[0][0] = count; demo_mode_demos[0][1] = 1;
		demo_mode_demos[1][0] = count; demo_mode_demos[1][1] = 3;
		demo_mode_demos[2][0] = count; demo_mode_demos[2][1] = 6;
		demo_mode_demo_data[0] = demo_mode_demo_data_Original_level1;
		demo_mode_demo_data[1] = demo_mode_demo_data_Original_level3;
		demo_mode_demo_data[2] = demo_mode_demo_data_Original_level6;
		/* 3 early levels from RobboIX */
		for (count = 0; count < level_pack_count - 1; count++) if (!strcmp(level_packs[count].name, "RobboIX")) break;
		demo_mode_demos[3][0] = count; demo_mode_demos[3][1] = 1;
		demo_mode_demos[4][0] = count; demo_mode_demos[4][1] = 4;
		demo_mode_demos[5][0] = count; demo_mode_demos[5][1] = 14;
		demo_mode_demo_data[3] = demo_mode_demo_data_RobboIX_level1;
		demo_mode_demo_data[4] = demo_mode_demo_data_RobboIX_level4;
		demo_mode_demo_data[5] = demo_mode_demo_data_RobboIX_level14;
		/* 3 early levels from RobboVII */
		for (count = 0; count < level_pack_count - 1; count++) if (!strcmp(level_packs[count].name, "RobboVII")) break;
		demo_mode_demos[6][0] = count; demo_mode_demos[6][1] = 2;
		demo_mode_demos[7][0] = count; demo_mode_demos[7][1] = 4;
		demo_mode_demos[8][0] = count; demo_mode_demos[8][1] = 5;
		demo_mode_demo_data[6] = demo_mode_demo_data_RobboVII_level2;
		demo_mode_demo_data[7] = demo_mode_demo_data_RobboVII_level4;
		demo_mode_demo_data[8] = demo_mode_demo_data_RobboVII_level5;
		/* 3 early levels from RobboVIII */
		for (count = 0; count < level_pack_count - 1; count++) if (!strcmp(level_packs[count].name, "RobboVIII")) break;
		demo_mode_demos[9][0] = count; demo_mode_demos[9][1] = 1;
		demo_mode_demos[10][0] = count; demo_mode_demos[10][1] = 2;
		demo_mode_demos[11][0] = count; demo_mode_demos[11][1] = 6;
		demo_mode_demo_data[9] = demo_mode_demo_data_RobboVIII_level1;
		demo_mode_demo_data[10] = demo_mode_demo_data_RobboVIII_level2;
		demo_mode_demo_data[11] = demo_mode_demo_data_RobboVIII_level6;

		/* The record slot is the last pointer */
		demo_mode_demo_data[DEMO_MODE_RECORD_SLOT] = demo_mode_demo_data_recorded;
		
	} else if (function == DEMO_MODE_TIMEOUT_INITIALISE) {
		demo_mode_timeout = DELAY_DEMO;

	} else if (function == DEMO_MODE_TIMEOUT_DECREMENT) {
		if (!--demo_mode_timeout) demo_mode(DEMO_MODE_ACTIVATE, 0);
		returnval = demo_mode_active;

	} else if (function == DEMO_MODE_ACTIVATE) {
		demo_mode_active = TRUE;
		demo_mode_frame = 0;
		/* * Store the currently selected pack
		   * Select the demo pack
		   * Store level_selected for the demo pack
		   * Select the demo level */
		demo_mode_original_selected_pack = selected_pack;
		if (++demo_mode_demos_index >= DEMO_MODE_DEMO_COUNT - 1) demo_mode_demos_index = 0;
		selected_pack = demo_mode_demos[demo_mode_demos_index][0];
		demo_mode_original_level_selected = level_packs[selected_pack].level_selected;
		level_packs[selected_pack].level_selected = demo_mode_demos[demo_mode_demos_index][1];
		#ifdef DEBUG_DEMO_MODE
			printf("*** Start %s ***\n", __func__);
			printf("Playing demo: pack %i level %i\n", demo_mode_demos[demo_mode_demos_index][0], demo_mode_demos[demo_mode_demos_index][1]);
			printf("*** Stop %s ***\n", __func__);
		#endif
		if (level_init()) {
			demo_mode(DEMO_MODE_DEACTIVATE, 0);			/* Restore user selected pack and level */
			demo_mode(DEMO_MODE_TIMEOUT_INITIALISE, 0);	/* Restart timer */
			game_mode = INTRO_SCREEN;
			show_introscreen(REDRAW_INITIALISE);
		} else {
			/* Game on */
			game_mode = GAME_ON;
			show_game_area(REDRAW_INITIALISE);
			/* Initialise the fade */
			show_game_area_fade(REDRAW_INITIALISE, 16);
		}

	} else if (function == DEMO_MODE_IS_ACTIVE) {
		returnval = demo_mode_active;

	} else if (function == DEMO_MODE_DEACTIVATE) {
		if (demo_mode_active) {
			demo_mode_active = FALSE;
			/* * Restore level_selected for the demo pack
			   * Restore the selected pack */
			level_packs[selected_pack].level_selected = demo_mode_original_level_selected;
			selected_pack = demo_mode_original_selected_pack;
		}

	} else if (function == DEMO_MODE_RECORD_INITIALISE) {
		for (count = 0; count < DEMO_MODE_LENGTH; count++) *(demo_mode_demo_data[DEMO_MODE_RECORD_SLOT] + count) = UNDEFINED;
		demo_mode_frame = 0;

	} else if (function == DEMO_MODE_RECORD) {
		if (demo_mode_frame >= DEMO_MODE_LENGTH) {
			printf("Demo recording reached end of buffer\n");
			*(demo_mode_demo_data[DEMO_MODE_RECORD_SLOT] + DEMO_MODE_LENGTH - 1) = ACTION_EXIT;
		} else {
			*(demo_mode_demo_data[DEMO_MODE_RECORD_SLOT] + demo_mode_frame++) = actionid;
		}

	} else if (function == DEMO_MODE_PLAYBACK) {
		returnval = *(demo_mode_demo_data[demo_mode_demos_index] + demo_mode_frame++);

	} else if (function == DEMO_MODE_DUMP) {
		printf("\nstatic int demo_mode_demo_data_%s_level%i[] = {", level_packs[selected_pack].name, level_packs[selected_pack].level_selected);
		for (count = 0; count < DEMO_MODE_LENGTH; count++) {
			if (*(demo_mode_demo_data[DEMO_MODE_RECORD_SLOT] + count) != UNDEFINED) printf("%i, ", *(demo_mode_demo_data[DEMO_MODE_RECORD_SLOT] + count));
		}
		printf("%c%c};\n\n", 8, 8);
	}

	return returnval;
}


