/* -*- Mode: c++ -*- 
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VRFHFSKMOD_H_
#define _VRFHFSKMOD_H_

#include <VrSigProc.h>

/***
  Frequency Hopping FSK Modulator
  VrFHFSKMod(int num,int* freqs,int fDev,int seqLen,int* seq,
             int bit_rate,int bits_per_output, int bph)
    num:  number of possible FSK center frequencies
    freqs:  ptr to array of center frequencies, in Hz
    fDev:  frequency separation, bit 0 = center - fDev, 1 = center + fDev
           if fDev=0, fDev is set to bit_rate/2
    seqLen:  length of hopping sequence
    seq:  hopping sequence, ptr to an array of length seqLen
    bit_rate:  data rate, in bits per sample
    bits_per_output: # of bits outputted in a waveform
    bph:  data bits per hop
***/

template<class oType>
class VrFHFSKMod : public VrSigProc<bit,oType> {
protected:
  int bitsPerOutput;
  int bitLength;
  int dataRate;
  oType*** waveforms;
  int bitsPerHop;
  int hops;
  int numCenterFreq;
  int freqDev;
  int* hopFrequency;
  int hopSeqLen;
  int* hopSeq;
  int freqNum;
  int bitsOutputted;
  oType* zeroWaveform;
  virtual void initialize();
public:   
  virtual void work(int n);  
  VrFHFSKMod(int num,int* freqs,int fDev,int seqLen,int* seq,
	     int bit_rate,int bpo,int bph);
};

template<class oType> void
VrFHFSKMod<oType>::work(int n)
{
  while (n>0) {
    if (bitsOutputted == bitsPerHop) {  /* hop! continue hopping, even with
					   NULL bits */
      bitsOutputted = 0;
      hops++;
      if (hops == hopSeqLen) hops = 0;
      freqNum = hopSeq[hops];
      //      printf("hopped to %d\n",freqNum);
    }
    
    bit* bitPtr = inputReadPtr(-bitsPerOutput+1);
    int wf_index = 0;
    int bitsConverted = 0;
    for (int j=bitsPerOutput-1;j>=0;j--,bitPtr++) {
      if (*bitPtr == -1) { // NULL bit
	bitsConverted = bitsPerOutput - 1 - j;
	break;
      }
      wf_index = wf_index | (*bitPtr << j);
    }
    
    if (*bitPtr == -1) { // NULL bit
      // fill the rest of the work request with a zero waveform
      //  can we do a memset of the o[0]?
      for (int i=0;i<n/bitLength-bitsConverted;i++) 
	outputWrite(zeroWaveform,bitLength);
      incReadPtr(validUnits()+2);  /* since the upstream module returned NULL for
				  the remainder of our request, ignore it */
      n = 0; 
      // resync hopping on NULL bit transmission ****/
      bitsOutputted = 0;
      hops = 0;
      freqNum = hopSeq[hops];
      printf("reset to %d\n",freqNum);
    }
    else {
      //      printf("%d (%d)\n",wf_index,bitsOutputted);
      outputWrite(&(waveforms[freqNum][wf_index][0]),bitLength*bitsPerOutput);
      incReadPtr(bitsPerOutput);
      n-=bitLength/bitsPerOutput;
      bitsOutputted+=bitsPerOutput;
    }
  }
  return;
}

template<class oType> void
VrFHFSKMod<oType>::initialize() {
  
  int maxAmp = (1 << (8*sizeof(oType) - 1)) - 1;

  /* to ensure continuous phase (each bit begins and ends at 0 phase):
     effective data rate is adjusted sf/bitLength
     frequency separation (2*freqDev) is a multiple of the data rate
     center freqs are adjusted so that f0 and f1 are multiples of the
       data rate (center - freqDev is a multiple of the data rate) */

  int sf = getOutputSamplingFrequency();
  bitLength = sf / dataRate;
  setOutputSize (bitLength*bitsPerOutput);

  if ((bitLength * dataRate) != sf) {
    dataRate = sf/bitLength;
    cout << "FHFSKMod: Adjusting data rate to: " << dataRate << endl;
  }

  if ((2*freqDev % dataRate) != 0) {
    freqDev = ((2*freqDev/dataRate)*dataRate + dataRate)/2;
    cout << "FHFSKMod: Adjusting frequency deviation to +-: "
	 << freqDev << endl;
  }

  
  /* 3-d array: 1st param: center freq number
                2nd: bit 0 or 1 */
  waveforms = (oType***)(new int[numCenterFreq]);
  for (int i=0;i<numCenterFreq;i++) {
    *(waveforms+i) = (oType**)(new int[(1<<bitsPerOutput)]);
    for (int k=0;k<(1 << bitsPerOutput);k++)
      *(*(waveforms+i)+k) = (oType*)(new oType[bitLength*bitsPerOutput]);
  }

  oType** base = (oType**)(new int[2]);
  *base = (oType*)new oType[bitLength];
  *(base+1) = (oType*)new oType[bitLength];

  for (int i=0;i<numCenterFreq;i++) {
    
    if (((hopFrequency[i] - freqDev) % dataRate) != 0) {
      hopFrequency[i] = ((hopFrequency[i]-freqDev)/dataRate)*dataRate
	+ dataRate + freqDev;
      cout << "FHFSKMod: Adjusting center frequency to: "
	   << hopFrequency[i] << endl;
    }

    for (int j=0;j<bitLength;j++) {
      base[0][j] = 
	(oType)(maxAmp*sin(2*M_PI*j*(hopFrequency[i] - freqDev)/(float)sf));
      base[1][j] = 
	(oType)(maxAmp*sin(2*M_PI*j*(hopFrequency[i] + freqDev)/(float)sf));
    }

    for (int k=0;k<(1 << bitsPerOutput);k++) {
      for (int l=0;l<bitsPerOutput;l++) {
	bit currBit = (k >> (bitsPerOutput - 1 - l)) & 0x1;
	memcpy(&(waveforms[i][k][l*bitLength]),
	       base[currBit],bitLength);
      }
    }
  }
  
  // create NULL waveform
  zeroWaveform = (oType*)new oType[bitLength*bitsPerOutput];
  memset(zeroWaveform,0,bitLength*bitsPerOutput*sizeof(oType));

  setHistory(bitsPerOutput);
}

template<class oType>
VrFHFSKMod<oType>::VrFHFSKMod(int numFreq, int* cf, int freq_dev,
			      int seqLen, int* seq,
			      int bit_rate, int bpo, int bph)
  :bitsPerOutput(bpo),dataRate(bit_rate),bitsPerHop(bph),hops(0),
   numCenterFreq(numFreq),hopSeqLen(seqLen),bitsOutputted(0)
{
  hopFrequency = cf;

  if (freq_dev == 0) 
    freqDev = bit_rate/2;
  else freqDev = freq_dev;

  hopSeq = seq;
}

#endif


