/* -*- Mode: c++ -*- */
/*
 * Copyright 2001,2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * This is a the main program for the complete ATSC Receiver!
 */

#include <config.h>
#include <make_GrMC4020Source.h>
#include <gr_FlowGraph.h>
#include <VrFixOffset.h>
#include <GrConvertSF.h>
#include <VrQuadratureDemod.h>
#include <GrFIRfilterFFF.h>
#include <gr_firdes.h>
#include <gr_fir_builder.h>
#include <VrNullSink.h>
#include <VrFileSource.h>
#include <VrFileSink.h>
#include <GrReal.h>
#include <GrAtscFPLL.h>
//#include <getopt.h>
#include <unistd.h>
#include <atsc_root_raised_cosine_bandpass.h>
#include <atsc_consts.h>
#include <GrAtscBitTimingLoop.h>
#include <GrAtscBitTimingLoop3.h>
#include <atsc_exp2_lp.h>
#include <GrRemoveDcFFF.h>
#include <GrAtscFieldSyncChecker.h>
#include <GrAtscEqualizer.h>
#include <create_atsc_equalizer.h>
#include <GrAtscFieldSyncDemux.h>
#include <GrAtscViterbiDecoder.h>
#include <GrAtscDeinterleaver.h>
#include <GrAtscRSDecoder.h>
#include <GrAtscDerandomizer.h>
#include <cstring>


static const double our_IF_freq = 5.75e6;	// IF freq of tuner module
static const double vanu_IF_freq = 5.9815e6;	// IF freq of Vanu test data


typedef atsc_mpeg_packet	final_oType;


static void 
usage (const char *name)
{
  cerr << "usage: " << name
       << " [-s {20 | 2x}] {-A | -f <filename>} [-o <filename>]\n";

  cerr << "  -A            : take input from A/D ch3\n";
  cerr << "  -f <filename> : take input from file of shorts\n";
  cerr << "  -r            : continuously repeat the file contents (loop)\n";
  cerr << "  -o <filename> : put output in filename, else VrNullSink\n";
  cerr << "  -s 20         : input sample rate is 20 MS/s\n";
  cerr << "  -s 2x         : input sample rate is 2x the ATSC symbol rate (21.52 MS/s)\n";
  cerr << "  ------ debug only ------\n";
  cerr << "  -l            : enable logging of intermediate results\n";
  cerr << "  -L            : enable LMS equalizer, else NOP equalizer\n";
  cerr << "  -V            : IF freq is that of vanu test data\n";
  cerr << "  -P <degrees>  : set initial phase of FPLL and don't update freq\n";
  cerr << "  -M <mu>       : set initial fractional delay\n";
  cerr << "  -N            : don't update fractional delay\n";
  cerr << "  -F <float>    : set bit timing loop filter tap\n";
  cerr << "  -R <float>    : set bit timing rate\n";


  exit (1);
}

int main (int argc, char **argv)
{
  // input rate values
  static const double IR_20  = 20e6;
  static const double IR_2X  = 2 * ATSC_SYMBOL_RATE;	// approx 21.52e6
  static const double IR_DEFAULT = IR_20;
  
  static const double DONT_SET_INITIAL_PHASE = -1;
  
  try {
    
    bool repeat_p = false;		// continuous loop through file
    bool input_from_ADC_p = false;
    bool clk_from_daughter_card_p = false;
    bool input_is_float_p = false;
    char *input_filename = 0;
    char *output_filename = 0;
    double IF_freq = our_IF_freq;
    double initial_phase = DONT_SET_INITIAL_PHASE;
    double input_rate = IR_DEFAULT;


    float mu = -1;		// valid if >= 0
    bool  no_update = false;
    float loop_filter_tap = -1;
    float loop_timing_rate = -1;

    bool  logging_p = false;
    bool  use_lms_p = false;

    int c;
    while ((c = getopt (argc, argv, "Af:ro:s:lLVP:M:NF:R:")) != EOF){
      switch (c){
      case 'A': input_from_ADC_p = true; 	break;

      case 'f':
	input_filename = optarg;
	input_is_float_p = false;
	break;

      case 'r': repeat_p = true;		break;

      case 'o': output_filename = optarg;	break;

      case 's':
	if (strcmp (optarg, "20") == 0){
	  input_rate = IR_20;
	}
	else if (strcmp (optarg, "2x") == 0){
	  input_rate = IR_2X;
	  clk_from_daughter_card_p = true;	// if we're using the A/D
	}
	else {
	  usage (argv[0]);
	}
	break;

	// debugging options

      case 'l': logging_p = true;		break;

      case 'L': use_lms_p = true;		break;

      case 'V': IF_freq = vanu_IF_freq;		break;

      case 'P':
	initial_phase = strtod (optarg, 0);
	break;

      case 'M': mu = strtod (optarg, 0);		break;
      case 'N': no_update = true;			break;
      case 'F': loop_filter_tap = strtod (optarg, 0);	break;
      case 'R': loop_timing_rate = strtod (optarg, 0);	break;

      case '?':
      default:
	usage (argv[0]);
      }
    }

    if (optind != argc)
      usage (argv[0]);

    if (input_filename && input_from_ADC_p){
      cerr << argv[0] << ": -f <filename> and -A are mutually exclusive\n";
      usage (argv[0]);
    }

    if (!(input_filename || input_from_ADC_p)){
      cerr << argv[0] << ": one of -f <filename> and -A must be specified\n";
      usage (argv[0]);
    }


    // freq of hdtv suppressed carrier pilot tone.  
    //
    // This assumes that the tuner is actually tuned to the middle of the 6
    // MHz channel.  The pilot is 0.31 MHz up from the bottom edge of the channel
    
    double pilot_freq = IF_freq - 3e6 + 0.31e6;


    cerr << "Input Sampling Rate: " << input_rate << endl;
    cerr << "Pilot Freq: " << pilot_freq << endl;

    VrSigProc		*vsrc;		// virtual source
    VrSource<short>	*source = 0;
    VrSink<final_oType>	*final_sink = 0;

    // optional logging sinks

    VrFileSink<float>			     *matched_log = 0;
    VrFileSink<float>			     *fpll_log = 0;
    VrFileSink<float>			     *lowpass_log = 0;
    VrFileSink<float>		             *nodc_log = 0;
    VrFileSink<float>			     *bt_data_log = 0;	// bit timing data
    VrFileSink<float>			     *bt_tags_log = 0;	// bit timing tags
    VrFileSink<float>			     *fsc_data_log = 0;	// field sync checker data
    VrFileSink<float>			     *fsc_tags_log = 0;	// field sync checker tags
    VrFileSink<float>			     *eq_data_log = 0;	// equalizer data
    VrFileSink<float>			     *eq_tags_log = 0;	// equalizer tags
    VrFileSink<atsc_soft_data_segment> 	     *demux_log = 0;
    VrFileSink<atsc_mpeg_packet_rs_encoded>  *viterbi_decoder_log  = 0;
    VrFileSink<atsc_mpeg_packet_rs_encoded>  *deinterleaver_log = 0;
    VrFileSink<atsc_mpeg_packet_no_sync>     *rs_decoder_log = 0;
    VrFileSink<atsc_mpeg_packet>	     *derandomizer_log = 0;


    // ================================================================
    //   Get data either from
    //		an external file or
    //		the ADC ch3 clocked with the daughter card
    // ================================================================

    if (input_filename){
      if (input_is_float_p){	// floats
	vsrc = new VrFileSource<float>(input_rate, input_filename, repeat_p);
      }
      else {			// shorts
	// --> float
	source = new VrFileSource<short>(input_rate, input_filename, repeat_p);
	GrConvertSF *converter = new GrConvertSF ();
	NWO_CONNECT (source, converter);
	vsrc = converter;
      }
    }
    else if (input_from_ADC_p){

      if (clk_from_daughter_card_p){
	// configure A/D to use external clock input 
	source = make_GrMC4020SourceS(input_rate,
					   MCC_CH3_EN
					   | MCC_ALL_1V
					   | MCC_CLK_AD_START_TRIG_IN);
      }
      else {
	// configure with internal time base
	source = make_GrMC4020SourceS(input_rate,
					   MCC_CH3_EN
					   | MCC_ALL_1V
					   | MCC_CLK_INTERNAL);
      }

      // short --> float
      VrFixOffset<short,float> *offset_fixer = new VrFixOffset<short,float>();

      NWO_CONNECT (source, offset_fixer);
      vsrc = offset_fixer;
    }
    else {
      assert (0);	// can't happen
    }

    // ================================================================
    //  apply the band pass matched filter (root raised cosine)
    //  (selects only the data of interest also...)
    // ================================================================

    // float --> float
    GrFIRfilterFFF *matched_filter =
      new GrFIRfilterFFF (1, new atsc_root_raised_cosine_bandpass (IF_freq));

    // ================================================================
    // Build the FPLL to track the carrier and down convert
    // ================================================================

    // float --> float
    GrAtscFPLL *fpll = new GrAtscFPLL (pilot_freq + 5e3);

    if (initial_phase != DONT_SET_INITIAL_PHASE){
      fpll->set_initial_phase (initial_phase * M_PI/180);
      fpll->set_no_update (true);
    }

    // ================================================================
    // low pass to kill the 2x carrier term here...
    // ================================================================

    // build low pass filter.

#if 0

    double transition_width = 0;

    // The lower edge of the transition band is 6 MHz (the channel width)
    // minus the offset to the pilot .31 MHz.  This is because we've
    // translated the pilot to DC in the FPLL.

    double lower_edge = 6e6 - 0.31e6;

    // The upper edge of the transition band is at twice the pilot.
    // We want the image of the pilot killed.  We could actually
    // set the upper edge at 2.75 (original band edge) + 3.06 (pilot_freq)
    // but that gives an even narrower transition band (more expensive).
    
    double upper_edge = IF_freq - 3e6 + pilot_freq; 	// tight spec
    // double upper_edge = 2 * pilot_freq;		// loose spec
    transition_width = upper_edge - lower_edge;

    vector<float> lp_coeffs =
      gr_firdes::low_pass (1.0,
			   input_rate,
			   (lower_edge + upper_edge) * 0.5,
			   transition_width,
			   gr_firdes::WIN_HAMMING);

    cerr << "Number of lp_coeffs: " << lp_coeffs.size () << endl;

    // float --> float
    GrFIRfilterFFF* lp_filter =
      new  GrFIRfilterFFF(1, lp_coeffs);

#else

    GrFIRfilterFFF* lp_filter = new GrFIRfilterFFF (1, new atsc_exp2_lp ());

#endif

    // ================================================================
    // remove residual DC from signal prior to bit timing loop
    // ================================================================

    // float --> float
    GrRemoveDcFFF *nodc = new GrRemoveDcFFF ();
    
    // ================================================================
    // Bit Timing Loop
    // ================================================================

    // float --> float,syminfo
    GrAtscBitTimingLoop3 *bt_loop =
      new GrAtscBitTimingLoop3 (input_rate / ATSC_SYMBOL_RATE);


    // ================================================================
    // Field Sync Checker
    // ================================================================

    // float,syminfo --> float,syminfo
    // data  --> data,tags
    GrAtscFieldSyncChecker *fsc = new GrAtscFieldSyncChecker ();
    
    // ================================================================
    // Equalizer
    // ================================================================

    atsc_equalizer *atsc_eq;
    if (use_lms_p)
      atsc_eq = create_atsc_equalizer_lms ();
    else
      atsc_eq = create_atsc_equalizer_nop ();
    
    // float,syminfo --> float,syminfo
    // data,tags  --> data,tags 
    GrAtscEqualizer *equalizer = new GrAtscEqualizer (atsc_eq);
    
    // ================================================================
    // Field Sync Demux
    // ================================================================

    // float,syminfo --> atsc_soft_data_segment
    // data,tags   --> atsc_soft_data_segment
    GrAtscFieldSyncDemux *demux = new GrAtscFieldSyncDemux();

    // ================================================================
    // Viterbi Decoder / Deinterleaver /
    // Reed-Solomon Decoder / Derandomizer
    // ================================================================

    // atsc_soft_data_segment --> atsc_mpeg_packet_rs_encoded
    GrAtscViterbiDecoder *viterbi_decoder = new GrAtscViterbiDecoder();

    // atsc_mpeg_packet_rs_encoded --> atsc_mpeg_packet_rs_encoded
    GrAtscDeinterleaver *deinterleaver = new GrAtscDeinterleaver();

    // atsc_mpeg_packet_rs_encoded --> atsc_mpeg_packet_no_sync
    GrAtscRSDecoder *rs_decoder = new GrAtscRSDecoder();

    // atsc_mpeg_packet_no_sync --> atsc_mpeg_packet
    GrAtscDerandomizer *derandomizer = new GrAtscDerandomizer();


    // ================================================================
    // final sink is either a file sink or a null sink
    // ================================================================

    if (output_filename)
      final_sink = new VrFileSink<final_oType>(output_filename);
    else
      final_sink = new VrNullSink<final_oType>();


    // create the empty flow graph
    gr_FlowGraph *fg = gr_FlowGraph::make ();

    // connect the modules together
    fg->connect (vsrc, matched_filter);
    fg->connect (matched_filter, fpll);
    fg->connect (fpll, lp_filter);
    fg->connect (lp_filter, nodc);
    fg->connect (nodc, bt_loop);
    fg->connect (bt_loop, 0, fsc);
    fg->connect (bt_loop, 1, fsc);
    fg->connect (fsc, 0, equalizer);
    fg->connect (fsc, 1, equalizer);
    fg->connect (equalizer, 0, demux);
    fg->connect (equalizer, 1, demux);
    fg->connect (demux, viterbi_decoder);
    fg->connect (viterbi_decoder, deinterleaver);
    fg->connect (deinterleaver, rs_decoder);
    fg->connect (rs_decoder, derandomizer);
    fg->connect (derandomizer, final_sink);

    if (logging_p){
      matched_log = new VrFileSink<float>("matched.rxout");
      fpll_log = new VrFileSink<float>("fpll.rxout");
      lowpass_log = new VrFileSink<float>("lowpass.rxout");
      nodc_log = new VrFileSink<float>("nodc.rxout");
      bt_data_log = new VrFileSink<float>("bt_data.rxout");
      bt_tags_log = new VrFileSink<float>("bt_tags.rxout");
      fsc_data_log = new VrFileSink<float>("fsc_data.rxout");
      fsc_tags_log = new VrFileSink<float>("fsc_tags.rxout");
      eq_data_log = new VrFileSink<float>("eq_data.rxout");
      eq_tags_log = new VrFileSink<float>("eq_tags.rxout");
      demux_log = new VrFileSink<atsc_soft_data_segment>("demux.rxout");
      viterbi_decoder_log = new VrFileSink<atsc_mpeg_packet_rs_encoded>("viterbi_decoder.rxout");
      deinterleaver_log = new VrFileSink<atsc_mpeg_packet_rs_encoded>("deinterleaver.rxout");
      rs_decoder_log = new VrFileSink<atsc_mpeg_packet_no_sync>("rs_decoder.rxout");
      derandomizer_log = new VrFileSink<atsc_mpeg_packet>("derandomizer.rxout");

      
      fg->connect (matched_filter,  matched_log);
      fg->connect (fpll, 	    fpll_log);
      fg->connect (lp_filter,	    lowpass_log);
      fg->connect (nodc,	    nodc_log);
      fg->connect (bt_loop, 0,	    bt_data_log);
      fg->connect (bt_loop, 1,	    bt_tags_log);
      fg->connect (fsc, 0, 	    fsc_data_log);
      fg->connect (fsc, 1, 	    fsc_tags_log);
      fg->connect (equalizer, 0,    eq_data_log);
      fg->connect (equalizer, 1,    eq_tags_log);
      fg->connect (demux,	    demux_log);
      fg->connect (viterbi_decoder, viterbi_decoder_log);
      fg->connect (deinterleaver,   deinterleaver_log);
      fg->connect (rs_decoder,      rs_decoder_log);
      fg->connect (derandomizer,    derandomizer_log);
    }

    fg->start ();    // start executing the flow graph (forks N threads)

    fg->wait ();     // wait (forever)

  } // end try

  catch (std::exception &e){
    cerr << "std library exception: " << e.what () << endl;
    exit (1);
  }
  catch (...) {
    cerr << "unknown exception thrown" << endl;
    exit (1);
  }
}
