/*
     This file is part of GNUnet.
     (C) 2007 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/directory.c
 * @brief code for opening directories
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include "fs.h"
#include "helper.h"
#include "meta.h"
#include "search.h"
#include <GNUnet/gnunet_util_crypto.h>
#include <GNUnet/gnunet_uritrack_lib.h>
#include <GNUnet/gnunet_namespace_lib.h>
#include <extractor.h>

#ifndef MINGW
static char *
selectFile ()
{
  GladeXML *uploadXML;
  GtkFileChooser *dialog;
  char *ret;

  uploadXML
    = glade_xml_new (GNUNET_GTK_get_glade_filename (),
                     "openDirectoryFileDialog", PACKAGE_NAME);
  GNUNET_GTK_connect_glade_with_plugins (uploadXML);
  dialog = GTK_FILE_CHOOSER (glade_xml_get_widget (uploadXML,
                                                   "openDirectoryFileDialog"));
  if (gtk_dialog_run (GTK_DIALOG (dialog)) != GTK_RESPONSE_CANCEL)
    ret = gtk_file_chooser_get_filename (dialog);
  else
    ret = NULL;
  gtk_widget_destroy (GTK_WIDGET (dialog));
  UNREF (uploadXML);
  return ret;
}
#else /* MINGW */
static char *
selectFile ()
{
  return plibc_ChooseFile (_("Choose the directory you want to open."),
                           OFN_FILEMUSTEXIST | OFN_SHAREAWARE);
}
#endif /* MINGW */

static int
spcb (const GNUNET_ECRS_FileInfo * fi,
      const GNUNET_HashCode * key, int isRoot, void *closure)
{
  SearchList *list = closure;
  fs_search_result_received (list, fi, list->uri);
  return GNUNET_OK;
}


void
on_open_menu_activate_fs (GtkWidget * dummy1, GtkWidget * dummy2)
{
  char *dn;
  char *directory_data;
  unsigned long long directory_data_len;
  size_t dlen;
  struct GNUNET_MetaData *md;
  int fd;
  SearchList *list;
  struct GNUNET_ECRS_URI *uri;
  GtkNotebook *notebook;

  dn = selectFile ();
  if (dn == NULL)
    return;
  if ((GNUNET_YES != GNUNET_disk_file_test (NULL,
                                            dn)) ||
      (GNUNET_OK !=
       GNUNET_disk_file_size (NULL, dn, &directory_data_len, GNUNET_YES)))
    {
      GNUNET_GTK_add_log_entry (_("Error accessing file `%s'."), dn);
      GNUNET_free (dn);
      return;
    }
  fd = GNUNET_disk_file_open (NULL, dn, O_LARGEFILE | O_RDONLY);
  if (fd == -1)
    {
      GNUNET_GTK_add_log_entry (_("Error opening file `%s'."), dn);
      GNUNET_free (dn);
      return;
    }
  dlen = (size_t) directory_data_len;
  directory_data = MMAP (NULL, dlen, PROT_READ, MAP_SHARED, fd, 0);
  if (directory_data == MAP_FAILED)
    {
      GNUNET_GTK_add_log_entry (_("Error mapping file `%s' into memory."),
                                dn);
      CLOSE (fd);
      GNUNET_free (dn);
      return;
    }
  uri = GNUNET_ECRS_keyword_string_to_uri (NULL, dn);
  md = NULL;
  list = fs_search_started (NULL, uri, 0, 0, NULL, GNUNET_FSUI_COMPLETED);
  GNUNET_ECRS_uri_destroy (uri);
  GNUNET_ECRS_directory_list_contents (NULL,
                                       directory_data, directory_data_len,
                                       &md, &spcb, list);
  if (md != NULL)
    GNUNET_meta_data_destroy (md);
  MUNMAP (directory_data, dlen);
  CLOSE (fd);
  GNUNET_free (dn);

  /* switch view -- select directory */
  notebook
    =
    GTK_NOTEBOOK (glade_xml_get_widget
                  (GNUNET_GTK_get_main_glade_XML (), "downloadNotebook"));
  gtk_notebook_set_current_page (notebook,
                                 gtk_notebook_page_num (notebook,
                                                        list->searchpage));

  notebook
    =
    GTK_NOTEBOOK (glade_xml_get_widget
                  (GNUNET_GTK_get_main_glade_XML (), "fsnotebook"));
  gtk_notebook_set_current_page (notebook,
                                 gtk_notebook_page_num (notebook,
                                                        glade_xml_get_widget
                                                        (GNUNET_GTK_get_main_glade_XML
                                                         (),
                                                         "fsdownloadvbox")));
}

/* end of directory.c */
