/*
**  NSAttributedString+Extensions.m
**
**  Copyright (c) 2004-2005
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "NSAttributedString+Extensions.h"

#include "Constants.h"
#include "ExtendedTextAttachmentCell.h"
#include "FilterManager.h"
#include "GNUMail.h"
#include "MailHeaderCell.h"
#include "MailWindowController.h"
#include "MimeType.h"
#include "MimeTypeManager.h"
#include "NSAttributedString+TextEnriched.h"
#include "NSColor+Extensions.h"
#include "NSFont+Extensions.h"
#include "NSUserDefaults+Extensions.h"
#include "Task.h"
#include "TaskManager.h"
#include "ThreadArcsCell.h"
#include "Utilities.h"

#include <Pantomime/CWFolder.h>
#include <Pantomime/CWInternetAddress.h>
#include <Pantomime/CWMessage.h>
#include <Pantomime/CWMIMEMultipart.h>
#include <Pantomime/CWMIMEUtility.h>
#include <Pantomime/CWPart.h>
#include <Pantomime/NSString+Extensions.h>

#define APPEND_HEADER(name, value) ({ \
  [maStr appendAttributedString: [NSAttributedString attributedStringWithString: [NSString stringWithFormat: @"%@: ", _(name)] \
						     attributes: headerNameAttribute] ]; \
  [maStr appendAttributedString: [NSAttributedString attributedStringWithString: value \
						     attributes: headerValueAttribute] ]; \
  [maStr appendAttributedString: [NSAttributedString attributedStringWithString: @"\n" \
						     attributes: nil] ]; \
})

//
//
//
static inline int levelFromString(NSString *theString, int start, int end)
{
  int i, level;
  unichar c;
  
  for (i = start,level = 0; i < end; i++)
    {
      c = [theString characterAtIndex: i];

      if (c == '>')
	{
	  level++;
	}
      else if (c > 32)
	{
	  break;
	}
    }

  return level;
}

//
//
//
@implementation NSAttributedString (GNUMailAttributedStringExtensions)

//
// This method returns a NSAttributedString object that has been built
// from the content of the message.
//
+ (NSAttributedString *) attributedStringFromContentForPart: (CWPart *) thePart
						 controller: (id) theController
{
  NSMutableAttributedString *maStr;
  NSMutableDictionary *tAttr;
  
  tAttr = [[NSMutableDictionary alloc] init];
  
  [tAttr setObject: [NSFont messageFont]  forKey: NSFontAttributeName];
  
  maStr = [[NSMutableAttributedString alloc] init];
  
  if ([[thePart content] isKindOfClass: [CWMIMEMultipart class]])
    {
      // We first verify if our multipart object is a multipart alternative.
      // If yes, we represent the best representation of the part.
      if ([thePart isMIMEType: @"multipart"  subType: @"alternative"])
	{
	  // We append our \n to separate our body part from the headers
	  [maStr appendAttributedString: [NSAttributedString attributedStringWithString: @"\n" 
							     attributes: nil]];
	  
	  // We then append the best representation from this multipart/alternative object
	  [maStr appendAttributedString: [NSAttributedString attributedStringFromMultipartAlternative:
							       (CWMIMEMultipart *)[thePart content]]];
	}
      // Then we verify if our multipart object is a multipart appledouble.
      else if ([thePart isMIMEType: @"multipart"  subType: @"appledouble"])
	{
	  // We append our \n to separate our body part from the headers
	  [maStr appendAttributedString: [NSAttributedString attributedStringWithString: @"\n" 
							     attributes: nil]];
	  
	  // We then append the best representation from this multipart/appledouble object
	  [maStr appendAttributedString: [NSAttributedString attributedStringFromMultipartAppleDouble:
							       (CWMIMEMultipart *)[thePart content]
							     controller: theController]];
	}
      // We have a multipart/mixed or multipart/related (or an other, unknown multipart/* object)
      else
	{
	  CWMIMEMultipart *aMimeMultipart;
	  CWPart *aPart;
	  int i;
     
	  aMimeMultipart = (CWMIMEMultipart *)[thePart content];
     
	  for (i = 0; i < [aMimeMultipart count]; i++)
	    {
	      // We get our part
	      aPart = [aMimeMultipart partAtIndex: i];
	      
	      // We recursively call our method to show all parts
	      [maStr appendAttributedString: [self attributedStringFromContentForPart: aPart  controller: theController]];
	    }
	}
    }
  // We have a message with Content-Type: application/* OR audio/* OR image/* OR video/*
  // We can also have a text/* part that was base64 encoded, but, we skip it. It'll be
  // treated as a NSString object. (See below).
  else if ([[thePart content] isKindOfClass: [NSData class]])
    {
      NSTextAttachment *aTextAttachment;
      ExtendedTextAttachmentCell *cell;
      NSFileWrapper *aFileWrapper;
      
      MimeType *aMimeType;
      NSImage *anImage;
      
      NSRect rectOfTextView;
      NSSize imageSize;
     
      aFileWrapper = [[NSFileWrapper alloc] initRegularFileWithContents: (NSData *)[thePart content]];
      
      if (![thePart filename])
	{
	  [aFileWrapper setPreferredFilename: @"unknown"];
	}
      else
	{
	  [aFileWrapper setPreferredFilename: [thePart filename]];
	}

      // We get the righ Mime-Type object for this part
      aMimeType = [[MimeTypeManager singleInstance] 
		    mimeTypeForFileExtension: [[aFileWrapper preferredFilename] pathExtension]];
      
      if (aMimeType && [aMimeType view] == DISPLAY_AS_ICON)
	{
	  anImage = [[MimeTypeManager singleInstance] bestIconForMimeType: aMimeType
						      pathExtension: [[aFileWrapper preferredFilename] pathExtension]];
	}
      else
	{
	  // If the image is a TIFF, we create it directly from a NSData.
	  if ([thePart isMIMEType: @"image"  subType: @"*"])
	    {
	      anImage = [[NSImage alloc] initWithData:(NSData *)[thePart content]];
	      AUTORELEASE(anImage);
	    }
	  //
	  // It's not an image, we query our MIME type manager in order to see 
	  // to get the right icon.
	  //
	  else
	    {
	      anImage = [[MimeTypeManager singleInstance] bestIconForMimeType: aMimeType
							  pathExtension: [[aFileWrapper preferredFilename] 
									   pathExtension]];
	    }
	}
      
      // If the image has been loaded sucessfully, it become our icon for our filewrapper
      if (anImage)
	{
	  [aFileWrapper setIcon: anImage];
	} 	
      
      // We now rescale the attachment if it doesn't fit in the text view. That could happen
      // very often for images.
      rectOfTextView = [[[[GNUMail lastMailWindowOnTop] windowController] textView] frame];
      imageSize = [[aFileWrapper icon] size];
      
      if (imageSize.width > rectOfTextView.size.width)
	{
	  double delta =  1.0 / (imageSize.width / (rectOfTextView.size.width-10));
	  [[aFileWrapper icon] setScalesWhenResized: YES];
	  [[aFileWrapper icon] setSize: NSMakeSize((imageSize.width*delta), imageSize.height*delta)];
	}
      
      // We now create our text attachment with our file wrapper
      aTextAttachment = [[NSTextAttachment alloc] initWithFileWrapper: aFileWrapper];
      
      // We add this attachment to our 'Save Attachment' menu
      [(GNUMail *)[NSApp delegate] addItemToMenuFromTextAttachment: aTextAttachment];
        
      cell = [[ExtendedTextAttachmentCell alloc] initWithFilename: [aFileWrapper preferredFilename]
						 size: [(NSData *)[thePart content] length] ];
      [cell setPart: thePart];
      
      [aTextAttachment setAttachmentCell: cell];
      
      // Cocoa bug
#ifdef MACOSX
      [cell setAttachment: aTextAttachment];
      [cell setImage: [aFileWrapper icon]];
#endif
      RELEASE(cell);
      RELEASE(aFileWrapper);
      
      // We separate the text attachment from any other line previously shown
      [maStr appendAttributedString: [NSAttributedString attributedStringWithString: @"\n" 
							 attributes: nil] ];
      
      [maStr appendAttributedString: [NSAttributedString attributedStringWithAttachment: aTextAttachment]];
      RELEASE(aTextAttachment);
    }
  //
  // We have a message/rfc822 as the Content-Type.
  //
  else if ([[thePart content] isKindOfClass: [CWMessage class]])
    {
      CWMessage *aMessage;
      
      aMessage = (CWMessage *)[thePart content];
      
      // We must represent this message/rfc822 part as an attachment
      if ( [thePart contentDisposition] &&
	   ([[thePart contentDisposition] caseInsensitiveCompare: @"attachment"] == NSOrderedSame) )
	{
	  NSTextAttachment *aTextAttachment;
	  ExtendedTextAttachmentCell *cell;
	  NSFileWrapper *aFileWrapper;
	  NSData *aData;
	  
	  aData = [aMessage rawSource];
	  
	  aFileWrapper = [[NSFileWrapper alloc] initRegularFileWithContents: aData];
	  
	  [aFileWrapper setPreferredFilename: @"message/rfc822 attachment"];
	  
	  aTextAttachment = [[NSTextAttachment alloc] initWithFileWrapper: aFileWrapper];
	  
	  // We add this attachment to our 'Save Attachment' menu
	  [(GNUMail *)[NSApp delegate] addItemToMenuFromTextAttachment: aTextAttachment];
          
	  cell = [[ExtendedTextAttachmentCell alloc] initWithFilename: [aFileWrapper preferredFilename]
						     size: [aData length] ];
	  [cell setPart: thePart];
	  
	  [aTextAttachment setAttachmentCell: cell];
          
          // Cocoa bug
#ifdef MACOSX
          [cell setAttachment: aTextAttachment];
          [cell setImage: [aFileWrapper icon]];
#endif
	  RELEASE(cell); 
	  RELEASE(aFileWrapper);
	  
	  [maStr appendAttributedString: [NSAttributedString attributedStringWithAttachment:
							       aTextAttachment]];
	  RELEASE(aTextAttachment);
	}
      // Its inline..
      else
	{	      
	  [maStr appendAttributedString: [self attributedStringFromHeadersForMessage: aMessage
					       showAllHeaders: NO
					       useMailHeaderCell: NO] ];
	  [maStr appendAttributedString: [self attributedStringFromContentForPart: aMessage  controller: theController]];
	}
    }
  //
  // We have a message with Content-Type: text/*
  // or at least, a part that we can represent directly in our textView.
  // 
  else if ([[thePart content] isKindOfClass: [NSString class]])
    { 
      NSString *aString;
      
      aString = (NSString *)[thePart content];

      // We must represent our NSString as an text attachment in our textView.
      if ([thePart contentDisposition] &&
	  ([[thePart contentDisposition] caseInsensitiveCompare: @"attachment"] == NSOrderedSame))
	{
	  NSTextAttachment *aTextAttachment;
	  ExtendedTextAttachmentCell *cell;
	  NSFileWrapper *aFileWrapper;
	  NSData *aData;
	  
	  aData = [aString dataUsingEncoding: NSUTF8StringEncoding];
	  aFileWrapper = [[NSFileWrapper alloc] initRegularFileWithContents: aData];
	  
	  if ( ![thePart filename] )
	    {
	      [aFileWrapper setPreferredFilename: @"unknown"];
	    }
	  else
	    {
	      [aFileWrapper setPreferredFilename: [thePart filename]];
	    }
	  
	  aTextAttachment = [[NSTextAttachment alloc] initWithFileWrapper: aFileWrapper];
	  
	  // We add this attachment to our 'Save Attachment' menu
	  [(GNUMail *)[NSApp delegate] addItemToMenuFromTextAttachment: aTextAttachment];

	  cell = [[ExtendedTextAttachmentCell alloc] initWithFilename: [aFileWrapper preferredFilename]
						     size: [aData length] ];
	  [cell setPart: thePart];
	  [aTextAttachment setAttachmentCell: cell];
          
          // Cocoa bug
#ifdef MACOSX
          [cell setAttachment: aTextAttachment];
          [cell setImage: [aFileWrapper icon]];
#endif
	  RELEASE(cell);
	  RELEASE(aFileWrapper);
	  
	  // We separate the text attachment from any other line previously shown
	  [maStr appendAttributedString: [NSAttributedString attributedStringWithString: @"\n" 
							     attributes: nil] ];
	  
	  [maStr appendAttributedString: [NSAttributedString attributedStringWithAttachment:
							       aTextAttachment]];
	  RELEASE(aTextAttachment);
	}
      // It's inline...
      else
	{
	  [maStr appendAttributedString: [NSAttributedString attributedStringFromTextPart: thePart]];
	}
    }
  //
  // We have something that we probably can't display.
  // Let's inform the user about this situation.
  // 
  else
    {
      CWMessage *aMessage;

      [maStr appendAttributedString: [NSAttributedString attributedStringWithString: _(@"Loading message...")
							 attributes: nil]];
     
      aMessage = (CWMessage *)thePart;
      
      if (![[TaskManager singleInstance] taskForService: [[aMessage folder] store]])
	{
	  Task *aTask;

	  [aMessage setProperty: [NSNumber numberWithBool: YES]  forKey: MessagePreloading];
	  
	  aTask = [[Task alloc] init];
	  [aTask setKey: [Utilities accountNameForFolder: [aMessage folder]]];
	  [aTask setOp: LOAD_ASYNC];
	  aTask->immediate = YES;
	  aTask->total_size = (float)[aMessage size]/(float)1024;
	  [aTask setMessage: aMessage];
	  [[TaskManager singleInstance] addTask: aTask];
	  RELEASE(aTask);
	}
      
      [[[TaskManager singleInstance] taskForService: [[aMessage folder] store]] addController: theController];
    }
  
  RELEASE(tAttr);

  return AUTORELEASE(maStr);
}


//
// This method returns a NSAttributedString object that has been built
// from the headers of the message.
//
+ (NSAttributedString *) attributedStringFromHeadersForMessage: (CWMessage *) theMessage
						showAllHeaders: (BOOL) showAllHeaders
					     useMailHeaderCell: (BOOL) useMailHeaderCell
{
  NSMutableDictionary *headerNameAttribute, *headerValueAttribute;
  NSMutableAttributedString *maStr; 

  NSMutableAttributedString *aMutableAttributedString;

  NSDictionary *allHeaders;
  NSArray *headersToShow;
  int i, count;

  maStr = [[NSMutableAttributedString alloc] init];
  

  // Attributes for our header names
  headerNameAttribute = [[NSMutableDictionary alloc] init];

  [headerNameAttribute setObject: [NSFont headerNameFont]
		       forKey: NSFontAttributeName];


  // Attributes for our header values
  headerValueAttribute = [[NSMutableDictionary alloc] init];

  [headerValueAttribute setObject: [NSFont headerValueFont]
			forKey: NSFontAttributeName];
			
  
  // We get all the message's headers
  allHeaders = [theMessage allHeaders];
  
  // We verify which headers of the message we show show.
  if (showAllHeaders)
    {
      headersToShow = [allHeaders allKeys];
    }
  else
    {
      headersToShow = [[NSUserDefaults standardUserDefaults] objectForKey: @"SHOWNHEADERS"];
    }

  count = [headersToShow count];
  
  for (i = 0; i < count; i++)
    {
      NSString *anHeader = [headersToShow objectAtIndex: i];
      
      if ([anHeader caseInsensitiveCompare: @"Date"] == NSOrderedSame &&
	  [theMessage receivedDate])
	{
	  if ([theMessage receivedDate])
	    {
	      APPEND_HEADER(@"Date", [[theMessage receivedDate] description]);
	    }
	}
      else if ([anHeader caseInsensitiveCompare: @"From"] == NSOrderedSame &&
	       [theMessage from])
	{
	  APPEND_HEADER(@"From", [[theMessage from] stringValue]);
	}
      else if ([anHeader caseInsensitiveCompare: @"Bcc"] == NSOrderedSame)
	{
	  NSString *bccStr = [NSString stringFromRecipients: [theMessage recipients]
				       type: PantomimeBccRecipient];
	  
	  if ([bccStr length] > 0)
	    {
	      APPEND_HEADER(@"Bcc", [bccStr substringToIndex: ([bccStr length]-2)]);
	    }
	}
      else if ([anHeader caseInsensitiveCompare: @"Cc"] == NSOrderedSame)
	{
	  NSString *ccStr= [NSString stringFromRecipients: [theMessage recipients]
				     type: PantomimeCcRecipient];
	  
	  if ([ccStr length] > 0)
	    {
	      APPEND_HEADER(@"Cc", [ccStr substringToIndex: ([ccStr length]-2)]);
	    }
	}
      else if ([anHeader caseInsensitiveCompare: @"Reply-To"] == NSOrderedSame)
	{ 
	  CWInternetAddress *anInternetAddress = [theMessage replyTo];
	  
	  if (anInternetAddress)
	    {
	      APPEND_HEADER(@"Reply-To", [anInternetAddress stringValue]);
	    }	
	}
      else if ([anHeader caseInsensitiveCompare: @"To"] == NSOrderedSame)
	{
	  NSString *toStr = [NSString stringFromRecipients: [theMessage recipients]
				      type: PantomimeToRecipient];
	  
	  if ([toStr length] > 0)
	    {
	      APPEND_HEADER(@"To", [toStr substringToIndex: ([toStr length]-2)]);
	    }
	}
      else if ([anHeader caseInsensitiveCompare: @"Content-Type"] == NSOrderedSame)
	{
	  NSString *aString;
	  
	  if ([theMessage charset])
	    {
	      aString = [NSString stringWithFormat: @"%@; charset=%@",
				  [theMessage contentType], [theMessage charset]];
	    }
	  else
	    {
	      aString = [theMessage contentType];
	    }
	  
	  APPEND_HEADER(@"Content-Type", aString);
	}
      else
	{
	  NSArray *allKeys = [allHeaders allKeys];
	  NSString *valueString = nil;
	  int j, c;
	  
	  c = [allKeys count];
	  
	  for (j = 0; j < c; j++)
	    {
	      if ([[[allKeys objectAtIndex: j] uppercaseString] isEqualToString: [anHeader uppercaseString]])
		{
		  if ([[allHeaders objectForKey: [allKeys objectAtIndex: j]] 
			isKindOfClass: [CWInternetAddress class]])
		    {
		      valueString = [[allHeaders objectForKey: [allKeys objectAtIndex: j]] stringValue];
		    }
		  else
		    {
		      valueString = [allHeaders objectForKey: [allKeys objectAtIndex: j]];
		    }
		  break;
		}
	    }
	  
	  if (valueString)
	    {
	      APPEND_HEADER(anHeader, valueString);
	    }
	}
    } // for (..) 
    
  if (useMailHeaderCell)
    {
      NSTextAttachment *aTextAttachment;
      MailHeaderCell *theMailHeaderCell;
      
      theMailHeaderCell = [[[GNUMail lastMailWindowOnTop] windowController] mailHeaderCell];
      [theMailHeaderCell setColor: [[FilterManager singleInstance] colorForMessage: theMessage]];
      [theMailHeaderCell setAttributedStringValue: maStr];
      [theMailHeaderCell resize: nil];
      
      // We now "embed" the header cell into a NSTextAttachment object
      // so we can add it to our mutable attributed string.
      aTextAttachment = [[NSTextAttachment alloc] init];
      [aTextAttachment setAttachmentCell: theMailHeaderCell];
      
      aMutableAttributedString = [[NSMutableAttributedString alloc] init];
      
      [aMutableAttributedString appendAttributedString: 
				  [NSMutableAttributedString attributedStringWithAttachment: aTextAttachment]];
      RELEASE(aTextAttachment);
      
      //
      // If we are using message thread, we show our visualization cell.
      //
      if ([[theMessage folder] allContainers])
	{
          ThreadArcsCell *theThreadArcsCell;

          theThreadArcsCell = [[[GNUMail lastMailWindowOnTop] windowController] threadArcsCell];

	  aTextAttachment = [[NSTextAttachment alloc] init];
	  [aTextAttachment setAttachmentCell: theThreadArcsCell];
	  [aMutableAttributedString appendAttributedString: [NSMutableAttributedString attributedStringWithAttachment: aTextAttachment]];
	  RELEASE(aTextAttachment);
#ifndef MACOSX
	  [aMutableAttributedString appendAttributedString: [NSAttributedString attributedStringWithString: @"\n"
										attributes: nil]];
#endif
	}
#ifdef MACOSX
      [aMutableAttributedString appendAttributedString: [NSAttributedString attributedStringWithString: @"\n\n"
									    attributes: nil]];
#else
      [aMutableAttributedString appendAttributedString: [NSAttributedString attributedStringWithString: @"\n"
									    attributes: nil]];
#endif
    }
  else
    {
      aMutableAttributedString = [[NSMutableAttributedString alloc] init];
      [aMutableAttributedString appendAttributedString: maStr];
      [aMutableAttributedString appendAttributedString: [NSAttributedString attributedStringWithString: @"\n"
									    attributes: nil] ];
    }
  
  RELEASE(maStr);
  RELEASE(headerNameAttribute);
  RELEASE(headerValueAttribute);

  return AUTORELEASE(aMutableAttributedString);
}


//
//
//
+ (NSAttributedString *) attributedStringFromMultipartAlternative: (CWMIMEMultipart *) theMimeMultipart
{
  NSString *aSubtype;
  CWPart *aPart;
  int i, index;
  
  if ([[NSUserDefaults standardUserDefaults] integerForKey: @"DEFAULT_MULTIPART_ALTERNATIVE_TYPE"] == TYPE_HTML)
    {
      aSubtype = @"html";
    }
  else
    {
      aSubtype = @"plain";
    }
  
  index = -1;

  // We search for our preferred part (text/html or text/plain) depending on aSubtype
  for (i = 0; i < [theMimeMultipart count]; i++)
    {
      aPart = [theMimeMultipart partAtIndex: i];
      
      if ([aPart isMIMEType: @"text"  subType: aSubtype])
	{
	  index = i;
	  break;
	}
    }

  // If we have found our preferred part, we use that one
  if (index >= 0)
    {
      return [self attributedStringFromTextPart: [theMimeMultipart partAtIndex: index]];
    }
  // We haven't, we show the first one!
  else
    {
      if ([theMimeMultipart count] > 0)
	{
	  return [self attributedStringFromTextPart: [theMimeMultipart partAtIndex: 0]];
	}
    }
  
  return [self attributedStringFromTextPart: nil];
}


//
//
//
+ (NSAttributedString *) attributedStringFromMultipartAppleDouble: (CWMIMEMultipart *) theMimeMultipart
						       controller: (id) theController
{
  NSMutableAttributedString *aMutableAttributedString;
  NSMutableDictionary *attributes;
  CWPart *aPart;
  int i;
  
  // We create a set of attributes (base font, color red)
  attributes = [[NSMutableDictionary alloc] init];
  [attributes setObject: [NSColor redColor]
	      forKey: NSForegroundColorAttributeName];

  aMutableAttributedString = [[NSMutableAttributedString alloc] init];

  for (i = 0; i < [theMimeMultipart count]; i++)
    {
      aPart = [theMimeMultipart partAtIndex: i];
    
      if ([aPart isMIMEType: @"application"  subType: @"applefile"])
	{
	  [aMutableAttributedString appendAttributedString: [self attributedStringWithString: _(@"(Decoded Apple file follows...)")
								  attributes: attributes]];
	}
      else
	{
	  // We first add a \n between our applefile description and the 'representation' of the attachment
	  [aMutableAttributedString appendAttributedString: [self attributedStringWithString: @"\n"  attributes: nil]];
	  
	  // We add the representation of our attachment
	  [aMutableAttributedString appendAttributedString: [self attributedStringFromContentForPart: aPart  controller: theController]];
	} 
    }
  
  // We add a \n to separate everything.
  [aMutableAttributedString appendAttributedString:
			      [NSAttributedString attributedStringWithString: @"\n" attributes: nil]];
  
  RELEASE(attributes);

  return AUTORELEASE(aMutableAttributedString);
}


//
//
//
+ (NSAttributedString *) attributedStringFromTextPart: (CWPart *) thePart
{
  NSMutableDictionary *textMessageAttributes;
  NSAttributedString *aAttributedString;
  NSString *aString;

  aAttributedString = nil;

  // Initial and trivial verifications.
  // We MUST be sure the content isn't nil and also that it's a valid NSString object.
  // Pantomime will return the raw source (NSData object) if the decoding operation has failed.
  if ( !thePart ||
       ![thePart content] || 
       ![[thePart content] isKindOfClass: [NSString class]] )
    {
      goto done;
    }

  // Initializations of the local vars
  textMessageAttributes = [[NSMutableDictionary alloc] init];
  aString = (NSString *)[thePart content];
  
  [textMessageAttributes setObject: [NSFont messageFont]
			 forKey: NSFontAttributeName];


  //
  // text/html
  //
  if ([thePart isMIMEType: @"text"  subType: @"html"])
    {
#ifdef MACOSX
      NSData *aData;
      
      aData = [aString dataUsingEncoding: [NSString encodingForPart: thePart]];      
      aAttributedString = [[NSAttributedString alloc] initWithHTML: aData
						      documentAttributes: nil];
      AUTORELEASE(aAttributedString);
#else
      aString = [CWMIMEUtility plainTextContentFromPart: thePart];
      aAttributedString = [NSAttributedString attributedStringWithString: aString
					      attributes: textMessageAttributes];
#endif
    }
  //
  // text/enriched
  //
  else if ([thePart isMIMEType: @"text"  subType: @"enriched"])
    {
      aAttributedString = [NSAttributedString attributedStringFromTextEnrichedString: aString];
    }
  //
  // text/rtf
  //
  else if ([thePart isMIMEType: @"text"  subType: @"rtf"])
    {
      NSData *aData;
	      
      aData = [aString dataUsingEncoding: [NSString encodingForPart: thePart]];
      
      aAttributedString = [[NSAttributedString alloc] initWithRTF: aData
						      documentAttributes: NULL];
      AUTORELEASE(aAttributedString); 
    }
  //
  // We surely got a text/plain part
  //
  else
    {
      NSMutableDictionary *plainTextMessageAttributes;
      
      if ( [[NSUserDefaults standardUserDefaults] 
	     objectForKey: @"USE_FIXED_FONT_FOR_TEXT_PLAIN_MESSAGES"] &&
	   [[NSUserDefaults standardUserDefaults] 
	     integerForKey: @"USE_FIXED_FONT_FOR_TEXT_PLAIN_MESSAGES"] == NSOnState )
	{
	  plainTextMessageAttributes = [[NSMutableDictionary alloc] init];
	  AUTORELEASE(plainTextMessageAttributes);
	  
	  [plainTextMessageAttributes setObject: [NSFont plainTextMessageFont]
				      forKey: NSFontAttributeName];
	}
      else
	{
	  plainTextMessageAttributes = textMessageAttributes;
	}
      
      aAttributedString = [NSAttributedString attributedStringWithString: aString
					      attributes: plainTextMessageAttributes];
    }

  RELEASE(textMessageAttributes);
  
 done:
  // We haven't found a text/plain part. Report this as a bug in GNUMail for not supporting
  // the other kind of parts.
  if ( !aAttributedString )
    {
      aAttributedString = [NSAttributedString attributedStringWithString: 
						_(@"No text part found. Please report this bug since GNUMail doesn't support this kind of part.")
					      attributes: nil];
    }
  
  return aAttributedString;
}


//
//
//
+ (NSAttributedString *) attributedStringWithString: (NSString *) theString
					 attributes: (NSDictionary *) theAttributes
{
  if (!theAttributes)
    {
      NSMutableDictionary *aMutableAttributedString;
      NSMutableDictionary *attributes;

      attributes = [[NSMutableDictionary alloc] init];
      [attributes setObject: [NSFont systemFontOfSize: 0] forKey: NSFontAttributeName];
      aMutableAttributedString = [[self alloc] initWithString: theString   attributes: attributes];
      RELEASE(attributes);
      
      return AUTORELEASE(aMutableAttributedString);
    }
  
  
  return AUTORELEASE([[NSAttributedString alloc] initWithString: theString attributes: theAttributes]);
}


//
//
//
#warning implement and use in the cell drawing code for the message headers
- (NSSize) sizeInRect: (NSRect) theRect
{
  if ([self size].width < theRect.size.width)
    {
      return [self size];
    }

  return NSZeroSize;
}

@end


//
//
//
@implementation NSMutableAttributedString (GNUMailMutableAttributedStringExtensions)

- (void) format
{
  NSMutableArray *allRanges;
  NSRange aRange, maxRange;
  NSString *aString;
  
  int c, i, len, index, offset;
  id attachment, cell;
  
  allRanges = [[NSMutableArray alloc] init];
  len = [self length];
  index = 0;
  
  maxRange = NSMakeRange(0, len);

  while (index < len)
    {
      attachment = [self attribute: NSAttachmentAttributeName
			 atIndex: index
			 longestEffectiveRange: &aRange
			 inRange: maxRange];
      
      if (attachment)
	{ 
	  cell = [attachment attachmentCell];
	  
	  if ([cell respondsToSelector: @selector(part)])
	    {
	      [allRanges addObject: [NSArray arrayWithObjects: attachment, [NSValue valueWithRange: aRange], nil]];
	    }
	}
      
      index = NSMaxRange(aRange);
      maxRange = NSMakeRange(index, len-index);
    }

  aString = [self string];
  c = [allRanges count];
  offset = 0;

  for (i = c-1; i >= 0; i--)
    {
      attachment = [[allRanges objectAtIndex: i] objectAtIndex: 0];
      cell = [attachment attachmentCell];
      
      aRange = [aString rangeOfString: [NSString stringWithFormat: @"<<%@>>", [[cell part] filename]]
			options: NSBackwardsSearch];
      
      if (aRange.location == NSNotFound)
        {
          aRange = [aString rangeOfString: [NSString stringWithFormat: @"<%@>", [[cell part] filename]]
			    options: NSBackwardsSearch];
        }
      
      if (aRange.length)
        {
	  NSRange r;

	  r = [[[allRanges objectAtIndex: i] lastObject] rangeValue];
	  r.location = r.location - offset;
	  
	  [self deleteCharactersInRange: r];
	  [self replaceCharactersInRange: aRange
		withAttributedString: [NSAttributedString attributedStringWithAttachment: attachment]];

	  offset = offset + aRange.length - 1;
        }
    }

  RELEASE(allRanges);
}


//
//
//
- (void) highlightAndActivateURL
{
  NSRange searchRange, foundRange;
  NSString *aString, *aPrefix;
  NSEnumerator *theEnumerator;
  NSArray *allPrefixes;

  int len;
  char c;

  allPrefixes = [NSArray arrayWithObjects: @"www.", @"http://", @"https://", @"ftp://", @"file://", nil];
  theEnumerator = [allPrefixes objectEnumerator];
    
  aString = [self string];
  len = [aString length];

  while ((aPrefix = (NSString *)[theEnumerator nextObject]))
    {
      searchRange = NSMakeRange(0, len);
      do
	{
	  foundRange = [aString rangeOfString: aPrefix
				options: 0
				range: searchRange];
	  
	  // If we found an URL...
	  if (foundRange.length > 0)
	    {
	      NSDictionary *linkAttributes;
	      NSURL *anURL;
	      int end;

	      // Restrict the searchRange so that it won't find the same string again
	      searchRange.location = end = NSMaxRange(foundRange);
	      searchRange.length = len - searchRange.location;
	      
	      // We assume the URL ends with whitespace
	      while ((end < len) && ((c = [aString characterAtIndex: end]) != '\n' && c != ' ' && c != '\t'))
		{
		  end++;
		}

	      // Set foundRange's length to the length of the URL
	      foundRange.length = end - foundRange.location;

	      // If our URL is ended with a ".", "!", "," ">", or ")", trim it
	      c = [aString characterAtIndex: (end - 1)];
	      
	      if (c == '.' || c == '!' || c == ',' || c == '?' || c == '>'
	      	  || c == ')')
		{
		  foundRange.length--;
		}

	      // We create our URL object from our substring
              // if we found just "www", we prepend "http://"
              if ([aPrefix caseInsensitiveCompare: @"www."] == NSOrderedSame)
		{
		  anURL = [NSURL URLWithString: [NSString stringWithFormat: @"http://%@", 
							  [aString substringWithRange: foundRange]]];
		}
	      else
		{
		  anURL = [NSURL URLWithString: [aString substringWithRange: foundRange]];
		}
	      
	      // Make the link attributes
	      linkAttributes = [NSDictionary dictionaryWithObjectsAndKeys: anURL, NSLinkAttributeName,
					     [NSNumber numberWithInt: NSSingleUnderlineStyle], NSUnderlineStyleAttributeName,
					     [NSColor blueColor], NSForegroundColorAttributeName,
					     NULL];
	      
	      // Finally, apply those attributes to the URL in the text
	      [self addAttributes: linkAttributes  range: foundRange];
	    }
	  
	} while (foundRange.length != 0);
    }
}


//
//
//
- (void) quote
{
  if ([[NSUserDefaults standardUserDefaults] integerForKey: @"COLOR_QUOTED_TEXT"  default: NSOnState] == NSOffState)
    {
      return;
    }
  else
    {   
      NSDictionary *attributes;
      NSString *aString;
 
      int i, j, len, level;
     
      aString = [self string];
      len = [aString length];
      i = j = level = 0;
      
      for (; i < len; i++)
	{
	  if ([aString characterAtIndex: i] == '\n')
	    {
	      if (i > j)
		{
		  level = levelFromString(aString, j, i);
		  
		  if (level)
		    {
		      attributes = [NSDictionary dictionaryWithObjectsAndKeys: 
						   [NSColor colorForLevel: level],
						 NSForegroundColorAttributeName, 
						 nil];
		      [self addAttributes: attributes  range: NSMakeRange(j,i-j)];
		    }
		}
	      j = i+1;
	    }
	}
      
      if (i > j)
	{
	  level = levelFromString(aString, j, i);

	  if (level)
	    {
	      attributes = [NSDictionary dictionaryWithObjectsAndKeys: 
					   [NSColor colorForLevel: level],
					 NSForegroundColorAttributeName, 
					 nil];
	      [self addAttributes: attributes  range: NSMakeRange(j,i-j)];
	    }
	}
    }
}

@end

