/*
**  TaskManager.m
**
**  Copyright (c) 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "TaskManager.h"

#include "ConsoleWindowController.h"
#include "Filter.h"
#include "FilterManager.h"
#include "GNUMail+TaskManager.h"
#include "Constants.h"
#include "LocalMailDelivery.h"
#include "MailboxManagerController.h"
#include "Task.h"
#include "Utilities.h"

#include <Foundation/NSArray.h>
#include <Foundation/NSAutoreleasePool.h>
#include <Foundation/NSConnection.h>
#include <Foundation/NSData.h>
#include <Foundation/NSDebug.h>
#include <Foundation/NSException.h>
#include <Foundation/NSLock.h>
#include <Foundation/NSObject.h>
#include <Foundation/NSProxy.h>
#include <Foundation/NSTimer.h>
#include <Foundation/NSRunLoop.h>
#include <Foundation/NSValue.h>

#include <Pantomime/Constants.h>
#include <Pantomime/FolderInformation.h>
#include <Pantomime/IMAPStore.h>
#include <Pantomime/Message.h>
#include <Pantomime/POP3CacheManager.h>
#include <Pantomime/POP3CacheObject.h>
#include <Pantomime/POP3Folder.h>
#include <Pantomime/POP3Store.h>
#include <Pantomime/Sendmail.h>
#include <Pantomime/SMTP.h>
#include <Pantomime/URLName.h>

#include <unistd.h>

#define LONG_DELAY 86400.0

#define ADD_CONSOLE_MESSAGE(format, args...) \
  [[ConsoleWindowController singleInstance] \
	 performSelectorOnMainThread: @selector(addConsoleMessage:) \
	 withObject: [NSString stringWithFormat: format, ##args] \
	 waitUntilDone: NO];

#define SET_LOCK_VALUE(b) ({ \
 [lock lock]; \
 mustStopCurrentTask = b; \
 [lock unlock]; \
})

#define SMTP_ERROR_OCCURED() ({ \
 [[NSApp delegate] performSelectorOnMainThread: @selector(runAlertPanel:) \
		   withObject: [NSDictionary dictionaryWithObjectsAndKeys: [NSString stringWithFormat: _(@"An error occured while sending the E-Mail. It might be a\nnetwork problem or an error in your sending preferences.\nLast response received from the server:\n\n(%d) %@\n\nTo save this E-Mail in the Drafts folder, open the Console window\nfrom the Windows menu and click on the \"Save in Drafts\" button."), [aSMTP lastResponseCode], [aSMTP lastResponse]], @"Message", _(@"Error!"), @"Title", nil] \
		   waitUntilDone: YES]; \
})

#define UPDATE_STATUS_LABEL(format, args...) \
  [[NSApp delegate] performSelectorOnMainThread: @selector(updateStatusLabelWithMessage:) \
		    withObject: [NSString stringWithFormat: format, ##args] \
		    waitUntilDone: NO];

static TaskManager *singleInstance;
static NSLock *lock;

//
//
//
@implementation TaskManager

- (id) init
{
  self = [super init];

  return self;
}


//
//
//
- (void) dealloc
{
  NSDebugLog(@"TaskManager: -dealloc");

  RELEASE(connection);
  RELEASE(lock);

  [super dealloc];
}


//
//
//
- (oneway void) fire
{
  if ( timer )
    {
      [timer fire];
    }
}


//
// This method will return IIF the application is about to terminate.
//
- (void) run: (id) thePorts
{
  NSAutoreleasePool *pool;
  
  pool = [[NSAutoreleasePool alloc] init];
  
  connection = [[NSConnection alloc] initWithReceivePort: [thePorts objectAtIndex: 0]
				     sendPort: [thePorts objectAtIndex: 1]];
  [connection setRootObject: self];
  [connection setRequestTimeout: LONG_DELAY];
  [connection setReplyTimeout: LONG_DELAY];

  // We create our timer, it'll get added to the runloop in the NSDefaultRunLoopMode
  timer = [NSTimer scheduledTimerWithTimeInterval: 5
		   target: self
		   selector: @selector(_nextTask)
		   userInfo: nil
		   repeats: YES];

  [[NSRunLoop currentRunLoop] run];
  
  NSDebugLog(@"TaskManager: Releasing the autorelease pool.");
  RELEASE(pool);
}


//
//
//
- (void) stop
{
  NSDebugLog(@"Stopping the TaskManager...");
  [timer invalidate];
  RELEASE(timer);
  NSDebugLog(@"Done stopping the TaskManager.");
}


//
//
//
- (void) stopCurrentTask
{
  SET_LOCK_VALUE(YES);
}


//
// Class methods
//
+ (TaskManager *) singleInstance
{
  if ( !singleInstance )
    {
      singleInstance = [[TaskManager alloc] init];
      lock = [[NSLock alloc] init];
    }

  return singleInstance;
}

@end



//
// TaskManager private interface
//
@implementation TaskManager (Private)

- (void) _executeActionUsingFilter: (Filter *) theFilter
                           message: (NSData *) theMessage
                              task: (Task *) theTask
{
  switch ( [theFilter actionEMailOperation] )
    {
    case BOUNCE:
    case FORWARD:
    case REPLY:
      NSLog(@"Unimplemented action - ignoring.");
      break;

    default:
      NSLog(@"Unknown action - ignoring.");
    }
}


//
//
//
- (void) _matchFilterRuleFromRawSource: (NSData *) theRawSource
				  task: (Task *) theTask
{
  FilterManager *aFilterManager;
  URLName *theURLName;
  Filter *aFilter;
  
  aFilterManager = (FilterManager *)[FilterManager singleInstance];
  
  aFilter = [aFilterManager matchedFilterForMessageAsRawSource: theRawSource
			    type: TYPE_INCOMING];
  
  if ( aFilter && [aFilter action] == BOUNCE_OR_FORWARD_OR_REPLY )
    {
      [self _executeActionUsingFilter: aFilter
	    message: theRawSource
	    task: theTask];
    }
  else if ( aFilter && [aFilter action] == PLAY_SOUND )
    {
      if ( [[NSFileManager defaultManager] fileExistsAtPath: [aFilter pathToSound]] )
	{
	  NSSound *aSound;
	  
	  aSound = [[NSSound alloc] initWithContentsOfFile: [aFilter pathToSound]  byReference: YES];
	  [aSound play];
	  RELEASE(aSound);
	}
    }
  
  // Even if we bounced (or forward or replied to the) message, or played a sound
  // when receiving it, we append it to our folder.
  theURLName = [aFilterManager matchedURLNameFromMessageAsRawSource: theRawSource
			       type: TYPE_INCOMING
			       key: [theTask key]
			       filter: aFilter];

  if ( [theTask origin] == ORIGIN_USER )
    {
      NSString *aFolderName;
      
      // We verify if our task owner is a MailWindowController. If it's the case,
      // we must check if we really need to add the the folder to our list of
      // filtered message folders.
      if ( [theTask owner] && 
	   [[theTask owner] respondsToSelector: @selector(dataView)] &&
	   [Utilities URLWithString: [theURLName stringValue]
		      matchFolder: [[theTask owner] folder]] )
	{
	  // Same folder, we skip it.
	  goto done;
	}
      
      if ( [[theURLName protocol] caseInsensitiveCompare: @"LOCAL"] == NSOrderedSame )
	{
	  aFolderName = [[NSString alloc] initWithFormat: _(@"Local - %@"), [theURLName foldername]];
	}
      else
	{
	  aFolderName = [[NSString alloc] initWithFormat: _(@"IMAP %@@%@ - %@"), [theURLName username],
					  [theURLName host], [theURLName foldername]];
	}
      
      [theTask setFilteredMessagesCount: [theTask filteredMessagesCount]+1];

      if ( ![[theTask filteredMessagesFolders] containsObject: aFolderName] )
	{
	  [[theTask filteredMessagesFolders] addObject: aFolderName];
	}
      
      RELEASE(aFolderName);
    }
  
 done:
  [[MailboxManagerController singleInstance] performSelectorOnMainThread: @selector(addMessage:)
					     withObject: [NSDictionary dictionaryWithObjectsAndKeys: theRawSource, @"Message",
								       theURLName, @"URLName", nil]
					     waitUntilDone: NO];

  // We set the application icon to GNUMail_Full.tiff
  [NSApp performSelectorOnMainThread: @selector(setApplicationIconImage:)
	 withObject: [NSImage imageNamed: @"GNUMail_Full.tiff"]
	 waitUntilDone: NO];
}


//
//
//
- (id) _mustStopTransfer
{
  NSNumber *aNumber;

  [lock lock];
  aNumber = [NSNumber numberWithBool: mustStopCurrentTask];
  [lock unlock];

  return aNumber;
}


//
//
//
- (void) _nextTask
{ 
  Task *aTask;
  
  aTask = [(GNUMail *)[connection rootProxy] nextTask];

  if ( aTask )
    {
      NSAutoreleasePool *pool;
      
      pool = [[NSAutoreleasePool alloc] init];
      RETAIN(aTask);

      NSDebugLog(@"TaskManager: Got a task. Op: %d - Key: %@", [aTask op], [[aTask key] description]);
      
      switch ( [aTask op] )
	{
	case SEND_SENDMAIL:
	  [self _sendUsingSendmailForTask: aTask];
	  break;
	  
	case SEND_SMTP:
	  [self _sendUsingSMTPForTask: aTask];
	  break;

	case RECEIVE_IMAP:
	  [self _receiveUsingIMAPForTask: aTask];
	  break;

	case RECEIVE_POP3:
	  [self _receiveUsingPOP3ForTask: aTask];
	  break;

	case RECEIVE_UNIX:
	  [self _receiveUsingUNIXForTask: aTask];
	  break;

	default:
	  NSDebugLog(@"Unknown task type. Ignoring and keeping in the queue.");
	}

      RELEASE(aTask);
      RELEASE(pool);
    }
  else
    {
      NSDebugLog(@"No task found in the queue");
    }
}


//
// We must get all body parts (download using body.peek[x]<from.increment>)
//
- (void) _receiveUsingIMAPForTask: (Task *) theTask
{
  NSDictionary *allValues, *aDictionary;
  NSString *aPassword, *aMechanism;
  IMAPStore *aStore;
  NSArray *subscribedFolders;
  int i;

  // We reset our stop ivar
  SET_LOCK_VALUE(NO);
  
  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		 objectForKey: [theTask key]] objectForKey: @"RECEIVE"];
  
  
  if ( [allValues objectForKey: @"USESECURECONNECTION"] &&
       [[allValues objectForKey: @"USESECURECONNECTION"] intValue] == NSOnState )
    {
      aStore = [[IMAPStore alloc] initSSLWithName: [allValues objectForKey: @"SERVERNAME"]
				  port: [[allValues objectForKey: @"PORT"] intValue]];
    }
  else
    {
      aStore = [[IMAPStore alloc] initWithName: [allValues objectForKey: @"SERVERNAME"]
				  port: [[allValues objectForKey: @"PORT"] intValue]];
    }
     
  // That should never fail since the password was just specified by the user
  // before the task was created.
  aPassword = [(GNUMail *)[connection rootProxy] passwordForKey: [theTask key]
			  type: IMAP];

  // We get our authentication mechanism
  aMechanism = [allValues objectForKey: @"AUTH_MECHANISM"];
  
  if ( aMechanism && [aMechanism isEqualToString: @"Password"] )
    {
      aMechanism = nil;
    }

  [aStore authenticate: [allValues objectForKey: @"USERNAME"]
	  password: aPassword
	  mechanism: aMechanism];
 
  subscribedFolders = [NSArray arrayWithArray: [[aStore subscribedFolderEnumerator] allObjects]]; 
 
  for (i = 0; i < [subscribedFolders count]; i++)
    {
      FolderInformation *aFolderInformation;
      NSString *aFolderName;

      if ( [[self _mustStopTransfer] boolValue] )
	{
	  break;
	}

      aFolderName = [subscribedFolders objectAtIndex: i];
   
      // No need to worry about STATUS'ing the selected mailbox since we're using
      // a different connection than the one in the main thread.
      aDictionary = [aStore folderStatus: [NSArray arrayWithObject: aFolderName]];
      
      // FIXME - the folder could have been deleted
      aFolderInformation = [[aDictionary allValues] lastObject];
      
      aDictionary = [NSDictionary dictionaryWithObjectsAndKeys: aFolderInformation, @"FOLDER_INFORMATION",
				  aFolderName, @"FOLDER_NAME",
				  [aStore name], @"STORE_NAME",
				  [aStore username], @"USERNAME", 
				  [aStore folderSeparator], @"FOLDER_SEPARATOR", nil];
      
      [[MailboxManagerController singleInstance] performSelectorOnMainThread: @selector(updateFolderInformation:)
						 withObject: aDictionary
						 waitUntilDone: NO];
    }

  [aStore close];
  RELEASE(aStore);

  [[NSApp delegate] performSelectorOnMainThread: @selector(taskCompleted)
		    withObject: nil
		    waitUntilDone: NO];
}


//
//
//
- (void) _receiveUsingPOP3ForTask: (Task *) theTask
{
  NSNumber *serverTypeValue, *portValue, *retainPeriodValue;
  NSDictionary *allValues;
  NSString *aServerName;
  
  // We reset our stop ivar
  SET_LOCK_VALUE(NO);

  // We get our values associated with the key
  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
		 objectForKey: [theTask key]] objectForKey: @"RECEIVE"];
  
  serverTypeValue = [allValues objectForKey: @"SERVERTYPE"];
  aServerName = [allValues objectForKey: @"SERVERNAME"];
  portValue = [allValues objectForKey: @"PORT"];
  retainPeriodValue = [allValues objectForKey: @"RETAINPERIOD"];
  
  // We use the default POP3 port if it's not defined.
  if ( !portValue )
    {
      portValue = [NSNumber numberWithInt: 110];
    }
  
  // We use the default retain period (365 days, if it's not defined)
  if ( !retainPeriodValue)
    {
      retainPeriodValue = [NSNumber numberWithInt: 365];
    }
  
  if ( !serverTypeValue || [serverTypeValue intValue] == POP3 )
    {
      NSString *aUsername, *aPassword;
      
      // We get our username
      aUsername = [allValues objectForKey: @"USERNAME"];
      
      // We get our password. We prompt the user if we need to.
      aPassword = [(GNUMail *)[connection rootProxy] passwordForKey: [theTask key]
			      type: POP3];
      
      if ( aPassword )
	{
	  NSString *mechanism;
	  POP3Folder *aFolder;
	  POP3Store *aStore;

	  BOOL leaveOnServer, manuallyStoppedTransfer;
	  int transferCount;
	  
	  leaveOnServer = ([[allValues objectForKey: @"LEAVEONSERVER"] intValue] == NSOnState ? YES : NO);
	  manuallyStoppedTransfer = NO;
	  transferCount = 0;

	  mechanism = nil;
	  aStore = nil;
	  
	  // We expect exceptions...
	  NS_DURING
	    {
	      if ( [allValues objectForKey: @"USESECURECONNECTION"] &&
		   [[allValues objectForKey: @"USESECURECONNECTION"] intValue] == NSOnState )
		{
		  UPDATE_STATUS_LABEL(_(@"Connecting using SSL to %@..."), aServerName)
		  aStore = (POP3Store *) [[POP3Store alloc] initSSLWithName: aServerName
							    port: [portValue intValue]];
		}
	      else
		{
		  UPDATE_STATUS_LABEL(_(@"Connecting to %@..."), aServerName)
		  aStore = (POP3Store *) [[POP3Store alloc] initWithName: aServerName
							    port: [portValue intValue]];
		  [[aStore tcpConnection] setStopTarget: self];
		  [[aStore tcpConnection] setStopSelector: @selector(_mustStopTransfer)];
		}
		  
	      // We verify that we got a valid connection with the POP3 server
	      if ( !aStore )
		{
		  [[NSApp delegate] performSelectorOnMainThread: @selector(runAlertPanel:)
				    withObject: [NSDictionary dictionaryWithObjectsAndKeys: 
								[NSString stringWithFormat: _(@"Unable to communicate with the POP3 server (%@)."),
									  aServerName], @"Message", _(@"Error!"), @"Title", nil]
				    waitUntilDone: YES];
		}
	      else
		{
		  // We verify if we should use APOP mode
		  if ( [allValues objectForKey: @"USEAPOP"] )
		    {
		      mechanism = ([[allValues objectForKey: @"USEAPOP"] intValue] == NSOnState ? @"APOP" : nil);
		    }

		  UPDATE_STATUS_LABEL(_(@"Login in %@"), aServerName)
		    
		  if ( [aStore authenticate: aUsername  password: aPassword  mechanism: mechanism] )
		    {
		      POP3CacheManager *aPOP3CacheManager;		      
		      NSString *aCacheFilename;

		      // We get our POP3 folder
		      aFolder = (POP3Folder *)[aStore folderForName: @"Inbox" prefetch: NO];
		      [aFolder setLeaveOnServer: leaveOnServer];
		      [aFolder setRetainPeriod: [retainPeriodValue intValue]];
		      
		      // We get our POP3 cache
		      aCacheFilename = [Utilities flattenPathFromString: [NSString stringWithFormat: @"%@ @ %@", aUsername, aServerName]
						  separator: @"/"];
		      aPOP3CacheManager = [POP3CacheManager pop3CacheObjectsFromDiskWithPath:
							      [NSString stringWithFormat: @"%@/POP3Cache_%@",
									GNUMailUserLibraryPath(),
									aCacheFilename]];
		      [aFolder setCacheManager: aPOP3CacheManager];
		      
		      NS_DURING
			{
			  transferCount = [self _transferMessagesFromPOP3Folder: aFolder
						task: theTask];
			  [theTask setReceivedMessagesCount: transferCount];
			}
		      NS_HANDLER
			{
			  ADD_CONSOLE_MESSAGE(_(@"Got a %@ while transferring mails from %@."), 
					      [localException name], [(POP3Store*)[aFolder store] name])
			  manuallyStoppedTransfer = YES;
			}
		      NS_ENDHANDLER {}
		      
		      // If we haven't manually stopped the transfer, we verify if we transferred any messages.
		      // After, we expunge our folder. If we have manually stopped the transfer, the POP3Store
		      // will simply get closed later.
		      if ( !manuallyStoppedTransfer )
			{
			  if ( transferCount == 0 && [theTask origin] == ORIGIN_USER )
			    {
			      // We show our "No New Messages" panel, if the user wants them!
			      if ( ![[NSUserDefaults standardUserDefaults] 
				      objectForKey: @"SHOW_NO_NEW_MESSAGES_PANEL"] ||
				   [[NSUserDefaults standardUserDefaults]
				     integerForKey: @"SHOW_NO_NEW_MESSAGES_PANEL"] == NSOnState )
				{
				  [[NSApp delegate] performSelectorOnMainThread: @selector(runAlertPanel:)
						    withObject: [NSDictionary dictionaryWithObjectsAndKeys: 
										[NSString stringWithFormat: _(@"There are no new messages on %@ @ %@."), 
											  aUsername, aServerName], @"Message", 
									      _(@"No New Messages..."), @"Title", nil]
						    waitUntilDone: YES];      
				}
			      
			      ADD_CONSOLE_MESSAGE(_(@"No new messages on server %@"), aServerName)
			    }
			  
			  // We now expunge our folder. It'll removed old messages if they have 'expired'.
			  UPDATE_STATUS_LABEL(_(@"Expunging the folder..."))
			  [aFolder expunge: NO];
			  [aFolder close];
			}
		      
		    } // if ( [aStore authenticate: password: ... ] )
		  else
		    {
		      // We inform our main thread the the password isn't valid for this server
		      [[NSApp delegate] performSelectorOnMainThread: @selector(authenticationFailed:)
					withObject: [NSDictionary dictionaryWithObjectsAndKeys: aServerName, @"Server",
								  aUsername, @"Username",
								  [NSNumber numberWithInt: POP3], @"Type", nil]
					waitUntilDone: YES];
		    }
		  
		  // We now close our store
		  // We set our mustStopCurrentTask to NO since closing the POP3 store sends a QUIT command
		  // on the socket and our TCPConnection class MUST send it.
		  SET_LOCK_VALUE(NO);
		  [aStore close];
		  DESTROY(aStore);
		} // else of if ( !aStore ) ...
	    }
	  NS_HANDLER
	    {
	      ADD_CONSOLE_MESSAGE(_(@"Got a %@ while estabilishing (or closing) the POP3 connection to server %@."), 
				  [localException name], aServerName)

	      // We force-close the TCP connection and we release our POP3Store, if we need to.
	      if ( aStore )
		{
		  [[aStore tcpConnection] close];
		  RELEASE(aStore);
		}
	    }
	  NS_ENDHANDLER {}
	  
	} // if ( password ) ...
      
    } // if (!serverTypeValue || ... )

  [[NSApp delegate] performSelectorOnMainThread: @selector(taskCompleted)
		    withObject: nil
		    waitUntilDone: NO];
}


//
//
//
- (void) _receiveUsingUNIXForTask: (Task *) theTask
{
  LocalMailDelivery *aLocalMailDelivery;
  NSDictionary *allValues;
  NSAutoreleasePool *pool;
  NSArray *anArray;
  int i;

  // We get our values associated with the key
  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
		 objectForKey: [theTask key]] objectForKey: @"RECEIVE"];

  aLocalMailDelivery = [[LocalMailDelivery alloc] initWithPathToSpool:
						    [allValues objectForKey: @"MAILSPOOLFILE"]];
  
  anArray = [aLocalMailDelivery messagesFromMailspoolFile];  
  pool = nil;
  
  for (i = 0; i < [anArray count]; i++)
    {
      if ( (i % 3) == 0 )
	{
	  TEST_RELEASE(pool);
	  pool = [[NSAutoreleasePool alloc] init];
	}
      
      [self _matchFilterRuleFromRawSource: [anArray objectAtIndex: i]
	    task: theTask];
    }
  
  TEST_RELEASE(pool);
  RELEASE(aLocalMailDelivery);

  [[NSApp delegate] performSelectorOnMainThread: @selector(taskCompleted)
		    withObject: nil
		    waitUntilDone: NO];
}


//
//
//
- (void) _sendUsingSendmailForTask: (Task *) theTask
{
  NSDictionary *allValues;
  Sendmail *aSendmail;
  
  NSString *aString;
  NSRange aRange;
  NSData *aData;

  BOOL messageWasSent;
  
  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		 objectForKey: [theTask sendingKey]] objectForKey: @"SEND"];
  
  // We initialize our local variables so that gcc shuts up
  messageWasSent = NO;
  aSendmail = nil;
  aData = nil;

  // We verify if the path is valid and the bin is executable
  aString = [allValues objectForKey: @"MAILER_PATH"];
  aRange = [aString rangeOfString: @" "];
  
  if ( aRange.location != NSNotFound )
    {
      aString = [aString substringToIndex: aRange.location];
    }

  if ( ![[NSFileManager defaultManager] isExecutableFileAtPath: aString] )
    {
      goto done;
    }
  
  // Everything is fine, let's send the message
  aSendmail = [[Sendmail alloc] initWithPathToSendmail: [allValues objectForKey: @"MAILER_PATH"]];
  
  if ( [[theTask message] respondsToSelector: @selector(isEqualToData:)] )
    {
      aData = [theTask message];
    }
  else
    {
      aData = [[theTask message] dataValue];
    }

  messageWasSent = [aSendmail sendMessageFromRawSource: aData];

 done:
  if ( messageWasSent )
    {
      // We do not append the bounced message (as NSData) to the folder.
      if ( ![[theTask message] respondsToSelector: @selector(isEqualToData:)] )
	{
	  FilterManager *aFilterManager;
	  URLName *theURLName;
	  
	  aFilterManager = (FilterManager *)[FilterManager singleInstance];
  
	  // Needs locking.
	  theURLName = [aFilterManager matchedURLNameFromMessageAsRawSource: aData
				       type: TYPE_OUTGOING
				       key: [theTask key]
				       filter: nil];
	  
	  [[MailboxManagerController singleInstance] performSelectorOnMainThread: @selector(addMessage:)
						     withObject: [NSDictionary dictionaryWithObjectsAndKeys: aData, @"Message",
									       theURLName, @"URLName", nil]
						     waitUntilDone: NO];
	}
      
      [[NSApp delegate] performSelectorOnMainThread: @selector(taskCompleted)
			withObject: nil
			waitUntilDone: NO];   
    }
  else
    {
      // An error occured. We warn the user about it.
      [[NSApp delegate] performSelectorOnMainThread: @selector(runAlertPanel:)
			withObject: [NSDictionary dictionaryWithObjectsAndKeys: _(@"An error occured while sending the E-Mail. The path to the\nmailer might be incorrect in your sending preferences."), @"Message", _(@"Error!"), @"Title", nil]
			waitUntilDone: YES];

      // We leave the task in the queue.
      NSDebugLog(@"Leaving the task in the queue + 5 min.");
      [theTask setDate: [AUTORELEASE([[NSDate alloc] init]) addTimeInterval: 300]];
      [theTask setRunning: NO];
      [[NSApp delegate] performSelectorOnMainThread: @selector(consoleWindowNeedsDisplay)
			withObject: nil
			waitUntilDone: NO];
    }
  
  RELEASE(aSendmail);
}


//
//
//
- (void) _sendUsingSMTPForTask: (Task *) theTask
{
  NSDictionary *allValues;
  NSNumber *portValue;
  NSData *aData;
  SMTP *aSMTP;

  BOOL messageWasSent, manuallyStoppedTransfer;

  // We reset our stop ivar
  SET_LOCK_VALUE(NO);

  messageWasSent = manuallyStoppedTransfer = NO;
  
  aSMTP = nil;
  aData = nil;
  
  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		 objectForKey: [theTask sendingKey]] objectForKey: @"SEND"];

  portValue = [allValues objectForKey: @"SMTP_PORT"];
  
  if ( !portValue )
    {
      portValue = [NSNumber numberWithInt: 25];
    }
  
  // We expect exceptions...
  NS_DURING
    {
      if ( [allValues objectForKey: @"USESECURECONNECTION"] &&
	   [[allValues objectForKey: @"USESECURECONNECTION"] intValue] == NSOnState )
	{
	  ADD_CONSOLE_MESSAGE(_(@"SMTP - connecting using SSL to %@..."), [allValues objectForKey: @"SMTP_HOST"])
	  aSMTP = [[SMTP alloc] initSSLWithName: [allValues objectForKey: @"SMTP_HOST"]
				port: [portValue intValue]];
	}
      else
	{
	  ADD_CONSOLE_MESSAGE(_(@"SMTP - connecting to %@..."), [allValues objectForKey: @"SMTP_HOST"])
	  aSMTP = [[SMTP alloc] initWithName: [allValues objectForKey: @"SMTP_HOST"]
				port: [portValue intValue]];
	}

      ADD_CONSOLE_MESSAGE(_(@"SMTP - Connected to %@!"), [allValues objectForKey: @"SMTP_HOST"])
      
      [[aSMTP tcpConnection] setStopTarget: self];
      [[aSMTP tcpConnection] setStopSelector: @selector(_mustStopTransfer)];
      
      // We verify if we need to authenticate ourself to the SMTP sever
      if ( [allValues objectForKey: @"SMTP_AUTH"] &&
	   [[allValues objectForKey: @"SMTP_AUTH"] intValue] == NSOnState )
	{
	  NSString *aPassword;
	  BOOL aBOOL;
	  
	  aPassword = [(GNUMail *)[connection rootProxy] passwordForKey: [theTask sendingKey]
				  type: OTHER];
	  aBOOL = NO;
	  
	  if ( aPassword )
	    {
	      ADD_CONSOLE_MESSAGE(_(@"SMTP - Authenticating to %@ using %@..."),
				  [allValues objectForKey: @"SMTP_HOST"],
				  [allValues objectForKey: @"SMTP_USERNAME"])
	      aBOOL = [aSMTP authenticate: [allValues objectForKey: @"SMTP_USERNAME"]
			     password: aPassword
			     mechanism: [allValues objectForKey: @"SMTP_AUTH_MECHANISM"]];
	    }
	  
	  // SMTP authentication failed, let's close the store and inform the main thread about that.
	  if ( !aBOOL )
	    {
	      [aSMTP close];
	      DESTROY(aSMTP);
	      
	      [[NSApp delegate] performSelectorOnMainThread: @selector(authenticationFailed:)
				withObject: [NSDictionary dictionaryWithObjectsAndKeys: [allValues objectForKey: @"SMTP_HOST"], @"Server",
							  [allValues objectForKey: @"SMTP_USERNAME"], @"Username",
							  [NSNumber numberWithInt: OTHER], @"Type", nil]
				waitUntilDone: YES];
	    }
	  
	  ADD_CONSOLE_MESSAGE(_(@"SMTP - Authenticated!"))
	}
      
      if ( aSMTP )
	{
	  // We get the raw source of the message to sent.
	  if ( [[theTask message] respondsToSelector: @selector(isEqualToData:)] )
	    {
	      aData = [theTask message];
	    }
	  else
	    {
	      aData = [[theTask message] dataValue];
	    }

	  // We send the message
	  ADD_CONSOLE_MESSAGE(_(@"SMTP - Sending message..."))
	  messageWasSent = [aSMTP sendMessageFromRawSource: aData];
	}
    }
  NS_HANDLER
    {
      if ( [[localException name] isEqualToString: PantomimeStopException] )
	{
	  ADD_CONSOLE_MESSAGE(_(@"Manual stop of sending thread."))
	  manuallyStoppedTransfer = YES;
	}
      else
	{
	  // We must inform the user that an error occured.
	  SMTP_ERROR_OCCURED();
	}
    } 
  NS_ENDHANDLER {}


  // If we sucessfully sent the message, we add it to our "Sent" folder.
  if ( messageWasSent )
    {
      // We do not append the bounced message (as NSData) to the folder.
      if ( ![[theTask message] respondsToSelector: @selector(isEqualToData:)])
	{
	  FilterManager *aFilterManager;
	  URLName *theURLName;
	  
	  aFilterManager = (FilterManager *)[FilterManager singleInstance];
	  
	  // Needs locking.
	  theURLName = [aFilterManager matchedURLNameFromMessageAsRawSource: aData
				       type: TYPE_OUTGOING
				       key: [theTask key]
				       filter: nil];

	  [[MailboxManagerController singleInstance] performSelectorOnMainThread: @selector(addMessage:)
						     withObject: [NSDictionary dictionaryWithObjectsAndKeys: aData, @"Message",
									       theURLName, @"URLName", nil]
						     waitUntilDone: NO];
	}
      
      // The message has been sent and appended to the Sent folder. Let's remove the task from the queue.
      [[NSApp delegate] performSelectorOnMainThread: @selector(taskCompleted)
			withObject: nil
			waitUntilDone: NO];
      ADD_CONSOLE_MESSAGE(_(@"SMTP - Sent!"))
    }
  else 
    {
      // We don't tell the user that the message wasn't sent if the user himself stopped the
      // sending operaiton.
      if ( !manuallyStoppedTransfer )
	{
	  SMTP_ERROR_OCCURED();
	}
	  
      // An error occured. We leave the task in the queue.
      NSDebugLog(@"Leaving the task in the queue + 5 min.");
      [theTask setDate: [AUTORELEASE([[NSDate alloc] init])  addTimeInterval: 300]];
      [theTask setRunning: NO];
      [[NSApp delegate] performSelectorOnMainThread: @selector(consoleWindowNeedsDisplay)
			withObject: nil
			waitUntilDone: NO];
    }
  
  // We finally close our SMTP connection, no matter what happened.
  if ( aSMTP )
    {
      NS_DURING
	{
	  // Using SMTP, we must close the connection immediately if we are stopping. We can't
	  // tell the SMTP server that we are stopping, in any way.
	  SET_LOCK_VALUE(NO);
	  [[aSMTP tcpConnection] close];
	  RELEASE(aSMTP);
	}
      NS_HANDLER
	{
	  // FIXME - maybe warn the user about that...
	  NSDebugLog(@"Error occured while closing the connection to the server...");
	}
      NS_ENDHANDLER {}
    }
}


//
// This method is used to transfer message from a POP3 folder
// to a local folder.
//
// It verifies for various filters to be sure to transfer to the
// right folder.
//
- (int) _transferMessagesFromPOP3Folder: (POP3Folder *) theFolder
				   task: (Task *) theTask
{
  POP3CacheObject *aPOP3CacheObject;
  NSString *aUID;
  NSData *aData;
 
  int i, count, transferCount;

  if ( !theFolder )
    {
      return 0;
    }

  count = [theFolder count];
  transferCount = 0;

  // We get all messages..
  for (i = 1; i <= count; i++)
    {
      // We ask for the UIDL of the message
      aUID = [theFolder UIDOfMessageAtIndex: i];  

      // Next we verify if we really need to transfer the message
      aPOP3CacheObject = [[theFolder cacheManager] findPOP3CacheObject: aUID];

      // We do...
      if ( !aPOP3CacheObject )
	{
	  UPDATE_STATUS_LABEL(_(@"Downloading message %d of %d..."), i, count)

	  // The UID hasn't been found so we first cache it
	  aPOP3CacheObject = [[POP3CacheObject alloc] initWithUID: aUID
						      date: [NSCalendarDate calendarDate]];
	  [[theFolder cacheManager] addObject: aPOP3CacheObject];
	  RELEASE(aPOP3CacheObject);

	  aData = [theFolder prefetchMessageAtIndex: i];
	  
	  // If it's a new message (aData isn't nil when it's a new message)
	  if ( aData )
	    {
	      [self _matchFilterRuleFromRawSource: aData
		    task: theTask];
	      transferCount++;
	      
	      // We synchronize our POPCacheManager
	      [[theFolder cacheManager] synchronize];
	    }
	  else
	    {
	      // The transfer has failed. Let's remove the object for our cache
	      [[theFolder cacheManager] removeObject: aPOP3CacheObject];
	    }
	}
      else
	{
	  UPDATE_STATUS_LABEL(_(@"Ignoring old message %d of %d..."), i, count)
	}
    } // for 
  
  return transferCount;
}

@end
