#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# win32/widgets/_base.py
#
# DESCRIPTION:
# A PyWin32 based user interface driver for GNUe forms.
#
# NOTES:
#

from types import StringType, UnicodeType
import win32api, win32gui, win32con, win32ui, commctrl
import struct, array ,re

from gnue.common import events

from gnue.forms.GFForm import *
from gnue.forms.uidrivers._base.widgets._base import *
from gnue.forms.uidrivers.win32.common import *

try:
  import ImageWin
except:
  pass

#TCN_SELCHANGE = (commctrl.TCN_FIRST - 1)
#TCN_SELCHANGING = (commctrl.TCN_FIRST - 2)

class Win32Base:
  _connectTable = {}

  def __init__(self, uidriver, styleEx, className, windowTitle, style, x,y, width,height, parent,menu=0,instance=0):
    if parent == 0:
      hparent = 0
    else:
      hparent = parent.GetHwnd()

    self._id = menu
    self._className = className
    self._parent = parent
    self._children = []
    if parent:
      parent._children.append(self)

    self._hwnd = win32gui.CreateWindowEx(styleEx, className, windowTitle, style, x,y, width,height, hparent, menu, instance, None)
    
    # this is only for SetFont
    self._PyCWnd = win32ui.CreateWindowFromHandle(self._hwnd)
    
    self._uiDriver = uidriver
    self._uiDriver._win32app._HwndToTkObj[self._hwnd] = self

    if className in ['STATIC', 'BUTTON', 'COMBOBOX', 'EDIT']:
      self.Show() 
      self._oldWndProc = win32gui.SetWindowLong(self._hwnd, win32con.GWL_WNDPROC, self._wndproc)

  def _wndproc(self, hwnd, msg, wParam, lParam):
    if msg == win32con.WM_LBUTTONDOWN:
      gfObject = self._uiDriver._IdToGFObj[self._id]
      uiObject = self._uiDriver._IdToUIObj[self._id]
  
      if 1: #not self.hasFocus():
      # Request Focus
        uiObject._eventHandler('requestFOCUS',gfObject,_form=gfObject._form)
        count = uiObject.widgets.index(self)
        uiObject._eventHandler('requestJUMPRECORD',count - gfObject._visibleIndex,_form=gfObject._form)

      if gfObject._type == 'GFButton':
        self.SetFocus()
        self._connectTable[self._id](self._id)
      elif gfObject.style == 'checkbox':
        uiObject._eventHandler('requestTOGGLECHKBOX',_form=gfObject._form)
      else:
        win32gui.CallWindowProc(self._oldWndProc, hwnd, msg, wParam, lParam)
        cursorPosition = self.getSelectedArea()[1]
        uiObject._eventHandler('requestCURSORMOVE', position=cursorPosition, _form=gfObject._form)

    elif msg == win32con.WM_LBUTTONUP:
      gfObject = self._uiDriver._IdToGFObj[self._id]
      if gfObject._type == 'GFEntry':
        selection1, selection2 = self.getSelectedArea()
        if selection1 != selection2:
          uiObject = self._uiDriver._IdToUIObj[self._id]
          uiObject._eventHandler('requestSELECTWITHMOUSE', \
                                           position1=selection1, position2=selection2, _form=gfObject._form)
          
      win32gui.CallWindowProc(self._oldWndProc, hwnd, msg, wParam, lParam)

    elif msg == win32con.WM_KEYDOWN:
      keycode = wParam
      gfObject = self._uiDriver._IdToGFObj[self._id]
      if (keycode in NOT_WM_CHAR_KEYS):

        if gfObject._type == 'GFButton' or (gfObject._type == 'GFEntry' and gfObject.style != 'dropdown'):
          action = None

          ShiftDown = (win32api.GetKeyState(win32con.VK_SHIFT) & 0x8000) and 1 or 0
          ControlDown = (win32api.GetKeyState(win32con.VK_CONTROL) & 0x8000) and 1 or 0
          AltDown = (win32api.GetKeyState(win32con.VK_MENU) & 0x8000) and 1 or 0

          command = GFKeyMapper.KeyMapper.getEvent(
            keycode,
            ShiftDown,
            ControlDown,
            AltDown)

          if command:
            action = events.Event('request%s' % command)

            # Add the object's _form to the outgoing event
            # rather than every event in the function
            action.__dict__.update({'_form':gfObject._form})
            uiObject = self._uiDriver._IdToUIObj[self._id]
            uiObject._eventHandler(action)

        else:
          return win32gui.CallWindowProc(self._oldWndProc, hwnd, msg, wParam, lParam)

    elif msg == win32con.WM_CHAR:
      # The TranslateMessage function generates a WM_CHAR message 
      # when the user presses any of the following keys:
      # Any character key 
      # backspace 
      # enter (carriage return) 
      # esc 
      # shift+enter (linefeed) 
      # tab 

      gfObject = self._uiDriver._IdToGFObj[self._id]
      action = None
      
      keycode = wParam
      
      ShiftDown = (win32api.GetKeyState(win32con.VK_SHIFT) & 0x8000) and 1 or 0
      ControlDown = (win32api.GetKeyState(win32con.VK_CONTROL) & 0x8000) and 1 or 0
      AltDown = (win32api.GetKeyState(win32con.VK_MENU) & 0x8000) and 1 or 0

      #
      # Sigh... a hack for using <enter> in multiline entries
      #
      if  keycode == 13 and \
          not ShiftDown and \
          not ControlDown and \
          not AltDown and \
          int (gConfigForms('enterIsNewLine')) and \
          (hasattr(gfObject,'Char__height') and gfObject.Char__height) > 1:

        command = 'NEWLINE'

      else:

        #
        # Get the event to process from the KeyMapper
        #
        command = GFKeyMapper.KeyMapper.getEvent(
          keycode,
          ShiftDown,
          ControlDown,
          AltDown)
        
      if command == 'NEWLINE':
        win32gui.SendMessage(hwnd, win32con.EM_LINESCROLL, 1, 0)
        action = events.Event('requestKEYPRESS', '\r\n',
                     text='\r\n',
                     code=10)

      elif command and not keycode in NOT_WM_CHAR_KEYS:
          if keycode == 13 and gfObject._type == 'GFButton':
            self._connectTable[self._id](self._id)
          else:
            action = events.Event('request%s' % command)

      elif gfObject._type == 'GFButton':
        action = events.Event('buttonActivated',gfObject)

      elif gfObject.style == 'checkbox':
        # <space> <=> <click>
        if keycode == 32:
            action = events.Event('requestTOGGLECHKBOX')
        else:
          # maybe some background error message here
          pass
      else:
        try:
          char = chr(keycode)
          import string
          if char in string.printable or char == "\n" or \
            128 <= keycode <= 255:
            action = events.Event('requestKEYPRESS', textDecode(char),
                         text=textDecode(char),
                         code=keycode)
        except ValueError:
          pass 

      if action:
      # Add the object's _form to the outgoing event
      # rather than every event in the function
        action.__dict__.update({'_form':gfObject._form})
        uiObject = self._uiDriver._IdToUIObj[self._id]
        uiObject._eventHandler(action)

      if gfObject._type == 'GFEntry':
        if gfObject.style == 'dropdown':
          return win32gui.CallWindowProc(self._oldWndProc, hwnd, msg, wParam, lParam)
      
    elif msg == win32con.WM_COMMAND:

      if win32api.HIWORD(wParam) == win32con.CBN_SELCHANGE:
        gfObject = self._uiDriver._IdToGFObj[self._id]
        uiObject = self._uiDriver._IdToUIObj[self._id]
        if gfObject.style == 'dropdown':
          selection = self.GetValue()
          string = gfObject._field.allowedValues()[1][selection]
          uiObject._eventHandler('requestREPLACEVALUE',object=gfObject,
                              index=selection, text=string,
                            _form=gfObject._form)
        return win32gui.CallWindowProc(self._oldWndProc, hwnd, msg, wParam, lParam)

    elif msg == win32con.WM_PAINT:
      gfObject = self._uiDriver._IdToGFObj[self._id]
      if gfObject._type == 'GFImage':
        win32gui.CallWindowProc(self._oldWndProc, hwnd, msg, wParam, lParam)
        hdc = win32gui.GetDC(self._hwnd)
        self.dib.expose(hdc)
        win32gui.ReleaseDC(self._hwnd,hdc)
      else:
        return win32gui.CallWindowProc(self._oldWndProc, hwnd, msg, wParam, lParam)

    else:
      return win32gui.CallWindowProc(self._oldWndProc, hwnd, msg, wParam, lParam)

  def SetFont(self,font):
    self._PyCWnd.SetFont(font,1)

  def GetId(self):
    return self._id

  def GetHwnd(self):
    return self._hwnd

  def Show(self):
    win32gui.ShowWindow(self._hwnd, win32con.SW_SHOWNORMAL)

  def Hide(self):
    try:
      win32gui.ShowWindow(self._hwnd, win32con.SW_HIDE)
    except:
      pass

  def GetParent(self):
    return self._parent

  def GetChildren(self):
    return self._children

  def SetFocus(self):
    win32gui.SetFocus(self._hwnd)

  def Enable(self, enabled):
    if enabled:
      win32gui.EnableWindow(self._hwnd, 1)
    else:
      win32gui.EnableWindow(self._hwnd, 0)

  def Destroy(self):
    win32gui.DestroyWindow(self._hwnd)

  def Connect(self, id, func):
    self._connectTable[id] = func


class Win32Window(Win32Base):
  _descriptionTable = {}
  _statusBar = None

  def addDescription(self, id, description):
    self._descriptionTable[id] = description

  def OnWMMenuselect(self, hwnd, msg, wParam, lParam):
    id = win32api.LOWORD(wParam)
    if self._descriptionTable.has_key(id):
      msg = self._descriptionTable[id]
    else:
      msg = ''
    win32gui.SendMessage(self._statusBar.GetHwnd(), commctrl.SB_SETTEXT, 0, msg)

  def OnWMClose(self, hwnd, msg, wParam, lParam):
    self._uiDriver.dispatchEvent('requestEXIT', _form=self._uiDriver._form)

  def OnWMSize(self, hwnd, msg, wParam, lParam):
    for child in self._children:
      # resizing statusbar as needed
      if child._className == commctrl.STATUSCLASSNAME:
        (left, top, right, bottom) = win32gui.GetClientRect(hwnd)
        x = right - left
        statwidths = struct.pack("iiiii",x-75-75-50-50,x-75-75-50,x-75-75,x-75,x)
        win32gui.SendMessage(child.GetHwnd(), commctrl.SB_SETPARTS, 5, statwidths)
        win32gui.SendMessage(child.GetHwnd(), win32con.WM_SIZE, 0, 0)
      # and toolbar too
      elif child._className == commctrl.TOOLBARCLASSNAME:
        win32gui.SendMessage(child.GetHwnd(), commctrl.TB_AUTOSIZE, 0, 0)
    
  def OnWMCommand(self, hwnd, msg, wParam, lParam):
    # menu/toolbar selection happend
    self._connectTable[wParam]()

  def OnWMNotify(self, hwnd, msg, wParam, lParam):
    # handling tabbed pages
    format = "iiiiiiiiiii"
    buf = win32gui.PyMakeBuffer(struct.calcsize(format), lParam)
    hwndFrom, idFrom, code, iItem, iSubItem, uNewState, uOldState, uChanged, actionx, actiony, lparam \
              = struct.unpack(format, buf)
    if code == -552: #commctrl.TCN_SELCHANGING
      window = self._uiDriver._win32app._HwndToTkObj[hwndFrom]
      pageId = win32gui.SendMessage(window.GetHwnd(), commctrl.TCM_GETCURSEL, 0, 0)
      window._children[pageId].Hide()
    elif code == -551: #commctrl.TCN_SELCHANGE
      window = self._uiDriver._win32app._HwndToTkObj[hwndFrom]
      pageId = win32gui.SendMessage(window.GetHwnd(), commctrl.TCM_GETCURSEL, 0, 0)
      window._children[pageId].Show()

      id = window._children[pageId].GetId()
      gfObject = self._uiDriver._IdToGFObj[id]
      uiObject = self._uiDriver._IdToUIObj[id]
      uiObject._uiForm._eventHandler('requestPAGE', pageId, _form=gfObject._form)


class Win32Entry(Win32Base):
    
  def SetValue(self, value):
    try:
      gfObject = self._uiDriver._IdToGFObj[self._id]
      if gfObject.style == 'checkbox':
        if value:
          val = win32con.BST_CHECKED
        else:
          val = win32con.BST_UNCHECKED
        win32gui.SendMessage(self._hwnd, win32con.BM_SETCHECK, val, 0)
      elif gfObject.style == 'dropdown':
        win32gui.SendMessage(self._hwnd, win32con.CB_SELECTSTRING, -1, value) # CB_SETCURSEL, value, 0)
      else:
        if hasattr(gfObject,'Char__height') and gfObject.Char__height > 1:
          corrvalue = re.sub(r'(?<!\r)\n', r'\r\n', value)
          win32gui.SetWindowText(self._hwnd, corrvalue)
        else:
          win32gui.SetWindowText(self._hwnd, str(value))
    except:
      pass

  def GetValue(self):
    gfObject = self._uiDriver._IdToGFObj[self._id]
    if gfObject.style == 'checkbox':
      val = win32gui.SendMessage(self._hwnd, win32con.BM_GETCHECK, 0, 0) & win32con.BST_CHECKED
      if val:
        return 1
      else:
        return 0
    elif gfObject.style == 'dropdown':
      return win32gui.SendMessage(self._hwnd, win32con.CB_GETCURSEL, 0, 0)
    else:
      return win32gui.GetWindowText(self._hwnd)

  def setCursorPosition(self, position):
    win32gui.SendMessage(self._hwnd, win32con.EM_SETSEL, position, position)

  def setSelectedArea(self, selection1, selection2):
    win32gui.SendMessage(self._hwnd, win32con.EM_SETSEL, selection1, selection2)

  def getSelectedArea(self):
    gs = win32gui.SendMessage(self._hwnd, win32con.EM_GETSEL, 0, 0)
    return (win32api.LOWORD(gs), win32api.HIWORD(gs))

class Win32Button(Win32Base):
  pass
#  def OnWMCommand(self, hwnd, msg, wParam, lParam):
#    print 'OnWMCommand in Win32Button', hwnd, msg, wParam, lParam
#    print self._connectTable[wParam]
#    self._connectTable[wParam](wParam)


class Win32Page(Win32Base):

  def OnWMCommand(self, hwnd, msg, wParam, lParam):
    window = self._uiDriver._win32app._HwndToTkObj[lParam]
    window.OnWMCommand(hwnd, msg, wParam, lParam)

  def OnWMVScroll(self, hwnd, msg, wParam, lParam):
    sbWindow = self._uiDriver._win32app._HwndToTkObj[lParam]
    id = sbWindow.GetId()
    gfObject = self._uiDriver._IdToGFObj[id]
    uiObject = self._uiDriver._IdToUIObj[id]
    scrollCode = win32api.LOWORD(wParam)

    if scrollCode == win32con.SB_THUMBTRACK:
      format = "IIiiIii"
      size = struct.calcsize(format)
      scrollinfo = struct.pack(format, size, win32con.SIF_ALL, 0, 0, 0, 0, 0)
      win32gui.SendMessage(lParam, win32con.SBM_GETSCROLLINFO, 0, scrollinfo)
      size, mask, min, max, page, pos, trackpos = struct.unpack(format, scrollinfo)
      recno = int(trackpos)+1
      action = events.Event('requestRECORDNUMBER',object=gfObject,
                            data=recno,
                          _form=gfObject._form)
    elif scrollCode == win32con.SB_PAGEDOWN:
      action = events.Event('requestJUMPROWSDOWN',object=gfObject, _form=gfObject._form)
    elif scrollCode == win32con.SB_PAGEUP:
      action = events.Event('requestJUMPROWSUP',object=gfObject, _form=gfObject._form)
    elif scrollCode == win32con.SB_LINEDOWN:
      action = events.Event('requestNEXTRECORD',object=gfObject, _form=gfObject._form)
    elif scrollCode == win32con.SB_LINEUP:
      action = events.Event('requestPREVRECORD',object=gfObject, _form=gfObject._form)
    elif scrollCode == win32con.SB_BOTTOM:
      action = events.Event('requestLASTRECORD',object=gfObject, _form=gfObject._form)
    elif scrollCode == win32con.SB_TOP:
      action = events.Event('requestFIRSTRECORD',object=gfObject, _form=gfObject._form)

    uiObject._uiForm._eventHandler(action)

class Win32Image(Win32Base):
    
  def SetValue(self, value):
    width, height = value.size
    win32gui.SetWindowPos(self._hwnd, 0, 0, 0, 0, 0, win32con.SWP_NOMOVE | win32con.SWP_NOZORDER)
    if width>1:
      win32gui.SetWindowPos(self._hwnd, 0, 0, 0, width, height, win32con.SWP_NOMOVE | win32con.SWP_NOZORDER)
      self.dib = ImageWin.Dib(value)


#
# UIHelper
#
# Provides a UI widget set of std functions to reduce the number of functions
# require in each UIclass listed below.
#
class UIHelper(UIWidget):
  def _addToCrossRef(self, widget, gfobject, uiobject):
    id = widget.GetId()
    self._uiDriver._IdToGFObj[id]=gfobject
    self._uiDriver._IdToUIObj[id]=uiobject

  def _deleteFromCrossRef(self, widget, object):
    id = widget.GetId()
    try:
      del self._uiDriver._IdToGFObj[id]
      del self._uiDriver._IdToUIObj[id]
    except:
      pass

  #
  # Override the base UIWidget functions
  #
  def show(self):
    for widget in self.widgets:
      widget.Show()

  def hide(self):
    for widget in self.widgets:
      widget.Hide()

          
  def showModal(self):
    for widget in self.widgets:
      widget.Show()

  def destroy(self):
    for widget in self.widgets:
      self.Destroy()


  def indexedFocus(self, index):
    self.widgets[index].SetFocus()

  def setValue(self, value, index=0, enabled=1):
    widget = self.widgets[index]

    # Check if foreign key changed
    gfObject = self._uiDriver._IdToGFObj[widget.GetId()]
    try:
      if gfObject.style == "dropdown" and \
          not gfObject._field._allowedValues == widget._origAllowedValues:
        widget._origAllowedValues = gfObject._field._allowedValues
        win32gui.SendMessage(widget.GetHwnd(), win32con.CB_RESETCONTENT, 0, 0)
        for value in gfObject._field.allowedValues()[1]:
          win32gui.SendMessage(widget.GetHwnd(), win32con.CB_ADDSTRING, 0, textEncode(value.encode))
        widget.SetValue("")
    except AttributeError:
      pass

    # display unicode chars properly
    value=textEncode(value) #.encode('mbcs') ?
      
    widget.SetValue(value)
    widget.Enable(enabled)


  def setCursorPosition(self, position, index=0):
    self.widgets[index].setCursorPosition(position)

  def setSelectedArea(self, selection1, selection2, index=0):
    self.widgets[index].setSelectedArea(selection1, selection2)

  def createWidget(self, event, spacer):
    newWidget = self._createWidget(event, spacer)
    newWidget.SetFont(self._uiDriver._font)
    self._addToCrossRef(newWidget,event.object,self)
    return newWidget

  def cleanup(self, object):
    for widget in self.widgets[:]:
      id = widget.GetId()
      del self._uiDriver._IdToGFObj[id]
      del self._uiDriver._IdToUIObj[id]
      self.widgets.pop(0)
      self._deleteFromCrossRef(widget, object)


#####################################################################
##
## Keymapper Support
##
#####################################################################
from gnue.forms import GFKeyMapper
from gnue.forms.GFKeyMapper import vk

# Translate from Win32 keystrokes to our virtual keystrokes
win32KeyTranslations = {
    vk.C     : 3,        vk.V        : 22,
    vk.X     : 24,        vk.A        : 1,
    vk.F1     : win32con.VK_F1,        vk.F2        : win32con.VK_F2,
    vk.F3     : win32con.VK_F3,        vk.F4        : win32con.VK_F4,
    vk.F5     : win32con.VK_F5,        vk.F6        : win32con.VK_F6,
    vk.F7     : win32con.VK_F7,        vk.F8        : win32con.VK_F8,
    vk.F9     : win32con.VK_F9,        vk.F10       : win32con.VK_F10,
    vk.F11    : win32con.VK_F11,       vk.F12       : win32con.VK_F12,
    vk.INSERT : win32con.VK_INSERT,    vk.DELETE    : win32con.VK_DELETE,
    vk.HOME   : win32con.VK_HOME,      vk.END       : win32con.VK_END,
    vk.PAGEUP : win32con.VK_PRIOR,     vk.PAGEDOWN  : win32con.VK_NEXT,
    vk.UP     : win32con.VK_UP,        vk.DOWN      : win32con.VK_DOWN,
    vk.LEFT   : win32con.VK_LEFT,      vk.RIGHT     : win32con.VK_RIGHT,
    vk.TAB    : win32con.VK_TAB,
    vk.ENTER  : win32con.VK_RETURN,    vk.BACKSPACE : win32con.VK_BACK }

GFKeyMapper.KeyMapper.setUIKeyMap(win32KeyTranslations)

NOT_WM_CHAR_KEYS =[
  win32con.VK_HOME,
  win32con.VK_END,
  win32con.VK_UP,
  win32con.VK_DOWN,
  win32con.VK_RIGHT,
  win32con.VK_LEFT,
  win32con.VK_INSERT,
  win32con.VK_DELETE,
  win32con.VK_PRIOR,
  win32con.VK_NEXT,
  win32con.VK_F1,
  win32con.VK_F2,
  win32con.VK_F3,
  win32con.VK_F4,
  win32con.VK_F5,
  win32con.VK_F6,
  win32con.VK_F7,
  win32con.VK_F8,
  win32con.VK_F9,
  win32con.VK_F10,
  win32con.VK_F11,
  win32con.VK_F12]
