#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# UIbase.py
#
# DESCRIPTION:
# A base class for UI modules.  This class is to be inherited by
# specific UI drivers.
#
# NOTES:
#
import sys, os, dircache

from gnue.common import events
from gnue.common.definitions.GObjects import *
from gnue.common.definitions.GRootObj import GRootObj
from gnue.common.utils.FileUtils import dyn_import

from gnue.forms.GFForm import *

#
# GFUserInterfaceBase
#
# Convience class to be inherited by other GFUserInterface's
# Provides basic functionality.
#
class GFUserInterfaceBase(GRootObj,events.EventAware):
  def __init__(self, eventHandler, name="Undefined",
               disableSplash=None, parentContainer=None):

    ############################################################
    #
    # Things you do not adjust in the individual UI drivers
    #
    ############################################################
    GRootObj.__init__(self, 'uiDriver', None, None)

    self._type = 'UIDriver'
    self.name = name

    self._disableSplash = disableSplash


    # Used when forms are embedded in navigator.
    # What parentContainer is depends on the
    # individual UIdriver and Navigator.
    self._parentContainer = parentContainer

    events.EventAware.__init__(self, eventHandler)

    #
    # Incomming events
    #
    self.registerEventListeners( {
            'gotoENTRY'        : self.switchFocus,
            'updateENTRY'      : self.updateEntry,
            'updateEntryEditor': self.updateEntryEditor,
            'gotoPAGE'         : self.gotoPage,
            'exitApplication'  : self.exitApplication,

            # User feedback events from the virtual form to the UI
            'beginWAIT'        : self.beginWait,
            'endWAIT'          : self.endWait,
            'formALERT'        : self.formAlert,
            'uiUPDATESTATUS'   : self.updateStatusBar,

            # Clipboard contents
            'setCLIPBOARD'     : self.setClipboardContents,
            'getCLIPBOARD'     : self.getClipboardContents,

            # Printout
            'uiPRINTOUT'       : self.printout,
          })

    #
    # Mapping - internal record status : displayed on status bar
    #
    self.statusValues = {'saved'     :'OK',
                         'modified'  :'MOD',
                         'deleted'   :'DEL',
                         'query'     :'QRY',
                         }

    #
    # Multi-form support
    #
    self._formNameToUIForm = {}             # Holds links the to the top level
                                            # UIForm objects held in memory
    self._formNameToGFForm = {}             # Holds links the to the top level
                                            # UIForm objects held in memory
    self._gfObjToUIWidget = {}              # dictionary of driver specific UIfoo
                                            # widgets with the GFObj used as the key

    self._gfObjToToolkitWidgets = {}        # the GFObj to UI widget cross ref


    #
    # Our local "clipboard"
    #
    self.__clipboard = None

    #
    # Import and register supported widgets in UI driver
    #
    self._supportedWidgets = {}

    basedir  = os.path.dirname(sys.modules[self.__module__].__file__)
    uiDriver = os.path.basename(basedir)
    basedir +='/widgets/'
    for widgetName in dircache.listdir(basedir):
      try:
        if widgetName[0] != '_':
          if os.path.isfile(basedir+widgetName) and os.path.splitext(widgetName)[1] == ".py":
            widgetName = os.path.splitext(widgetName)[0]
            widget = dyn_import('gnue.forms.uidrivers.%s.widgets.%s' %(uiDriver,widgetName))
          elif os.path.isdir(basedir+widgetName):
            widget = dyn_import('gnue.forms.uidrivers.%s.widgets.%s' %(uiDriver,widgetName))
          else:
            raise ImportError, "How the fsck did I get here?"

          try:
            self._supportedWidgets[widget.configuration['provides']]=widget
          except Exception,mesg:
            raise ImportError, mesg

      except ImportError, mesg:
        GDebug.printMesg(2,"%s.widgets.%s doesn't appear to be a valid ui widget" % (uiDriver,widgetName))
        GDebug.printMesg(5,' --> %s' % (mesg))

    ############################################################
    #
    # Things you may need to  adjust in the individual UI drivers
    #
    ############################################################

    #
    # Dialogs required by base
    #
    # TODO: Once <dialog> works we may be able to
    # do away with these
    #
    self._DIALOGS = {}


  def _buildUI(self,object, formName):
    try:
      #
      # Look up the parent GObj of the current obj
      #
      if object._type == "GFPage":
        parent = object.findParentOfType(None)
      else:
        parent = object.getParent()

      # find the ui widget that corrosponds with that parent
      uiParent = self
      if parent:
        try:
          uiParent = self._gfObjToUIWidget[parent]
        except KeyError:
          pass

      event = self._updateCreateEvent(events.Event('CreateUIWidget',None,
                   object=object,
                   parent=uiParent,
                   #container=self.currentWidget[0],
                   textWidth=self.textWidth,
                   textHeight=self.textHeight,
                   widgetWidth=self.widgetWidth,
                   widgetHeight=self.widgetHeight,
                   interface=self,
                   initialize=1))

      uiWidget = self._supportedWidgets[object._type].configuration['baseClass'](event)
      uiWidget._form = self._form

      #
      # Add to the cross refernce
      #
      self._gfObjToUIWidget[object]=uiWidget
      GDebug.printMesg(0, "OBJECT%s "% object)
      GDebug.printMesg(0, "WIDGET %s " % uiWidget)

      #
      # If the current object is a GFForm then add it to the
      # dictionary.
      #

      if object._type == 'GFForm':
        self._formNameToUIForm[formName] = uiWidget
        self._formNameToGFForm[formName] = object

      #
      # GFObject to list of widget set specific widgets
      # associated with it
      #
      # Note: The list would be empty on hidden fields
      if not uiWidget.widgets == []:
        self._gfObjToToolkitWidgets[object] = uiWidget.widgets

      GDebug.printMesg(2, "Widget is %s" % uiWidget.widgets)

    except KeyError:
      pass


  #############################################################################
  #
  # Public Interface
  #
  # The interface exposed to the forms backend
  #
  #

  #
  # buildForm
  #
  # when passed a GObj tree constructed from a .gfd file
  # it handles the creatation of the user interface
  #
  def buildForm(self, form, formName):

    self._form = form

    # Create the UI from the GFForm passed in
    form.walk(self._buildUI, formName=formName)

    self._gfObjToUIWidget[form].phaseInit()

    self._formNameToUIForm[formName]._gfObjToToolkitWidgets = self._gfObjToToolkitWidgets
    self._formNameToUIForm[formName]._gfObjToUIWidget = self._gfObjToUIWidget
    self._formNameToUIForm[formName]._form = form


  #
  # activateForm
  #

  def activateForm(self,formName, modal=0):

    self._form = self._formNameToGFForm[formName]
    self._activateForm(self._form, modal)


  #############################################################################
  #
  # EVENT FUNCTIONS
  #
  # Handles incoming events and calls UI instance specific functions to
  # execute the actions.  These events come from the forms back end.
  #

  #
  # switchFocus
  #
  # Moves the focus to a specific UI widget
  #
  def switchFocus(self, event):
    object = event.object
    if object: # Some pages might not have any widgets that can be active
      self._gfObjToUIWidget[object].indexedFocus(object._visibleIndex)

    self.dispatchEvent('beginEDITMODE', object, _form=object._form)

  #
  # updateEntry
  #
  # Updates all visible toolkit widgets
  # tied to a specific GFObject
  #
  def updateEntry(self,event):
    if event.data.hidden or event.data._type == 'GFButton':
      return

    entry = event.data
    field = entry._field
    handler = entry._displayHandler
    prehandling = handler.editing
    handler.editing = 0
    index = entry._visibleIndex
    block = entry._block
    currentRecord = block._resultSet.getRecordNumber()

    # Fill the prior spots
    for count in range(index):
      value = handler.getDisplayFiller(block._resultSet.getRecord(currentRecord- \
                              (index-count)).getField(field.field))

      self._gfObjToUIWidget[entry].setValue(value,count)

    # Fill current spot
    value = handler.getDisplayFiller(entry.getValue())
    self._gfObjToUIWidget[entry].setValue(value, index)

    # Fill trailing spots
    #
    # You must skip the matching index but
    # you do not want to just add 1 to count
    # as the formulas would then be off
    count = index

    lastRow = block._resultSet.getRecordCount()
    while count < int(entry._rows):
      if count != index:
        cr = currentRecord+(count-index)
        rec = block._resultSet.getRecord(cr)

        if rec == None:
          # Blank the displayed widget
          value = handler.getDisplayFiller(None)
          # Don't ask... but it's needed
          lastRow -= 1
        else:
          value = handler.getDisplayFiller(rec.getField(field.field))

        self._gfObjToUIWidget[entry].setValue(value, count, cr <= lastRow)
      count += 1

    handler.editing = prehandling

  #
  # updateEntryEditor
  #
  # updates the displayed value and sets
  # the cursor position for a specific GFObject
  #
  def updateEntryEditor(self, event):
    index = event.object._visibleIndex
    widget = self._gfObjToUIWidget[event.object]
    widget.setValue(event.display, index)
    widget.setCursorPosition(event.cursor, index)

    if event.selection != None:
      selection1, selection2 = event.selection
      widget.setSelectedArea(selection1, selection2, index)

  #
  # Clipboard routines
  #
  # If a particular UI has a system-wide clipboard,
  # these methods should be overridden to use that
  # clipboard.
  #
  def getClipboardContents(self, event):
    GDebug.printMesg(5, "Getting clipboard '%s'" % self.__clipboard)
    event.__result__ = "%s" % self.__clipboard

  def setClipboardContents(self, event):
    GDebug.printMesg(5, "Setting clipboard '%s'" % event.text)
    self.__clipboard = "%s" % event.text


  #############################################################################
  #
  # Optional Functions
  #
  # UIDrivers can override the following functions
  #

  #
  # beginWait
  # Called whenever forms goes into a "wait" state in which user cannot
  # interact with interface (e.g., while waiting for a query or a commit)
  #
  def beginWait (self, event):
    pass

  #
  # endWait
  #
  # Called whenever forms leaves a "wait" state
  #
  def endWait (self, event):
    pass

  # exitApplication
  #
  # Form has told the application to close so call the UIs private exit routine
  #
  def exitApplication(self,event):
    self._exit(event._formName)

  #
  # _updateCreateEvent
  #
  # Can be used by UI drivers to add more attributes to the event that
  # creates a widget.  Called by the _stdConstructor during the building of the UI
  #
  def _updateCreateEvent(self, event):
    return event

  #
  # printout
  #
  # Perform the default printout/"screen print" function for
  # this uidriver
  #
  def printout(self, event):
    pass

  #############################################################################
  #
  # Required Functions
  #
  # UIDrivers must implement the following features

  def _activateForm(self,modal=0):
    GDebug.printMesg(0, "Fatal UI driver error.  Required function not implemented")
    sys.exit(1)

  def _exit(self):
    GDebug.printMesg(0, "Fatal UI driver error.  Required function not implemented")
    sys.exit(1)


#
# newSkool
#

  #
  # gotoPage
  #
  def gotoPage(self,event):
    self._gfObjToUIWidget[event.data]._uiForm.gotoPage(event)


  #
  # updateStatusBar
  #
  # Calls the UIForm's _setStatusBar function to update status
  #
  def updateStatusBar(self,event):
    status = None
    if event.recordStatus != None:
      status = self.statusValues[event.recordStatus]

    insertValue = None
    if event.insertMode:
      insertValue = event.insertMode and 'INS' or  'OVR'
    # TODO:
    self._gfObjToUIWidget[event._form]._uiForm._setStatusBar(event.tip, status, insertValue,
                                                            event.currentRecord, event.maxRecord,
                                                            event.currentPage, event.maxPage)

