# #
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# GFBlock.py
#
# DESCRIPTION:
"""
Classes making up the Block object
"""
# NOTES:
#
# HISTORY:
#

from gnue.forms.GFObjects.GFDataSource import GFDataSource

from gnue.common.apps import GDebug
from gnue.common.datasources import GConditions
from GFContainer import GFContainer
from gnue.common import events

import string

# These should really go somewhere else
TRUE = 1
FALSE = 0


############################################################
# GFBlock
#
#
class GFBlock(GFContainer, events.EventAware):
  def __init__(self, parent=None):
    GFContainer.__init__(self, parent, 'GFBlock')

    self.mode = 'normal'
    self._convertAstericksToPercent = gConfigForms('AsterickWildcard')

    self._inits = [self.initialize]

    #self._datasource = None
    self._resultSet = None
    self._dataSourceLink = None

    self._currentRecord = 0
    self._recordCount = 0
    self._queryDefaults = {}
    self._queryValues = {}
    self._lastQueryValues = {}
    self._autocreate =  int(gConfigForms('autocreate'))
    self._gap = 0
    self._rows = 1

    self._scrollbars= []

    # Populated by GFEntry's initialize
    self._entryList = []

    # Populated by GFField's initialize
    self._fieldMap = {}
    self._fieldList = []

    #
    # Trigger exposure
    #
    self._validTriggers = {
##                  'ON-SWITCH':      'On-Switch',    ## Deprecated
                  'ON-NEWRECORD':   'On-NewRecord',
                  'PRE-COMMIT':     'Pre-Commit',
                  'POST-COMMIT':    'Post-Commit',
                  'PRE-QUERY':      'Pre-Query',
                  'POST-QUERY':     'Post-Query',
                  'PRE-MODIFY':     'Pre-Modify',
                  'PRE-INSERT':     'Pre-Insert',
                  'PRE-DELETE':     'Pre-Delete',
                  'PRE-UPDATE':     'Pre-Update',
                  'PRE-FOCUSOUT':   'Pre-FocusOut',
                  'POST-FOCUSOUT':  'Post-FocusOut',
                  'PRE-FOCUSIN':    'Pre-FocusIn',
                  'POST-FOCUSIN':   'Post-FocusIn',
                  'PRE-CHANGE':     'Pre-Change',
                  'POST-CHANGE':    'Post-Change',
                  }


    self._triggerGlobal = 1
    self._triggerFunctions={'clear':{'function':self.processClear,
                                     'description':''},
                            'commit':{'function':self.commit,
                                      'description':''},
                            'deleteRecord':{'function':self.deleteRecord,
                                            'description':''},
                            'gotoRecord':{'function':self.jumpRecord,
                                          'description':''},
                            'firstRecord':{'function':self.firstRecord,
                                           'description':'Navigates the block to the first record it contains.'},
                            'isEmpty':{'function':self.isEmpty,
                                       'description':'Returns True if block is empty.'},
                            'isSaved':{'function':self.isSaved,
                                       'description':'Returns True if block contains no modified records.'},
                            'lastRecord':{'function':self.lastRecord,
                                      'description':'Navigates the block to the last record it contains.'},
                            'newRecord':{'function':self.newRecord,
                                         'description':''},
                            'nextRecord':{'function':self.nextRecord,
                                          'description':'Navigates the block to the next record in sequence.'},
                            'prevRecord':{'function':self.prevRecord,
                                          'description':'Navigates the block to the previous record in sequence.'},
                            'jumpRecords':{'function':self.jumpRecords,
                                          'description':'Navigates the specified number of records.'},
                            'rollback':{'function':self.processRollback,
                                        'description':'Clears all records regardless of state from the block'},
                            'initQuery':{'function':self.initQuery,
                                        'description':'Prepares the block for query input.'},
                            'copyQuery':{'function':self.copyQuery,
                                        'description':'Prepares the block for query input.'},
                            'cancelQuery':{'function':self.cancelQuery,
                                        'description':'Cancels query input.'},
                            'executeQuery':{'function':self.processQuery,
                                            'description':'Executes the current query.'},
                            }

    self._triggerProperties={'parent':  {'get':self.getParent}}

     
  # Iterator support (Python 2.2+)
  def __iter__(self):
    """
    Iterator support for Python 2.2+
    
    Allows you to do:
      for foo in myBlock:
        ....
    """

    return _BlockIter(self)


  def _buildObject(self):

    if hasattr(self, 'rows'):
      self._rows = self.rows

    if hasattr(self, 'rowSpacer'):
      self._gap = self.rowSpacer

    if hasattr(self,'datasource'):
      self.datasource = string.lower(self.datasource)

    return GFContainer._buildObject(self)

  #
  # Primary initialization
  #
  def initialize(self):
    self._form = form = self.findParentOfType('GFForm')
    self._logic = logic = self.findParentOfType('GFLogic')

    self._lastValues = {}

    logic._blockList.append(self)
    logic._blockMap[self.name] = self

    # Initialize our events system
    events.EventAware.__init__(self, form._instance.eventController)

    # Get all focusable items, ordered correctly
    self._focusOrder = self.getFocusOrder()

    # Create a stub/non-bound datasource if we aren't bound to one
    if not hasattr(self,'datasource') or not self.datasource:
      ds = GFDataSource(self)
      self.datasource = ds.name = "__dts_%s" % id(self)
      form._datasourceDictionary[ds.name] = ds
      ds._buildObject()
      ds.phaseInit()

    self._dataSourceLink = form._datasourceDictionary[self.datasource]

    # We will monitor our own resultSet changes
    self._dataSourceLink.registerResultSetListener(self._loadResultSet)

    self.walk(self.__setChildRowSettings)

  def __setChildRowSettings(self, object):
    if hasattr(object,'rows'):
      rows = object._rows = object.rows
    else:
      rows = object._rows = self._rows

    if hasattr(object,'rowSpacer'):
      object._gap = object.rowSpacer
    else:
      object._gap = self._gap


  #
  # isSaved
  #
  # returns a positive value if the datasource the block is associated
  # with is saved
  #
  def isSaved(self):
    return self._resultSet == None or not self._resultSet.isPending()

  #
  # deleteRecord
  #
  # Doesn't really delete the record but marks it for
  # deletion during next commit
  #
  def deleteRecord(self):
    self._resultSet.current.delete()

  #
  # Load and process a resultset
  #
  def _loadResultSet(self, resultSet):
    self._resultSet = resultSet
    self._currentRecord = -1
    self._recordCount = 0
    resultSet._block = self

    if self._resultSet.firstRecord():
      self.switchRecord(0)
      self._recordCount = self._resultSet.getRecordCount()
    else:
      # TODO: This means no results were returned from a query.
      # TODO: It probably shouldn't create a new record :)  Display a message?
      self.newRecord()

  #
  # isEmpty()
  #
  def isEmpty(self):
    return self._resultSet.current.isEmpty()

  #
  #
  # Moves the proper record into editing position
  #
  def switchRecord(self, adjustment):

    self._currentRecord = self._resultSet.getRecordNumber()
    for field in self._fieldList:
      for entry in field._entryList:
        # This loop is probably better somewhere else
        entry.recalculateVisible( adjustment, self._currentRecord, self._recordCount)
      self._form.updateUIEntry(field)
      self._form.refreshUIEvents()

    # Adjusting scrollbars
    for sb in self._scrollbars:
      sb.adjustScrollbar(self._currentRecord, self._recordCount)

  #
  # newRecord
  #
  # Adds a record to the current records in memory
  #
  def newRecord(self):

    # Focus out
    self.processTrigger('PRE-FOCUSOUT')
    self.processTrigger('POST-FOCUSOUT')

    if self._resultSet.insertRecord():

      # Set the defaultToLast fields
      for field, value in self._lastValues.items():
        if value is not None:
          self._resultSet.current.setField(field, value, 0)

      self._recordCount = self._resultSet.getRecordCount()
      self.switchRecord(1)

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('ON-NEWRECORD')
      self.processTrigger('POST-FOCUSIN')

  def nextRecord(self):
    if not self._resultSet.isLastRecord():

      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      # TODO: I'm not 100% sure of the definition of
      # TODO: Post-FocusOut  In this context!
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.nextRecord()
      self._recordCount = self._resultSet.getRecordCount()
      self.switchRecord(1)

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

    elif self._autocreate and not self.isEmpty() and not self.restrictInsert:
      self.newRecord()

  def lastRecord(self):
    if not self._resultSet.isLastRecord():
      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.lastRecord()
      self.switchRecord(0)

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

  def firstRecord(self):
    if not self._resultSet.isFirstRecord():
      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.firstRecord()
      self.switchRecord(0)

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

  def prevRecord(self):
    if not self._resultSet.isFirstRecord():

      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.prevRecord()
      self.switchRecord(-1)

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

  def commit(self):
    #Commented out (dimas)
    #Does not work properly anyway
    #self._form.changeFocus(self)
    #self._form.commit()
    
    #TODO: Add error handling
    #TODO: Check how triggers performed
    #TODO: original code is in GFForm.commit
    self._precommitRecord = self._currentRecord

    self.processCommit()

  def jumpRecord(self, recordNumber):
    # If recordNumber is negative, move relative to last record
    if recordNumber < 0:
      recordNumber += self._resultSet.getRecordCount()
    if recordNumber < 0:
      raise "Invalid record number"
    
    if recordNumber != self._resultSet.getRecordNumber():
      # Focus out
      self.processTrigger('PRE-FOCUSOUT')
      self.processTrigger('POST-FOCUSOUT')

      if not self._resultSet.setRecord(recordNumber):
        self._resultSet.lastRecord()

      jump = self._resultSet.getRecordNumber() - self._currentRecord
      self._currentRecord = self._resultSet.getRecordNumber()
      self.switchRecord(jump)

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

      
  def jumpRecords(self, adjustment):
    targetRecord = self._resultSet.getRecordNumber() + adjustment
    
    if targetRecord < 0:
      targetRecord = 0
    elif targetRecord > self._resultSet.getRecordCount():
      targetRecord = self._resultSet.getRecordCount()

    self.jumpRecord(targetRecord)
    self._form._instance.updateRecordCounter(self._form) 
  
  #
  # processCommit
  #
  def processCommit(self):
    GDebug.printMesg(1, "processing commit on block %s"%self.name,1)

    self.mode='commit'
    
    self._resultSet.setRecord(self._precommitRecord)

    # Backstep thru the record adjusting for any prior records that
    # will be deleted.  Keeps the record postition properly adjusted
    # during deletes.
    for record in self._resultSet._cachedRecords:
      if record.isDeleted():
        self._resultSet.prevRecord()
        self._precommitRecord -= 1
      if record == self._resultSet.current:
        break

    if not self._dataSourceLink.hasMaster():
      self._resultSet.post()
      self._dataSourceLink._dataObject.commit()

    self._recordCount = self._resultSet.getRecordCount()

    # Make sure the block is aware of the current
    # position in the records.  If this is removed
    # ui corruption can occur in zipcode.gfd if
    # you commit while in or after a record that
    # is to be deleted.

    self.jumpRecord(self._precommitRecord)

    # If all our records were deleted, create an empty record
    if not self._recordCount:
      self.newRecord()
    else:
      self.jumpRecord(self._resultSet.getRecordNumber())

    self.mode='normal'


  #
  # processClear
  #
  def processClear(self):
    self._dataSourceLink._dataObject.rollback()
    if not self._dataSourceLink.hasMaster():
      self._dataSourceLink.createEmptyResultSet()
    else:
      self._dataSourceLink.createEmptyResultSet(masterRecordSet=self._resultSet._masterRecordSet)
    self._currentRecord = 0
    self._recordCount = 0
    self.switchRecord(0)


  #
  # processRollback
  #
  # if recovering=False, then the user requested a rollback
  # if recovering=True, then a commit or such failed and we need to clean up
  # (but not lose state information)
  #
  def processRollback(self, recovering=False):
    if not recovering:
      self._currentRecord = 0
      self._recordCount = 0
      self._dataSourceLink.createEmptyResultSet()
    self._dataSourceLink._dataObject.rollback()
    self.switchRecord(0)

  #
  # initQuery and processQuery
  #
  def initQuery(self):

    # If Enter-Query is hit once, enter query mode
    # If Enter-Query is hit twice, bring back conditions from last query.
    # If Enter-Query is hit thrice, cancel the query and go into normal mode.

    if self.mode != 'query':
        self.mode = 'query'
        self._query2 = int(gConfigForms("RememberLastQuery"))
        self._queryValues = {}
        self._queryValues.update(self._queryDefaults)
        self.switchRecord(0)

  def copyQuery(self): 
    self._query2 = 0
    self._queryValues = {}
    self._queryValues.update(self._lastQueryValues)
    self.switchRecord(0)
  
  def cancelQuery(self):
    self.mode = 'normal'
    self.switchRecord(0)

  def processQuery(self):
    # Set the maxList to a single master block
    # as a placeholder
    maxList = [self._dataSourceLink._dataObject]
    while maxList[0]._masterObject:
      maxList = [maxList[0]._masterObject]

    # Find the longest master/detail chain that
    # contains query values.  This will become
    # the chain that is queried
    for block in self._logic._blockList:
      if block._queryValues.keys():
        templist = [block._dataSourceLink._dataObject]
        while templist[-1]._masterObject:
          templist.append(templist[-1]._masterObject)
        if len(maxList) < len(templist): maxList = templist

    # Store block states
    for block in self._logic._blockList:
      block.mode = 'normal'
      block._lastQueryValues = {}
      block._lastQueryValues.update(block._queryValues)

    # graft in the sloppy query stuff if needed
    for dataobject in maxList:
      for block in self._logic._blockList:
        if dataobject == block._dataSourceLink._dataObject:
          break
      for entry in block._entryList:
        if hasattr(entry._field,'sloppyQuery') and block._queryValues.has_key(entry._field):
          block._queryValues[entry._field] = "%"+ string.join(list(block._queryValues[entry._field]),"%")+"%"

    # Build the sql required by the detail blocks
    SQL = ""
    for dataobject in maxList[:-1]:

      for block in self._logic._blockList:
        if dataobject == block._dataSourceLink._dataObject:
          break

      block.processTrigger('PRE-QUERY')
      for field in block._fieldList:
        field.processTrigger('PRE-QUERY')

      conditions = _generateConditional(block)
      SQL = self._dataSourceLink.getQueryString(conditions,1,SQL)

    for block in self._logic._blockList:
     if maxList[-1] == block._dataSourceLink._dataObject:
      break
    rootBlock = block

    conditions = _generateConditional(rootBlock)
    rootBlock._dataSourceLink.createResultSet(conditions, sql=SQL)

    rootBlock._recordCount = rootBlock._resultSet.getRecordCount()

    for block in self._logic._blockList:
      block.processTrigger('POST-QUERY')
      for field in block._fieldList:
        field.processTrigger('POST-QUERY')

    # Adjusting scrollbars
    for sb in self._scrollbars:
      sb.adjustScrollbar(self._currentRecord, self._recordCount)


  def registerScrollbar(self, sb):
    self._scrollbars.append(sb)

#
# _generateConditional
#
# Creates a GCondtional tree based upon the values currently
# stored in the form.
#
def _generateConditional(block):
    conditionLike = {}
    conditionEq = {}
    # Get all the user-supplied parameters from the entry widgets
    for entry, val in block._queryValues.items():
        if entry._bound and entry.isQueryable() and len(str(val)):
          if entry.typecast == 'text':
            if block._convertAstericksToPercent:
              print val
              try:
                val = str(val).replace('*','%')
              except ValueError:
                pass
            print val, val.find('%'), val.find('_')
            if (val.find('%') >= 0 or val.find('_') >= 0):
              conditionLike[entry.field] = val
            else:
              conditionEq[entry.field] = val
          else:
##            GDebug.printMesg(5,'Adding conditional AND (%s=%s)' % (entry.field, val))
            conditionEq[entry.field] = val

    if len(conditionLike.keys()) and len(conditionEq.keys()):
##        GDebug.printMesg(5,'Combining like w/and (%s, %s)' % (conditionLike, conditionEq))
        conditions = GConditions.combineConditions( \
            GConditions.buildConditionFromDict(conditionLike, GConditions.GClike), \
            GConditions.buildConditionFromDict(conditionEq, GConditions.GCeq) )

    elif len(conditionLike.keys()):
        conditions = GConditions.buildConditionFromDict(conditionLike, GConditions.GClike)

    elif len(conditionEq.keys()):
        conditions = GConditions.buildConditionFromDict(conditionEq, GConditions.GCeq)

    else:
        conditions = {}

    return conditions




# A simple resultset iterator
# Lets you use ResultSets as:
#
#   for record in myResultSet:
#      blah
#
# NOTE: Python 2.2+  (but it won't get called in
#    Python 2.1 or below, so not a problem)
#
class _BlockIter:
  def __init__(self, block):
    self.block = block
    self.new = True
    self.done = False

  def __iter__(self):
    return self

  def next(self):
    if self.done:
      raise StopIteration
    elif self.new:
      self.block.jumpRecord(0)
      self.new = False
    elif self.block._resultSet.isLastRecord():
      self.done = True
      raise StopIteration
    else:
      self.block.nextRecord()
    
    if self.block.isEmpty() and self.block._resultSet.isLastRecord(): 
      self.done = True
      raise StopIteration
    
    return self.block

