#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# UndoManager.py
#
# DESCRIPTION:
#
# NOTES:
#

__all__ = ['UndoManager']

import types

class UndoManager:
  def __init__(self, instance):
    self.objects = []

    self.history = []
    self.future = []
    self.instance = instance

    self.deleteMap = {}

    # True if we are in the middle of a REDO action
    self.redo = 0

    # Non-zero if we are in the middle of an UndoGroup
    self.groupCount = 0
    self.group = []

    instance.registerEventListeners({
                       'ObjectDeleted'       : self.catchEvent,
                       'ObjectCreated'       : self.catchEvent,
                       'ObjectModified'      : self.catchEvent,
                       'BeginUndoGroup'      : self.beginGroup,
                       'EndUndoGroup'        : self.endGroup,
                       'RequestUndo'         : self.undo,
                       'RequestRedo'         : self.redo,
                      })


  def catchEvent(self, event):
    if not self.redo:
      history = History(self, event)
      if self.groupCount:
        self.group.append(history)
      else:
        self.history.append ([history])
      self.future = []
      self.history = self.history[-50:]
      self.updateEvents()

  def beginGroup(self, event):
    self.groupCount += 1
    self.future = []

  def endGroup(self, event):
    self.groupCount -= 1
    if not self.groupCount:
      self.history.append(self.group)
      self.history = self.history[-50:]
      self.group = []

  def undo(self, event):
    self.redo = 1
    if self.history:
      events = self.history.pop()
      events.reverse()
      for event in events:
        event.reverse(self)

    self.updateEvents()
    self.redo = 0

  def redo(self, event):
    self.redo = 1

    # TODO

    self.updateEvents()
    self.redo = 0

  def updateEvents(self):
    if self.history:
      self.instance.dispatchEvent('Enable:RequestUndo')
    else:
      self.instance.dispatchEvent('Disable:RequestUndo')

    if 0: #TODO# self.future:
      self.instance.dispatchEvent('Enable:RequestRedo')
    else:
      self.instance.dispatchEvent('Disable:RequestRedo')

  # Return the current instance of this object
  def getCurrObject(self, object):
    print "Finding current version of %s" % object
    while 1:
      try:
        object = self.deleteMap[id(object)]
      except KeyError:
        print "Found  %s" % object
        return object

class History:
  def __init__(self, manager, event):
    self.event = event
    if event.__event__ == 'ObjectDeleted':
      self.parentindex = event.object.getParent ()._children.index(event.object)
      self.parent = id(event.object.getParent ())

  def reverse(self, manager):
    orig = self.event
    print "-----"
    print orig.object
    object = manager.getCurrObject(orig.object)
    print object
    if orig.__event__ == 'ObjectCreated':
      manager.instance.dispatchEvent('ObjectDeleted',
          object=orig.object, originator=__name__)
    elif orig.__event__ == 'ObjectModified':
      for key in orig.new.keys():
        del object.__dict__[key.replace(':','__')]
      for key, value in orig.old.items():
        object.__dict__[key.replace(':','__')] = value
      manager.instance.dispatchEvent('ObjectModified',
          object=orig.object, originator=__name__,
          new=orig.old, old=orig.new)
    elif orig.__event__ == 'ObjectDeleted':
      attrs = {}
      for key, val in object.__dict__.items():
        if key[0] != '_' and type(val) != types.MethodType:
          attrs[key] = val
      parent = manager.getCurrObject(object.getParent ())
      obj = object.__class__(parent)
      obj.__dict__.update(attrs)
      obj._buildObject()
      manager.deleteMap[id(object)] = obj

      # Move the child to its old, relative position
      # in the parent's child list
      obj.getParent ()._children.pop()
      obj.getParent ()._children.insert(self.parentindex, obj)

      # Notify our peeps...
      manager.instance.dispatchEvent('ObjectCreated', originator=__name__,
          object=obj)

