#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# forms/events/Arrange.py
#
# DESCRIPTION:
# Code to handle the various Auto-Arrange events
#
# NOTES:
#

class Arrange:
  def __init__(self, instance):
    self.instance = instance
    self.enabled = 1
    instance.registerEventListeners( {
       'Forms:AlignLeft':       self.leftEdge,
       'Forms:AlignRight':      self.rightEdge,
       'Forms:AlignHCenter':    self.hCenter,
       'Forms:AlignTop':        self.topEdge,
       'Forms:AlignBottom':     self.bottomEdge,
       'Forms:AlignVCenter':    self.vCenter,
       'Forms:SmartAlign':      self.smartAlign,
       'ObjectSelected':        self.objectSelected } )
    self.selection = None

  def _getDimensions(self):
    objects = []
    x=99999
    y=99999
    mx = -1
    my = 1
    for object in self.selection:
      try:
        x = min(x, object['Char:x'])
        y = min(y, object['Char:y'])
      except KeyError:
        continue
      try:
        mx = max(mx, object['Char:x'] + object['Char:width'] - 1)
      except KeyError:
        mx = max(mx, object['Char:x'] + 9)
      try:
        my = max(my, object['Char:y'] + object['Char:width'] - 1)
      except KeyError:
        pass
      objects.append(object)
##    print x,y,mx, mx
    return (objects, x, y, mx, my)


  def leftEdge(self, event):
    event.dispatchAfter('BeginUndoGroup')
    objects, x, y, mx, my = self._getDimensions()
    for object in objects:
      oldx = object['Char:x']
      if oldx != x:
        object.Char__x = x
        event.dispatchAfter('ObjectModified', originator=__name__,
                                 object=object,
                                 old={'Char:x': oldx},
                                 new={'Char:x': x})
    event.dispatchAfter('EndUndoGroup')


  def rightEdge(self, event):
    event.dispatchAfter('BeginUndoGroup')
    objects, x, y, mx, my = self._getDimensions()
    for object in objects:
      oldx = object['Char:x']
      try:
        newx = mx - object['Char:width'] + 1
      except KeyError:
        newx = mx - 9
      if oldx != newx:
        object.Char__x = newx
        event.dispatchAfter('ObjectModified', originator=__name__,
                                 object=object,
                                 old={'Char:x': oldx},
                                 new={'Char:x': newx})
    event.dispatchAfter('EndUndoGroup')


  def hCenter(self, event):
    event.dispatchAfter('BeginUndoGroup')
    objects, x, y, mx, my = self._getDimensions()
    bx = (mx - x + 1)/2
    for object in objects:
      oldx = object['Char:x']
      try:
        newx = int(bx - (object['Char:width']/2) + 1)
      except KeyError:
        newx = mx - 6
      if oldx != newx:
        object.Char__x = newx
        event.dispatchAfter('ObjectModified', originator=__name__,
                                 object=object,
                                 old={'Char:x': oldx},
                                 new={'Char:x': newx})
    event.dispatchAfter('EndUndoGroup')


  def topEdge(self, event):
    event.dispatchAfter('BeginUndoGroup')
    objects, x, y, mx, my = self._getDimensions()
    for object in objects:
      oldy = object['Char:y']
      if oldy != y:
        object.Char__y = y
        event.dispatchAfter('ObjectModified', originator=__name__,
                                 object=object,
                                 old={'Char:y': oldy},
                                 new={'Char:y': y})
    event.dispatchAfter('EndUndoGroup')


  def bottomEdge(self, event):
    event.dispatchAfter('BeginUndoGroup')
    objects, x, y, mx, my = self._getDimensions()
    for object in objects:
      oldy = object['Char:y']
      try:
        newy = my - object['Char:height'] + 1
      except KeyError:
        newy = my
      if oldy != newy:
        object.Char__y = newy
        event.dispatchAfter('ObjectModified', originator=__name__,
                                 object=object,
                                 old={'Char:y': oldy},
                                 new={'Char:y': newy})
    event.dispatchAfter('EndUndoGroup')


  def vCenter(self, event):
    event.dispatchAfter('BeginUndoGroup')
    objects, x, y, mx, my = self._getDimensions()
    by = (my - y + 1)/2
    for object in objects:
      oldy = object['Char:y']
      try:
        newy = int(by - (object['Char:height']/2) + 1)
      except KeyError:
        newy = by
      if oldy != newy:
        object.Char__y = newy
        event.dispatchAfter('ObjectModified', originator=__name__,
                                 object=object,
                                 old={'Char:y': oldy},
                                 new={'Char:y': newy})
    event.dispatchAfter('EndUndoGroup')


  def smartAlign(self, event):
    objects, x, y, mx, my = self._getDimensions()
    entries = [[]] * (my - y + 1)
    nonentries = [[]] * (my - y + 1)
    for object in objects:
      if object._type in ('entry','button','image'):
        entries[object['Char:y'] - y].append( (object['Char:x'],object))
      else:
        nonentries[object['Char:y'] - y].append( (object['Char:x'],object))

    # TODO



  def objectSelected(self, event):
    try:
      self.selection = event.selection
    except:
      self.selection = [event.object]

    #
    # Enable/disable our menu options
    # depending on if enough items are
    # selected or not.
    #
    evt = None
    if len(self.selection) > 2 and not self.enabled:
      evt = 'Enable'
    elif self.enabled:
      evt = 'Disable'

    if evt:
      for task in ('FormsAlignLeft',
                   'FormsAlignRight',
                   'FormsAlignHCenter',
                   'FormsAlignTop',
                   'FormsAlignBottom',
                   'FormsAlignVCenter',
                   'FormsSmartAlign'):
        self.instance.dispatchEvent('%s:%s' % (evt, task))




