#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# reports/Instance.py
#
# DESCRIPTION:
# Creates a "report" instance
#
# NOTES:
#

from gnue.designer.reports import properties as reportsProperties
from gnue.designer.base.Instance import BaseInstance
from gnue.common.utils.FileUtils import dyn_import
from gnue.common.apps import GDebug
import wizards

# Tool support...
from LayoutEditor import LayoutEditor
from TreeView import TreeView
from gnue.designer.base.tools.PropertyEditor import PropertyEditor
from gnue.designer.base.tools.DataSourceEditor import DataSourceEditor
from gnue.designer.base.tools.SchemaViewer import SchemaViewer
from gnue.designer.base.tools.TriggerEditor import TriggerEditor
from gnue.designer.base.tools.EventEditor import EventEditor

from gnue.reports.base import GRReport, GRParser, GRSources, GRParameters, GRSortOptions
from gnue.reports.base import layout as GRLayout


class Instance(BaseInstance):

  def __init__(self, app, *args, **params):
    self.properties = reportsProperties

    #self._tableMappings = {}
    BaseInstance.__init__(self, app, *args, **params)

    report = self.rootObject
    report._parameters = report.findChildOfType('GRParameters') or \
        self.incubator.createObject(report, 'parameters', report, select=0)
    report._sortoptions = report.findChildOfType('GRSortOptions') or \
        self.incubator.createObject(report, 'sortoptions', report, select=0)
    report._sources = report.findChildOfType('GRSources') or \
        self.incubator.createObject(report, 'sources', report, select=0)
    report._layout = report.findChildOfType('GRLayout') or \
        self.incubator.createObject(report, 'layout', report, select=0)



  def loadBuffer(self, buffer):
    report = GRParser.loadFile (buffer, self, initialize=0)
    report.name = 'report'

    # Find which filter is in use... yes, this is ugly
    namespace = report.findChildOfType('GRLayout')._findNamespace()
    filter = namespace.split(':')[-1]

    try:
      filtersupport = dyn_import('gnue.designer.reports.%s.FilterSupport' % filter)
      usingDefault = 0
    except ImportError:
      GDebug.printMesg(2,'No filter support for %s; using default')
      filtersupport = dyn_import('gnue.designer.reports.Default.FilterSupport')
      usingDefault = 1

    self.filter = filtersupport.FilterSupport(self, report)
    if usingDefault:
      self.filter.setNamespace(namespace)

    self.incubator = self.filter.INCUBATORCLASS(self)

    return report


  def loadEmpty(self, style=None):
    report = GRReport.GRReport()
    report.title = "Untitled report"
    report.name = 'report'
    GRParameters.GRParameters(report)._buildObject()
    GRSortOptions.GRSortOptions(report)._buildObject()
    GRSources.GRSources(report)._buildObject()
    GRLayout.GRLayout(report)._buildObject()

    # Find which filter is in use... yes, this is ugly
    namespace = style or "GNUe:Reports:Standard"
    filter = (namespace).split(':')[-1]

    try:
      filtersupport = dyn_import('gnue.designer.reports.%s.FilterSupport' % filter)
      usingDefault = 0
    except ImportError:
      GDebug.printMesg(2,'No filter support for %s; using default')
      filtersupport = dyn_import('gnue.designer.reports.Default.FilterSupport')
      usingDefault = 1

    self.filter = filtersupport.FilterSupport(self, report)
    if usingDefault:
      self.filter.setNamespace(namespace)

    self.incubator = self.filter.INCUBATORCLASS(self)

    self.filter.initAsNew()

    report._buildObject()
    return report


  def inventoryObject(self, object):
    pass


  def createTools(self):
    self.addTool('propertyEditor', _('Property Inspector'), PropertyEditor,
                      hotkey='F10', menuGroup=301)
    self.addTool('treeEditor',     _('Object Navigator'), TreeView,
                      hotkey='F8', menuGroup=301)
    self.addTool('schemaViewer',   _('Schema Navigator'), SchemaViewer,
                      menuGroup=301)

    self.addTool('datasourceEditor',_('Data Source Editor'), DataSourceEditor,
                      hotkey='F9', menuGroup=311)
    #self.addTool('groupEditor',   _('Group Editor'), GroupEditor,
    #                  hotkey='F5', menuGroup=311)
    #self.addTool('fieldEditor',   _('Field Editor'), FieldEditor,
    #                  hotkey='F6', menuGroup=311)

    self.addTool('eventEditor',    _('Trigger Inspector'), EventEditor,
                      hotkey='F7', menuGroup=321)
    self.addTool('triggerEditor',  _('Code Editor'), TriggerEditor,
                      hotkey='F12', menuGroup=321)

    # Add filter-specific tools
    self.filter.addTools()


  def createWizards(self):
    self.loadWizards(wizards)

  # Used by TemplateParser to build a wizard.current dict
  def buildWizardCurrentDict(self):
    report = self.rootObject
    return { 'report': report,
             'sources': report._sources,
             'parameters': report._parameters,
             'layout': report._layout,
             'sortoptions': report._sortoptions,
             'object': self._currentObject }

  def preSave(self):

    # Resort the main sections to a logical order
    report = self.rootObject
    for o in [report._parameters, report._sortoptions,
              report._sources, report._layout]:
      report._children.pop(report._children.index(o))
      if len(o._children):
        report._children.append(o)


