#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# BlockEditor.py
#
# DESCRIPTION:
# Implements the "Block Editor" tool
#
# NOTES:
#

__all__ = ['BlockEditor']

from gnue.designer.base.ToolBase import *
from gnue.designer.base.MultiObjectGridEditor import *
from gnue.designer.base.ObjectList import *
from gnue.designer.base.ToolBar import BaseToolBar
from gnue.forms.GFObjects.GFBlock import GFBlock
from gnue.common.datasources.GDataSource import GDataSource
from wxPython.wx import *

class BlockEditor(ToolBase):

  runtime_section = "FormsBlockEditor"
  default_dock = 'frame' # 'right-20'
  uses_toolbar = 1
##  uses_feedback_bar = 1

  def init(self):
    self.grid = _BlockEditorGrid(self, self.instance)
    EVT_SIZE(self, self.__OnSize)

  def __OnSize(self, event):
    self.grid.SetSize(self.GetClientSize())

  def createToolbar(self, parent):
    return _BlockToolbar(parent, self.instance)



######################################################################
#
#
class _BlockEditorGrid(MultiObjectGridEditor):

  base_gobject = GFBlock

  def init(self):
    self.datasources = ObjectList(self.instance, GDataSource,
                                  'name', self.__datasourcesUpdated)
    self.instance.registerEventListeners({'BlockEditor:Delete': self.__OnDelete})


  def getColumnDefinitions(self):
    return  (
        {'Name':      'Name',     # Internal name
         'Attribute': 'name',     # Attribute (can be a method)
         'Label':     _('Name'),  # Column label
        },
        {'Name':      'Rows',     # Internal name
         'Attribute': 'rows',     # Attribute (can be a method)
         'Label':     _('# Rows'),  # Column label
        },
        {'Name':      'DataSource',     # Internal name
         'Attribute': 'datasource',     # Attribute (can be a method)
         'Label':     _('Data Source'), # Column label
         'ValidSet':  self.__getDataSources,
        },
        {'Name':      'BaseTable',          # Internal name
         'Attribute': self.__getBaseTable,  # Attribute (can be a method)
         'ReadOnly':  1,                    # Can the user edit this field?
         'Label':     _('Base Table'),      # Column label
        },
      )

  #
  # Given a datasource, find the base table
  #
  def __getBaseTable(self, object):
    try:
      if not object.datasource:
        return ""
    except AttributeError:
      return ""

    try:
      datasource = self.datasources[object.datasource]
    except AttributeError:
      return ""

    if hasattr(datasource,'table') and datasource.table:
      return datasource.table
    elif datasource.type == 'static':
      return _("(Custom List)")
    else:
      return ""

  #
  #
  #
  def __getDataSources(self):
    return self.datasources.getDict()


  #
  # Called if any of the datasources have been updated
  #
  def __datasourcesUpdated(self):
    self.updateValidSet('DataSource')

  #
  # User selected the delete icon
  #
  def __OnDelete(self, event):
    if self.current:
      self.instance.incubator.deleteObject(self.instance.rootObject,
                                           self.current, firstRun=0)


######################################################################
#
#
ToolbarMapping = 'Wizard:AddBlock,wizard_block;' \
               + 'BlockEditor:Delete,tb_delete'

class _BlockToolbar(BaseToolBar):
##  ICON_SIZE=(16,16)
  def init(self):
    self.addToolbarMapping(ToolbarMapping)
