#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# ObjectList.py
#
# DESCRIPTION:
# Maintains a list of GObjects, keeping it in synch with any
# attribute changes, additions, or deletions.
#
# Listeners can be notified when this list changes in any way.
#
# NOTES:
# This list isn't meant for code that needs to finely track
# changes to objects, but just need a current list of objects.
#

__all__ = ['ObjectList']


class ObjectList:
  def __init__(self, instance, base, nameAttr, updateMethod=None):
    self.base = base
    self.name = nameAttr
    self.listeners = updateMethod and [updateMethod] or []
    self.objects = []
    instance.registerEventListeners({
                       'ObjectCreated'       : self.__objectCreated,
                       'ObjectModified'      : self.__objectModified,
                       'ObjectDeleted'       : self.__objectDeleted,
                      })
    instance.rootObject.walk(self.__inventory)

  def addListener(self, listener):
    self.listeners.append(listener)

  def removeListener(self, listener):
      self.listeners.remove(listener)

  def __inventory(self, object):
    if isinstance(object, self.base):
      self.objects.append(object)

  def __objectModified (self, event):
    if isinstance(event.object, self.base) and \
         self.name in event.old.keys() or \
         self.name in event.new.keys():
      for listener in self.listeners:
        listener()

  def __objectCreated (self, event):
    if isinstance(event.object, self.base):
      self.objects.append(event.object)
      for listener in self.listeners:
        listener()

  def __objectDeleted (self, event):
    try:
      self.objects.remove(event.object)
    except ValueError:
      return
    for listener in self.listeners:
      listener()

  def getDict(self):
    rs = {}
    for object in self.objects:
      try:
        rs[object.__dict__[self.name]] = object
      except KeyError:
        pass
    return rs

  def getList(self):
    return tuple(self.objects)

  def __getitem__(self, name):

    # Name can either be the object's "name", or an index value
    try:
      name = name.lower()
    except AttributeError:
      # Index reference, not name reference
      return self.objects[name]

    # Name reference
    for object in self.objects:
      try:
        if object.__dict__[self.name].lower() == name:
          return object
      except KeyError:
        pass
    raise KeyError

  def __getslice__(self, i, j):
    return self.objects[i:j]

  def __len__(self):
    return len(self.objects)

