#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# Instance.py
#
# DESCRIPTION:
# This is the base instance class for a designer session. Every
# open file will be associated with its own Instance.
#
# NOTES:
#

__all__ = ['BaseInstance']

import sys, os, time, dircache, string
from wxPython.wx import *
from gnue.common.apps import GDebug
from gnue.common.apps import RuntimeSettings
from gnue.common.utils.FileUtils import dyn_import
from gnue.common.events import EventController, Event
from gnue.designer.base.MenuBar import MenuBar
from gnue.designer.base.UndoManager import UndoManager
from gnue.designer.base.ToolBase import ToolBase
from gnue.designer.base.docks.Docker import Docker
from gnue.designer.base.ObjectList import ObjectList
from gnue.designer.base import TemplateBase
from gnue.designer.base.TemplateParser import WizardRunner
from PrimaryToolBar import PrimaryToolBar
from gnue.designer import VERSION, PACKAGE
from ModuleSupport import SupportedModules

TITLE=PACKAGE

class BaseInstance(wxFrame, EventController):

  ########################################################################
  #
  # Virtual functions...
  #
  # Override these functions in your Instance
  #

  wizardRunner = WizardRunner

  # Called to load the object from a file buffer/handle
  def loadFile(self, buffer):
    return None

  # Called to load an "empty" object.  Create the root object
  # and any initial child objects (if appropriate)
  def loadEmpty(self, style=None):
    return None

  # Called in a tree-like fashion for every object
  # whether loaded in the beginning or later added.
  def inventoryObject(self, object):
    pass

  # Called just before saving the file using GObject's dumpTree
  # You might do some sanity checks, etc.
  def preSave(self):
    return

  #
  def createTools(self):
    pass

  #
  def createWizards(self):
    pass

  #
  def initMenu(self):
    pass

  # Instances can add to the primary toolbar
  def initToolBar(self):
    pass


  # Used by TemplateParser to build a wizard.current dict
  def buildWizardCurrentDict(self):
    return {'object': self._currentObject}

  ########################################################################
  #
  # BaseInstance-specific stuff
  # If you need to overwrite something here,
  # then odds are we really need to provide
  # a hook of some sorts.
  #
  def __init__(self, app, location=None, buffer=None, style=None):
    wxFrame.__init__(self, NULL, -1, "")
    EventController.__init__(self)

    # During startup, we will cache all events
    # so we can execute them when everything has
    # been initialized.
    self.startCachingEvents()

    # Register ourself with RuntimeSettings
    RuntimeSettings.init(configFilename="gnue-des.ini", homeConfigDir=".gnue")
    RuntimeSettings.registerInstance(self)

    # Tell RuntimeSettings that we have information to save
    self.runtime_section = self.properties.nickname + 'Layout'
    RuntimeSettings.registerRuntimeSettingHandler(self, self)

    # And the MRU
    RuntimeSettings.registerRuntimeSettingHandler(self, app.mru)

    # Set up the Undo/Redo Manager
    UndoManager(self)


    self._isdirty = 0
    self._makeBackup = 1
    self._isNew = 1

    self._app = app
    self.connections = app.connections
    self.nameMappings = {}
    self.usedNames = []

    self._pages = []

    self.objectLists = {}

    self._path = ""

    self.globalAccelerators = []
    self.globalAcceleratorListeners = []

    self._nameMappers = {} # {GDataSource: {'name': [(GFBlock,'name')]}}


    self.registerEventListeners({
                       # Menu/Toolbar stuff
                       'RequestSave'         : self.OnSave,
                       'RequestSaveAs'       : self.OnSaveAs,
                       'RequestClose'        : self.OnClose,

                       # Object stuff
                       'ObjectSelected'      : self.__onSetCurrentObject,
                       'ObjectCreated'       : self.__onCreateObject,
                       'ObjectModified'      : self.__onModifyObject,
                       'ObjectDeleted'       : self.__onDeleteObject,
                      })

    if location == None:

      if buffer != None:
        self.__loadFromBuffer(buffer)
        self.makeDirty()
      else:
        self.__createEmptyInstance(style)

    else:

      if not os.access (location, os.R_OK):
        self.Show(1)
        if wxMessageDialog(self,
          _('The requested file does not exist.\n') +
          _('Do you want to create this file?') +
          _('\n\nFile: %s') \
            % location, _("File Not Found"), wxYES_NO|wxICON_QUESTION ).ShowModal() == wxID_NO:
          sys.exit()
        self.Show(0)
        self.__createEmptyInstance(style)
        self._path = location
        self.makeDirty()

      elif not os.access (location, os.W_OK):
        self.Show(1)
        if wxMessageDialog(self, \
          _('The requested file is Read Only.\n') +
          _('To save any changes, you will \n') +
          _('be required to do a "Save As..."\n\nFile: %s') \
            % location, _("Read Only Warning"), wxOK|wxCANCEL|wxICON_EXCLAMATION ).ShowModal() == wxID_CANCEL:
          sys.exit()
        self.Show(0)
        self.__loadFromFile(location)
      else:
        self.__loadFromFile(location)

    self.statusbar = self.CreateStatusBar()
    self.SetStatusText('Welcome to GNUe Designer -- Do not expose to direct sunlight, do not feed after midnight, and do not get wet.')
    self.menubar = MenuBar(self)
    self.toolbar = PrimaryToolBar(self, self)
    self.SetToolBar(self.toolbar)


    StartupStatus(_('Inventorying Document Objects'))
    self.rootObject.walk(self.__inventory)

    StartupStatus(_('Creating User Interface'))

    # Create the supplemental tools
    self._toolCache = []

    # Set up the menu system
    self._initMenu()
    self.initMenu()

    self.createTools()
    self._initTools()
    self.createWizards()
    self.initToolBar()

    # Finalize menu bar
    self.menubar.finalize()
    self._app.mru.addMenu(self.menubar.getMenu('File|Open Recent|'), self)

    # TODO: This is a hack to disable any menu items
    # TODO: for actions we've yet to implement
    for action in ('RequestRevert','RequestRedo','RequestCopy',
                   'RequestPaste','RequestPasteSpecial','RequestCut'):
      self.dispatchEvent('Disable:%s' % action)


    # Build accelerator list
    accel = wxAcceleratorTable(self.globalAccelerators)
    for child in [self] + self.globalAcceleratorListeners:
      child.SetAcceleratorTable(accel)

    # Add ourselve to the main app's instance list
    self._app.addInstance(self)

    # Fit the widgets to the screen
    self.Fit()

    self.SetSize((
       RuntimeSettings.getint(self.runtime_section, 'width', 550),
       RuntimeSettings.getint(self.runtime_section, 'height', 400)))

    self.SetPosition((
       RuntimeSettings.getint(self.runtime_section, 'x', -1),
       RuntimeSettings.getint(self.runtime_section, 'y', -1)))

    self.dispatchEvent('ObjectSelected', originator=self,
                       object=self.rootObject)

    self.Show(true)
    self.Refresh()

    EVT_CLOSE(self, self.OnClose)

    self.stopCachingEvents()

  def _initMenu(self):
    # Add the [sub]menus
    for location, text, grouping in (
          ('File', _('&File'), 100),
          ('File|New', _('&New'), 100),
          ('File|Open Recent', _('&Open Recent'), 200.1),
          ('File|Connect To', _('&Connect To'), 400),
          ('Edit', _('&Edit'), 200),
          ('Insert',_('&Insert'), 300),
          ('Modify',_('&Modify'), 400),
          ('Tools',_('&Tools'), 500),
          ('View',_('&View'), 800),
          ('Help',_('&Help'), 999)):

      self.menubar.addMenu(location, text, grouping)

    for location, event, text, hotkey, help, grouping in (
       ('File', 'RequestOpen', _('&Open'), 'Ctrl+O', _("Open an existing document"), 200.1),
       ('File|Open Recent', 'XXXX', _('&Foo'), None, _("This is a placeholder for the Open Recent menu"), 200.1),
       ('File|New', 'RequestNewWizard', _('From &Wizard...'), None, _("Create a new document using a wizard"), 900.1),
       ('File', 'RequestSave', _("&Save"), 'Ctrl+S', _("Save the current document"),300.1),
       ('File', 'RequestSaveAs', _("Save &As..."),None, _("Save the current document under a new name"), 300.2),
       ('File', 'RequestSaveAll', _("Save A&ll"),None,_("Save all open document"),300.3),
       ('File', 'RequestRevert', _("Reload"), None, _("Reload the current document as of its last save (abandoning any changes)"), 500),
       ('File', 'RequestClose', _("&Close"), 'Ctrl+W', _("Close the current document"), 990),
       ('File', 'RequestExit', _("E&xit"), None, _("Exit GNUe Designer"), 995),
       ('Edit', 'RequestUndo', _("&Undo"), 'Ctrl+Z', _("Undo the last action"), 100.1),
       ('Edit', 'RequestRedo', _("&Redo"), 'Ctrl+Y', _("Redo the last undo action"), 100.2),
       ('Edit', 'RequestCut', _("Cu&t"), 'Ctrl+X', _("Cut the current object and move to the clipboard"), 200.1),
       ('Edit', 'RequestCopy', _("&Copy"), 'Ctrl+C', _("Copy the current object to the clipboard"), 200.2),
       ('Edit', 'RequestPaste', _("&Paste"), 'Ctrl+V', _("Paste the current object on the clipboard"), 200.3),
       ('Edit', 'RequestPasteSpecial', _("Paste &Special..."), None, _("Paste the current object on the clipboard with special attributes"), 200.4),
       ('Modify','RequestDelete', _("&Delete Item"), 'Delete', _("Delete the current object"),100),
       ('Help', 'RequestAbout', _("&About GNUe Designer"), None, _("More information about GNUe Designer"), 900),
     ):
      self.menubar.addAction(location, text, event,
                      grouping, canDisable=1,
                      icon=None, hotkey=hotkey, help=help)

    # Add supported tools to File|New
    for tool in SupportedModules:
       self.menubar.addAction('File|New', '&%s' % tool.properties.nickname,'RequestNew', 100,
        help=_('Create a new %s') % string.lower(tool.properties.nickname), eventdata={'type': tool.properties.module })

    # Add connections
    for conn in self.connections.getAllConnectionParameters().keys():
      id = wxNewId()
      self.menubar.addAction('File|Connect To', conn, 'Connect:%s' % conn,
               help=_("Login to %s connection") % conn,
               eventdata={'connection': conn})
      self.registerEventListeners({'Connect:%s' % conn:self.__OnConnectTo})


  def _initTools(self):
    self.docker = Docker(self, self, self._toolCache)

  def addTool(self, id, title, baseclass, hotkey=None, menuGroup=499):
    self._toolCache.append( (id, title, baseclass, hotkey, menuGroup) )


  # Return, or create, an ObjectList based on the xml tag
  def getObjectList(self, tag):
    try:
      return self.objectLists[tag]
    except KeyError:
      defin = self.incubator.elements[tag]
      baseClass = defin['BaseClass']

      # Determine the "name" attribute"
      nameAttr = None
      try:
        if defin['Attributes'].has_key('name'):
          nameAttr = 'name'
        else:
          for attribute, props in defin['Attributes'].items():
            try:
              if props['Unique']:
                nameAttr = attribute
                break
            except KeyError:
              pass
      except KeyError:
        pass

      list = ObjectList(self, baseClass, nameAttr)
      self.objectLists[tag] = list
      return list

  def __loadFromFile(self, location):
    try:
      self._path = location
      fileHandle = open(location,'r')
      self.__loadFromBuffer(fileHandle)
      fileHandle.close()
      self.makeClean()
      self._isNew = 0
    except IOError, msg:
      print "\n%s %s\n\nUnable to open file '%s'. \nUnexpected read error:\n  %s.\n" % (TITLE, VERSION, location, msg)
      sys.exit()
    self._app.mru.addLocation(location)


  def addNameMapper(self, instance, attribute, childinst, childattr):
    try:
      instm = self._nameMappers[instance]
    except KeyError:
      instm = {}
      self._nameMappers[instance] = instm
    try:
      attrm = instm[attribute]
    except KeyError:
      attrm = []
      instm[attribute] = attrm
    attrm.append ( (childinst, childattr) )


  def __loadFromBuffer(self, fileHandle):
    self.rootObject = self.loadBuffer(fileHandle)


  def __createEmptyInstance(self, style):
    self.rootObject = self.loadEmpty(style)
    self.makeClean()
    self._isNew = 1


  #
  #  Used by RuntimeSettings
  #
  def saveRuntimeSettings(self):
    x, y = self.GetPositionTuple()
    width, height = self.GetSizeTuple()
    settings = { 'x': abs(x),
                 'y': abs(y),
                 'height': height,
                 'width': width  }

    return ( self.runtime_section, settings )


  # Do we need to be saved?
  def isDirty(self):
    return self._isdirty

  # Mark our form as "dirty" (unsaved changes)
  def makeDirty(self):
    if not self._isdirty:
      self._isdirty = 1
      if self._path == "":
        self.SetTitle( TITLE + _(" - <New %s> *") % self.properties.nickname)
      else:
        self.SetTitle( TITLE + " - " + self._path + " *")
    self.dispatchEvent('MakeDirty')

  # Mark our form as "clean" (no unsaved changes)
  def makeClean(self):
    self._isdirty = 0
    self._isnew = 0
    if self._path == "":
      self.SetTitle( TITLE + _(" - <New %s>") % self.properties.nickname)
    else:
      self.SetTitle( TITLE + " - " + self._path)
    self.dispatchEvent('MakeClean')


  # Take an object and mangle it all up
  # until it is useful to us
  def __inventory (self, object):

    # Add a GObjectHelper to intercept __getitem__ calls on GObject
    GObjectHelper(self, object)

    if object != self.rootObject:

      # just a hack for designer.reports.TreeView.py (btami)
      if object._xmlnamespace:
        if object.name:
          object._description = object.name + "(out:)"
        else:
         object._description = object._xmltag + "(out:)"

      # Assign an ID if none exists
      # TODO: in the future, I want to change these to id=".." instead of name=".."
      if hasattr(object, 'name'):
        if (object.name == None or ( \
            object.name[:3] == "__<" and \
            object.name[-3:] == ">__")):
          object.name = self.getNextGenericName(object._type[2:])
        self.nameMappings[string.lower(object.name)] = object
        self.usedNames.append(object.name.lower())

    # Now, give the tool-specific instances a chance
    self.inventoryObject(object)


  def __onSetCurrentObject (self, event):
    object = event.object
    handler = event.originator
    self._currentObject = object

  def __onCreateObject (self, event):
    self.__inventory(event.object)
    self.makeDirty()

  def __onModifyObject (self, event):
    object = event.object
    # Check for any name dependencies
    # (i.e., GFEntry is dependent on GFBlock.name and GFField.name)
    # Automatically create any change events for the child objects
    if hasattr(object, '_nameDependencies'):
      for key in object._nameDependencies.keys():
        masterattr, slaveattr = object._nameDependencies[key]
        if masterattr in event.new.keys():
          oa = {slaveattr: key[slaveattr]}
          na = {slaveattr: object[masterattr]}
          key.__dict__.update(na)
          self.dispatchEvent( 'ObjectModified',
                              object=key,
                              originator=self,
                              old=oa, new=na )
    self.makeDirty()

  def __onDeleteObject (self, event):
    object = event.object

    # Delete the actual object from its parent
    object._parent._children.remove(object)
    ##object._parent._children.pop(object._parent._children.index(object))

    self.makeDirty()


  def getNextGenericName (self, type):
    return self.getUniqueName(type.capitalize() + '1')


  def getUniqueName(self, name, limitingObject=None):

    try:
      usedNames = limitingObject.usedNames
    except:
      usedNames = self.usedNames

    if name.lower() not in usedNames:
      usedNames.append(name.lower())
      return name

    index = len(name) - 1

    while index > 0 and '0' <= name[index] <= '9':
      index -= 1

    index += 1
    if index >= len(name):
      start = 1
      base = name
    else:
      start = int(name[index:])
      base = name[:index]

    while ("%s%s" % (base, start)).lower() in usedNames:
      start += 1

    rv = "%s%s" % (base, start)
    usedNames.append(rv.lower())
    if rv.lower() not in self.usedNames:
      self.usedNames.append(rv.lower())
    return rv



  def save(self):

    if self.preSave():
      print "Not saving definition"
      return

    location = self._path
    fileHandle = None
    fileHandle2 = None
    if self._makeBackup:
      try:
        fileHandle = open(location,'r')
        fileHandle2 = open(location + "~",'w')
        fileHandle2.writelines(fileHandle.readlines())
      except:
        pass
      else:
        if fileHandle != None:
          fileHandle.close()
        if fileHandle2 != None:
          fileHandle2.close()

    self._makeBackup = 0

    fileHandle = open(location,'w')

    fileHandle.write('<?xml version="1.0" encoding="%s"?>\n\n' % \
           gConfig('textEncoding'))

    fileHandle.write('<!--  %s (%s)\n      Saved on: %s  -->\n\n' \
       % (TITLE, VERSION, \
          time.strftime("%Y-%m-%d %H:%M:%S",time.localtime(time.time()))))
    fileHandle.write(self.rootObject.dumpXML(treeDump=1).encode(gConfig('textEncoding')))
    fileHandle.close()

    self._app.mru.addLocation(location)
    self.makeClean()



  def __OnConnectTo(self, event):
    conn = event.connection
    tempDO = self.connections.getDataObject(conn, 'object')
    try:
      self.connections.requestConnection(tempDO, conn)
      self.dispatchEvent('ConnectionEstablished',connection=conn)
      self.dispatchEvent('Disable:Connect:%s' % conn)
    except:
      print _("Unable to connect to %s") % conn


  def OnSave(self, event):
    if not len(self._path):
      self.OnSaveAs(event)
    else:
      self.save()


  def OnSaveAs(self, event):
    wildcard = ""
    # Make the "default" file extension for a tool
    # appear before the other extensions.
    filterIndex = [self.properties.defaultFileExtension]
    wildcard += "%s (*.%s)|*.%s|" % \
     ( self.properties.fileExtensions[self.properties.defaultFileExtension],
       self.properties.defaultFileExtension,
       self.properties.defaultFileExtension)

    for type in self.properties.fileExtensions.keys():
      if type != self.properties.defaultFileExtension:
        wildcard += "%s (*.%s)|*.%s|" % \
           ( self.properties.fileExtensions[type], type, type)
        filterIndex.append(type)

    dlg = wxFileDialog(self, _("Save %s As...") % self.properties.description, defaultDir=os.getcwd(),
                           wildcard = wildcard,
                           style=wxSAVE)

    if dlg.ShowModal() == wxID_OK:
      path = dlg.GetPath()
      if len(path) < 4 or not (path[-4] == '.' and string.lower(path[-3:]) in self.properties.fileExtensions.keys()):
        path += "." + filterIndex[dlg.GetFilterIndex()]

      if os.path.isfile(path):
        dlg = wxMessageDialog(NULL,
              _('The file "%s".\n' % path) +
              _("exists. Overwrite?"),
              _("Unsaved Changes"), style=wxYES_NO|wxICON_WARNING)
        save = dlg.ShowModal()
        dlg.Destroy()
        if save == wxID_NO:
          self.OnSaveAs(event)
          return

      self._path = path
      self.SetTitle (TITLE + " - " + self._path)
      self.save()

    dlg.Destroy()


  def OnClose(self, event):
    if self.isDirty():
      dlg = wxMessageDialog(NULL,
              _("This document has unsaved changes.\n") +
              _("Save changes before closing?"),
              _("Unsaved Changes"), style=wxYES_NO|wxCANCEL|wxICON_WARNING)
      save = dlg.ShowModal()
      dlg.Destroy()
      if save == wxID_YES:
        self.OnSave(event)
      elif save == wxID_CANCEL:
        event.Veto()
        return

    RuntimeSettings.saveRuntimeSettings(self)
    self._app.mru.removeMenu(self.menubar.getMenu('File|Open Recent|'), self)
    self._app.removeInstance(self)
    self.Destroy()


  #################################################################
  #
  #################################################################

  def loadWizards(self, package):
    templates = []

    basedir = os.path.dirname(package.__file__)
    processed = []  # Base file names processed (e.g., base of Simple.py*
                  # is Simple) This will keep us from importing Simple
                  # three times if Simple.py, Simple.pyc, and Simple.lib
                  # all exist.

    for dir in dircache.listdir(basedir):
      base = string.split(dir,'.')[0]
      if not dir[0] in ('.','_') and not base in processed:
        processed.append(base)
        try:
          templates.append(dyn_import(
             '%s.%s' % (package.__name__,base)).TemplateInformation)
        except ImportError, mesg:
          GDebug.printMesg(2,"%s.%s doesn't appear to be a valid wizard" % (package.__name__, base))
          GDebug.printMesg(5,' --> %s' % (mesg))
        except AttributeError:
          GDebug.printMesg(2,'Wizard %s for package %s is missing'
                   ' an \'TemplateInformation\' attribute.' %
                           (base,package.__name__))

    for template in templates:
      try:
        location = template['MenuLocation']
        try:
          location, translation, grouping = location
          grouping = float(grouping)
        except:
          location, translation = location
          grouping = 499.0

        if location:
          self.wizardRunner(template, self)
          self.menubar.addAction(location=location, text=translation,
                                  event='Wizard:%s' % template['BaseID'],
                                  grouping=grouping, canDisable=1,
                                  eventdata={'template':template})
      except ValueError:
        continue

#
class GObjectHelper:
  def __init__(self, instance, object):
    self.instance = instance
    self.object = object
    object._setItemHook = self._setItemHook

  # Replace the getitem hooks from GObject
  # This is for the wizards, so they can do
  # entry['Char:x']-type calls and get back
  # what they expected.

  def _setItemHook(self, key, value):
    ek = key.replace(':','__')
    object = self.object
    try:
      ov = {key: object.__dict__[ek]}
    except KeyError:
      ov = {}
    object.__dict__[key.replace(':','__')] = value
    self.instance.dispatchEvent('ObjectModified',
       object = object, new={key:value}, old=ov, originator = '__inline__')


#
# Helper class used by the tool-specific packages
#
class ModuleProperties:
  xmlOpeningTag = 'undefined'
  short = 'undefined'
  application = 'GNUe Tool'
  description = 'undefined'
  fileExtensions = {}
  defaultFileExtension = 'undefined'

