#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# EventController.py
#
# DESCRIPTION:
#
# NOTES:
#

from EventAware import EventAware
from Event import Event

class EventController(EventAware):
  """
  An EventController is responsible for dispatching events to 
  registered listeners.
  """
  def __init__(self):
    # Note: Don't call EventAware.__init__
    #   ... that would be fugly.

    # Store a dictionary of registered events
    self.__incomingEvents = {}

  
  def registerEventListeners(self, events):
    """
    Registers an event listener for a specific event
    
    @param events: A dictionary of {'eventName': listenerFuction} pairs
    @type events: dict
    """
    for event in events.keys():
      try:
        self.__incomingEvents[event].append(events[event])
      except KeyError:
        self.__incomingEvents[event] = [events[event]]

  def startCachingEvents(self):
    """
    Causes the event controller to start storing events
    rather than dispatching them.  It will continue to 
    store events until L{stopCachingEvents} is called.
    """
    self.__cache = []

  def stopCachingEvents(self):
    """
    Notifies the event controller that is should 
    start processing events again.  Any previously 
    cached events are dispatched.
    """
    cache = self.__cache
    del self.__cache
    for event, arg, parms in cache:
      self.dispatchEvent(event, *arg, **parms)

  def dispatchEvent(self, event, *args, **parms):

    # If we are caching our events, cache it:
    try:
      self.__cache.append((event, args, parms))
      return
    except AttributeError:
      pass

    # Hackery so dispatchEvent can be passed
    # either an Event() object, or a text string
    # identifying the type of event.  If the
    # latter, an event is created on the fly.
    try:
      event.__event__
    except:
      event = Event(event, *args, **parms)

    handlers = []

    if self.__incomingEvents.has_key(event.__event__):
      handlers = self.__incomingEvents[event.__event__]

    if self.__incomingEvents.has_key('__before__'):
      handlers = self.__incomingEvents['__before__'] + handlers

    if self.__incomingEvents.has_key('__after__'):
      handlers = handlers + self.__incomingEvents['__after__']

    for handler in handlers:
      handler(event)
      if event.__error__ or event.__dropped__:
        break

    # Fire any "dispatchAfter" events
    for args, parms in event.__after__:
      self.dispatchEvent(*args, **parms)

    return event.__result__
