#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2003-2004 Free Software Foundation
#
# FILE:
# pyro/ClientAdapter.py
#
# DESCRIPTION:
# Set of classes that implement the pyro client adapter for GNUe Comm.
#
# NOTES:
# Requires pyro from  http://sourceforge.net/Projects/pyro
#
# Client Parameters:
#
#    url         The complete URI address of the XML-RPC service.
#                e.g, url="https://www.domain.com:9876/service.php"
#    ..or..
#    transport   The service transport (either "http" or "https")
#    host        The hostname or IP address of the service
#    port        The port that the service is located on
#

from gnue.common.rpc import GCommBase
from gnue.common.rpc import GComm
from gnue.common.apps import GConfig, GDebug
from gnue import paths

import string, sys

try:
  import Pyro.naming, Pyro.core, Pyro.protocol
  
except ImportError:
  tmsg = _("\nUnable to load pyro.  To use the Pyro interface, \n") \
         + _("please install xmlrpc from:\n") \
         + "    http://sourceforge.net/projects/pyro " \
         + _("the appropriate DEBIAN package is python-pyro.deb")
  raise GComm.AdapterInitializationError, tmsg


##############################################################################
#
# ClientAdapter
#
class ClientAdapter(GCommBase.Client):

  def __init__(self, params):

    self._timeout = 1.0
    
    #
    # check for the correct parameters
    #
    
    try:

      #
      #  test for an url parameter
      # 
      try:
        url = params['url']
        if not len(url):
          raise KeyError

        # split url into server + host + path parts
        import urlparse
        
        transport, netloc, path, params, query, fragment = \
                   urlparse.urlparse(url)
        host, port=split(netloc,':',1)
        
      except KeyError:
        #
        #  transport parameter
        #
        try:
          transport = params['transport']
        except KeyError:
          transport = 'http'

        if (transport!='http'):
           tmsg = _("py-xmlrpc just support http as a transport. ") +\
                  _("If you need https please have a look at the ") +\
                  _("'pw_xmlrpc' adapter.")
           raise GComm.AdapterConfigurationError,  tmsg
        #
        #   path parameter
        #
        try:
          path = params['path']
        except KeyError:
          path = '/'

        #
        #   port parameter
        #
        try:
          port = params['port']          
        except KeyError:
          port = 8765
        
          
        #
        #   host parameter
        #
        try:
          host = params['host']
        except KeyError:
          host = 'localhost'

          
      try:
        self._url = "%s://%s:%s%s" % \
                    ( transport, params['host'], port,path )
      except:
        pass
        
      # setLoglevel
      if hasattr(params,'loglevel'):
        pass
      
      # care for non-integer port settings
      if type(port)!=type(12):
        port=string.atoi(port)

      self.pyro_group=':GComm'

      # initialize the client and set the default namespace group
      Pyro.core.initClient()
      Pyro.config.PYRO_NS_DEFAULTGROUP=self.pyro_group

      locator = Pyro.naming.NameServerLocator()
      print 'Searching Naming Service...',
      self.ns = locator.getNS()
      print 'Naming Service found at',self.ns.URI.address,'('+(Pyro.protocol.getHostname(self.ns.URI.address) \
                                                               or '??')+') port',self.ns.URI.port
    except KeyError:
      tmsg = _("To use XML-RPC, you must either specify a 'url' or\n") +\
             _("a 'host' and 'port' setting.")
      raise GComm.AdapterConfigurationError,  tmsg
    except:
      tmsg = _("Unable to initialize the XML-RPC interface at %s") % url
      raise GComm.AdapterInitializationError, tmsg


  def request(self, service, params={}):
    # TODO: add support for more than one baseproxy
    # resolve the Pyro object
    self._baseproxy = _ProxyObject(self, service)
    return self._baseproxy
    print 'binding to object',service
    try:
      URI=self.ns.resolve(service+'.self')
      print 'URI:',URI
    except Pyro.core.PyroError,x:
      print 'Couldn\'t bind object (%s), nameserver says: %s' % (service,x)
      raise SystemExit
    
    obj = Pyro.core.getProxyForURI(URI)
    return obj
    #obj = Pyro.core.getAttrProxyForURI(URI)
    self._baseproxy = _ProxyObject(self, service)
    return self._baseproxy


  def errorHandler(self,src, exc):
    # doesn't work correct, so is being disabled
    if exc[1]==(111, 'Connection refused'):
      # cannot raise a new error, so I try to change it
      exc=(GComm.AdapterInitializationError,
           _("Unable to initialize the XML-RPC interface at %s: ")\
           % src.getDesc() +  _("Connection refused"), None )
    print "Connection to ",src.getDesc()," failed."
    print exc
    # sys.exit(0)
    #import traceback
    #sys.stderr.write()
    #traceback.print_exception(exc[0], exc[1], exc[2])
    #return xmlrpc.ONERR_KEEP_WORK
    return xmlrpc.ONERR_KEEP_DEF
  
  def close(self):
    # do a cleanup of the proxy objects and the object stubs
    # on the XMLRPC server side

    # 1. close the proxy objects for services (static, client only)
    self._baseproxy=None    
    
    # 2. close the proxy objects for objects (dynamic, client+server)
    for i in self._objectProxys:
      i._close()      
    self._objectProxys=[] 
    
    # close the client it self
    self._client=None    

  # just in case the user forgot to delete some objects
  def __del__(self):
    if self._client!=None:
      self.close()
    GCommBase.Client.__del__()

  def setTimeout(self,timeout):
    self._timeout=timeout
    
  def setAuthentification(self,name,passw):
    self._auth_name=name
    self._auth_passw=passw


  def runMethod(self, method, *args, **params):
    # print "Calling method: %s with attr %s " % method,string.join(args,",")
    
    # check for special parameters
    i=0
    while i<len(args):
      # care for special types
      if hasattr(args[i],'_type'):
        if args[i]._type=='base64':
          args[i]=xmlrpc.base64(args[i]._value)
        elif args[i]._type=='binary':
          args[i]=xmlrpc.base64(args[i]._value)
        elif args[i]._type=='boolean':
          args[i]=xmlrpc.boolean(args[i]._value)
        elif args[i]._type=='datetime':
          args[i]=xmlrpc.datetime(args[i]._value)
        else:
          if hasattr(args[i],'_value'):
            args[i]=args[i]._value
      i=i+1

            
    # call the method
    if 1: #try:
          print 'binding to object',method
          try:
            URI=self.ns.resolve(method)
            print 'URI:',URI
          except Pyro.core.PyroError,x:
            print 'Couldn\'t bind object for method %s, nameserver says: %s' % (method,x)
            raise SystemExit
    
          obj = Pyro.core.getAttrProxyForURI(URI)
          print obj
          result=obj(args)

    if 0: #except:      
      msg="Error: %s" % (sys.exc_value,)
      
      print 
      print "********************************************\n"
      print msg,"\n\n"
      print "********************************************"
      sys.exit(0)
           
    return result




##############################################################################
#
# ProxyObject
#
class _ProxyObject(GCommBase.ProxyObject):

  def __call__(self, *args, **params):
    if not hasattr(self,'_parent'):
      return
    if not hasattr(self._parent,'_pyrobind'):
      try:
        URI=self._adapter.ns.resolve(string.join(self._parent._attrPath,'.')+'.self')
      except Pyro.core.PyroError,x:
        print 'Couldn\'t bind object %s, nameserver says: %s' % (string.join(self._parent._attrPath,'.')+'.self',x)
        raise SystemExit      
      self._parent._pyrobind = Pyro.core.getProxyForURI(URI)
    
    return getattr(self._parent._pyrobind, self._attr)(*args, **params)

  # TODO: Create new getattr/setattr commands

  def _close(self):
    # if it is a dynamic object then close it
    if self._attrPath==[self._attr]:
      self._adapter.runMethod(self._attr+'._close')
