#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# GParser.py
#
# DESCRIPTION:
# Class that contains a SAX2-based XML processor for GNUe
#
# NOTES:
#

import sys, copy, types
from gnue.common.definitions.GObjects import GObj
from gnue.common.definitions.GRootObj import GRootObj
from gnue.common.logic.GTrigger import GTrigger
from gnue.common.utils.FileUtils import openResource

try:
  from xml.sax import saxutils
  import xml.sax
except ImportError:
  print """
   This GNUe tool requires pythons XML module be installed.
   Typically this is the case, however some GNU/Linux distro's
   like Debian distribute this as a seperate package

   To install this package...
     On Debian: apt-get install python-xml

"""


import string
from gnue.common.apps import GDebug
from gnue.common.formatting import GTypecast
from gnue.common.definitions.GParserHelpers import GContent


#######################################################
#
# Error classed raised for markup errors
#
class MarkupError(StandardError):
  pass


#######################################################
#
# loadXMLObject
#
# This method loads an object from an XML file and
# returns that object.  If initialize is 1 (default),
# then the object is initialized and ready to go.
# Setting initialize to 0 is useful for a design
# environment where the object is not actually
# being used, but simply loaded as a document.
#
# "attributes" is a dictionary containing extra
# attributes that should be attached to this object.
#
#  e.g., if attributes={myproperty:[0,1,2]}, then
#    before the object is initialized or returned,
#    object.myproperty == [0,1,2].
#
#######################################################

def loadXMLObject(stream, handler, rootType, xmlFileType,
  initialize=1, attributes={}, initParameters={}):

  # Create a parser
  parser = xml.sax.make_parser()

  # Set up some namespace-related stuff for the parsers
  parser.setFeature(xml.sax.handler.feature_namespaces, 1)

  # Allow for parameter external entities
  ## Does not work with expat!!! ##
  ##parser.setFeature(xml.sax.handler.feature_external_pes, 1)

  # Create a stack for the parsing routine
  object = None

  # Create the handler
  dh = handler()
  dh.initValidation()

  # Tell the parser to use our handler
  parser.setContentHandler(dh)  

  parser.parse(stream)

  object = dh.getRoot()

  if not object:
    tmsg = _("Error loading %s: empty definition file") % (xmlFileType)
    raise MarkupError, tmsg
  elif object._type != rootType:
    tmsg = _("Error loading %s: not a valid %s definition (expected: %s, got: %s)") % \
        (xmlFileType, xmlFileType, rootType, object._type) 
    raise MarkupError, tmsg

  dh.finalValidation()

  # Set the root object's attributes
  #
  # There should only be 1 root object but GNUe Forms
  # allows for nested forms so we have to walk the tree
  #
  #
  #object.__dict__.update(attributes)

  object.walk(addAttributesWalker,attributes=attributes)

  if initialize:
    GDebug.printMesg(10,"Initializing the object tree starting at %s" %(object))
    object.phaseInit(dh._phaseInitCount)



  return object
#######################################################
#
# addAttributesWalker
#
#######################################################
def addAttributesWalker(object, attributes={}):
  if isinstance(object,GRootObj):
    object.__dict__.update(attributes)



#######################################################
#
# normalise_whitespace
#
# Remove redundant whitespace from a string ( from xml-howto )
#
#######################################################
def normalise_whitespace(text):
  return string.join( string.split(text), ' ')


def default(attrs, key, default):
  try:
    return attrs[key]
  except KeyError:
    return default


#######################################################
#
# xmlHandler
#
# This class is called by the XML parser to
# process the xml file.
#
#######################################################
class xmlHandler(xml.sax.ContentHandler):
  def __init__(self):

    self.xmlElements = {}
    self.xmlMasqueradeNamespaceElements = None
    self.xmlNamespaceAttributesAsPrefixes = 0

    self.xmlStack = []
    self.nameStack = []
    self.bootstrapflag = 0
    self.uniqueIDs = {}
    self.root = None
    self._phaseInitCount = 0

    self._requiredTags = []
    self._singleInstanceTags = []
    self._tagCounts = {}

  #
  # Called by client code to get the "root" node
  #
  def getRoot(self):
    return self.root

  #
  # Builds structures need to verify requirements in the file
  #
  def initValidation(self):
    #
    # Build list of tags along with a list of
    # require tags
    #
    for element in self.xmlElements.keys():
      self._tagCounts[element] = 0
      try:
        if self.xmlElements[element]['Required'] == 1:
          self._requiredTags.append(element)
      except KeyError:
        pass

      try:
        if self.xmlElements[element]['SingleInstance'] == 1:
          self._singleInstanceTags.append(element)
      except KeyError:
        pass

  def finalValidation(self):
    # TODO: too simple a validation need to be per object instance
    #for element in self._singleInstanceTags:
    #  if self._tagCounts[element] > 1:
    #    raise MarkupError, _("File has multiple instances of <%s> when only one allowed") % (element)

    for element in self._requiredTags:
      if self._tagCounts[element] < 1:
        tmsg = _("File is missing required tag <%s>") % (element)
        raise MarkupError, tmsg


  #
  # Called by the internal SAX parser whenever
  # a starting XML element/tag is encountered.
  #
  def startElementNS(self, qtag, qname, saxattrs):

    ns, name = qtag
    attrs = {}
    loadedxmlattrs = {}

    if not ns:
      #
      # No namespace qualifier
      #
      GDebug.printMesg(50, "<%s>" % name)

      try:
        baseAttrs = self.xmlElements[name].get('Attributes',{})
      except KeyError:
        tmsg = _("Error processing <%s> tag [I do not know what a <%s> tag does]") \
                % (name, name)
        raise MarkupError, tmsg

      xmlns = {}

      for qattr in saxattrs.keys():
        attrns, attr = qattr

        if attrns:
          if not self.xmlNamespaceAttributesAsPrefixes:
            tmsg = _("Unexpected namespace on attribute")
            raise MarkupError, tmsg
          prefix = attrns.split(':')[-1]
          attrs[prefix + '__' + attr] = saxattrs[qattr]
          xmlns[prefix] = attrns

        else:

          # Typecasting, anyone?  If attribute should be int, make it an int
          try:            
            attrs[attr] = baseAttrs[attr].get('Typecast',GTypecast.text)(saxattrs[qattr])
            loadedxmlattrs[attr] = attrs[attr]
          except KeyError:
            tmsg = _('Error processing <%s> tag [I do not recognize the "%s" attribute')\
                % (name, attr)
            raise MarkupError, tmsg

          # this error shouldn't occure anymore
          #except UnicodeError:
          #  tmsg = _('Error processing <%s> tag [Encoding error: invalid character in "%s" attribute;]')\
          #      % (name, attr)
          #  raise MarkupError, tmsg

          #except StandardError, msg:
          #  raise
          #  tmsg = _('Error processing <%s> tag [invalid type for "%s" attribute; value is "%s"]')\
          #      % (name, attr, saxattrs[qattr])
          #  raise MarkupError, tmsg

          # If this attribute must be unique, check for duplicates
          if baseAttrs[attr].get('Unique',0): # default (baseAttrs[attr],'Unique',0):
            if self.uniqueIDs.has_key('%s' % (saxattrs[qattr])):
              tmsg = _('Error processing <%s> tag ["%s" attribute should be unique; duplicate value is "%s"]')\
                % (name, attr, saxattrs[qattr]) 
              raise MarkupError, tmsg

      for attr in baseAttrs.keys():
        if not attrs.has_key(attr):

          # Pull default values for missing attributes
          if baseAttrs[attr].has_key ('Default'):
            attrs[attr] = baseAttrs[attr].get('Typecast', GTypecast.text) (baseAttrs[attr]['Default'])# default(baseAttrs[attr],'Typecast', GTypecast.text) (baseAttrs[attr]['Default'])

          # Check for missing required attributes
          elif baseAttrs[attr].get('Required', 0): #default(baseAttrs[attr], 'Required', 0):
            tmsg = _('Error processing <%s> tag [required attribute "%s" not present]')\
                % (name, attr)
            raise MarkupError, tmsg

      attrs['_xmlnamespaces'] = xmlns

      if self.bootstrapflag:
        if self.xmlStack[0] != None:
          object = self.xmlElements[name]['BaseClass'](self.xmlStack[0])
      else:
        object = self.xmlElements[name]['BaseClass']()
        self.root = object
        self.bootstrapflag = 1

      self._tagCounts[name] += 1

      object._xmltag = name

    elif self.xmlMasqueradeNamespaceElements:
      #
      # namespace qualifier and we are masquerading
      #

      GDebug.printMesg(50, "<%s:%s>" % (ns,name))

      for qattr in saxattrs.keys():
        attrns, attr = qattr

        attrs[attr] = saxattrs[qattr]
        loadedxmlattrs[attr] = saxattrs[qattr]

      try:
        object = self.xmlMasqueradeNamespaceElements(self.xmlStack[0])
      except IndexError:
        tmsg = _("Error processing <%s:%s> tag: root element needs to be in default namespace") %\
               (ns,name)
        raise MarkupError, tmsg
	    
      object._xmltag = name
      object._xmlnamespace = ns
      object._listedAttributes = loadedxmlattrs.keys()

    else:
      #
      # namespace qualifier and we are not masquerading
      #
      print _("WARNING: Markup includes namespaces, but the current tool does not include namespace support!")
      sys.exit()


    # Save the attributes loaded from XML file
    # (i.e., attributes that were not defaulted)
    object._loadedxmlattrs = loadedxmlattrs

    # Set the attributes
    object.__dict__.update(attrs)

    self.xmlStack.insert(0, object)
    self.nameStack.insert(0, name)


  #
  # Called by the internal SAX parser whenever
  # text (not part of a tag) is encountered.
  #
  def characters(self, text):

    if self.xmlStack[0] != None:

      # Masqueraging namespace elements, then keep content
      xmlns = self.xmlMasqueradeNamespaceElements and \
          isinstance(self.xmlStack[0],self.xmlMasqueradeNamespaceElements)

      # Should we keep the text?
      if xmlns or self.xmlElements[self.nameStack[0]].get('MixedContent',0):

        if xmlns or self.xmlElements[self.nameStack[0]].get('KeepWhitespace',0):
          GContent(self.xmlStack[0], text)
        else:
          # Normalize
          if len(string.replace(string.replace(string.replace(text,' ',''),'\n',''),'\t','')):
            text = normalise_whitespace (text)
          else:
            text = ""
          if len(text):
            GContent(self.xmlStack[0], text)


  #
  # Called by the internal SAX parser whenever
  # an ending XML tag/element is encountered.
  #
  def endElementNS(self, qtag, qname):
    ns, name = qtag
    self.nameStack.pop(0)
    child = self.xmlStack.pop(0)
    inits = child._buildObject()
    self._phaseInitCount = (inits != None and inits > self._phaseInitCount \
                            and inits or self._phaseInitCount)
    GDebug.printMesg(50, "</%s>" % name)


class GImportItem(GObj):
  def __init__(self, parent=None, type="GCImport-Item"):
    GObj.__init__(self, parent, type=type)
    self._loadedxmlattrs = {} # Set by parser
    self._inits = [self.primaryInit]
    self._xmlParser = self.findParentOfType(None)._xmlParser

  def _buildObject(self):
    if hasattr(self,'_xmltag'):
      self._type = 'GC%s' % self._xmltag
    if not hasattr(self,'_importclass'):
      self._importclass = self._xmlParser\
         .getXMLelements()[string.lower(self._type[9:])]['BaseClass']
    return GObj._buildObject(self)

  def primaryInit(self):
     #
     # Open the library and convert it into objects
     #
     handle = openResource(self.library)
     form = self._xmlParser.loadFile(handle, self.findParentOfType(None)._app, initialize=0)
     handle.close()
     id = 'id'
     if hasattr(self,'name'):
         id = 'name'
     #
     # Configure the imported object, assign as a child of self
     #
     rv = self.__findImportItem(self, form, id)
     if rv != None:
       rv._parent = self
       rv._IMPORTED = 1
       self._children.append(rv)
       #
       # transfer attributes reassigned during the import
       #
       for key in self._loadedxmlattrs.keys():
         if key[0] != '_':
           rv.__dict__[key] = self._loadedxmlattrs[key]
           GDebug.printMesg (5, ">>> Moving %s" % key)
       rv._buildObject()
     else:
         tmsg =  _("Unable to find an importable object named %s in %s") \
            % (self.name, self.library)
         raise MarkupError, tmsg

  #
  # __findImportItem
  #
  # finds the item in the object tree with the
  # same name and instance type
  #
  def __findImportItem(self, find, object, id):
     if isinstance(object, find._importclass) and \
        hasattr(object, id) and \
        object.__dict__[id] == find.__dict__[id]:
       return object
     elif hasattr(object,'_children'):
       rv = None
       for child in object._children:
         rv = self.__findImportItem(find, child, id)
         if rv:
           break
       return rv
     else:
       return None


class GImport(GObj):
  def __init__(self, parent=None):
    GObj.__init__(self, parent, type="GCImport")
    self.library = ""
    self._form = None
    self._inits = [self.primaryInit]
    self._xmlParser = self.findParentOfType(None)._xmlParser

  def primaryInit(self):
    handle = openResource(self.library)
    form = self._xmlParser.loadFile(handle, self.findParentOfType(None)._app, initialize=0)
    handle.close()

    for attribute in self._loadedxmlattrs.keys():
      if attribute != 'library':
        importAll =  self._loadedxmlattrs[attribute] == "*"
        importNames = string.split(string.replace(self._loadedxmlattrs[attribute],' ',''),',')

        instanceType = self._xmlParser.getXMLelements()[string.lower(attribute)]['BaseClass']

        if importAll or len(importNames):
          for child in form._children:
            if isinstance(child,instanceType) and \
               (importAll or child.name in importNames):
              child._parent = self
              child._IMPORTED = 1
              self._children.append(child)
              child._buildObject()

def buildImportableTags(rootTag, elements):
    #
    # Scans xml elements and looks for Importable = 1
    # Items with this set can be imported
    # If an object needs to be importable,
    # simply add its tag name to the tuple below
    # and make sure it has a "name" attribute
    # (otherwise we don't know how to reference
    # it in the imported file).
    #
    importElement = {'BaseClass': GImport,
                     'Attributes': {'library': {
                                      'Required': 1,
                                      'Typecast': GTypecast.name },
                                   },
                     'ParentTags': rootTag,
                     }

    for key in elements.keys():
     if elements[key].has_key('Importable') and elements[key]['Importable']:
       name = "import-%s" % key
       copy._deepcopy_dispatch[types.FunctionType] = copy._deepcopy_atomic
       copy._deepcopy_dispatch[types.ClassType] = copy._deepcopy_atomic
       copy._deepcopy_dispatch[type(int)] = copy._deepcopy_atomic

       p = copy.deepcopy(elements[key])
       p['BaseClass'] = GImportItem

       if not p.has_key('Attributes'):
         p['Attributes'] = {}

       p['Attributes']['library'] = {
          'Required': 1,
          'Typecast': GTypecast.name }
       p['MixedContent'] = 0
       p['Required'] = 0
       elements[name] = p

       importElement['Attributes'][key] =  {
         'Typecast': GTypecast.name,
         'Default': ""  }

    if len(importElement['Attributes'].keys()):
      elements['import'] = importElement
    return elements
