#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# interbase/DBdriver.py
#
# DESCRIPTION:
# Driver to provide access to data via the Kinterbasdb Interbase/Firebird Python Driver
# Requires Kinterbasdb > 3.0 (http://kinterbasdb.sourceforge.net/)
#
# NOTES:
#
#   Supported attributes (via connections.conf or <database> tag)
#
#     host=      This is the Interbase host for your connection  (required)
#     dbame=      This is the Interbase database to use (required)
#

__all__ = ['Connection']

####                                   ####
#### IF YOU MODIFY ANY CONNECTION      ####
#### ATTRIBUTES, PLEASE UPDATE info.py ####
####                                   ####

from string import upper, lower, rstrip
import sys
from gnue.common.datasources import GDataObjects, GConditions, GConnections
from gnue.common.apps import GDebug
from gnue.common.datasources.drivers import DBSIG2
from DataObject import *
from gnue.common.datasources.drivers.interbase.Schema.Discovery.Introspection import Introspection


try:
  import kinterbasdb as SIG2api
except ImportError:
  raise GConnections.DependencyError, ('Kinterbasdb', None)


######################################################################
#
#  GConnection object for Interbase drivers
#
class Connection(DBSIG2.Connection):

  defaultBehavior = Introspection
  _driver = SIG2api
  _DatabaseError = SIG2api.DatabaseError
  supportedDataObjects = {
    'object': DataObject_Object,
    'sql':    DataObject_SQL
  }
  # The date/time format used in insert/select statements
  # (based on format used for time.strftime())
  _dateTimeFormat = "cast('%Y-%m-%d %H:%M:%S' as timestamp)"

  def connect(self, connectData={}):
    GDebug.printMesg(1,"Interbase database driver initializing")

    try:
      ib_encoding = ib_encTable[self._encoding]
    except KeyError:
      GDebug.printMesg(1,_("Encoding '%s' is not supported by interbase dbdriver.") % \
                       self._encoding + _('Using default encoding.'))
      ib_encoding = ''

    if ib_encoding:
      GDebug.printMesg(1,'Setting interbase client_encoding to %s (%s)' % (ib_encoding,self._encoding))

    try:
      self.native = SIG2api.connect( \
                 user=str(connectData['_username']), \
                 password=str(connectData['_password']), \
                 charset=ib_encoding, \
                 database=connectData['dbname'], \
                 host=connectData['host'])
    except self._DatabaseError, value:
      raise GDataObjects.LoginError, value


  # Return the current date, according to database
  def getTimeStamp(self):
    return self.__singleQuery("select cast('now' as date) from rdb$database")

  # Return a sequence number from sequence 'name'
  def getSequence(self, name):
    return self.__singleQuery("select gen_id(%s,1) from rdb$database" % name)

  # Used internally
  def __singleQuery(self, statement):
    cursor = self.native.cursor()
    try:
      cursor.execute(statement)
      rv = cursor.fetchone()
      cursor.close()
    except mesg:
      GDebug.printMesg(1,"**** Unable to execute extension query")
      GDebug.printMesg(1,"**** %s" % mesg)
      cursor.close()
      return None

    try:
      return rv[0]
    except:
      return None

# RDB$CHARACTER_SETS.RDB$CHARACTER_SET_NAME
ib_encTable =  {'ascii'     :  'ASCII',
                ''          :  'BIG_5',
                ''          :  'CYRL',
                'cp437'     :  'DOS437',
                'cp850'     :  'DOS850',
                'cp852'     :  'DOS852',
                'cp857'     :  'DOS857',
                'cp860'     :  'DOS860',
                'cp861'     :  'DOS861',
                'cp863'     :  'DOS863',
                'cp865'     :  'DOS865',
                ''          :  'EUCJ_0208',
                ''          :  'GB_2312',
                'iso8859-1' :  'ISO8859_1',
                'iso8859-2' :  'ISO8859_2',
                'iso8859-3' :  'ISO8859_3',
                'iso8859-4' :  'ISO8859_4',
                'iso8859-5' :  'ISO8859_5',
                'iso8859-6' :  'ISO8859_6',
                'iso8859-7' :  'ISO8859_7',
                'iso8859-8' :  'ISO8859_8',
                'iso8859-9' :  'ISO8859_9',
                'iso8859-13':  'ISO8859_13',
                ''          :  'KSC_5601',
                ''          :  'NEXT',
                ''          :  'NONE',
                ''          :  'OCTETS',
                ''          :  'SJIS_0208',
                'utf-8'     :  'UNICODE_FSS',
                'cp1250'    :  'WIN1250',
                'cp1251'    :  'WIN1251',
                'cp1252'    :  'WIN1252',
                'cp1253'    :  'WIN1253',
                'cp1254'    :  'WIN1254',
                'cp1255'    :  'WIN1255',
                'cp1256'    :  'WIN1256',
                'cp1257'    :  'WIN1257'}
