#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# GDataSource.py
#
# DESCRIPTION:
# Class that implements a provider-independent DataSource object
#
# NOTES:
#
# HISTORY:
#

from gnue.common.apps import GDebug
from gnue.common.datasources import GDataObjects
from gnue.common.definitions import GObjects
import sys, string, types
from gnue.common.datasources import GConnections
from gnue.common.formatting import GTypecast
from gnue.common.datasources import GConditions

########################################################################
#
# Class that handles DataSources.  This is a subclass of GObj, which
# means this class can be created from XML markup and stored in an
# Object tree (e.g., a Forms tree).
#
########################################################################

class GDataSource(GObjects.GObj):

  def __init__(self, parent=None, type="GDataSource"):
    GObjects.GObj.__init__(self, parent, type)
    self.type = "object"
    self.connection = None
    self._connections = None
    self._dataObject = None
    self._connectionComment = ""
    self._fieldReferences = {}
    self._unboundFieldReferences = {}
    self._defaultValues = {}

    self._inits =[self.primaryInit, self.secondaryInit, self.tertiaryInit]
    self._currentResultSet = None
    self._resultSetListeners = []
    self._toplevelParent = None # Needs to be set by subclass
                                      # so that _topObject gets set
    self._topObject = None

    #
    # trigger support
    #
    self._triggerGlobal = 1
    self._triggerFunctions = {'createResultSet':{'function':self.createResultSet,
                                                 },
                              'simpleQuery':{'function':self.triggerSimpleQuery,
                                             },
                              'delete':{'function':self.deleteCurrentRecordsetEntry
                                        },
                              'call':{'function':self.callFuncOfCurrentRecordsetEntry
                                        },
                              'getCondition':{'function':self.getCondition},
                              'setCondition':{'function':self.setCondition},
                              }

    self._triggerProperties = {'extensions':{'get':self.getExtensions,
                                           'direct':1}}

  def __getattr__(self, attr):
    if self._dataObject and attr[1] != '_' and hasattr(self._dataObject,attr):
      return getattr(self._dataObject,attr)
    else:
      raise AttributeError, attr

  def _buildObject(self):

    # Added 0.5.0 -- Delete before 1.0
    if hasattr(self,'database'):
      self.connection = self.database
      del self.database

    try:
      for field in string.split(self.explicitfields,','):
        self._fieldReferences[field] = 1
    except AttributeError:
      pass
    try:
      # TODO: This might be deprecated;
      # TODO: should be using <sorting> tags
      for field in string.split(self.order_by,','):
        if field[-5:].lower() == ' desc':
          field = field[:-5]
        self._fieldReferences[field] = 1
    except AttributeError:
      pass
    return GObjects.GObj._buildObject(self)


  def triggerSimpleQuery(self,maskDict):
    queryDict = {}
    okToProcess = 1
    for key in maskDict.keys():
      queryDict[key]= str(maskDict[key])
      if not len(queryDict[key]):
        okToProcess = 0
        break

    conditions = GConditions.buildConditionFromDict(queryDict,GConditions.GClike)
    resultSet = self.createResultSet(conditions)
    recordCount = resultSet.getRecordCount()

    returnList = []
    for count in range(recordCount):
      record = resultSet.getRecord(count)
      resultDict = {}
      for key in record._fields.keys():
        resultDict[key]=record.getField(key) or ""
      returnList.append(resultDict)
    return returnList

  def deleteCurrentRecordsetEntry(self):
    self._currentResultSet.getPostingRecordset().delete()

  def callFuncOfCurrentRecordsetEntry(self,name,params):
    n=self._currentResultSet.getRecordNumber()
    rset=self._currentResultSet.getRecord(n)
    if hasattr(rset,'callFunc'):
      return rset.callFunc(name,params)
    else:
      tmsg = _("Backend doesn't support the trigger 'call' function")
      raise StandardError, tmsg

  #
  # get/set the static condition assosiated with a datasource
  # the static condition is build out of the <condition> child
  # elements below a datasource XML definition
  #
  def setCondition(self, mycondition):
    dataObject._staticCondition = mycondition
#   dataObject.invalidateCachedConditions()

  def getCondition(self):
    return dataObject._staticCondition

  #
  # get the dbdriver extension object
  #
  def getExtensions(self):
    return self.extensions

  #
  # This method should be called after the object is created
  # but before any other methods are called
  #
  def setConnectionManager(self, connectionManager):
    self._connections = connectionManager

  def initialize(self):
    if not self.connection:
      # We are a connectionless datasource (virtual?)
      # We have to bind to something, so bind to empty or static driver
      if not self.type=="static":
        from gnue.common.datasources.drivers.special.unbound import Driver
        GDebug.printMesg (7, 'Using empty data driver')
        dataObject = Driver.supportedDataObjects['object'](None)

      else:
        from gnue.common.datasources.drivers.special.static import Connection
        GDebug.printMesg (7, 'Using static data driver')
        dataObject = Connection.supportedDataObjects['object'](None)

        for child in self._children:
          if isinstance(child, GStaticSet):
            dataObject._staticSet = child
            break


    elif self._connections:
      self.connection = string.lower(self.connection)
      # This will throw a GConnections.NotFoundError if an unknown
      # connection name is specified.  The calling method should
      # catch this exception and handle it properly (exit w/message)
      dataObject = \
         self._connections.getDataObject(self.connection, self.type)
      GDebug.printMesg (7, "GDataSource.py bound to %s " % self._dataObject)

    self.name = string.lower(self.name)
    self._topObject._datasourceDictionary[self.name]=self

    dataObject._fieldReferences = self._fieldReferences
    dataObject._unboundFieldReferences = self._unboundFieldReferences
    dataObject._defaultValues = self._defaultValues
    dataObject._dataSource = self

    # TODO: Short-term hack to allow primary key support
    try:
      dataObject._primaryIdField = self.primarykey
      dataObject._primaryIdFormat = "%s = '%%s'" % self.primarykey
      dataObject._primaryIdChecked = 1
    except AttributeError:
      pass

    for child in self._children:
      if isinstance(child, GConditions.GCondition):
        dataObject._staticCondition = child
        break

    # Copy all attributes from XML to the dataObject
    tagAttributes = getXMLelements()['datasource']['Attributes']
    for attribute in tagAttributes.keys():
      if self.__dict__.has_key(attribute):
        dataObject.__dict__[attribute] = self.__dict__[attribute]
      else:
        try:
          dataObject.__dict__[attribute] = tagAttributes[attribute]['Default']
        except KeyError:
          pass
    self._dataObject = dataObject


  def connect(self):
    if self.connection != None:
      self._connections.requestConnection(self._dataObject)


  def getDataObject(self):
    return self._dataObject


  def referenceField(self, field, defaultValue=None):
    GDebug.printMesg(7,'Field %s implicitly referenced' % field)
    self._fieldReferences[field] = ""

    if defaultValue != None:
      self._defaultValues[field] = defaultValue

  def referenceFields(self, fields):
    for field in fields:
      if isinstance (field, types.StringType) or \
         isinstance (field, types.UnicodeType):
        self.referenceField(field)
      else:
        self.referenceField(*field)

  def referenceUnboundField(self, field, defaultValue=None):
    GDebug.printMesg(7,'Field %s implicitly referenced' % field)
    self._unboundFieldReferences[field] = 1

    if defaultValue != None:
      self._defaultValues[field] = defaultValue


  #
  # The following are a simple wrapper around the datasource's dataobject
  # to hide the dataobject from the app programmer
  #
  def hasMaster(self):
    return self._dataObject != None and self._dataObject.hasMaster()

  def createResultSet(self, conditions={}, readOnly=0, sql=""):
      resultSet= self._dataObject.createResultSet(conditions,readOnly,sql=sql)
      self.__setResultSet( resultSet )
      return resultSet

  def addDetailDataObject(self, dataObject, handler=None):
      self._dataObject.addDetailDataObject(dataObject, handler)

  def createEmptyResultSet(self, readOnly=0,masterRecordSet=None):
      resultSet = self._dataObject.createEmptyResultSet(readOnly, masterRecordSet=masterRecordSet)
      self.__setResultSet( resultSet )
      return resultSet

  def getQueryString(self,conditions={},forDetailSQL=None,additionalSQL=""):
    return self._dataObject.getQueryString(conditions, forDetailSQL,additionalSQL)

  #
  # Master/detail stuff
  #

  # Called by dbdrivers whenever this datasource's master has changed
  def masterResultSetChanged(self, masterResultSet, detailResultSet):
    self._masterResultSet = masterResultSet
    self.__setResultSet( detailResultSet )

  def __setResultSet(self, resultSet):
    self._currentResultSet = resultSet
    # Notify all the listeners (i.e., blocks) that the result set changed
    for listener in self._resultSetListeners:
      listener(resultSet)

  def registerResultSetListener(self, listener):
    self._resultSetListeners.append(listener)

  def primaryInit(self):
    self._topObject = self.findParentOfType(self._toplevelParent)
    GDebug.printMesg(10,"Setting %s to connect mgr %s" %(self.name,self._topObject._connections))
    self.setConnectionManager(self._topObject._connections)
    self.initialize()
    self.connect()
    self.extensions = self._dataObject.triggerExtensions    

  # TODO: Merged into GDataSource per the TODOs in reports and forms however
  # TODO: self._topObject._datasourceDictionary implies that the top object
  # TODO: always has a specifc structure.  This is a bad thing :(  Maybe GRootObj
  # TODO: should contain a getDatasourceDict()?
  #
  def secondaryInit(self):

    if hasattr(self, 'master') and self.master:

      self.master = string.lower(self.master)
      GDebug.printMesg(3,"Linking detail '%s' to master '%s'" \
                         % (self.name, self.master) )

      if self._topObject._datasourceDictionary.has_key(self.master):
        self._topObject._datasourceDictionary[self.master] \
            .getDataObject().addDetailDataObject(self.getDataObject(),
                                                 self)
      else:
        tmsg = _("Detail source '%s' references non-existant master '%s'") \
                         % (self.name, self.master)
        raise StandardError, tmsg

  def tertiaryInit(self):
    if hasattr(self, 'prequery'):
      if not self.hasMaster() and self.prequery:
        self.createResultSet()

  #
  # Hooks for record-level triggers
  #

  def _beforeCommitInsert(self, record):
    return 1

  def _beforeCommitUpdate(self, record):
    return 1

  def _beforeCommitDelete(self, record):
    return 1

  def _onModification(self, record):
    return 1

  def _onRecordLoaded(self, record):
    return 1



######
#
#
#
######
class GSql(GObjects.GObj):
  def __init__(self, parent=None):
     GObjects.GObj.__init__(self, parent, type="GDSql")

######
#
# Static Datasource Support
#
######
class GStaticSet(GObjects.GObj):
  def __init__(self, parent=None):
     GObjects.GObj.__init__(self, parent, type="GDStaticSet")

class GStaticSetRow(GObjects.GObj):
  def __init__(self, parent=None):
     GObjects.GObj.__init__(self, parent, type="GDStaticSetRow")

class GStaticSetField(GObjects.GObj):
  def __init__(self, parent=None):
     GObjects.GObj.__init__(self, parent, type="GDStaticSetField")


######
#
#
#
######
class GConnection(GObjects.GObj):
  def __init__(self, parent=None):
    GObjects.GObj.__init__(self, parent, "GCConnection")
    self.comment = ""
    self.name = ""
    self._inits =[self.initialize]

  def _buildObject(self):
    self.name = string.lower(self.name)
    return GObjects.GObj._buildObject(self)

  def initialize(self):
    # Add our database connection information to the connections
    # manager, then let it handle everything from there.
    root = self.findParentOfType(None)
    root._instance.connections.\
        addConnectionSpecification(self.name, {
           'name': self.name,
           'provider': self.provider,
           'dbname': self.dbname,
           'host': self.host } )



######
#
# Used by client GParsers to automatically pull supported xml tags
#
######

#
# Return any XML elements associated with
# GDataSources.  Bases is a dictionary of tags
# whose values are update dictionaries.
# For example: bases={'datasource': {'BaseClass':myDataSource}}
# sets xmlElements['datasource']['BaseClass'] = myDataSource
#
def getXMLelements(updates={}):

  xmlElements = {
      'datasource': {
         'BaseClass': GDataSource,
         'Importable': 1,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Unique':   1,
               'Typecast': GTypecast.name },
            'type':        {
               'Typecast': GTypecast.name,
               'Default':  "object" },
            'connection':    {
               'Typecast': GTypecast.name },
            'database':    {
               'Typecast': GTypecast.name, 
               'Deprecated': 'Use {connection} attribute instead' },
            'table':       {
               'Typecast': GTypecast.name },
            'cache':       {
               'Typecast': GTypecast.whole,
               'Default':  5 },
            'prequery':    {
               'Typecast': GTypecast.boolean,
               'Default':  0 },
            'distinct':    {
               'Typecast': GTypecast.boolean,
               'Default':  0 },
            'order_by':    {
               'Typecast': GTypecast.text },
            'master':      {
               'Typecast': GTypecast.name },
            'masterlink':  {
               'Typecast': GTypecast.text },
            'detaillink':  {
               'Typecast': GTypecast.text },
            # TODO: Short-term hack
            'explicitfields': {
               'Typecast': GTypecast.text },
            'primarykey': {
               'Typecast': GTypecast.text } },
         'ParentTags': None },
      'staticset': {
         'BaseClass': GStaticSet,
#  TODO: This should be replaced by a SingleInstanceInParentObject
#        instead of SingleInstance (in the whole file)
#         'SingleInstance': 1,
         'Attributes': {
            'fields':        {
               'Typecast': GTypecast.text,
               'Required': 1 } },
         'ParentTags': ('datasource',) },
      'staticsetrow': {
         'BaseClass': GStaticSetRow,
         'ParentTags': ('staticset',) },
      'staticsetfield': {
         'BaseClass': GStaticSetField,
         'Attributes': {
            'name':        {
               'Typecast': GTypecast.text,
               'Required': 1 },
            'value':        {
               'Typecast': GTypecast.text,
               'Required': 1 } },
         'ParentTags': ('staticsetrow',) },
      'connection': {
         'BaseClass': GConnection,
         'Attributes': {
            'name': {
               'Required': 1,
               'Unique': 1,
               'Typecast': GTypecast.name,
               'Description': 'TODO' },
            'provider': {
               'Required': 1,
               'Typecast': GTypecast.name,
               'Description': 'TODO' },
            'dbname': {
               'Required': 0,
               'Typecast': GTypecast.text,
               'Description': 'TODO' },
            'service': {
               'Required': 0,
               'Typecast': GTypecast.text,
               'Description': 'TODO' },
            'comment': {
               'Required': 0,
               'Typecast': GTypecast.text,
               'Description': 'TODO' },
            'host': {
               'Required': 0,
               'Typecast': GTypecast.text,
               'Description': 'TODO' } },
         'ParentTags': None,
         'Description': 'TODO' },
  }

  # Add conditional elements
  xmlElements.update(
      GConditions.getXMLelements(
          {'condition':{'ParentTags':('datasource',) } } ))

  for alteration in updates.keys():
    xmlElements[alteration].update(updates[alteration])

  # Connections will have the same parent as datasources
  xmlElements['connection']['ParentTags'] = xmlElements['datasource']['ParentTags']

  return xmlElements


#
# Wrapper for standalone DataSources
# (i.e., not in context of a GObj tree)
#
def DataSourceWrapper(connections=None, fields=(), attributes={}, init=1, unicodeMode=0):
  source = _DataSourceWrapper()
  
  if connections: 
    source.setConnectionManager(connections)
  
  if init: 
    source.buildAndInitObject(**attributes)
  else: 
    source.buildObject(**attributes)
    
  if fields: 
    source.referenceFields(fields)
  
  return source
  
  
class _DataSourceWrapper(GDataSource): 
  def __init__(self, *args, **parms): 
    GDataSource.__init__(self, *args, **parms)
    self._datasourceDictionary={}
    self._toplevelParent = self._type
    

