/***************************************************************************
                       plotting.cpp  -  GDL routines for plotting
                             -------------------
    begin                : July 22 2002
    copyright            : (C) 2002-2011 by Marc Schellens et al.
    email                : m_schellens@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "includefirst.hpp"

#include <memory>

// PLplot is used for direct graphics
#include <plplot/plstream.h>

#include "initsysvar.hpp"
#include "envt.hpp"
#include "graphics.hpp"
#include "plotting.hpp"
#include "math_utl.hpp"

#ifdef _MSC_VER
#define isfinite _finite
#define isnan _isnan
#endif

namespace lib {

  using namespace std;

  // local helper function
  void GetMinMaxVal( DDoubleGDL* val, double* minVal, double* maxVal)
  {
    DLong minE, maxE;
    const bool omitNaN = true;
    val->MinMax( &minE, &maxE, NULL, NULL, omitNaN);
    if( minVal != NULL) *minVal = (*val)[ minE];
    if( maxVal != NULL) *maxVal = (*val)[ maxE];
  }


  PLFLT AutoTick(DDouble x)
  {
    if( x == 0.0) return 1.0;

    DLong n = static_cast<DLong>( floor(log10(x/3.5)));
    DDouble y = (x / (3.5 * pow(10.,static_cast<double>(n))));
    DLong m;
    if (y >= 1 && y < 2)
      m = 1;
    else if (y >= 2 && y < 5)
      m = 2;
    else if (y >= 5)
      m = 5;

    PLFLT intv = (PLFLT) (m * pow(10.,static_cast<double>(n)));
    return intv;
  }


  PLFLT AutoIntv(DDouble x)
  {
    if( x == 0.0) {
      //      cout << "zero"<<endl;
      return 1.0;
    }

    DLong n = static_cast<DLong>( floor(log10(x/2.82)));
    DDouble y = (x / (2.82 * pow(10.,static_cast<double>(n))));
    DLong m;
    if (y >= 1 && y < 2)
      m = 1;
    else if (y >= 2 && y < 4.47)
      m = 2;
    else if (y >= 4.47)
      m = 5;

    //    cout << "AutoIntv" << x << " " << y << endl;

    PLFLT intv = (PLFLT) (m * pow(10.,static_cast<double>(n)));
    return intv;
  }

  //improved version of "AutoIntv" for:
  // 1/ better managing ranges when all the data have same value
  // 2/ mimic IDL behavior when data are all positive
  // please notice that (val_min, val_max) will be changed
  // and "epsilon" is a coefficient if "extended range" is expected
  PLFLT AutoIntvAC(DDouble &val_min, DDouble &val_max, DLong NoZero, bool log)
  {
    PLFLT intv = 1.;
    int cas = 0 ;
    DDouble x;
    bool debug = false ;
    if (debug) {cout << "init: " <<  val_min << " " << val_max << endl;}

    if (log)
    {
      if (val_min == 0 || val_max == 0) return intv;
      val_min = log10(val_min);
      val_max = log10(val_max);
    }

    // case "all below ABS((MACHAR()).xmin)
    if ((abs(val_min) < 1e-38) && (abs(val_max) < 1e-38))
      {
	val_min=DDouble(-1.);
	val_max=DDouble( 1.);
	intv = (PLFLT) (2.);
	cas = 1 ;
      }

    // case "all values are equal"
    if (cas == 0)
      {
	x=val_max-val_min;
	if (abs(x) < 1e-30) {
	  DDouble val_ref;
          val_ref=val_max;
          if (0.98*val_min < val_ref) { // positive case
            val_max=1.02*val_ref;
            val_min=0.98*val_ref;
          } else {     // negative case
            val_max=0.98*val_ref;
            val_min=1.02*val_ref;
          }
          if (debug) {cout << "Rescale : " << val_min << " " << val_max << endl;}
	}
      }

    // case "all data positive, must start at Zero" (mimic IDL behavior)
    if ((cas == 0) && (val_min >= 0.0) && (NoZero == 0))
      {
	cas = 2 ;
	DDouble resu, val_norm ;
	// we used redundant scale (1.,1.2 and 10., 12. to avoid roundoff problem in log10)
	DDouble levels[12]={1.,1.2,1.5,2.,2.5,3.,4.,5.,6.,8.,10.,12.};
	int nb_levels= 12;

	DLong n = static_cast<DLong>( floor(log10(val_max)));
	DDouble scale= pow(10.,static_cast<double>(n));

	val_norm=val_max/scale;

	resu=levels[0];
	for (int c = 0; c < nb_levels; c++) {
	  if ((val_norm > levels[c]) && (val_norm <= levels[c+1])) resu=levels[c+1] ;
	}
	val_min=0.0;
	val_max=resu*scale;
	intv = (PLFLT)(val_max);
      }

    // general case (only negative OR negative and positive)
    if (cas == 0)
      {
	x=val_max-val_min;
	intv = AutoIntv( x);
	val_max = ceil(val_max/intv) * intv;
	val_min = floor(val_min/intv) * intv;
      }

    if (debug) {cout << "cas: "<< cas << " new range: "<<  val_min << " " << val_max << endl;}

    if (log)
    {
      val_min = pow(10, val_min);
      val_max = pow(10, val_max);
    }

    return intv;
  }

  // !P
  void GetPData( DLong& p_background,
		 DLong& p_noErase, DLong& p_color, DLong& p_psym,
		 DLong& p_linestyle,
		 DFloat& p_symsize, DFloat& p_charsize, DFloat& p_thick,
		 DString& p_title, DString& p_subTitle, DFloat& p_ticklen)
  {
    static DStructGDL* pStruct = SysVar::P();
    static unsigned backgroundTag = pStruct->Desc()->TagIndex( "BACKGROUND");
    static unsigned noEraseTag = pStruct->Desc()->TagIndex( "NOERASE");
    static unsigned colorTag = pStruct->Desc()->TagIndex( "COLOR");
    static unsigned psymTag = pStruct->Desc()->TagIndex( "PSYM");
    static unsigned linestyleTag = pStruct->Desc()->TagIndex( "LINESTYLE");
    static unsigned symsizeTag = pStruct->Desc()->TagIndex( "SYMSIZE");
    static unsigned charsizeTag = pStruct->Desc()->TagIndex( "CHARSIZE");
    static unsigned thickTag = pStruct->Desc()->TagIndex( "THICK");
    static unsigned ticklenTag = pStruct->Desc()->TagIndex( "TICKLEN");
    static unsigned titleTag = pStruct->Desc()->TagIndex( "TITLE");
    static unsigned subTitleTag = pStruct->Desc()->TagIndex( "SUBTITLE");
    p_background =
      (*static_cast<DLongGDL*>( pStruct->GetTag( backgroundTag, 0)))[0];
    p_noErase =
      (*static_cast<DLongGDL*>( pStruct->GetTag( noEraseTag, 0)))[0];
    p_color =
      (*static_cast<DLongGDL*>( pStruct->GetTag( colorTag, 0)))[0];
    p_psym =
      (*static_cast<DLongGDL*>( pStruct->GetTag( psymTag, 0)))[0];
    p_linestyle =
      (*static_cast<DLongGDL*>( pStruct->GetTag( linestyleTag, 0)))[0];
    p_symsize =
      (*static_cast<DFloatGDL*>( pStruct->GetTag( symsizeTag, 0)))[0];
    p_charsize =
      (*static_cast<DFloatGDL*>( pStruct->GetTag( charsizeTag, 0)))[0];
    p_thick =
      (*static_cast<DFloatGDL*>( pStruct->GetTag( thickTag, 0)))[0];
    p_title =
      (*static_cast<DStringGDL*>( pStruct->GetTag( titleTag, 0)))[0];
    p_subTitle =
      (*static_cast<DStringGDL*>( pStruct->GetTag( subTitleTag, 0)))[0];
    p_ticklen =
      (*static_cast<DFloatGDL*>( pStruct->GetTag( ticklenTag, 0)))[0];
  }

  // !X, !Y, !Z
  void GetAxisData( DStructGDL* xStruct,
		    DLong& style, DString& title, DFloat& charSize,
		    DFloat& margin0, DFloat& margin1, DFloat& ticklen)
  {
    static unsigned styleTag = xStruct->Desc()->TagIndex( "STYLE");
    static unsigned marginTag = xStruct->Desc()->TagIndex( "MARGIN");
    static unsigned axisTitleTag = xStruct->Desc()->TagIndex( "TITLE");
    static unsigned axischarsizeTag = xStruct->Desc()->TagIndex( "CHARSIZE");
    static unsigned ticklenTag = xStruct->Desc()->TagIndex( "TICKLEN");
    style =
      (*static_cast<DLongGDL*>( xStruct->GetTag( styleTag, 0)))[0];
    title =
      (*static_cast<DStringGDL*>( xStruct->GetTag( axisTitleTag, 0)))[0];
    charSize =
      (*static_cast<DFloatGDL*>( xStruct->GetTag( axischarsizeTag, 0)))[0];
    margin0 =
      (*static_cast<DFloatGDL*>( xStruct->GetTag( marginTag, 0)))[0];
    margin1 =
      (*static_cast<DFloatGDL*>( xStruct->GetTag( marginTag, 0)))[1];
    ticklen =
      (*static_cast<DFloatGDL*>( xStruct->GetTag( ticklenTag, 0)))[0];
  }

  void GetUserSymSize(EnvT *e,  GDLGStream *a, DDouble& UsymConvX, DDouble& UsymConvY)
  {
    DDouble *scaleX, *scaleY;
    GetSFromPlotStructs(&scaleX, &scaleY);
    // get subpage in mm
    PLFLT scrXL, scrXR, scrYB, scrYT;
    a->gspa( scrXL, scrXR, scrYB, scrYT);
    PLFLT scrX = scrXR-scrXL;
    PLFLT scrY = scrYT-scrYB;
    // get char size in mm (default, actual)
    PLFLT defH, actH;
    a->gchr( defH, actH);
    //get symsize
    static DStructGDL* pStruct = SysVar::P();
    DFloat symsize = (*static_cast<DFloatGDL*>
		      (pStruct->GetTag( pStruct->Desc()->TagIndex("SYMSIZE"), 0)))[0];
    e->AssureFloatScalarKWIfPresent( "SYMSIZE", symsize);
    if( symsize <= 0.0) symsize = 1.0;
    UsymConvX=0.5*symsize*(defH/scrX)/scaleX[1];
    UsymConvY=0.5*symsize*(defH/scrY)/scaleY[1];

  }

  void AdjustAxisOpts(string& xOpt, string& yOpt,
    DLong xStyle, DLong yStyle, DLong xTicks, DLong yTicks,
    string& xTickformat, string& yTickformat, DLong xLog, DLong yLog
  ) // {{{
  {
    if ((xStyle & 8) == 8) xOpt = "b";
    if ((yStyle & 8) == 8) yOpt = "b";

    if (xTicks == 1) xOpt += "t"; else xOpt += "st";
    if (yTicks == 1) yOpt += "tv"; else yOpt += "stv";

    if (xTickformat != "(A1)") xOpt += "n";
    if (yTickformat != "(A1)") yOpt += "n";

    if( xLog) xOpt += "l";
    if( yLog) yOpt += "l";

    if ((xStyle & 4) == 4) xOpt = "";
    if ((yStyle & 4) == 4) yOpt = "";
  } // }}}

  void CheckMargin( EnvT* e, GDLGStream* actStream,
		    DFloat xMarginL,
		    DFloat xMarginR,
		    DFloat yMarginB,
		    DFloat yMarginT,
		    PLFLT& xMR,
		    PLFLT& xML,
		    PLFLT& yMB,
		    PLFLT& yMT)
  {
    // get subpage in mm
    PLFLT scrXL, scrXR, scrYB, scrYT;
    actStream->gspa( scrXL, scrXR, scrYB, scrYT);
    PLFLT scrX = scrXR-scrXL;
    PLFLT scrY = scrYT-scrYB;

    // get char size in mm (default, actual)
    PLFLT defH, actH;
    actStream->gchr( defH, actH);

    xML = xMarginL * actH / scrX;
    xMR = xMarginR * actH / scrX;

    // factor 1.111 by ACoulais on 16/12/2010. Consequences on CONVERT_COORD
    const float yCharExtension = 1.5*1.11111;
    yMB = yMarginB * actH / scrY * yCharExtension;
    yMT = yMarginT * actH / scrY * yCharExtension;

    if( xML+xMR >= 1.0)
      {
	Message( e->GetProName() + ": XMARGIN to large (adjusted).");
	PLFLT xMMult = xML+xMR;
	xML /= xMMult * 1.5;
	xMR /= xMMult * 1.5;
      }
    if( yMB+yMT >= 1.0)
      {
	Message( e->GetProName() + ": YMARGIN to large (adjusted).");
	PLFLT yMMult = yMB+yMT;
	yMB /= yMMult * 1.5;
	yMT /= yMMult * 1.5;
      }
  }
 void Clipping( DDoubleGDL* clippingD,
		 DDouble& xStart,
		 DDouble& xEnd,
		 DDouble& minVal,
		 DDouble& maxVal)
  { //do nothing, clipping should not be done thusly.
  }
 // temporary ignor clipping
//  void Clipping( DDoubleGDL* clippingD,
//		 DDouble& xStart,
//		 DDouble& xEnd,
//		 DDouble& minVal,
//		 DDouble& maxVal)
//  {
//    SizeT cEl=clippingD->N_Elements();
//
//    // world coordinates
//    DDouble wcxs, wcxe,wcys, wcye;
//
//    if(cEl >= 1) wcxs=(*clippingD)[0]; else wcxs=0;
//    if(cEl >= 2) wcys=(*clippingD)[1]; else wcys=0;
//    if(cEl >= 3) wcxe=(*clippingD)[2]; else wcxe=wcxs;
//    if(cEl >= 4) wcye=(*clippingD)[3]; else wcye=wcys;
//
//    if(wcxe < wcxs ) wcxe=wcxs;
//    if(wcye < wcys ) wcye=wcys;
//
//    //     // viewport (0..1)
//    //     DDouble cxs, cxe,cys, cye;
//    //     cxs=(-xStart+wcxs)*(1-0)/(xEnd-xStart);
//    //     cxe=(-xStart+wcxe)*(1-0)/(xEnd-xStart);
//    //     cys=(-yStart+wcys)*(1-0)/(yEnd-yStart);
//    //     cye=(-yStart+wcye)*(1-0)/(yEnd-yStart);
//    //     actStream->vpor(cxs, cxe, cys, cye);
//
//    xStart=wcxs; xEnd=wcxe; minVal=wcys; maxVal=wcye;
//  }
  void setIsoPort(GDLGStream* actStream,
  PLFLT x1,
  PLFLT x2,
  PLFLT y1,
  PLFLT y2,
  PLFLT aspect)
  {
    PLFLT X1, X2, Y1, Y2, X1s, X2s, Y1s, Y2s, displacx,displacy, scalex,scaley,offsetx,offsety;
    if (aspect <= 0.0)
    {
      actStream->vpor(x1, x2, y1, y2);
      return;
    }
    // here we need too compensate for the change of aspect due to eventual !P.MULTI plots
     actStream->vpor(x1, x2, y1, y2); //ask for non-iso window
     actStream->gvpd(X1, X2, Y1, Y2); //get viewport values
     //compute relation desiredViewport-page viewport x=scalex*X+offsetx:
     scalex=(x2-x1)/(X2-X1);
     offsetx=(x1*X2-x2*X1)/(X2-X1);
     scaley=(y2-y1)/(Y2-Y1);
     offsety=(y1*Y2-y2*Y1)/(Y2-Y1);
     //ask for wiewport scaled to isotropic by plplot
     actStream->vpas(x1, x2, y1, y2, aspect);
     //retrieve values
     actStream->gvpd(X1s, X2s, Y1s, Y2s);
     //measure displacement
     displacx=X1s-X1;
     displacy=Y1s-Y1;
     //set wiewport scaled by plplot, displaced, as vpor using above linear transformation
     x1=(X1s-displacx)*scalex+offsetx;
     x2=(X2s-displacx)*scalex+offsetx;
     y1=(Y1s-displacy)*scaley+offsety;
     y2=(Y2s-displacy)*scaley+offsety;
     actStream->vpor(x1, x2, y1, y2);
}

  bool SetVP_WC( EnvT* e,
		 GDLGStream* actStream,
		 DFloatGDL* pos,
		 DDoubleGDL* clippingD,
		 bool xLog, bool yLog,
		 DFloat xMarginL,
		 DFloat xMarginR,
		 DFloat yMarginB,
		 DFloat yMarginT,
		 // input/output
		 DDouble xStart,
		 DDouble xEnd,
		 DDouble yStart,
		 DDouble yEnd,
         DLong iso   )
  {
    //    cout << "xStart " << xStart << "  xEnd "<<xEnd<<endl;
    //    cout << "yStart " << yStart << "  yEnd "<<yEnd<<endl;


    PLFLT xMR;
    PLFLT xML;
    PLFLT yMB;
    PLFLT yMT;

    CheckMargin( e, actStream,
		 xMarginL,
		 xMarginR,
		 yMarginB,
		 yMarginT,
		 xMR, xML, yMB, yMT);

    // viewport - POSITION overrides
    static bool kwP=FALSE;
    static bool do_iso=FALSE;
    static PLFLT aspect=0.0;
    static PLFLT positionP[ 4]={0,0,0,0};
    static PLFLT position[ 4];
    DStructGDL* pStruct = SysVar::P();

    // Get !P.position values
    if(pStruct != NULL) {
      static unsigned positionTag = pStruct->Desc()->TagIndex( "POSITION");
      for( SizeT i=0; i<4; ++i)
	positionP[i] = (PLFLT)
	  (*static_cast<DFloatGDL*>(pStruct->GetTag( positionTag, 0)))[i];
    }

    // If pos == NULL (oplot, /OVERPLOT etc. Reuse previous values)
    if (pos == NULL)
    {
      // If position keyword previously set
      if (kwP)
      {
// Creates a viewport with the specified normalized subpage coordinates.
        if (do_iso) setIsoPort(actStream,position[0], position[2], position[1], position[3], aspect);
        else actStream->vpor(position[0], position[2], position[1], position[3]);
      }
      else
      {
        // If !P.position not set
        if (positionP[0] == 0 && positionP[1] == 0 &&
            positionP[2] == 0 && positionP[3] == 0)
        {
          if (do_iso) setIsoPort(actStream,position[0], position[2], position[1], position[3], aspect);
          else actStream->vpor(position[0], position[2], position[1], position[3]);
      }
        else
        {
          // !P.position set
          if (do_iso) setIsoPort(actStream,positionP[0], positionP[2], positionP[1], positionP[3], aspect);
          else actStream->vpor(positionP[0], positionP[2], positionP[1], positionP[3]);
        }
      }
    }
    else //New Plot
    {
      if (iso == 1) // Check ISOTROPIC first
      {
        do_iso = TRUE;
        if ((xLog) && (yLog))
        {
          aspect = abs(log10(yEnd/yStart) / log10(xEnd/xStart));
        } else if (xLog)
        {
          aspect = abs((yEnd-yStart) / log10(xEnd/xStart));
        } else if (yLog)
        {
          aspect = abs( log10(yEnd/yStart) / (xEnd-xStart));
        } else
        {
           aspect = abs((yEnd-yStart)/(xEnd-xStart));
        }
      }
      else
      {
        do_iso = FALSE;
        aspect = 0.0; // vpas with aspect=0.0 equals vpor.
      }

      // New plot without POSITION=[] as argument
      if (pos == (DFloatGDL*) 0xF)
      {
        kwP = false;
        //compute isotropic ratio & save values

        // If !P.position not set use default values
        if (positionP[0] == 0 && positionP[1] == 0 &&
            positionP[2] == 0 && positionP[3] == 0)
        {

          // Set to default values
          position[0] = xML;
          position[1] = yMB;
          position[2] = 1.0 - xMR;
          position[3] = 1.0 - yMT;
          if (do_iso) setIsoPort(actStream,position[0], position[2], position[1], position[3], aspect);
          else actStream->vpor(position[0], position[2], position[1], position[3]);
         }
        else
        {
          // Use !P.position values
          if (do_iso) setIsoPort(actStream,positionP[0], positionP[2], positionP[1], positionP[3], aspect);
          else actStream->vpor(positionP[0], positionP[2], positionP[1], positionP[3]);
        }
      }
      else       // Position keyword set
      {
        kwP = true;
        for (SizeT i = 0; i < 4 && i < pos->N_Elements(); ++i) position[ i] = (*pos)[ i];
        if (do_iso) setIsoPort(actStream,position[0], position[2], position[1], position[3], aspect);
        else actStream->vpor(position[0], position[2], position[1], position[3]);
      }
    }

    // CLIPPING
    if( clippingD != NULL)
	Clipping( clippingD, xStart, xEnd, yStart, yEnd);

    // for OPLOT start and end values are already log
    // SA: changing only local variables!
    if( pos != NULL)
      {
	if( xLog) //normally xStart at this point should never be <=0!
	  {
	    if( xStart <= 0.0) xStart = -12; else xStart = log10( xStart);
	    if( xEnd   <= 0.0) return false; else xEnd = log10( xEnd);
	  }
	if( yLog) //normally yStart at this point should never be <=0!
	  {
	    if( yStart <= 0.0) yStart = -12; else yStart = log10( yStart);
	    if( yEnd <= 0.0) return false; else yEnd = log10( yEnd);
	  }
      }
//    cout << "VP wind: "<<xStart<<" "<<xEnd<<" "<<yStart<<" "<<yEnd<<endl;
     //   printf("data lim (setv): %f %f %f %f\n", xStart, xEnd, yStart, yEnd);
    // set world coordinates
    actStream->wind( xStart, xEnd, yStart, yEnd);
//       cout << "xStart " << xStart << "  xEnd "<<xEnd<<endl;
//        cout << "yStart " << yStart << "  yEnd "<<yEnd<<endl;

   return true;
  }

  void UpdateSWPlotStructs(GDLGStream* actStream, DDouble xStart, DDouble xEnd, DDouble yStart, DDouble yEnd, bool xLog, bool yLog)
  {
    // Get viewpoint parameters and store in WINDOW & S
    PLFLT p_xmin, p_xmax, p_ymin, p_ymax;
    actStream->gvpd (p_xmin, p_xmax, p_ymin, p_ymax);

    DStructGDL* Struct = NULL;
    if (xLog) xStart=log10(xStart);
    if (xLog) xEnd=log10(xEnd);
    if (yLog) yStart=log10(yStart);
    if (yLog) yEnd=log10(yEnd);

    Struct = SysVar::X();
    static unsigned windowTag = Struct->Desc()->TagIndex( "WINDOW");
    static unsigned sTag = Struct->Desc()->TagIndex( "S");
    if (Struct != NULL)
    {
      (*static_cast<DFloatGDL*>( Struct->GetTag( windowTag, 0)))[0] = p_xmin;
      (*static_cast<DFloatGDL*>( Struct->GetTag( windowTag, 0)))[1] = p_xmax;

      (*static_cast<DDoubleGDL*>( Struct->GetTag( sTag, 0)))[0] =
        (p_xmin*xEnd - p_xmax*xStart) / (xEnd - xStart);
      (*static_cast<DDoubleGDL*>( Struct->GetTag( sTag, 0)))[1] =
        (p_xmax - p_xmin) / (xEnd - xStart);
    }

    Struct = SysVar::Y();
    if(Struct != NULL)
    {
      (*static_cast<DFloatGDL*>( Struct->GetTag( windowTag, 0)))[0] = p_ymin;
      (*static_cast<DFloatGDL*>( Struct->GetTag( windowTag, 0)))[1] = p_ymax;

      (*static_cast<DDoubleGDL*>( Struct->GetTag( sTag, 0)))[0] =
        (p_ymin*yEnd - p_ymax*yStart) / (yEnd - yStart);
      (*static_cast<DDoubleGDL*>( Struct->GetTag( sTag, 0)))[1] =
        (p_ymax - p_ymin) / (yEnd - yStart);
    }
  }

  void GetSFromPlotStructs(DDouble **sx, DDouble **sy)
  {
    static DStructGDL* xStruct = SysVar::X();
    static DStructGDL* yStruct = SysVar::Y();
    unsigned sxTag = xStruct->Desc()->TagIndex( "S");
    unsigned syTag = yStruct->Desc()->TagIndex( "S");
    *sx = &(*static_cast<DDoubleGDL*>( xStruct->GetTag( sxTag, 0)))[0];
    *sy = &(*static_cast<DDoubleGDL*>( yStruct->GetTag( syTag, 0)))[0];
  }

  void GetWFromPlotStructs(DFloat **wx, DFloat **wy)
  {
    static DStructGDL* xStruct = SysVar::X();
    static DStructGDL* yStruct = SysVar::Y();
    unsigned xwindowTag = xStruct->Desc()->TagIndex( "WINDOW");
    unsigned ywindowTag = yStruct->Desc()->TagIndex( "WINDOW");
    *wx = &(*static_cast<DFloatGDL*>( xStruct->GetTag( xwindowTag, 0)))[0];
    *wy = &(*static_cast<DFloatGDL*>( yStruct->GetTag( ywindowTag, 0)))[0];
  }

  void GetUsym(DLong **n, DInt **do_fill, DFloat **x, DFloat **y)
  {
    static DStructGDL* usymStruct = SysVar::USYM();
    unsigned nTag = usymStruct->Desc()->TagIndex( "DIM");
    unsigned fillTag = usymStruct->Desc()->TagIndex( "FILL");
    unsigned xTag = usymStruct->Desc()->TagIndex( "X");
    unsigned yTag = usymStruct->Desc()->TagIndex( "Y");

    *n = &(*static_cast<DLongGDL*>( usymStruct->GetTag( nTag, 0)))[0];
    *do_fill = &(*static_cast<DIntGDL*>( usymStruct->GetTag( fillTag, 0)))[0];
    *x = &(*static_cast<DFloatGDL*>( usymStruct->GetTag( xTag, 0)))[0];
    *y = &(*static_cast<DFloatGDL*>( usymStruct->GetTag( yTag, 0)))[0];
  }

  void SetUsym(DLong n, DInt do_fill, DFloat *x, DFloat *y)
  {
    static DStructGDL* usymStruct = SysVar::USYM();
    unsigned xTag = usymStruct->Desc()->TagIndex( "X");
    unsigned yTag = usymStruct->Desc()->TagIndex( "Y");
    unsigned nTag = usymStruct->Desc()->TagIndex( "DIM");
    unsigned fillTag = usymStruct->Desc()->TagIndex( "FILL");

    (*static_cast<DLongGDL*>( usymStruct->GetTag( nTag, 0)))[0] = n;
    (*static_cast<DIntGDL*>( usymStruct->GetTag( fillTag, 0)))[0] = do_fill;

     for (int i=0; i<n; i++)
     {
        (*static_cast<DFloatGDL*>( usymStruct->GetTag( xTag, 0)))[i] = x[i];
        (*static_cast<DFloatGDL*>( usymStruct->GetTag( yTag, 0)))[i] = y[i];
     }
   }

  void DataCoordLimits(DDouble *sx, DDouble *sy, DFloat *wx, DFloat *wy,
    DDouble *xStart, DDouble *xEnd, DDouble *yStart, DDouble *yEnd, bool clip_by_default)
  {
    *xStart = (wx[0] - sx[0]) / sx[1];
    *xEnd   = (wx[1] - sx[0]) / sx[1];
    *yStart = (wy[0] - sy[0]) / sy[1];
    *yEnd   = (wy[1] - sy[0]) / sy[1];
    //   cout << *xStart <<" "<< *xEnd << " "<< *yStart <<" "<< *yEnd << ""<< endl;

    // patch from Joanna (tracker item no. 3029409, see test_clip.pro)
    if (!clip_by_default) {
      //      cout << "joanna" << endl;
      DFloat wxlen = wx[1] - wx[0];
      DFloat wylen = wy[1] - wy[0];
      DFloat xlen = *xEnd - *xStart;
      DFloat ylen = *yEnd - *yStart;
      *xStart = *xStart - xlen/wxlen * wx[0];
      *xEnd = *xEnd + xlen/wxlen * (1 - wx[1]);
      *yStart = *yStart - ylen/wylen * wy[0];
      *yEnd = *yEnd + ylen/wylen * (1 - wy[1]);
    }
    //    cout << *xStart <<" "<< *xEnd << " "<< *yStart <<" "<< *yEnd << ""<< endl;
  }


  void ac_histo(GDLGStream *a, int i_buff, PLFLT *x_buff, PLFLT *y_buff, bool xLog )
  {
    PLFLT x,x1,y,y1,val;
    for ( int jj=1; jj<i_buff; ++jj)
    {
      x1=x_buff[jj-1];
      x=x_buff[jj];
      y1=y_buff[jj-1];
      y=y_buff[jj];
      // cf patch 3567803
      if (xLog) {
	//  val=log10((pow(10.0,x1)+pow(10.0,x))/2.0);
	val=x1+log10(0.5+0.5*(pow(10.0,x-x1)));
      }
      else {
	val=(x1+x)/2.0;
      }
      a->join(x1,y1,val,y1);
      a->join(val,y1,val,y);
      a->join(val,y,x,y);
    }
  }

  void getNormalizedCoordinatesFromPLPLOT(GDLGStream *a, DDouble wx, DDouble wy, DDouble *nx, DDouble *ny)
  {
    // from current values derive the relative coordinates in the sense of plplot
    DDouble s1,s2;
    PLFLT nxmin,nxmax,nymin,nymax,wxmin,wxmax,wymin,wymax;
    a->gvpd(nxmin,nxmax,nymin,nymax);//norm of current box
    a->gvpw(wxmin,wxmax,wymin,wymax);
//   fprintf(stderr,"World: x=[%lf,%lf] y=[%lf,%lf] Norm: x=[%lf,%lf] y=[%lf,%lf]\n",wxmin,wxmax,wymin,wymax,nxmin,nxmax,nymin,nymax);
    s1=(nxmax-nxmin)/(wxmax-wxmin);
    s2=nxmin;
    *nx=s1*(wx-wxmin)+s2;
    s1=(nymax-nymin)/(wymax-wymin);
    s2=nymin;
    *ny=s1*(wy-wymin)+s2;
  }
  void getWorldCoordinatesFromPLPLOT(GDLGStream *a, DDouble nx, DDouble ny, DDouble *wx, DDouble *wy)
  {
    // from current values derive the world coordinates in the sense of plplot
    DDouble s1,s2;
    PLFLT nxmin,nxmax,nymin,nymax,wxmin,wxmax,wymin,wymax;
    a->gvpd(nxmin,nxmax,nymin,nymax); //norm of current box
    a->gvpw(wxmin,wxmax,wymin,wymax); //world of current box
//   fprintf(stderr,"World: x=[%lf,%lf] y=[%lf,%lf] Norm: x=[%lf,%lf] y=[%lf,%lf]\n",wxmin,wxmax,wymin,wymax,nxmin,nxmax,nymin,nymax);
    s1=(wxmax-wxmin)/(nxmax-nxmin);
    s2=wxmin;
    *wx=s1*(nx-nxmin)+s2;
    s1=(wymax-wymin)/(nymax-nymin);
    s2=wymin;
    *wy=s1*(ny-nymin)+s2;
 }

 static DDouble savedPointX=0.0;
 static DDouble savedPointY=0.0;
 void saveLastPoint(GDLGStream *a, DDouble wx, DDouble wy)
  {
   DDouble nx;
   DDouble ny;
   getNormalizedCoordinatesFromPLPLOT(a, wx, wy, &savedPointX, &savedPointY);
//   fprintf(stderr,"Saved norm: %lf %lf\n",savedPointX,savedPointY);
  }
  void getLastPoint(GDLGStream *a, DDouble* wx, DDouble* wy)
  {
   getWorldCoordinatesFromPLPLOT(a, savedPointX, savedPointY , wx, wy);
//   fprintf(stderr,"Got norm: %lf %lf giving %lf %lf world\n", savedPointX, savedPointY, *wx, *wy);
  }
  //CORE PLOT FUNCTION -> Draws a line along xVal, yVal

  template <typename T> bool draw_polyline(EnvT *e, GDLGStream *a,
  T * xVal, T* yVal,
  DDouble minVal, DDouble maxVal, bool doMinMax,
  bool xLog, bool yLog,
  DLong psym, bool append){
    bool line = false;
    bool valid = true;
    DLong psym_ = 0;

    if (psym < 0) {
      line = true;
      psym_ = -psym;
    } else if (psym == 0) {
      line = true;
      psym_ = psym;
    } else {
      psym_ = psym;
    }

    //usersym
    DFloat *userSymX, *userSymY;
    DLong *userSymArrayDim;
    DInt *do_fill;
    if (psym_ == 8) {
      GetUsym(&userSymArrayDim, &do_fill, &userSymX, &userSymY);
      if (*userSymArrayDim == 0) {
        e->Throw("No user symbol defined.");
      }
    }
    DDouble UsymConvX, UsymConvY;
    GetUserSymSize(e, a, UsymConvX, UsymConvY);

    DLong minEl = (xVal->N_Elements() < yVal->N_Elements()) ?
    xVal->N_Elements() : yVal->N_Elements();
    // if scalar x
    if (xVal->N_Elements() == 1 && xVal->Rank() == 0)
      minEl = yVal->N_Elements();
    // if scalar y
    if (yVal->N_Elements() == 1 && yVal->Rank() == 0)
      minEl = xVal->N_Elements();

    bool mapSet = false;
  #ifdef USE_LIBPROJ4
    // Map Stuff (xtype = 3)
    LPTYPE idata;
    XYTYPE odata;

    get_mapset(mapSet);

    DDouble xStart, xEnd;
    get_axis_crange("X", xStart, xEnd);

    if (mapSet) {
      ref = map_init();
      if (ref == NULL) {
        e->Throw("Projection initialization failed.");
      }
    }
  #endif

    // is one of the 2 "arrays" a singleton or not ?

    PLFLT y, y1, yMapBefore, y_ref;
    int flag_y_const = 0;
    y_ref = static_cast<PLFLT>((*yVal)[0]);
    if (yVal->N_Elements() == 1 && yVal->Rank() == 0) flag_y_const = 1;

    PLFLT x, x1, xMapBefore, x_ref;
    int flag_x_const = 0;
    x_ref = static_cast<PLFLT>((*xVal)[0]);
    if (xVal->N_Elements() == 1 && xVal->Rank() == 0) flag_x_const = 1;

    // AC 070601 we use a buffer to use the fast ->line method
    // instead of the slow ->join one.
    // 2 tricks:
    // trick 1/ size of buffer is limited to 1e4 (compromize syze/speed) in order to be able to manage very
    //    large among of data whitout duplicating all the arrays
    // trick 2/ when we have a NaN or and Inf, we realize the plot, then reset.

    int debug_ac = 0;

    int n_buff_max = 500000; // idl default seems to be more than 2e6 !!

    if (minEl < n_buff_max) n_buff_max = append ? minEl + 1 : minEl;
    int i_buff = 0;
    PLFLT *x_buff = new PLFLT[n_buff_max];
    PLFLT *y_buff = new PLFLT[n_buff_max];

    // flag to reset Buffer when a NaN or a Infinity are founded
    int reset = 0;

    // translation plplot symbols - 8th symbol is superseded by USERSYM.
    const PLINT codeArr[] = {0, 2, 3, 1, 11, 7, 6, 5, 4};
    bool isBad = FALSE;
    for (int i = 0; i < minEl; ++i) {
      isBad = FALSE;
      if (append) //start with the old point
      {
        getLastPoint(a, &x, &y);
        i--; //to get good counter afterwards
        append = FALSE; //and stop appending after!
        if (xLog) x = pow(10, x);
        if (yLog) y = pow(10, y);
      } else {
        if (!flag_x_const) x = static_cast<PLFLT>((*xVal)[i]);
        else x = x_ref;
        if (!flag_y_const) y = static_cast<PLFLT>((*yVal)[i]);
        else y = y_ref;
      }
  #ifdef USE_LIBPROJ4
      if (mapSet && !e->KeywordSet("NORMAL")) {
        idata.lam = x * DEG_TO_RAD;
        idata.phi = y * DEG_TO_RAD;
        if (i > 0) {
          xMapBefore = odata.x;
          yMapBefore = odata.y;
        }
        odata = PJ_FWD(idata, ref);
        x = odata.x;
        y = odata.y;
      }
  #endif
      isBad = (!isfinite(x) || !isfinite(y) || isnan(x) || isnan(y));
      if (doMinMax) isBad = (isBad || (y < minVal) || (y > maxVal));
      if (isBad) {
        reset = 1;
        if (i_buff > 0) {
          if (line) {
            a->line(i_buff, x_buff, y_buff);
          }
          if ((psym_ > 0 && psym_ < 8) || psym_ == 9) {
            a->poin(i_buff, x_buff, y_buff, codeArr[psym_]);
          }
          if (psym_ == 8) {
            PLFLT *xx = new PLFLT[*userSymArrayDim];
            PLFLT *yy = new PLFLT[*userSymArrayDim];
            for (int j = 0; j < i_buff; ++j) {
              if (debug_ac) {
                cout << "j: " << j << ", X: " << x_buff[j] << ", Y: " << y_buff[j] << endl;
              };
              for (int kk = 0; kk < *userSymArrayDim; kk++) {
                xx[kk] = x_buff[j] + userSymX[kk] * UsymConvX;
                yy[kk] = y_buff[j] + userSymY[kk] * UsymConvY;
              }
              if (*do_fill == 1) {
                a->fill(*userSymArrayDim, xx, yy);
              } else {
                a->line(*userSymArrayDim, xx, yy);
              }
            }
          }
          if (psym_ == 10) {
            ac_histo(a, i_buff, x_buff, y_buff, xLog);
          }
          i_buff = 0;
        }
        continue;
      }

  #ifdef USE_LIBPROJ4
      if (mapSet && !e->KeywordSet("NORMAL")) {
        if (i > 0) //;&& (i_buff >0))
        {
          x1 = xMapBefore;
          if (!isfinite(xMapBefore) || !isfinite(yMapBefore)) continue;

          // Break "jumps" across maps (kludge!)
          if (fabs(x - x1) > 0.5 * (xEnd - xStart)) {
            reset = 1;
            if ((i_buff > 0) && (line)) {
              a->line(i_buff, x_buff, y_buff);
              //		  x_buff[0]=x_buff[i_buff-1];
              //y_buff[0]=y_buff[i_buff-1];
              i_buff = 0;
            }
            continue;
          }
        }
      }
  #endif
      //note: here y is in minVal maxVal
      if (xLog) if (x <= 0.0) continue;
        else x = log10(x);
      if (yLog) if (y <= 0.0) continue;
        else y = log10(y);

      x_buff[i_buff] = x;
      y_buff[i_buff] = y;
      i_buff = i_buff + 1;

      //	cout << "nbuf: " << i << " " << i_buff << " "<< n_buff_max-1 << " " << minEl-1 << endl;

      if ((i_buff == n_buff_max) || ((i == minEl - 1) && !append) || ((i == minEl) && append)) {
        if (line) {
          a->line(i_buff, x_buff, y_buff);
        };
        if ((psym_ > 0 && psym_ < 8) || psym_ == 9) {
          a->poin(i_buff, x_buff, y_buff, codeArr[psym_]);
        }
        if (psym_ == 8) {
          PLFLT *xx = new PLFLT[*userSymArrayDim];
          PLFLT *yy = new PLFLT[*userSymArrayDim];
          for (int j = 0; j < i_buff; ++j) {
            if (debug_ac) {
              cout << "j: " << j << ", X: " << x_buff[j] << ", Y: " << y_buff[j] << endl;
            };
            for (int kk = 0; kk < *userSymArrayDim; kk++) {
              xx[kk] = x_buff[j] + userSymX[kk] * UsymConvX;
              yy[kk] = y_buff[j] + userSymY[kk] * UsymConvY;
            }
            if (*do_fill == 1) {
              a->fill(*userSymArrayDim, xx, yy);
              //to be tested: provided we define a 'non-gradient' gradient before this should work
              //                a->gradient(*userSymArrayDim,xx,yy,0.0);
            } else {
              a->line(*userSymArrayDim, xx, yy);
            }
          }
        }
        if (psym_ == 10) {
          ac_histo(a, i_buff, x_buff, y_buff, xLog);
        }

        // we must recopy the last point since the line must continue (tested via small buffer ...)
        x_buff[0] = x_buff[i_buff - 1];
        y_buff[0] = y_buff[i_buff - 1];
        i_buff = 1;
      }
    }

    delete[] x_buff;
    delete[] y_buff;
    //save last point
    saveLastPoint(a, x, y);
    return (valid);
  }
  // explicit instantiation for SpDDouble
  template bool draw_polyline(EnvT*, GDLGStream*, Data_<SpDDouble>*, Data_<SpDDouble>*, DDouble, DDouble, bool, bool, bool, DLong, bool);

  //[XYZ]MARGIN kw decoding
  void gkw_axis_margin(EnvT *e, string axis,DFloat &start, DFloat &end)
  {
    DStructGDL* Struct;
    if(axis=="X") Struct = SysVar::X();
    if(axis=="Y") Struct = SysVar::Y();

    if(Struct != NULL)
      {
	static unsigned marginTag = Struct->Desc()->TagIndex( "MARGIN");
	start =
	  (*static_cast<DFloatGDL*>( Struct->GetTag( marginTag, 0)))[0];
	end =
	  (*static_cast<DFloatGDL*>( Struct->GetTag( marginTag, 0)))[1];
      }

    string MarginName=axis+"MARGIN";
    BaseGDL* Margin=e->GetKW(e->KeywordIx(MarginName));
    if(Margin !=NULL)
      {
	if(Margin->N_Elements() > 2)
	  e->Throw("Keyword array parameter "+MarginName+
		   " must have from 1 to 2 elements.");
	auto_ptr<DFloatGDL> guard;
	DFloatGDL* MarginF = static_cast<DFloatGDL*>
	  ( Margin->Convert2( GDL_FLOAT, BaseGDL::COPY));
	guard.reset( MarginF);
	start = (*MarginF)[0];
	if( MarginF->N_Elements() > 1)
	  end = (*MarginF)[1];
      }
  }

  //BACKGROUND COLOR
  void gkw_background(EnvT *e, GDLGStream *a, bool kw)
  {
    static DStructGDL* pStruct = SysVar::P();
    DLong background =
      (*static_cast<DLongGDL*>
       (pStruct->GetTag( pStruct->Desc()->TagIndex("BACKGROUND"), 0)))[0];
    if(kw)
      e->AssureLongScalarKWIfPresent( "BACKGROUND", background);

    // Get decomposed value
    Graphics* actDevice = Graphics::GetDevice();
    DLong decomposed = actDevice->GetDecomposed();
    if (decomposed != 0 && decomposed != 1) {decomposed=0;}

    a->Background( background, decomposed);
  }

  //COLOR
  void gkw_color(EnvT *e, GDLGStream *a)
  {
    // Get COLOR from PLOT system variable
    static DStructGDL* pStruct = SysVar::P();
    DLong color =
      (*static_cast<DLongGDL*>
       (pStruct->GetTag( pStruct->Desc()->TagIndex("COLOR"), 0)))[0];

    // Get # of colors from DEVICE system variable
    DVar *var=FindInVarList(sysVarList,"D");
    DStructGDL* s = static_cast<DStructGDL*>( var->Data());
    DLong ncolor = (*static_cast<DLongGDL*>
                    (s->GetTag(s->Desc()->TagIndex("N_COLORS"), 0)))[0];

    if (ncolor > 256 && color == 255) color = ncolor - 1;

    e->AssureLongScalarKWIfPresent( "COLOR", color);

    // Get decomposed value
    Graphics* actDevice = Graphics::GetDevice();
    DLong decomposed = actDevice->GetDecomposed();
    if (decomposed != 0 && decomposed != 1) {decomposed=0;}
    a->Color( color, decomposed, 2);
  }

  // helper for NOERASE (but also used in XYOUTS)
  void handle_pmulti_position(EnvT *e, GDLGStream *a)
  {
    // !P.MULTI is ignored if POSITION kw or !P.POSITION or !P.REGION is specified
    // TODO: !P.REGION!

    DFloatGDL* pos = NULL;

    // system variable
    static DStructGDL* pStruct = SysVar::P();
    pos = static_cast<DFloatGDL*>(pStruct-> GetTag( pStruct->Desc()->TagIndex("POSITION"), 0));
    if ((*pos)[0] == (*pos)[2]) pos = NULL;

    // keyword
    if (pos == NULL)
    {
      DSub* pro = e->GetPro();
      int positionIx = pro->FindKey( "POSITION");
      if (positionIx != -1) pos = e->IfDefGetKWAs<DFloatGDL>( positionIx);
    }

    if (pos != NULL) a->NoSub();
  }

  //NOERASE
  void gkw_noerase(EnvT *e,GDLGStream *a, bool noe)
  {
    DLong noErase=0;
    DLongGDL* pMulti = SysVar::GetPMulti();
    static DStructGDL* pStruct = SysVar::P();

    if(!noe)
      {
	noErase = (*static_cast<DLongGDL*>
		   ( pStruct->
		     GetTag( pStruct->Desc()->TagIndex("NOERASE"), 0)))[0];
	if(e->KeywordSet("NOERASE")) {
	  noErase=1;
	}
      }
    else
      {
	noErase=1;
      }

    a->NextPlot( !noErase);
    handle_pmulti_position(e, a);
  }

  //PSYM
  void gkw_psym(EnvT *e, DLong &psym)
  {
    static DStructGDL* pStruct = SysVar::P();
    psym= (*static_cast<DLongGDL*>
	   (pStruct->GetTag(pStruct->Desc()->TagIndex("PSYM"), 0)))[0];

    e->AssureLongScalarKWIfPresent( "PSYM", psym);
    if( psym > 10 || psym < -8 || psym == 9)
      e->Throw(
			  "PSYM (plotting symbol) out of range.");
  }

    //SYMSIZE
  void gkw_symsize(EnvT *e, GDLGStream *a)
  {
    static DStructGDL* pStruct = SysVar::P();
    DFloat symsize = (*static_cast<DFloatGDL*>
		      (pStruct->GetTag( pStruct->Desc()->TagIndex("SYMSIZE"), 0)))[0];
    e->AssureFloatScalarKWIfPresent( "SYMSIZE", symsize);
    if( symsize <= 0.0) symsize = 1.0;
    a->ssym(0.0, symsize);
  }

  //CHARSIZE
  void gkw_charsize(EnvT *e, GDLGStream *a, DFloat &charsize, bool kw)
  {
    static DStructGDL* pStruct = SysVar::P();
    charsize = (*static_cast<DFloatGDL*>
			(pStruct->GetTag
			 ( pStruct->Desc()->TagIndex("CHARSIZE"), 0)))[0];
    if(kw)
      e->AssureFloatScalarKWIfPresent( "CHARSIZE", charsize);

    if( charsize <= 0.0) charsize = 1.0;
    a->schr(0.0, charsize);
  }
  //OLD CHARSIZE (for xyouts only?)
  void gkw_charsize_xyouts(EnvT *e, GDLGStream *a, DFloat &charsize)
  {
    static DStructGDL* pStruct = SysVar::P();
    charsize = (*static_cast<DFloatGDL*>
			(pStruct->GetTag
			 ( pStruct->Desc()->TagIndex("CHARSIZE"), 0)))[0];
    //imagine CHARSIZE & SIZE: we prefer CHARSIZE of course
    if(e->KeywordSet("SIZE")) e->AssureFloatScalarKWIfPresent( "SIZE", charsize);
    e->AssureFloatScalarKWIfPresent( "CHARSIZE", charsize);

    if( charsize <= 0.0) charsize = 1.0;
    a->schr(0.0, charsize);
  }
  //THICK
  void gkw_thick(EnvT *e, GDLGStream *a)
  {
    static DStructGDL* pStruct = SysVar::P();
    DFloat thick = (*static_cast<DFloatGDL*>
		    (pStruct->GetTag( pStruct->Desc()->TagIndex("THICK"), 0)))[0];

    e->AssureFloatScalarKWIfPresent( "THICK", thick);
    if( thick <= 0.0) thick = 1.0;
    a->wid( static_cast<PLINT>(floor( thick-0.5)));
  }

  //LINESTYLE
  void gkw_linestyle(EnvT *e, GDLGStream *a)
  {
    static DStructGDL* pStruct = SysVar::P();
    DLong linestyle=
      (*static_cast<DLongGDL*>
       (pStruct->GetTag( pStruct->Desc()->TagIndex("LINESTYLE"), 0)))[0];

    // if the LINESTYLE keyword is present, the value will be change
    DLong temp_linestyle=-1111;
    e->AssureLongScalarKWIfPresent( "LINESTYLE",temp_linestyle);

    bool debug=false;
    if (debug) {
      cout << "temp_linestyle " <<  temp_linestyle << endl;
      cout << "     linestyle " <<  linestyle << endl;
    }
    if (temp_linestyle != -1111) {linestyle=temp_linestyle;}//+1;
    if (linestyle < 0 ) {linestyle=0;}
    if (linestyle > 5 ) {linestyle=5;}

      // see
      // file:///home/coulais/SoftsExternes/plplot-5.5.3/examples/c++/x09.cc
      // file:///home/coulais/SoftsExternes/plplot-5.5.3/doc/docbook/src/plstyl.html

    if (linestyle == 0) { // solid (continuous line)
      static PLINT nbp=0;
      a->styl(nbp, NULL, NULL);
    }
    if (linestyle == 1) { // dots
      static PLINT nbp=1;
      static PLINT mark[] = {75};
      static PLINT space[] = {1500};
      a->styl(nbp, mark, space);
    }
    if (linestyle == 2) { // dashed
      static PLINT nbp=1;
      static PLINT mark[] = {1500};
      static PLINT space[] = {1500};
      a->styl(nbp, mark, space);
  }
    if (linestyle == 3) { // dash dot
      static PLINT nbp=2;
      static PLINT mark[] = {1500,100};
      static PLINT space[] = {1000,1000};
      a->styl(nbp, mark, space);
    }
    if (linestyle == 4) { // dash dot dot
      static PLINT nbp=4;
      static PLINT mark[] = {1500,100,100,100};
      static PLINT space[] = {1000,1000,1000,1000};
      a->styl(nbp, mark, space);
    }
    if (linestyle == 5) { // long dash
      static PLINT nbp=1;
      static PLINT mark[] = {3000};
      static PLINT space[] = {1500};
      a->styl(nbp, mark, space);
    }
  }

  //TITLE
  void gkw_title(EnvT* e, GDLGStream *a, PLFLT ad)
  {
    DLong thick=0;
    e->AssureLongScalarKWIfPresent("CHARTHICK",thick);
    a->wid(thick);

    static DStructGDL* pStruct = SysVar::P();
    static unsigned titleTag = pStruct->Desc()->TagIndex( "TITLE");
    static unsigned subTitleTag = pStruct->Desc()->TagIndex( "SUBTITLE");
    DString title =
      (*static_cast<DStringGDL*>( pStruct->GetTag( titleTag, 0)))[0];
    DString subTitle =
      (*static_cast<DStringGDL*>( pStruct->GetTag( subTitleTag, 0)))[0];
    e->AssureStringScalarKWIfPresent( "TITLE", title);
    e->AssureStringScalarKWIfPresent( "SUBTITLE", subTitle);

    a->schr( 0.0, 1.25*ad);
    a->mtex("t",1.25,0.5,0.5,title.c_str());
    a->schr( 0.0, ad); // charsize is reset here
    a->mtex("b",5.4,0.5,0.5,subTitle.c_str());
    a->wid(0);
  }

  //crange to struct

  void set_axis_crange(string axis, DDouble Start, DDouble End, bool log)
  {
    DStructGDL* Struct = NULL;
    if (axis == "X") Struct = SysVar::X();
    if (axis == "Y") Struct = SysVar::Y();
    if (axis == "Z") Struct = SysVar::Z();
    if (Struct != NULL)
    {
      int debug = 0;
      if (debug) cout << "Set     :" << Start << " " << End << endl;

      static unsigned crangeTag = Struct->Desc()->TagIndex("CRANGE");
      if (log)
      {
        (*static_cast<DDoubleGDL*> (Struct->GetTag(crangeTag, 0)))[0] = log10(Start);
        (*static_cast<DDoubleGDL*> (Struct->GetTag(crangeTag, 0)))[1] = log10(End);
        if (debug) cout << "set log" << Start << " " << End << endl;
      }
      else
      {
        (*static_cast<DDoubleGDL*> (Struct->GetTag(crangeTag, 0)))[0] = Start;
        (*static_cast<DDoubleGDL*> (Struct->GetTag(crangeTag, 0)))[1] = End;
      }
    }
  }

  //CRANGE from struct
  void get_axis_crange(string axis, DDouble &Start, DDouble &End)
  {
    DStructGDL* Struct=NULL;
    if(axis=="X") Struct = SysVar::X();
    if(axis=="Y") Struct = SysVar::Y();
    if(axis=="Z") Struct = SysVar::Z();
    if(Struct!=NULL)
    {
	int debug=0;
	if (debug) cout << "Get     :" << Start << " " << End << endl;

	static unsigned crangeTag = Struct->Desc()->TagIndex( "CRANGE");
	Start = (*static_cast<DDoubleGDL*>( Struct->GetTag( crangeTag, 0)))[0];
	End = (*static_cast<DDoubleGDL*>( Struct->GetTag( crangeTag, 0)))[1];

	static unsigned typeTag = Struct->Desc()->TagIndex( "TYPE");
	if ((*static_cast<DLongGDL*>(Struct->GetTag( typeTag, 0)))[0] == 1)
	  {
	    Start=pow(10.,Start);
	    End=pow(10.,End);
	    if (debug) cout << "Get log :" << Start << " " << End << endl;
	  }
    }
  }

   void get_axis_type(string axis,bool &log)
  {
    DStructGDL* Struct;
    if(axis=="X") Struct = SysVar::X();
    if(axis=="Y") Struct = SysVar::Y();
    if(axis=="Z") Struct = SysVar::Z();
    if(Struct != NULL) {
      static unsigned typeTag = Struct->Desc()->TagIndex( "TYPE");
      if ((*static_cast<DLongGDL*>(Struct->GetTag( typeTag, 0)))[0] == 1)
	log = 1;
      else
	log=0;
    }
  }

  void get_mapset(bool &mapset)
  {
    DStructGDL* Struct = SysVar::X();
    if(Struct != NULL) {
      static unsigned typeTag = Struct->Desc()->TagIndex( "TYPE");

      if ((*static_cast<DLongGDL*>(Struct->GetTag( typeTag, 0)))[0] == 3)
	mapset = 1;
      else
	mapset = 0;
    }
  }

  void set_mapset(bool mapset)
  {
    DStructGDL* Struct = SysVar::X();
    if(Struct!=NULL)
      {
	static unsigned typeTag = Struct->Desc()->TagIndex( "TYPE");
	(*static_cast<DLongGDL*>( Struct->GetTag( typeTag, 0)))[0] = mapset;
      }
  }


  //axis type (log..)
  void set_axis_type(string axis, bool Type)
  {
    DStructGDL* Struct=NULL;
    if(axis=="X") Struct = SysVar::X();
    if(axis=="Y") Struct = SysVar::Y();
    if(axis=="Z") Struct = SysVar::Z();
    if(Struct!=NULL)
      {
	static unsigned typeTag = Struct->Desc()->TagIndex("TYPE");
	(*static_cast<DLongGDL*>(Struct->GetTag(typeTag, 0)))[0] = Type;
      }
  }

  void gkw_axis_charsize(EnvT* e, string axis, DFloat &charsize)
  {
    DStructGDL* Struct;
    if(axis=="X") Struct = SysVar::X();
    if(axis=="Y") Struct = SysVar::Y();

    if(Struct != NULL)
      {
	static unsigned charsizeTag = Struct->Desc()->TagIndex("CHARSIZE");
	charsize =
	  (*static_cast<DFloatGDL*>( Struct->GetTag( charsizeTag, 0)))[0];
      }

    string Charsize_s=axis+"CHARSIZE";
    e->AssureFloatScalarKWIfPresent( Charsize_s, charsize);
    if(charsize <=0.0) charsize=1.0;
  }


  //STYLE
  void gkw_axis_style(EnvT *e, string axis, DLong &style)
  {
    DStructGDL* Struct;
    if(axis=="X") Struct = SysVar::X();
    if(axis=="Y") Struct = SysVar::Y();
    if(Struct != NULL)
      {
	static unsigned styleTag = Struct->Desc()->TagIndex( "STYLE");
	style =
	  (*static_cast<DLongGDL*>( Struct->GetTag( styleTag, 0)))[0];
      }

    string StyleName=axis+"STYLE";

  }

  void gkw_axis_title(EnvT *e, string axis, DString &title)
  {
    DStructGDL* Struct;
    if(axis=="X") Struct = SysVar::X();
    if(axis=="Y") Struct = SysVar::Y();

    if(Struct != NULL)
      {
	static unsigned titleTag = Struct->Desc()->TagIndex("TITLE");
	title =
	  (*static_cast<DStringGDL*>( Struct->GetTag( titleTag, 0)))[0];
      }

    string TitleName=axis+"TITLE";
    e->AssureStringScalarKWIfPresent( TitleName, title);

  }

  //GET RANGE

  void gkw_axis_range(EnvT *e, string axis, DDouble &start, DDouble &end,
                      DLong &ynozero)
  {
    DStructGDL* Struct;
    if (axis == "X") Struct = SysVar::X();
    if (axis == "Y") Struct = SysVar::Y();
    if (Struct != NULL)
    {
      DDouble test1, test2;
      static unsigned rangeTag = Struct->Desc()->TagIndex("RANGE");
      test1 = (*static_cast<DDoubleGDL*> (Struct->GetTag(rangeTag, 0)))[0];
      test2 = (*static_cast<DDoubleGDL*> (Struct->GetTag(rangeTag, 0)))[1];
      if (!(test1 == 0.0 && test2 == 0.0))
      {
        start = test1;
        end = test2;
      }
    }
    string RangeName = axis + "RANGE";
    BaseGDL* Range = e->GetKW(e->KeywordIx(RangeName));
    if (Range != NULL)
    {
      if (Range->N_Elements() != 2)
        e->Throw("Keyword array parameter " + RangeName +
                 " must have 2 elements.");
      auto_ptr<DDoubleGDL> guard;
      DDoubleGDL* RangeF = static_cast<DDoubleGDL*>
	(Range->Convert2(GDL_DOUBLE, BaseGDL::COPY));
      guard.reset(RangeF);
      start = (*RangeF)[0];
      end = (*RangeF)[1];
      if (axis == "Y") ynozero = 1;
    }
  }
  //current value of margin of axis 'axis'
  void get_axis_margin(string axis, DFloat &low, DFloat &high)
  {
    DStructGDL* Struct=NULL;
    if(axis=="X") Struct = SysVar::X();
    if(axis=="Y") Struct = SysVar::Y();
    if(Struct!=NULL)
      {
	static unsigned marginTag = Struct->Desc()->TagIndex( "MARGIN");
	low = (*static_cast<DFloatGDL*>( Struct->GetTag( marginTag, 0)))[0];
	high = (*static_cast<DFloatGDL*>( Struct->GetTag( marginTag, 0)))[1];
      }
  }
  void usersym(EnvT *e)
{
      DFloatGDL *xyVal, *xVal, *yVal;
      auto_ptr<BaseGDL> p0_guard;
      DLong n;
      DInt do_fill;
      DFloat *x, *y;
      SizeT nParam = e->NParam();

      if (nParam == 1) {
 	   BaseGDL* p0 = e->GetNumericArrayParDefined( 0)->Transpose( NULL); //hence [1024,2]

	xyVal = static_cast<DFloatGDL*>
	  (p0->Convert2( GDL_FLOAT, BaseGDL::COPY));
	p0_guard.reset( p0); // delete upon exit

	if(xyVal->Rank() != 2 || xyVal->Dim(1) != 2)
          e->Throw(e->GetParString(0)+" must be a 2-dim array of type [2,N] in this context.");

            if (xyVal->Dim(0) > 1024)
            {
                e->Throw("Max array size for USERSYM is 1024");
            }
            n = xyVal->Dim(0);
            // array is in the good order for direct C assignement
            x=&(*xyVal)[0];
            y=&(*xyVal)[n];
       } else {
            xVal = e->GetParAs< DFloatGDL > (0);
            if (xVal->Rank() != 1)
                e->Throw(e->GetParString(0)+" must be a 1D array in this context: ");

            yVal = e->GetParAs< DFloatGDL > (1);
            if (yVal->Rank() != 1)
                e->Throw("Expression must be a 1D array in this context: " + e->GetParString(1));

            if (xVal->Dim(0)!= yVal->Dim(0))
            {
               e->Throw("Arrays must have same size ");
            }

            if (xVal->Dim(0) > 1024)
            {
                e->Throw("Max array size for USERSYM is 1024");
            }
            n = xVal->Dim(0);
            x=&(*xVal)[0];
            y=&(*yVal)[0];
        }
        do_fill=0;
        if (e->KeywordSet("FILL")) {
        do_fill=1;
        }
        SetUsym(n,do_fill, x, y);
  }
} // namespace
