//  Gnomoradio - gnomoradio/song-rater.cc
//  Copyright (C) 2003  Matt Gerginski
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <libintl.h>
#include "song-rater.h"
#include "icons/star-icons.h"

#define _(String) gettext (String)
#define gettext_noop(String) String
#define N_(String) gettext_noop (String)

using namespace std;
using namespace Glib;
using namespace Gdk;
using namespace Gtk;

bool Gnomoradio::SongRater::initialized = false;
RefPtr<Gdk::Pixbuf> Gnomoradio::SongRater::good_pixbuf;
RefPtr<Gdk::Pixbuf> Gnomoradio::SongRater::bad_pixbuf;
RefPtr<Gdk::Pixbuf> Gnomoradio::SongRater::blank_pixbuf;
RefPtr<Gdk::Pixbuf> Gnomoradio::SongRater::neutral_pixbuf;
RefPtr<Gdk::Pixbuf> Gnomoradio::SongRater::empty_pixbuf;


Gnomoradio::SongRater::SongRater (int value) :
	compact(true),
	adj(value, 0, 0)
{
	update_rater();

	adj.signal_value_changed().connect(sigc::mem_fun(*this, &Gnomoradio::SongRater::on_adj_value_changed));
}

Gnomoradio::SongRater::SongRater (int value, int lower, int upper) :
	compact(false),
	minus(_(" - ")),
	plus(_(" + ")),
	adj(value, lower, upper)
{
        pack_start(minus, PACK_SHRINK);

	int total_images = upper - lower + 1; // total number of images +1 for zero

	if (!initialized) {
		try {
			blank_pixbuf   = Gdk::Pixbuf::create_from_inline(-1, star_blank);
			neutral_pixbuf = Gdk::Pixbuf::create_from_inline(-1, star_neutral);
			good_pixbuf    = Gdk::Pixbuf::create_from_inline(-1, star_good);
			bad_pixbuf     = Gdk::Pixbuf::create_from_inline(-1, star_bad);
			empty_pixbuf   = Gdk::Pixbuf::create_from_inline(-1, star_empty);
			
		} catch (...) {
		}

		initialized = true;
	}

	for (int i = 0; i < total_images; ++i) {
		images.push_back(manage(new Gtk::Image(blank_pixbuf)));
		
		event_boxes.push_back(manage(new EventBox));
		
		event_boxes[i]->set_events(BUTTON_PRESS_MASK);

		event_boxes[i]->add(*images[i]);

		pack_start(*event_boxes[i], PACK_EXPAND_PADDING, 2);

		event_boxes[i]->signal_button_press_event().connect(sigc::bind<int>(sigc::mem_fun(*this, &Gnomoradio::SongRater::on_image_clicked), i));
	}

	/* assign neutral_pixbuf to image representing "0" rating */
	images[0- lower]->set(neutral_pixbuf);

	pack_start(plus, PACK_SHRINK);

	minus.signal_clicked().connect(sigc::mem_fun(*this, &Gnomoradio::SongRater::on_minus_clicked));
	plus.signal_clicked().connect(sigc::mem_fun(*this, &Gnomoradio::SongRater::on_plus_clicked));

	update_rater();

	adj.signal_value_changed().connect(sigc::mem_fun(*this, &Gnomoradio::SongRater::on_adj_value_changed));
}

Gnomoradio::SongRater::~SongRater ()
{

}

void Gnomoradio::SongRater::on_plus_clicked ()
{
	if (adj.get_value() < adj.get_upper())
		adj.set_value(adj.get_value() + 1);
}

void Gnomoradio::SongRater::on_minus_clicked ()
{
	if (adj.get_value() > adj.get_lower())
		adj.set_value(adj.get_value() - 1);
}

bool Gnomoradio::SongRater::on_image_clicked (GdkEventButton* event, int image_number)
{
	adj.set_value(image_number - adj.get_upper());

	return true;
}

void Gnomoradio::SongRater::on_adj_value_changed ()
{
	update_rater();

	signal_value_changed.emit();
}

void Gnomoradio::SongRater::update_rater ()
{
	int value        = (int) adj.get_value();

	if (compact)
	{
		RefPtr<Gdk::Pixbuf> pixbuf_being_used = good_pixbuf;

		if (value < 0) {
			value *= -1;
		
			pixbuf_being_used = bad_pixbuf;
		}

		for (int i = 0; i < value; ++i) {
			/* Is there a space for this image?  If not, add one */
			if (images.size() < i) {
				images.push_back(manage(new Gtk::Image(pixbuf_being_used)));
				pack_start(*images[i], PACK_SHRINK);
			} else			
				images[i]->set(pixbuf_being_used);
		}

		int size = images.size();

		/* get rid of leftovers */
		for (int i = value; i < size; ++i)
			images[i]->set(empty_pixbuf);
	}

	else // not compact
	{
		int middle_image = (int) (0 - adj.get_lower());
		int upper_image  = (int) (middle_image + adj.get_upper());
		int lower_image  = (int) (middle_image + adj.get_lower());

		if (value <= 0) {
			/* set images above middle_image to blank */
			for (int img = middle_image + 1; img <= upper_image; ++img)
				images[img]->set(blank_pixbuf);

			/* start at the middle image, and go backward by 1.  If the rating is lower
			 * or equal, make that image "bad".  If the rating becomes higher, make that
			 * image blank  
			 */
			for (int img = middle_image - 1; img >= lower_image; --img) {
				if (value - adj.get_lower() <= img)
					images[img]->set(bad_pixbuf);
				else
					images[img]->set(blank_pixbuf);  
			}
		}

		if (value >= 0) {
			/* set images below middle_image to blank */
			for (int img = middle_image - 1; img >= lower_image; --img)
				images[img]->set(blank_pixbuf);

			/* same as above, this time going forward */
			for (int img = middle_image + 1; img <= upper_image; ++img) {
				if (value + adj.get_upper() >= img)
					images[img]->set(good_pixbuf);
				else
					images[img]->set(blank_pixbuf);  
			}
		}
	}
}
