// -*- c++ -*-

//  Gnomoradio - roboradio/song.h
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version)
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __ROBORADIO_SONG_H
#define __ROBORADIO_SONG_H

#include <vector>
#include <map>
#include <set>
#include <sigc++/sigc++.h>
#include <glibmm.h>

namespace Roboradio
{
	typedef unsigned int Time;
	typedef unsigned long AbsTime;

	enum Playback
	{
		STOP,
		PLAY,
		PAUSE
	};

	struct SongStatus
	{
		bool available, ready;
		int playing, upcoming;

		SongStatus ()
			: available(false),
			  ready(false),
			  playing(0),
			  upcoming(0) {}
	};

	class SongRef;

	class Song : public SigC::Object
	{
	public:
		Song (const Glib::ustring &loc);
		virtual ~Song ();
		
		void set_info (const Glib::ustring &key, const Glib::ustring &value);
		Glib::ustring get_info (const Glib::ustring &key) const;
		void get_info (std::vector<Glib::ustring> &keys, std::vector<Glib::ustring> &values) const;
		
		int get_rating () const { return rating; }
		void set_rating (int new_rating);
		void thumbs_up () { set_rating(rating + 1); }
		void thumbs_down () { set_rating(rating - 1); }

		SongStatus get_status () const { return status; }
		Time get_length () const { return length; }
		const Glib::ustring& get_url () const { return url; }

		time_t last_played () const { return last_play; }
		unsigned int times_played () const { return times_play; }

		// use Ref objects instead
		void ref ();
		void unref ();
		void playing_ref ();
		void playing_unref ();
		void upcoming_ref ();
		void upcoming_unref ();

		SigC::Signal0<void> signal_info_changed;
		SigC::Signal1<void,Glib::ustring> signal_url_changed;
		SigC::Signal1<void,Time> signal_length_changed;
		SigC::Signal1<void,SongStatus> signal_status_changed;
		SigC::Signal1<void,int> signal_rating_changed;
		SigC::Signal1<void,unsigned int> signal_import_progress;

		SigC::Signal1<void,Time> signal_position_changed;
		SigC::Signal0<void> signal_done;

		static std::vector<SongRef> get_known_songs ();

		static SigC::Signal1<void,SongRef> signal_global_new_song;
		static SigC::Signal1<void,SongRef> signal_global_song_info_changed;
		static SigC::Signal1<void,SongRef> signal_global_song_rating_changed;
		static SigC::Signal1<void,SongRef> signal_global_song_length_changed;
		static SigC::Signal1<void,SongRef> signal_global_song_status_changed;
		static SigC::Signal2<void,SongRef,unsigned int> signal_global_song_import_progress;
		static SigC::Signal1<void,SongRef> signal_global_song_done;

		static const int min_rating = -3;
		static const int max_rating = 3;

		virtual void obtain_available_info () {}
		virtual bool importable () { return false; }
		virtual void import () {}
		//virtual int get_import_progress ();

		virtual Playback get_playback () const = 0;
		virtual void set_playback (Playback pb) = 0;
		virtual void seek (Time pos) = 0;
		virtual Time get_position () = 0;

		friend class State;
		friend class SongRef;

	protected:
		void set_status_available (bool a);
		void set_status_ready (bool r);
		void set_length (Time len);
		void set_url (const Glib::ustring &new_url);
		void set_import_progress (unsigned int p);

		void done ();

	private:
		int refcnt;

		Glib::ustring url;
		Time length;
		int rating;
		std::map<Glib::ustring,Glib::ustring> info;
		SongStatus status;
		time_t last_play;
		unsigned int times_play;
	};

	template <class T>
	class ref_ptr
	{
	protected:
		T *s;
	public:
		explicit ref_ptr (T *p = 0) : s(p) { if (s) s->ref(); }
		ref_ptr (const ref_ptr<T> &ref) : s(ref.s) { if (s) s->ref(); }
		~ref_ptr () { if (s) s->unref(); }

		ref_ptr<T> & operator= (const ref_ptr<T> &p) {
			if (s != p.s) {
				if (s)
					s->unref();
				s = p.s;
				if (s)
					s->ref();
			}
			return *this;
		}
		T& operator* () const { return *s; }
		T* operator-> () const { return s; }
	};

	class SongRef : public ref_ptr<Song>
	{
	public:
		explicit SongRef (Song *p = 0) : ref_ptr<Song>(p) {}
		SongRef (const Glib::ustring &url,
			 bool check_ready = true,
			 bool known_available = false);
	};
}

#endif
