//  Gnomoradio - roboradio/song-rainbow.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <vector>
#include <list>

#include "roboradio/song-rainbow.h"
#include "roboradio/init.h"
#include "rainbow/rdf-resource.h"
#include "rainbow/license.h"

using namespace std;
using namespace SigC;
using namespace Glib;
using namespace Rainbow;
using namespace Roboradio;

bool Roboradio::SongRainbow::initialized = false;

Roboradio::SongRainbow::SongRainbow (const ustring &loc)
	: SongLocal(loc),
	  fetching_info(0),
	  import_once_info_is_obtained(false)
{
	if (!initialized) {
		Roboradio::Init::get_rainbow().hub().signal_must_uncache.connect(slot(&Roboradio::SongRainbow::uncache_as_appropriate));
		initialized = true;
	}
}

Roboradio::SongRainbow::~SongRainbow ()
{
}

bool Roboradio::SongRainbow::importable ()
{
	return (&Roboradio::Init::get_rainbow());
}

void Roboradio::SongRainbow::download_resource ()
{
	Roboradio::Init::get_rainbow().hub().prepare_resource(resource);
}

void Roboradio::SongRainbow::import ()
{
	if (&*resource) {
		download_resource();
	} else {
		import_once_info_is_obtained = true;
		if (!fetching_info)
			obtain_available_info();
	}
}

void Roboradio::SongRainbow::uncache ()
{
	if (&*resource)
		Roboradio::Init::get_rainbow().hub().uncache_resource(resource);
}

void Roboradio::SongRainbow::obtain_available_info ()
{
	fetching_info++;
	RdfResource::get_and_do(get_url(), slot(*this, &SongRainbow::on_song_rdf_downloaded));
}

ustring Roboradio::SongRainbow::get_filename () const
{
	return get_status().ready ? resource->get_filename() : ustring();
}

void Roboradio::SongRainbow::download_done_callback (bool success)
{
	if (success) {
		set_status_available(true);
		set_status_ready(true);
		SongLocal::obtain_available_info();
	} else {
		set_import_progress(0);
		set_status_available(false);
	}
}

void Roboradio::SongRainbow::downloading_callback (unsigned int progress)
{
	set_import_progress(progress);
}

void Roboradio::SongRainbow::found_info_callback (bool success)
{
}

void Roboradio::SongRainbow::file_deleted_callback ()
{
	set_status_ready(false);
}

void Roboradio::SongRainbow::on_song_rdf_downloaded (xmlpp::Element *song,
						     Rainbow::ref_ptr<RdfResource> rdf)
{
	fetching_info--;
	if (!song)
		return;

	xmlpp::Node::NodeList children = song->get_children();
	for (xmlpp::Node::NodeList::iterator i = children.begin(); i != children.end(); ++i) {
		xmlpp::Element *el = dynamic_cast<xmlpp::Element*>(*i);
		if (!el)
			continue;
		xmlpp::TextNode *content = el->get_child_text();
		xmlpp::Attribute *resource_link = el->get_attribute("resource");
		if (el->get_name() == "title" || el->get_name() == "track") {
			if (content)
				set_info(el->get_name(), content->get_content());
		} else if (el->get_name() == "isPartOf") {
			if (content)
				set_info("album", content->get_content());
			else if (resource_link) {
				fetching_info++;
				RdfResource::get_and_do(RdfResource::absolute_uri(resource_link->get_value(), rdf->get_base_uri()),
							slot(*this, &SongRainbow::on_album_rdf_downloaded));
			}
		} else if (el->get_name() == "creator") {
			if (content)
				set_info("artist", content->get_content());
			else if (resource_link) {
				ustring abs = RdfResource::absolute_uri(resource_link->get_value(), rdf->get_base_uri());
				xmlpp::Element *secondary = rdf->get_secondary_info(abs);
				if (secondary) {
					on_artist_rdf_downloaded(secondary, rdf); // rdf should not be used here because it is not the same rdf file really, but for all uses of the function we are calling, it works correctly
				} else {
					fetching_info++;
					RdfResource::get_and_do(abs, slot(*this, &SongRainbow::on_artist_rdf_downloaded));
				}
			}
		} else if (el->get_name() == "audiofile") {
			if (resource_link)
				audiofiles.insert(RdfResource::absolute_uri(resource_link->get_value(),
									       rdf->get_base_uri()));
		} else if (el->get_name() == "license") {
			if (resource_link) {
				set_info("license", resource_link->get_value());
			}
		}
	}

	on_audiofiles_determined();
}

void Roboradio::SongRainbow::create_hub_resource ()
{
	// FIXME: using the last audiofile (alphabetically) may not always be right, and rbegin is also listed in the function below
	resource = Init::get_rainbow().hub().create(*audiofiles.rbegin());

	if (resource->downloaded())
		download_done_callback(true);

	resource->signal_found_info.connect(slot(*this, &SongRainbow::found_info_callback));
	resource->signal_download_done.connect(slot(*this, &SongRainbow::download_done_callback));
	resource->signal_downloading.connect(slot(*this, &SongRainbow::downloading_callback));
	resource->signal_file_deleted.connect(slot(*this, &SongRainbow::file_deleted_callback));
}

void Roboradio::SongRainbow::on_audiofiles_determined ()
{
	if (!&Roboradio::Init::get_rainbow())
		return;

	if (audiofiles.size()) {
		set_status_available(true);
		if (import_once_info_is_obtained) {
			create_hub_resource();
			download_resource();
		} else if (&*Init::get_rainbow().hub().find(*audiofiles.rbegin()))
			create_hub_resource();
	}
}

void Roboradio::SongRainbow::on_album_rdf_downloaded (xmlpp::Element *album,
						      Rainbow::ref_ptr<RdfResource> rdf)
{
	fetching_info--;
	if (!album)
		return;

	xmlpp::Node::NodeList children = album->get_children();
	for (xmlpp::Node::NodeList::iterator i = children.begin(); i != children.end(); ++i) {
		xmlpp::Element *el = dynamic_cast<xmlpp::Element*>(*i);
		if (!el)
			continue;
		xmlpp::TextNode *content = el->get_child_text();
		xmlpp::Attribute *resource_link = el->get_attribute("resource");
		if (el->get_name() == "title") {
			if (content)
				set_info("album", content->get_content());
		} else if (el->get_name() == "retail") {
			if (resource_link)
				set_info("retail",
					 RdfResource::absolute_uri(resource_link->get_value(),
								   rdf->get_base_uri()));
		} else if (el->get_name() == "img") {
			xmlpp::Attribute *src = el->get_attribute("src");
			if (src)
				set_info("coverart",
					 RdfResource::absolute_uri(src->get_value(),
								   rdf->get_base_uri()));
		}
	}
}

void Roboradio::SongRainbow::on_artist_rdf_downloaded (xmlpp::Element *artist,
						       Rainbow::ref_ptr<RdfResource> rdf)
{
	fetching_info--;
	if (!artist)
		return;

	xmlpp::Node::NodeList children = artist->get_children();
	for (xmlpp::Node::NodeList::iterator i = children.begin(); i != children.end(); ++i) {
		xmlpp::Element *el = dynamic_cast<xmlpp::Element*>(*i);
		if (!el)
			continue;
		xmlpp::TextNode *content = el->get_child_text();
		xmlpp::Attribute *resource_link = el->get_attribute("resource");
		if (el->get_name() == "title") {
			if (content)
				set_info("artist", content->get_content());
		} else if (el->get_name() == "website") {
			if (resource_link)
				set_info("artistsite",
					 RdfResource::absolute_uri(resource_link->get_value(),
								   rdf->get_base_uri()));
		}
	}
}

namespace Roboradio
{
	class SortByLastPlay
	{
	public:
		bool operator () (const SongRainbow *x, const SongRainbow *y) const {
			return x->last_played() < y->last_played();
		}
	};
}

void Roboradio::SongRainbow::uncache_as_appropriate (size_t kb)
{
	ssize_t remaining = kb;
	vector<SongRef> known = get_known_songs();
	list<SongRainbow*> rating[max_rating + 2];
	// note: for our purposes, rating of max_rating + 1
	//       implies that it is upcoming (in a radio list)

	for (vector<SongRef>::iterator i = known.begin(); i != known.end(); ++i) {
		SongRainbow *sr = dynamic_cast<SongRainbow*>(&**i);
		if (!sr || !&*sr->resource || !sr->resource->downloaded()
		    || sr->get_status().playing)
			continue;

		if (sr->get_status().upcoming > 0)
			rating[max_rating + 1].push_back(sr);
		else if (sr->get_rating() < 0) {
			remaining -= sr->resource->get_allocated_size() / 1024;
			sr->uncache();
		} else
			rating[sr->get_rating()].push_back(sr);
	}
	
	if (remaining < 0)
		return;
	
	// remove oldest unrated songs
	for (int r = 0; r <= max_rating + 1; ++r) {
		rating[r].sort(SortByLastPlay());
		
		for (list<SongRainbow*>::iterator i = rating[r].begin(); i != rating[r].end(); ++i) {
			remaining -= (*i)->resource->get_allocated_size() / 1024;
			(*i)->uncache();
			if (remaining < 0)
				return;
		}
	}
}
