//  Gnomoradio - gnomoradio/player-controls.cc
//  Copyright (C) 2003  Jim Garrison, Matt Gerginski
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "player-controls.h"
#include "main-window.h"

// image files
#include "control-play.xpm"
#include "control-stop.xpm"
#include "control-pause.xpm"
#include "control-next.xpm"
#include "control-previous.xpm"

using namespace std;
using namespace SigC;
using namespace Gtk;
using namespace Gdk;
using namespace Wraporadio;
using namespace Gnomoradio;

PlayerControls::PlayerControls (Wraporadio::Init *wrap,
				PlaylistView *pl_view,
				MainWindow *main_win)
	: timer("0:00"),
	  wr(wrap),
	  view(pl_view),
	  mw(main_win),
	  position(0, 1, 1),
	  user_changing_position(false),
	  play_img(Gdk::Pixbuf::create_from_xpm_data(control_play_xpm)),
	  stop_img(Gdk::Pixbuf::create_from_xpm_data(control_stop_xpm)),
	  pause_img(Gdk::Pixbuf::create_from_xpm_data(control_pause_xpm)),
	  next_img(Gdk::Pixbuf::create_from_xpm_data(control_next_xpm)),
	  prev_img(Gdk::Pixbuf::create_from_xpm_data(control_previous_xpm))
{
	// prepare images
	play_img.set(Gdk::Pixbuf::create_from_xpm_data(control_play_xpm));
	stop_img.set(Gdk::Pixbuf::create_from_xpm_data(control_stop_xpm));
	pause_img.set(Gdk::Pixbuf::create_from_xpm_data(control_pause_xpm));
	next_img.set(Gdk::Pixbuf::create_from_xpm_data(control_next_xpm));
	prev_img.set(Gdk::Pixbuf::create_from_xpm_data(control_previous_xpm));

	stop_img.show();
	pause_img.show();

	prev.add(prev_img);
	state.add(play_img);
	next.add(next_img);
	stop_img.show();
	pause_img.show();

	prev.signal_clicked().connect(slot(*this, &PlayerControls::on_prev_clicked));
	state.signal_clicked().connect(slot(*this, &PlayerControls::on_state_clicked));
	next.signal_clicked().connect(slot(*this, &PlayerControls::on_next_clicked));

	prev.set_sensitive(false);
	next.set_sensitive(false);

	position.signal_button_press_event()
		.connect(slot(*this, &PlayerControls::on_position_click_event), false);
	position.signal_button_release_event()
		.connect(slot(*this, &PlayerControls::on_position_click_event), false);

	position.set_draw_value(false);
	position.set_sensitive(false);

	song_info.set_line_wrap();
	song_info.set_justify(JUSTIFY_CENTER);

	EventBox *song_info_event_box = manage(new EventBox);
	song_info_event_box->add(song_info);
	song_info_event_box->set_events(Gdk::BUTTON_PRESS_MASK);
	song_info_event_box->signal_button_press_event()
		.connect(slot(*this, &PlayerControls::on_button_press_event));

	// packing
	HBox *button_hbox = manage(new HBox(true, 4));
	button_hbox->pack_start(prev);
	button_hbox->pack_start(state);
	button_hbox->pack_start(next);

	Alignment *button_align = manage(new Alignment(.5, .5, 0, 0));
	button_align->add(*button_hbox);

	HBox *position_hbox = manage(new HBox(false, 4));
	position_hbox->pack_start(timer, PACK_SHRINK);
	position_hbox->pack_start(position, PACK_EXPAND_WIDGET);

	Frame *song_info_frame = manage(new Frame);
	song_info_frame->add(*song_info_event_box);
	song_info_frame->set_shadow_type(SHADOW_IN);

	VBox *pos_song_info_vbox = manage(new VBox(false, 2));
	pos_song_info_vbox->pack_start(*song_info_frame);
	pos_song_info_vbox->pack_start(*position_hbox);

	pack_start(*button_align, PACK_SHRINK, 8);
	pack_start(*pos_song_info_vbox, PACK_EXPAND_WIDGET, 8);
	show_all();

	pl_view->signal_different_list.connect(slot(*this, &PlayerControls::on_viewed_playlist_changed));
	wr->player()->signal_state_changed.connect(slot(*this, &PlayerControls::on_state_changed));
	wr->player()->signal_song_changed.connect(slot(*this, &PlayerControls::on_song_changed));
	wr->player()->signal_state_changed.connect(slot(*this, &PlayerControls::on_song_changed));
}

void Gnomoradio::PlayerControls::on_prev_clicked ()
{
	wr->player()->prev();
}

void Gnomoradio::PlayerControls::on_state_clicked ()
{
	if (SongList(wr->player()->get_song_list()) == view->current_list())
		wr->player()->pause();
	else if (&*wr->player()->get_song())
		wr->player()->stop();
	else
		wr->player()->play(view->current_list().sl);
}

void Gnomoradio::PlayerControls::on_next_clicked ()
{
	wr->player()->next();
}

void Gnomoradio::PlayerControls::on_song_changed ()
{
	// update position slider and song info
	song_position_changed_connection.disconnect();
	if (&*wr->player()->get_song()) {
		position.set_sensitive(true);
		unsigned int len = wr->player()->get_song()->get_length();
		position.set_range(0, len ? len : 1);
		song_position_changed_connection = wr->player()->get_song()->signal_position_changed
			.connect(slot(*this, &PlayerControls::on_song_position_changed));

		Glib::ustring artist = wr->player()->get_song()->get_info("artist");
		Glib::ustring title  = wr->player()->get_song()->get_info("title");

		if (!title.empty() && !artist.empty())
			song_info.set_text(wr->player()->get_song()->get_info("artist") + " - " + wr->player()->get_song()->get_info("title"));
		else if (artist.empty())
			song_info.set_text(title);
		else if (title.empty())
			song_info.set_text(artist);
	} else {
		position.set_value(0);
		timer.set_text("0:00");
		position.set_sensitive(false);
		song_info.set_text("");
	}
}

void Gnomoradio::PlayerControls::on_state_changed ()
{
	// update state buttons
	if (&*wr->player()->get_song()) {
		if (view->current_list() != wr->player()->get_song_list()) {
			//stop
			state.remove();
			state.add(stop_img);
			prev.set_sensitive(true);
			next.set_sensitive(true);
		} else if (wr->player()->get_paused()) {
			//play
			state.remove();
			state.add(play_img);
			prev.set_sensitive(false);
			next.set_sensitive(false);
		} else {
			//pause
			state.remove();
			state.add(pause_img);
			prev.set_sensitive(true);
			next.set_sensitive(true);
		}
	} else {
		//play
		state.remove();
		state.add(play_img);
		prev.set_sensitive(false);
		next.set_sensitive(false);
	}
}

void Gnomoradio::PlayerControls::on_song_position_changed (Wraporadio::Time pos)
{
	if (!user_changing_position) {
		position.set_value(pos);
		timer.set_text(format_time(pos));
	}
}

void Gnomoradio::PlayerControls::on_viewed_playlist_changed (Wraporadio::SongList list)
{
	on_state_changed();
}

bool Gnomoradio::PlayerControls::on_position_click_event (GdkEventButton *event)
{
	if (event->button == 1) {
		if (event->type == GDK_BUTTON_PRESS) {
			user_changing_position = true; 
			position_timer_connection = position.signal_value_changed().connect(SigC::slot(*this, &PlayerControls::sync_timer_to_position_change));
		} else if (event->type == GDK_BUTTON_RELEASE) {
			user_changing_position = false;
			position_timer_connection.disconnect();
			if (&*wr->player()->get_song())
				wr->player()->get_song()->seek(Time(position.get_value()));
		}
	}

	return false;
}

void Gnomoradio::PlayerControls::sync_timer_to_position_change ()
{
	timer.set_text(format_time((unsigned int)position.get_value()));
}

Glib::ustring Gnomoradio::PlayerControls::format_time (Wraporadio::Time pos)
{
	int seconds, minutes, hours;
	
	hours = pos / 3600;
	pos -= hours * 3600;
	minutes = pos / 60;
	seconds = pos - minutes * 60;
	
	if (hours) {
		return((Glib::ustring) g_strdup_printf("%d:%02d:%02d", hours, minutes, seconds));
	} else {
		return((Glib::ustring) g_strdup_printf("%d:%02d", minutes, seconds));
	}
}

bool Gnomoradio::PlayerControls::on_button_press_event (GdkEventButton *event)
{
	if (event->type == GDK_BUTTON_PRESS
	    && event->button == 1
	    && &*wr->player()->get_song())
		mw->find_current_song();

	return true;
}
