/*
 *
 * gnome-scanneroption.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "gnomescanneroption.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#define GNOME_SCANNER_OPTION_ERROR			(g_type_qname (GNOME_TYPE_SCANNER_OPTION))
#define	GNOME_SCANNER_OPTION_PARENT_CLASS(klass)	(G_OBJECT_CLASS (g_type_class_peek_parent ((klass))))
#define GET_PRIVATE(obj)				(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCANNER_OPTION, GnomeScannerOptionPrivate))

typedef struct _GnomeScannerOptionPrivate	GnomeScannerOptionPrivate;

struct _GnomeScannerOptionPrivate {

  gboolean		dispose_has_run;

  /* PROPERTIES */
  gpointer			id;
  GValue			*value;
};

enum {
  PROP_0,
  PROP_ID,
  PROP_NAME,
  PROP_TITLE,
  PROP_DESC,
  PROP_VALUE,
  PROP_TYPE,
  PROP_UNIT,
  PROP_CONSTRAINT,
  PROP_FLAGS
};

enum {
  SIGNAL_CHANGED,
  N_SIGNALS
};

static guint signals[N_SIGNALS];


/* GOBJECT STUFFS */
void	gnome_scanner_option_set_property 		(GObject *obj,
							 guint property_id,
							 const GValue *value,
							 GParamSpec *pspec);

void	gnome_scanner_option_get_property 		(GObject *obj,
							 guint property_id,
							 GValue *value,
							 GParamSpec *pspec);

G_DEFINE_TYPE (GnomeScannerOption, gnome_scanner_option, G_TYPE_OBJECT);

void
gnome_scanner_option_class_init (GnomeScannerOptionClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property = gnome_scanner_option_set_property;
  gobject_class->get_property = gnome_scanner_option_get_property;

  g_type_class_add_private (gobject_class,
			    sizeof (GnomeScannerOptionPrivate));

  /* PROPERTIES */

  g_object_class_install_property (gobject_class,
				   PROP_ID,
				   g_param_spec_pointer ("id",
							 "Id",
							 "Unique identifier used by the backend.",
							 G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_NAME,
				   g_param_spec_string ("name",
							"Name",
							"String without spaces nor special characters describing the option",
							"",
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_TITLE,
				   g_param_spec_string ("title",
							"Title",
							"String containing the title of the option.",
							"",
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_DESC,
				   g_param_spec_string ("desc",
							"Description",
							"Long description of the option.",
							"",
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_VALUE,
				   g_param_spec_pointer ("value",
							 "Value",
							 "#GValue containing the value of the option",
							 G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
				   PROP_TYPE,
				   g_param_spec_gtype ("type",
						       "Type",
						       "The type of the option.",
						       G_TYPE_NONE,
						       G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_UNIT,
				   g_param_spec_int ("unit",
						     "Unit",
						     "Physical unit of options value if float or integer.",
						     0,
						     G_MAXINT,
						     GNOME_SCAN_UNIT_NONE,
						     G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_CONSTRAINT,
				   g_param_spec_pointer ("constraint",
							 "Constraint",
							 "The constraint value.",
							 G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_FLAGS,
				   g_param_spec_flags ("flags",
						       "Flags",
						       "Options flags.",
						       GNOME_TYPE_SCANNER_OPTION_FLAGS,
						       GNOME_SCANNER_OPTION_INACTIVE,
						       G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  /* SIGNALS */

  /**
   * GnomeScannerOption::changed:
   * @option: The emitting #GnomeScannerOption
   *
   * Emitted when the value is changed.
   */
  signals[SIGNAL_CHANGED] = g_signal_new ("changed",
					  GNOME_TYPE_SCANNER_OPTION,
					  G_SIGNAL_RUN_FIRST,
					  G_STRUCT_OFFSET (GnomeScannerOptionClass, changed),
					  NULL,
					  NULL,
					  g_cclosure_marshal_VOID__VOID,
					  G_TYPE_NONE,
					  0);
}


void
gnome_scanner_option_init (GnomeScannerOption *option)
{
  GnomeScannerOptionPrivate *priv = GET_PRIVATE (option);

  priv->id		= NULL;
  priv->value		= NULL;
  option->name		= NULL;
  option->title		= NULL;
  option->desc		= NULL;
  option->unit		= GNOME_SCAN_UNIT_NONE;
  option->constraint	= NULL;
  option->flags		= 0;
}

void
gnome_scanner_option_set_property (GObject *obj,
				   guint property_id,
				   const GValue *value,
				   GParamSpec *pspec)
{
  GnomeScannerOption* option = GNOME_SCANNER_OPTION (obj);
  GnomeScannerOptionPrivate *priv = GET_PRIVATE (option);

  switch (property_id) {
  case PROP_ID:
    priv->id = g_value_get_pointer (value);
    break;
  case PROP_NAME:
    option->name = g_strdup (g_value_get_string (value));
    break;
  case PROP_TITLE:
    option->title = g_strdup (g_value_get_string (value));
    break;
  case PROP_DESC:
    option->desc = g_strdup (g_value_get_string (value));
    break;
  case PROP_VALUE:
    priv->value = g_value_get_pointer (value);
    break;
  case PROP_TYPE:
    break;
  case PROP_UNIT:
    option->unit = g_value_get_int (value);
    break;
  case PROP_CONSTRAINT:
    option->constraint = g_value_get_pointer (value);
    break;
  case PROP_FLAGS:
    option->flags = g_value_get_flags (value);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id,pspec);
    break;
  }
}

void
gnome_scanner_option_get_property (GObject *obj,
				   guint property_id,
				   GValue *value,
				   GParamSpec *pspec)
{
  GnomeScannerOption *option = GNOME_SCANNER_OPTION (obj);
  GnomeScannerOptionPrivate *priv = GET_PRIVATE (option);

  switch (property_id) {
  case PROP_ID:
    g_value_set_pointer (value, priv->id);
    break;
  case PROP_NAME:
    g_value_set_string (value, option->name);
    break;
  case PROP_TITLE:
    g_value_set_string (value, option->title);
    break;
  case PROP_DESC:
    g_value_set_string (value, option->desc);
    break;
  case PROP_VALUE:
    g_value_set_pointer (value, priv->value);
    break;
  case PROP_TYPE:
    g_value_set_int (value, G_VALUE_TYPE (priv->value));
    break;
  case PROP_UNIT:
    g_value_set_int (value, option->unit);
    break;
  case PROP_CONSTRAINT:
    g_value_set_pointer (value, option->constraint);
    break;
  case PROP_FLAGS:
    g_value_set_flags (value, option->flags);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,property_id,pspec);
    break;
  }
}




/**
 * gnome_scanner_option_get_int:
 * @option: a #GnomeScannerOption
 * 
 * Convenient function similar to g_value_get_int() to get an integer
 * value from a #GnomeScannerOption.
 * 
 * Return value: An integer value or 0 if the option doesn't holds an integer
 */
gint
gnome_scanner_option_get_int (GnomeScannerOption *option)
{
  g_return_val_if_fail (GNOME_IS_SCANNER_OPTION (option), 0);
  g_return_val_if_fail (G_VALUE_HOLDS_INT (GET_PRIVATE (option)->value), 0);
  return g_value_get_int (GET_PRIVATE (option)->value);
}

/**
 * gnome_scanner_option_set_int:
 * @option: a #GnomeScannerOption 
 * @value: an integer
 * 
 * Convenient function to set an integer value to an option.
 */
void
gnome_scanner_option_set_int (GnomeScannerOption *option,
			      gint value)
{
  g_return_if_fail (GNOME_IS_SCANNER_OPTION (option));
  g_value_set_int (GET_PRIVATE (option)->value, value);
}

/**
 * gnome_scanner_option_get_float:
 * @option: a #GnomeScannerOption
 * 
 * Convenient function similar to g_value_get_float() for a
 * #GnomeScannerOption.
 * 
 * Return value: a float or 0
 */
gfloat
gnome_scanner_option_get_float (GnomeScannerOption *option)
{
  g_return_val_if_fail (GNOME_IS_SCANNER_OPTION (option), 0);
  g_return_val_if_fail (G_VALUE_HOLDS_FLOAT (GET_PRIVATE (option)->value), 0);
  return g_value_get_float (GET_PRIVATE (option)->value);
}

/**
 * gnome_scanner_option_set_float:
 * @option: a #GnomeScannerOption
 * @value: a float
 *
 * Convenient function similar to g_value_set_float() for a
 * #GnomeScannerOption.
 * 
 */
void
gnome_scanner_option_set_float (GnomeScannerOption *option,
				gfloat value)
{
  g_return_if_fail (GNOME_IS_SCANNER_OPTION (option));
  g_value_set_float (GET_PRIVATE (option)->value, value);
}

/* PROPERTIES */

/**
 * gnome_scanner_option_get_id:
 * @option: a #GnomeScannerOption
 * 
 * Return the unique id of the option. This function should be called
 * only by #GnomeScanBackend.
 * 
 * Return value: The option unique id
 */
gpointer
gnome_scanner_option_get_id (GnomeScannerOption *option)
{
  g_return_val_if_fail (GNOME_IS_SCANNER_OPTION (option), 0);
  return GET_PRIVATE (option)->id;
}

/**
 * gnome_scanner_option_get_name:
 * @option: a #GnomeScannerOption
 * 
 * Returns a string containing the name of the option.
 * 
 * Return value: The option's name
 */
const gchar*
gnome_scanner_option_get_name (GnomeScannerOption *option)
{
  g_return_val_if_fail (GNOME_IS_SCANNER_OPTION (option), NULL);
  return option->name;
}

/**
 * gnome_scanner_option_get_title:
 * @option: a #GnomeScannerOption
 * 
 * Returns a string containing the title of the option.
 * 
 * Return value: The option's title
 */
const gchar*
gnome_scanner_option_get_title (GnomeScannerOption *option)
{
  g_return_val_if_fail (GNOME_IS_SCANNER_OPTION (option), NULL);
  return option->title;
}

/**
 * gnome_scanner_option_get_desc:
 * @option: a #GnomeScannerOption
 * 
 * Returns a string containing a description of the option
 * 
 * Return value: The option's description
 */
const gchar*
gnome_scanner_option_get_desc (GnomeScannerOption *option)
{
  g_return_val_if_fail (GNOME_IS_SCANNER_OPTION (option), NULL);
  return option->desc;
}

/**
 * gnome_scanner_option_get_value:
 * @option: a #GnomeScannerOption
 * 
 * Return the value of the option as a #GValue..
 * 
 * Return value: The option's value
 */
GValue*
gnome_scanner_option_get_value (GnomeScannerOption *option)
{
  g_return_val_if_fail (GNOME_IS_SCANNER_OPTION (option), NULL);
  return GET_PRIVATE (option)->value;
}

/**
 * gnome_scanner_option_get_flags:
 * @option: a #GnomeScannerOption
 *
 * Get Option's flags.
 * 
 * Returns: Option's flags
 * See: #GnomeScannerOptionFlags
 **/
GnomeScannerOptionFlags
gnome_scanner_option_get_flags (GnomeScannerOption *option)
{
  g_return_val_if_fail (GNOME_IS_SCANNER_OPTION (option), 0);
  return option->flags;
}

/**
 * gnome_scanner_option_set_value:
 * @option: a #GnomeScannerOption
 * @value: a #GValue to copy to the option value
 * 
 * The value is copied to the option value this mean that value's
 * types must correspond.
 * 
 */
void
gnome_scanner_option_set_value (GnomeScannerOption *option,
				GValue *value)
{
  g_return_if_fail (GNOME_IS_SCANNER_OPTION (option));
  g_value_copy (GET_PRIVATE (option)->value, value);
  g_signal_emit_by_name (option,
			 "changed");
}

/**
 * gnome_scanner_option_get_value_type:
 * @option: a #GnomeScannerOption
 * 
 * Retreive the type of the option's value.
 * 
 * Return value: The type of the option's value
 */
GType
gnome_scanner_option_get_value_type (GnomeScannerOption *option)
{
  g_return_if_fail (GNOME_IS_SCANNER_OPTION (option));
  return G_VALUE_TYPE (GET_PRIVATE (option)->value);
}

/**
 * gnome_scanner_option_get_unit:
 * @option: a #GnomeScannerOption
 * 
 * Retreive the unit of the option value.
 * 
 * Return value: the unit of the option
 */
GnomeScanUnit
gnome_scanner_option_get_unit (GnomeScannerOption *option)
{
  g_return_val_if_fail (GNOME_IS_SCANNER_OPTION (option), GNOME_SCAN_UNIT_NONE);
  return option->unit;
}

/**
 * gnome_scanner_option_get_constraint:
 * @option: a #GnomeScannerOption
 * 
 * The option constraint or NULL.
 * 
 * Returns: The option constraint.
 **/
GnomeScannerOptionConstraint*
gnome_scanner_option_get_constraint (GnomeScannerOption *option)
{
  g_return_val_if_fail (GNOME_IS_SCANNER_OPTION (option), NULL);
  return option->constraint;
}

