/* gnome-rdp
 * Copyright (C) 2005-2006 Balazs Varkonyi
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
using System;
using Gtk;
using Gdk;
using Glade;
using System.Diagnostics;
using System.Data;
using Mono.Data.SqliteClient;
using System.Threading;
using Mono.Unix;
using System.Collections;
using System.Reflection;
using System.IO;

namespace GnomeRDP 
{
	public class MainApp
	{		
#pragma warning disable 0649
		[Widget]Gtk.Window			wndMain;
		[Widget]ScrolledWindow 		scrolledwindow1;
		[Widget]ToolButton			tbNew;
		[Widget]ToolButton			tbProperties;
		[Widget]ToolButton			tbDelete;
		[Widget]ToolButton			tbNewCategory;
		[Widget]ToolButton			tbConnect;
		[Widget]MenuBar				mainMenu;
		[Widget]Statusbar			sbStatus;
#pragma warning restore 0649
		
		private Gtk.Action actionQuit;
		private Gtk.Action actionNew;
		private Gtk.Action actionProperties;
		private Gtk.Action actionDelete;
		private Gtk.Action actionConnect;
		private Gtk.Action actionNewCategory;
		private Gtk.Action actionAbout;
		private static string tmpFileName = System.IO.Path.GetTempFileName();
		private string par = "";
		private ProcessCaller prc;
		private SessionTreeStore sessionTreeStore;
		private SessionTreeView	sessionTreeView;		
		private bool startVisible = true; 
		private StatusIcon trayIcon;
		
		public static void Main(string[] args)
		{
			new MainApp(args);
		}	

		public MainApp(string[] args) 
		{
			Catalog.Init("gnome-rdp", Defines.LOCALE_DIR);
 
			if (this.ParseArguments(args))
			{	
				Gdk.Threads.Init();
				Application.Init ();
	
				Stream glade = Assembly.GetExecutingAssembly().GetManifestResourceStream("gui.glade");
				Glade.XML gxml = new Glade.XML(glade, "wndMain", "gnome-rdp");
				gxml.Autoconnect (this);
	
				this.CreateActions();
				if (this.InitializeComponent())
				{
					this.CreateMenu();
					CreatePopupTrayMenu();
					Gdk.Threads.Enter();
					wndMain.Visible = startVisible;
					Application.Run ();
					Gdk.Threads.Leave();
				}
			}
		}
		
		private bool ParseArguments(string[] args)
		{
			bool continueRunning = true;
			foreach (string arg in args)
			{
				switch (arg)
				{
					case "--start-hidden":
					case "-m":
						startVisible = false;
						break;
					case "--version":
					case "-v":
						PrintOutVersion();
						continueRunning = false;
						break;
					case "--help":
					case "-h":
						PrintOutHelp();
						continueRunning = false;
						break;
					case "--debug":
					case "-d":
						Defines.DEBUG = true;
						break;
				}
			}
			return continueRunning;
		}
		
		private void PrintOutVersion()
		{
			Console.WriteLine(Defines.PACKAGE + " " + Defines.VERSION);
			Console.WriteLine("Copyright (c) 2005-2008 Balazs Varkonyi");
		}

		private void PrintOutHelp()
		{
			Console.WriteLine(Catalog.GetString("Usage: gnome-rdp [options]") + "\n");
			Console.WriteLine(Catalog.GetString("Options:") + "\n");
			Console.WriteLine("\t--start-hidden | -m: " + 
				Catalog.GetString("start minimized to tray"));
			Console.WriteLine("\t--version      | -v: " + 
				Catalog.GetString("display version information"));
			Console.WriteLine("\t--help         | -h: " + 
				Catalog.GetString("display this help message") + "\n");
		}
		
		private void CreatePopupTrayMenu()
		{
			this.trayIcon = new StatusIcon(Gdk.Pixbuf.LoadFromResource("gnome-rdp-icon.png")); 
            this.trayIcon.Visible = true;
            this.trayIcon.Activate += OnTrayIconClicked;
            this.trayIcon.PopupMenu += OnTrayIconPopup;
			this.trayIcon.Tooltip = Catalog.GetString("gnome-rdp");			
		}
		
		private bool InitializeComponent()
		{
			bool result = true;
			
			wndMain.Icon = Gdk.Pixbuf.LoadFromResource("gnome-rdp-icon.png");
			wndMain.DeleteEvent += new Gtk.DeleteEventHandler(OnWindowDeleteEvent);
			wndMain.Title = "Gnome-RDP";
			wndMain.Resizable = true;

			sessionTreeStore = new SessionTreeStore();
			sessionTreeView = new SessionTreeView(sessionTreeStore, false);
			if (!System.IO.File.Exists(Defines.DatabasePath))
			{
				Options.CheckDatabase();
				Configuration.CreateDatabase();
			}
			if (!Configuration.CheckDatabaseVersion())
			{
				MessageDialog md = new MessageDialog(this.wndMain, 
					DialogFlags.Modal, MessageType.Error, ButtonsType.Close, 
					Catalog.GetString("Invalid database version!"));
				md.Run();
				md.Destroy();
				return false;
			}
			//ApplicationOptions.VerifyDatabase();
			
			this.RefreshSessionList();
			scrolledwindow1.Add(sessionTreeView);
			sessionTreeView.ExpandAll();
			sessionTreeView.WidgetEvent += new Gtk.WidgetEventHandler(OnWidgetEvent);
			sessionTreeView.RowActivated += OnConnectClicked;
			tbNew.Clicked += OnNewClicked;
			tbProperties.Clicked += OnOptionsClicked;
			tbDelete.Clicked += OnDeleteClicked;
			tbNewCategory.Clicked += OnNewCategoryClicked;
			tbConnect.Clicked += OnConnectClicked;
			scrolledwindow1.ShowAll();
			return result;
		}
		
		private void CreateActions()
		{
			actionQuit 			= new Gtk.Action("quit", 		 Catalog.GetString("_Quit"),		null, "gtk-quit");
			actionNew 			= new Gtk.Action("new", 	 	 Catalog.GetString("_New"), 		null, "gtk-new");
			actionProperties 	= new Gtk.Action("properties", 	 Catalog.GetString("_Properties"), 	null, "gtk-properties");
			actionDelete	 	= new Gtk.Action("delete", 		 Catalog.GetString("_Delete"), 		null, "gtk-delete");
			actionConnect	 	= new Gtk.Action("connect",		 Catalog.GetString("_Connect"), 	null, "gtk-connect");
			actionNewCategory 	= new Gtk.Action("newcategory",	 Catalog.GetString("_New Group"), 	null, "gtk-add");
			actionAbout		 	= new Gtk.Action("about", 		 Catalog.GetString("_About"), 		null, "gtk-about");
			
			actionQuit.Activated 			+= OnQuitClicked;
			actionNew.Activated  			+= OnNewClicked;
			actionProperties.Activated 		+= OnOptionsClicked;
			actionDelete.Activated 			+= OnDeleteClicked;
			actionConnect.Activated			+= OnConnectClicked;
			actionNewCategory.Activated		+= OnNewCategoryClicked;
			actionAbout.Activated 			+= OnAboutClicked;
		}
		
		private void CreateMenu()
		{
			//File
			Gtk.Menu fileMenu = new Gtk.Menu();
			Gtk.MenuItem fileItem = new Gtk.MenuItem(Catalog.GetString("_File"));
			fileItem.Submenu = fileMenu;
			fileMenu.Append(actionQuit.CreateMenuItem());
			
			//Session
			Gtk.Menu sessionMenu = new Gtk.Menu();
			Gtk.MenuItem sessionItem = new Gtk.MenuItem(Catalog.GetString("_Session"));
			sessionItem.Submenu = sessionMenu;
			sessionMenu.Append(actionNew.CreateMenuItem());
			sessionMenu.Append(actionProperties.CreateMenuItem());
			sessionMenu.Append(actionDelete.CreateMenuItem());
			sessionMenu.Append(actionConnect.CreateMenuItem());

			//Group
			Gtk.Menu categoryMenu = new Gtk.Menu();
			Gtk.MenuItem categoryItem = new Gtk.MenuItem(Catalog.GetString("_Group"));
			categoryItem.Submenu = categoryMenu;
			categoryMenu.Append(actionNewCategory.CreateMenuItem());
			
			//Options
			Gtk.MenuItem optionsItem = new Gtk.MenuItem(Catalog.GetString("_Options"));
			Gtk.Menu optionsSubMenu = new Gtk.Menu();
			optionsItem.Submenu = optionsSubMenu;
			Gtk.CheckMenuItem optionsUseKeyringItem = new CheckMenuItem(Catalog.GetString("_Use Keyring"));
			optionsUseKeyringItem.Active = Options.UseKeyring;
			optionsUseKeyringItem.Activated += OnUseKeyringClicked;
			optionsSubMenu.Append(optionsUseKeyringItem);
			
			//Help
			Gtk.Menu helpMenu = new Gtk.Menu();
			Gtk.MenuItem helpItem = new Gtk.MenuItem(Catalog.GetString("_Help"));
			helpItem.Submenu = helpMenu;
			helpMenu.Append(actionAbout.CreateMenuItem());
			
			mainMenu.Append(fileItem);
			mainMenu.Append(sessionItem);
			mainMenu.Append(categoryItem);
			mainMenu.Append(optionsItem);
			mainMenu.Append(helpItem);
			mainMenu.ShowAll();
		}
		
		private void RefreshSessionList()
		{
			sessionTreeStore.Clear();
			RefreshSessionListRecursive(0, new TreeIter());
		}

		private void RefreshSessionListRecursive(int parentId, TreeIter parentIter)
		{
			foreach (Configuration cfg in Configuration.LoadSessions())
			{
				if ((parentId == 0) && (cfg.ParentId == 0))
				{
					TreeIter iter = sessionTreeStore.AppendValuesSession(cfg);
					if (cfg.IsCategory)
					{
						this.RefreshSessionListRecursive(cfg.Id, iter);
					}
				}
				else
				{
					if (cfg.ParentId == parentId)
					{
						TreeIter iter = sessionTreeStore.AppendValuesSession(parentIter, cfg);
						if (cfg.IsCategory)
						{
							this.RefreshSessionListRecursive(cfg.Id, iter);
						}
					}
				}
			}
		}

		public void DoWork(Configuration config)
		{
			if (config.Protocol == 0)
			{
				prc = new ProcessCaller(
					"rdesktop", 
					par, 
					"/", 
					false, 
					"", 
					this.wndMain);
				prc.StartProcess();

			}
			if (config.Protocol == 1)
			{
//                VNC vnc = new VNC();
                
				if (config.Password.Length > 0) 
				{
                    prc = new ProcessCaller(
                                            "vncpasswd",
                                            tmpFileName,
                                            "/",
                                            true,
                                            config.Password,
                                            this.wndMain);
                    prc.StartProcess();
                    Thread.Sleep(1000);
                }
                
                Console.WriteLine(par);
				prc = new ProcessCaller(
					"xtightvncviewer", 
					par, 
					"/", 
					false,
					config.Password, 
					this.wndMain);
				prc.StartProcess();
			}
			if (config.Protocol == 2)
			{
				prc = new ProcessCaller(
					"gnome-terminal",
					par,
					"/",
					true, 
					config.Password, 
					this.wndMain);					
				prc.StartProcess();
			}
		}

		private bool OnQuit()
		{
			int result = (int)Gtk.ResponseType.Yes;
			if (ProcessCaller.ConnectionCount > 0)
			{
				result = QuitQuestion();
				if (result == (int)Gtk.ResponseType.Yes)
				{	
					foreach (Process process in ProcessCaller.Processes)
					{
						if (!process.HasExited)
						{
							process.Kill();
						}	
					}
				}
			}
			if (result == (int)Gtk.ResponseType.Yes)
			{
				Application.Quit ();
				return false;
			}
			return true;
		}

		private int QuitQuestion()
		{
			int result;
			string msg = "<big><b>" + Catalog.GetString("There is some opened session!") + "</b></big>\n\n" +
				Catalog.GetString("Are you sure to quit the application?") + "\n" +
				"<i>" + Catalog.GetString("Opened sessions will be killed.") + "</i>";
			MessageDialog md = new MessageDialog(this.wndMain, DialogFlags.Modal, MessageType.Question, ButtonsType.YesNo, msg);
			result = md.Run();
			md.Destroy();
			return result;
		}
		
		private void parseConfiguration(Configuration config)
		{
			par = "";
			if (config.Protocol == 0)
			{
				switch (config.SrvType)
				{
					case 0:
						par += "-4";
						break;
					case 1:
						par += "-5";
						break;
				}
				
				if (config.User.Length > 0) 
				{
					if (config.User.Contains(" "))
					{
						par += string.Format(" -u \"{0}\"", config.User);
					}
					else
					{
						par += " -u " + config.User;
					}
				}
				if (config.Password.Length > 0) 
				{
					par += " -p " + config.Password;
				}
				if (config.Domain.Length > 0) 
				{
					par += " -d " + config.Domain;
				}
				switch (config.ColorDepth) 
				{
					case 0:
						par += " -a 8";
						break;
					case 1:
						par += " -a 15";
						break;
					case 2:
						par += " -a 16";
						break;
					case 3:
						par += " -a 24";
						break;
				}
				if (config.ScreenResolutionX == 0)
				{
					par += " -f";
				}
				else
				{
					par += " -g " + config.ScreenResolutionX + "x" + config.ScreenResolutionY;	
				}
				switch (config.SoundRedirection)
				{
					case 0:
						par += " -r sound:off";
						break;		
					case 1:
						par += " -0 -r sound:remote";
						break;
					case 2:
						par += " -0 -r sound:local";
						break;
				}
				par += " -k " + config.KeyboardLang;		
				par += " " + config.Computer;
			}
			if (config.Protocol == 1)
			{
				par ="";
				if (config.CompressionLevel == 1)
				{
					par += " -encodings \"zlib hextile raw\" "; 
				}else if (config.CompressionLevel == 2)
				{
					par += " -encodings \"hextile zlib raw\" "; 
				}else if (config.CompressionLevel == 3)
				{
					par += " -encodings \"raw zlib hextile\" "; 
				}
				if (config.ImageQuality == 1)
				{
					par += " -quality 2 ";
				}else if (config.ImageQuality == 2)
				{
					par += " -quality 5 ";
				}else if (config.ImageQuality == 3)
				{
					par += " -quality 9 ";
				}else if (config.ImageQuality == 4)
				{
					par += " -truecolor ";
				}
				if ((config.ImageQuality >0) || (config.CompressionLevel>0))
				{
					par += " -AutoSelect=0 ";
				}
				if (config.WindowMode > 0)
				{
					par += " -fullscreen ";
				}
				if (config.ConnectionType == 0)
				{
					par += " -viewonly ";
				}else if (config.ConnectionType == 2)
				{
					par += " -shared";
				}
				if (config.Password.Length > 0) 
				{
					par += " -passwd ";
					par += tmpFileName+" ";
				}
				par += config.Computer;
			}
			if (config.Protocol == 2)
			{
				par += "--command=\"ssh";
				if (config.TerminalSize == 1) 
				{
					par = " --full-screen --command=\"ssh";
				}						
				if (config.User.Length > 0) 
				{
					par += " -l " + config.User;
				}

				par += " -e none -t " + config.Computer + "\"";
			}
		}		
		
		private void OnWindowDeleteEvent (object sender, DeleteEventArgs a) 
		{
			wndMain.Visible = false;
			a.RetVal = true;
		}

		private void OnQuitClicked (object sender, EventArgs a) 
		{
			OnQuit();
		}		

		private void OnConnectClicked (object sender, EventArgs a) 
		{	
			Configuration cfg = new Configuration();
			switch (sender.ToString())
			{
				case "Gtk.ToolButton":
				case "GnomeRDP.SessionTreeView":
					cfg = sessionTreeView.config;
					break;
				case "Gtk.ImageMenuItem":
					ImageMenuItem item = (ImageMenuItem)sender;
					foreach (Configuration config in Configuration.LoadSessions())
					{
						if (config.Id.ToString() == item.Name)
						{
							cfg = config;
						}
					}
					break;					
			}
			this.parseConfiguration(cfg);
			if ((cfg.Computer.Length > 0) && (!cfg.IsCategory))  
			{	
				DoWork(cfg);
			} 			
			else 
			{
				TreeIter iter;
				sessionTreeView.Selection.GetSelected(out iter);
				if (sessionTreeView.GetRowExpanded(sessionTreeView.Model.GetPath(iter)))
				{
					sessionTreeView.CollapseRow(sessionTreeView.Model.GetPath(iter));
				}
				else
				{
					sessionTreeView.ExpandRow(sessionTreeView.Model.GetPath(iter), false);
				}
				/*
				MessageDialog md = new MessageDialog(wndMain, DialogFlags.DestroyWithParent, MessageType.Error, ButtonsType.Close, "<b><big>" + Catalog.GetString("Computer isn't specified") + "</big></b>\n\n" + Catalog.GetString("Please specify the name or IP address of remote host."));
				md.Run();
				md.Destroy();
				*/
			}
		}
		
		private void OnNewClicked (object sender, EventArgs a)
		{
			Configuration cfg = new Configuration();
			OptionsDialog optionsDialog = new OptionsDialog(cfg);
			optionsDialog.Run();
			this.RefreshSessionList();
		}
		
		private void OnOptionsClicked (object sender, EventArgs a)
		{
			TreeIter iter;
			TreePath path;
			sessionTreeView.Selection.GetSelected(out iter);
			path = sessionTreeView.Model.GetPath(iter);
			if (sessionTreeView.config.IsCategory)
			{
				CategoryDialog categoryDialog = new CategoryDialog(sessionTreeView.config);
				categoryDialog.Run();
			}
			else
			{
				OptionsDialog optionsDialog = new OptionsDialog(sessionTreeView.config);
				optionsDialog.Run();
			}
			this.RefreshSessionList();
			sessionTreeView.ExpandToPath(path);
		}
		
		private void OnDeleteClicked (object sender, EventArgs a)
		{
			string msg = "<b><big>" + Catalog.GetString("Confirm delete operation") + 
				"</big></b>\n\n" + Catalog.GetString("Are you sure do you want to delete") + 
				" <b>'" + sessionTreeView.config.SessionName + "'</b> ";
			if (sessionTreeView.config.IsCategory)
			{
				msg += Catalog.GetString("group and it's all subitem?");
			}
			else
			{
				msg += Catalog.GetString("session?");
			}
			MessageDialog md = new MessageDialog(wndMain, DialogFlags.DestroyWithParent, 
				MessageType.Question, ButtonsType.YesNo, msg);
			int dresult = md.Run();
												
			if (dresult == (int)Gtk.ResponseType.Yes) 
			{
				sessionTreeView.config.DeleteSession();
				this.RefreshSessionList();
			}
			md.Destroy();		
		}	
				
		private void OnAboutClicked (object sender, EventArgs a)
		{
			AboutDialog aboutDialog = new AboutDialog();
			aboutDialog.Run();
		}
				
		private void OnWidgetEvent(object sender, WidgetEventArgs a)
		{
			if ((sessionTreeView.config != null) &&
				(!sessionTreeView.config.IsCategory))
			{
				tbProperties.Sensitive = true;
				tbDelete.Sensitive = true;
				tbConnect.Sensitive = true;
				actionProperties.Sensitive = true;
				actionDelete.Sensitive = true;
				actionConnect.Sensitive = true;
				sbStatus.Pop(0);
				sbStatus.Push(0, sessionTreeView.config.SessionName);
			}
			else
			{
				if ((sessionTreeView.config != null) &&
					(sessionTreeView.config.IsCategory))
				{
					tbDelete.Sensitive = true;
					tbProperties.Sensitive = true;
					actionDelete.Sensitive = true;
					actionProperties.Sensitive = true;
				}
				else
				{
					tbDelete.Sensitive = false;
					tbProperties.Sensitive = false;
					actionDelete.Sensitive = false;
					actionProperties.Sensitive = false;					
				}
				tbConnect.Sensitive = false;	
				actionConnect.Sensitive = false;
				sbStatus.Pop(0);
				sbStatus.Push(0, "");
			}	
		}
		
		private void OnNewCategoryClicked(object sender, EventArgs a)
		{
			CategoryDialog categoryDialog = new CategoryDialog(new Configuration());
			categoryDialog.Run();
			this.RefreshSessionList();
		}
		
		private void OnUseKeyringClicked(object sender, EventArgs a)
		{
			bool value = !Options.UseKeyring;
			
			((Gtk.CheckMenuItem)sender).Active = value;
			Options.UseKeyring = value;
		}
						
		private void OnTrayIconClicked(object sender, EventArgs a)
		{
			if (wndMain.Visible)
			{
				wndMain.Visible = false;
			}
			else
			{
				wndMain.Visible = true;
			}
		}
		
		private void OnTrayIconPopup(object sender, EventArgs a)
		{
			Menu popupMenu = this.CreateSessionsPopupMenu(0, null);
			popupMenu.Append(new SeparatorMenuItem());
			popupMenu.Append(actionQuit.CreateMenuItem());
			popupMenu.ShowAll();
			popupMenu.Popup();			
		}
		
		private Menu CreateSessionsPopupMenu(int parentId, Menu parentMenu)
		{
			Menu submenu = new Menu();
			foreach (Configuration cfg in Configuration.LoadSessions())
			{
				if (cfg.ParentId == parentId)
				{
					Gtk.Image img = new Gtk.Image();
					switch (cfg.Protocol)
					{
						case 0:
							img.Pixbuf = Gdk.Pixbuf.LoadFromResource("rdp_16.png");
							break;
						case 1:
							img.Pixbuf = Gdk.Pixbuf.LoadFromResource("vnc_16.png");
							break;
						case 2:
							img.Pixbuf = Gdk.Pixbuf.LoadFromResource("ssh_16.png");
							break;
						default:
							img.Pixbuf = Gdk.Pixbuf.LoadFromResource("group_16.png");
							break;
					}
					ImageMenuItem item = new ImageMenuItem(cfg.SessionName);
					item.Name = cfg.Id.ToString();
					item.Image = img;
					submenu.Append(item);
					if (cfg.IsCategory)
					{
						Menu smenu = this.CreateSessionsPopupMenu(cfg.Id, submenu);
						item.Submenu = smenu;
					}
					else
					{
						item.Activated += new EventHandler(OnConnectClicked);
					}
				}
			}
			return submenu;
		}		
	}
}
