/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* connection-list.cpp
 *
 * Copyright (C) 2005 Takuo KITAME.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * Author: Takuo KITAME.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdarg.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>

#include "utils.h"
#include "connection-list.h"
#include "gnome-peercast.h"

#define DELAY_UPDATE 1000 /* 1 second */

enum
{
	LIST_INDEX,   /* INDEX */
	LIST_TYPE,   /* Type */
	LIST_STATUS,   /* Status */
	LIST_TIME,   /* uptime */
	LIST_HOST,   /* Host Name and port */
	LIST_COLOR,  /* Host color */
	LIST_POS,    /* Sync Position */
	LIST_AGENT,  /* Agent Name */
	LIST_RATE,   /* Kbits/s */
	N_COLUMNS
};


static GtkTreeModel *
create_model (void)
{
	GtkListStore * store;

	store = gtk_list_store_new (N_COLUMNS,
				    G_TYPE_UINT, /* INDEX */
				    G_TYPE_STRING, /* TYPE */
				    G_TYPE_STRING, /* Status */
				    G_TYPE_STRING, /* uptime */
				    G_TYPE_STRING, /* host  */
				    G_TYPE_STRING, /* color  */
				    G_TYPE_UINT,   /* pos */
				    G_TYPE_STRING, /* agent */
				    G_TYPE_STRING   /* RATE */
		);
	return GTK_TREE_MODEL (store);
}


static void
add_columns (GtkTreeView *treeview)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Index"), renderer,
							"text", LIST_INDEX,
							NULL);
	gtk_tree_view_column_set_visible (col, FALSE);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Type"), renderer,
							"text", LIST_TYPE,
							NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Status"), renderer,
							"text", LIST_STATUS,
							NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Time"),renderer,
							"text", LIST_TIME,
							NULL);
	g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("IP:Port"), renderer,
							"text", LIST_HOST,
							"foreground", LIST_COLOR,
							NULL);
	gtk_tree_view_column_set_resizable (col, TRUE);
	gtk_tree_view_append_column (treeview, col);


	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Position"), renderer,
							"text", LIST_POS,
							NULL);
	g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Agent"), renderer,
							"text", LIST_AGENT,
							NULL);
	gtk_tree_view_column_set_resizable (col, TRUE);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Bitrate"), renderer,
							"text", LIST_RATE,
							NULL);
	g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
	gtk_tree_view_append_column (treeview, col);
}

static gboolean
cl_find_row_by_index (GtkTreeView *tv, GtkTreeIter *iter, guint i)
{
	GtkTreeModel *model;
	guint index;

	model = gtk_tree_view_get_model (tv);
	if (gtk_tree_model_get_iter_first (model, iter)) {
		do {
			gtk_tree_model_get (model, iter,
					    LIST_INDEX, &index, -1);
			if (i == index)
				return TRUE;
		} while (gtk_tree_model_iter_next (model, iter));
	}

	return FALSE; /* not found */
}

static gboolean
update_connection_list (gpointer data)
{
	ConnectionList *cl = (ConnectionList *)data;
	GtkTreeView *treeview = cl->getTreeView ();
	GtkTreeModel *model;
	GtkListStore *store;
	GtkTreeIter iter;
	Servent *s;

	s = servMgr->servents;

	model = gtk_tree_view_get_model (treeview);
	store = GTK_LIST_STORE (model);

	while (s)
	{
		if (cl_find_row_by_index (treeview, &iter, s->serventIndex)) {
			if (s->type == Servent::T_NONE)
				gtk_list_store_remove (store, &iter);
		}
		if (s->type != Servent::T_NONE)
			cl->setServent (s);
		s = s->next;
	}

	return TRUE;
}

void
ConnectionList::stopSelected (void)
{
	Servent *s = selectedServent ();

	if (!s) {
		return ;
	}

	Host h = s->getHost();
	{
		guint ip = h.ip;
		gushort port = h.port;
		Host h(ip,port);
		gchar hostName[64];

		h.toStr (hostName);
		LOG_DEBUG ("Servent [%s] stopped", hostName);
	}

	s->thread.active = false;
}

ConnectionList::ConnectionList (GtkWidget *w)
{
	static gint timeout_source = 1;
	GtkTreeModel *model = create_model ();

	debug_print (1, "New object: ConnectionList");

	treeview = GTK_TREE_VIEW (w);

	gtk_tree_view_set_model (treeview, model);

	add_columns (treeview);

	selection = gtk_tree_view_get_selection (treeview);

	timeout_source = g_timeout_add (DELAY_UPDATE,
					update_connection_list, this);
}

Servent *
ConnectionList::selectedServent (void)
{
	Servent *servent = NULL;
	GtkTreeModel *model;
	GtkTreeIter iter;
	guint index;

	if (! gtk_tree_selection_get_selected (selection, &model, &iter))
		return NULL;

	gtk_tree_model_get (model, &iter,
			    LIST_INDEX, &index,
			    -1);

	servent = servent_find_by_index (index);

	return servent;
}

void
ConnectionList::setServent (Servent *servent)
{
	GtkListStore *store;
	GtkTreeIter iter;
	GnomePeercastApp *gap = (GnomePeercastApp *)peercastApp;

	if (servent->type == Servent::T_NONE) return;

	store = GTK_LIST_STORE (gtk_tree_view_get_model (treeview));

	Host h = servent->getHost();
	{
		gchar hoststr[256];
		guint ip = h.ip;
		gushort port = h.port;
		Host h(ip,port);
		gchar hostName[64];
		gchar timestr[64];
		gchar ratestr[64];
		gchar h_name[128];
		guint tnum = 0;

		h.toStr (hostName);
		if (gap->getConfigBool (CONF_GUI_PREFIX"/name_resolve_connection") &&
		    ClientSocket::getHostname (h_name, ip))
			g_snprintf (hoststr, sizeof (hoststr),
				    "%s[%s]", hostName, h_name);
		else
			g_snprintf (hoststr, sizeof (hoststr), "%s", hostName);

		if (servent->lastConnect)
			tnum = sys->getTime () - servent->lastConnect;

		make_time_str (timestr, sizeof (timestr), tnum);

		if (servent->sock) {
			guint tot = servent->sock->bytesInPerSec + servent->sock->bytesOutPerSec;
			snprintf (ratestr, sizeof(ratestr), "%.1f", BYTES_TO_KBPS (tot));
		} else
			snprintf (ratestr, sizeof(ratestr), "%d", 0);

		if (!cl_find_row_by_index (treeview, &iter, servent->serventIndex))
			gtk_list_store_append (store, &iter);

		if (servent->type == Servent::T_RELAY) {
			gchar *color;
			FW_TYPE t = servent_firewall_type (servent);
			switch (t) {
			case FW_TYPE_ON:
				color = gap->getConfigString (CONF_GUI_PREFIX"/color_firewalled");
				break;				
			case FW_TYPE_PUSH:
				color = gap->getConfigString (CONF_GUI_PREFIX"/push_relay");
				break;
			case FW_TYPE_OFF:
				color = gap->getConfigString (CONF_GUI_PREFIX"/color_unfirewalled");
				break;
			case FW_TYPE_UNKNOWN:
			default:
				color = g_strdup ("Black");
			}

			gtk_list_store_set (store, &iter,
					    LIST_INDEX,  servent->serventIndex,
					    LIST_TYPE,   servent->getTypeStr (),
					    LIST_STATUS, servent->getStatusStr (),
					    LIST_TIME, timestr,
					    LIST_HOST, hoststr,
					    LIST_COLOR, color,
					    LIST_POS, servent->syncPos,
					    LIST_AGENT, servent->agent.cstr(),
					    LIST_RATE, ratestr,
					    -1);
			g_free (color);
		} else if (servent->type == Servent::T_DIRECT) {
			gchar *color;
			color = gap->getConfigString (CONF_GUI_PREFIX"/color_direct");
			gtk_list_store_set (store, &iter,
					    LIST_INDEX,  servent->serventIndex,
					    LIST_TYPE,   servent->getTypeStr (),
					    LIST_STATUS, servent->getStatusStr (),
					    LIST_TIME, timestr,
					    LIST_HOST, hoststr,
					    LIST_COLOR, color,
					    LIST_POS, servent->syncPos,
					    LIST_AGENT, servent->agent.cstr(),
					    LIST_RATE, ratestr,
					    -1);
			g_free (color);
		} else if (servent->status == Servent::S_CONNECTED) {
			gchar p[64];
			snprintf (p, sizeof(p), "%d/%d",
				  servent->gnuStream.packetsIn,
				  servent->gnuStream.packetsOut);
			gtk_list_store_set (store, &iter,
					    LIST_INDEX,  servent->serventIndex,
					    LIST_TYPE,   servent->getTypeStr (),
					    LIST_STATUS, servent->getStatusStr (),
					    LIST_TIME, timestr,
					    LIST_HOST, hoststr,
					    LIST_POS, p,
					    LIST_AGENT, "-",
					    -1);
		} else {
			gtk_list_store_set (store, &iter,
					    LIST_INDEX,  servent->serventIndex,
					    LIST_TYPE,   servent->getTypeStr (),
					    LIST_STATUS, servent->getStatusStr (),
					    LIST_TIME, timestr,
					    LIST_HOST, hoststr,
					    LIST_AGENT, "-",
					    -1);
		}
	}
}

void
ConnectionList::deleteServent (Servent *servent)
{
	GtkListStore *store;
	GtkTreeIter iter;

	store = GTK_LIST_STORE (gtk_tree_view_get_model (treeview));
	if (! cl_find_row_by_index (treeview, &iter, servent->serventIndex)) {
		/* not found */
		/* FIXME: already removed? */
		return;
	}
	gtk_list_store_remove (store, &iter);
}
