/* wavfile.c
   WAV interface
   Copyright (C) 2004 Linus Walleij

This file is part of the GNOMAD package.

GNOMAD is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

You should have received a copy of the GNU General Public License
along with GNOMAD; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. 

Much of the code in this file was derived from the getid3() code,
written in PHP. The C implementation here is however made from
scratch.

*/

#include "common.h"
#include "metadata.h"
#include "filesystem.h"
#include "util.h"
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

static guint32 le_to_guint(guchar *data)
{
  return data[3] << 24 | data[2] << 16 | data[1] << 8 | data[0];
}

/* -------------------------------------- */
/* EXPORTED FUNCTIONS                     */
/* -------------------------------------- */

void
get_tag_for_wavfile (metadata_t *meta)
{
  gint fd;
  guchar header[46];
  gchar *tmppath = filename_fromutf8(meta->path);
  gint n;

  // g_print("Getting WAV tag info for %s...\n", meta->path);
  fd = (gint) open(tmppath, O_RDONLY, 0);
  if (fd < 0) {
    g_free(tmppath);
    return;
  }
  // g_print("Opened file\n");
  g_free(tmppath);

  // Read in some stuff...
  n = read(fd,header,46);
  if (n == 46) {
    // Hardcoded RIFF header
    if ((header[0] == 'R' || header[0] == 'r') &&
	(header[1] == 'I' || header[1] == 'i') &&
	(header[2] == 'F' || header[2] == 'f') &&
	(header[3] == 'F' || header[3] == 'f') &&
	(header[8] == 'W' || header[8] == 'w') &&
	(header[9] == 'A' || header[9] == 'a') &&
	(header[10] == 'V' || header[10] == 'v') &&
	(header[11] == 'E' || header[11] == 'e') &&
	(header[12] == 'f' || header[12] == 'F') &&
	(header[13] == 'm' || header[13] == 'M') &&
	(header[14] == 't' || header[14] == 'T') &&
	header[15] == ' ') {
      // This is indeed a RIFF/WAVE file
      guint32 chunksize = le_to_guint(&header[4]);
      guint32 fmtchunksize = le_to_guint(&header[16]);
      guint32 samplerate = le_to_guint(&header[24]);
      guint32 byterate = le_to_guint(&header[28]);
      guint32 calctime;

      // Calculate the run time. Remove the format chunk size,
      // the four remaining bytes of the RIFF header and the
      // data header (8 bytes).
      calctime = (chunksize - fmtchunksize - 8 - 4) / byterate;
      if (calctime == 0) {
	// Atleast one second, please.
	calctime = 1;
      }

      //g_print("RIFF/WAVE chunksize: %d bytes\n", chunksize);
      //g_print("Sample rate: %d samples/s\n", samplerate);
      //g_print("Byte rate: %d bytes/s\n", byterate);
      //g_print("Calculated time: %d seconds\n", calctime);
      if (meta->length != NULL) {
	g_free(meta->length);
      }
      meta->length = seconds_to_mmss(calctime);
    }
  }

  close(fd);
}
